/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Santé)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------      */                                                                    


#include "Transformer.h"
#include "vtkMatrix4x4.h"

/*
	CONSTRUCTOR: Initializes the two points with empty vectors, the angle in 0.
*/
Transformer::Transformer() 
{
	std::vector<int> empty (3,0); 
	_angle= 0;
	_centerPoint=empty;
	_scaleX=1;
	_scaleY=1;
	//If the transform already exists, we delete it before we create a new transform 
	//and set the matrix with the identity matrix
	_transform= vtkTransform::New();
	_matrix = vtkMatrix4x4::New();
	_matrix->Identity();
	_transform->SetMatrix(_matrix);	
}

/*
	DESTRUCTOR
*/
Transformer::~Transformer()
{
	//We delete the existing transform
	if (_transform != NULL ) { _transform->Delete(); }
	if (_matrix != NULL ) { _matrix->Delete(); }
}

/*
	SETS A NEW TRANSFORM
*/
void Transformer::SetTransform(vtkTransform *transform)
{
	_transform=transform;
}

/*
	SETS CENTER POINT
*/
void Transformer::SetCenterPoint(std::vector<int> point)
{
    _centerPoint=point;
}

/*
	SETS THE ANGLE
*/
void Transformer::SetAngle(double angle)
{
	_angle=angle;
}

/*
	SETS THE X SCALE
*/
void Transformer::SetScaleX(double scaleX)
{
	_scaleX=scaleX/100.0;
}

/*
	SETS THE Y SCALE
*/
void Transformer::SetScaleY(double scaleY)
{
	_scaleY=scaleY/100.0;
}

/*
 SETS THE Y SCALE
 */
void Transformer::SetScaleZ(double scaleZ)
{
	_scaleZ=scaleZ/100.0;
}



/*
	GETS THE RESULTANT TRANSFORM
*/
vtkTransform *Transformer::GetResult()
{
	return _transform;
}

/*
	MAKES THE TRANSFORMATIONS
*/
void Transformer::Run()
{
	_transform->Identity();
	_transform->PostMultiply();
	if(_centerPoint.size() < 3)
	{
		_transform->Translate(-_centerPoint[0], -_centerPoint[1], 0);
		std::cout << "NTU Transformer2D Center Points: " << _centerPoint[0] << " " << _centerPoint[1] << std::endl;
	}
	else
	{
		_transform->Translate(-_centerPoint[0], -_centerPoint[1], -_centerPoint[2]);
		std::cout << "NTU Transformer2D Center Points: " << _centerPoint[0] << " " << _centerPoint[1] << " " << _centerPoint[2] << std::endl;
	}

	_transform->RotateWXYZ(_angle, 0, 0, 1);
	//std::cout << "NTU Transformer2D Angle: " << _angle << std::endl;

	if(_scaleX < 0 || _scaleY < 0 || _scaleZ < 0)
	{
		_transform->Scale(100, 100, 100);
		//std::cout << "NTU Transformer2D Scale: " << _scaleX << " " << _scaleY << " " << _scaleZ << std::endl;
	}
	else
	{
		_transform->Scale(_scaleX, _scaleY,_scaleZ);
		//std::cout << "NTU Transformer2D Scale: " << _scaleX << " " << _scaleY << " " << _scaleZ << std::endl;
	}		
	_transform->Inverse();
	_transform->Update();
}
