/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Santé)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------      */                                                                    


#include "Transformer3D.h"
#include "vtkMatrix4x4.h"

/*
	CONSTRUCTOR: Initializes the two points with empty vectors, the angle in 0.
*/
Transformer3D::Transformer3D() 
{
	//If the transform already exists, we delete it before we create a new transform 
	//and set the matrix with the identity matrix
	_transform= vtkTransform::New();
	vtkMatrix4x4 *_matrix = vtkMatrix4x4::New();
	_matrix->Identity();
	_transform->SetMatrix(_matrix);
	_firstPoint[0] = 0;
	_firstPoint[1] = 0;
	_firstPoint[2] = 0;
	_secondPoint[0] = 0;
	_secondPoint[1] = 0;
	_secondPoint[2] = 0;
}

/*
	DESTRUCTOR
*/
Transformer3D::~Transformer3D()
{
	//We delete the existing transform
	if (_transform != NULL ) { _transform->Delete(); }
}

/*
	SETS A NEW TRANSFORM
*/
void Transformer3D::SetTransform(vtkTransform *transform)
{
	_transform=transform;
}

/*
	SETS FIRST POINT
*/
void Transformer3D::SetFirstTranslation(double* first)
{
    _firstPoint[0]=first[0];
	_firstPoint[1]=first[1];
	_firstPoint[2]=first[2];
}

/*
	SETS SECOND POINT
*/
void Transformer3D::SetSecondTranslation(double* second)
{
    _secondPoint[0]=second[0];
	_secondPoint[1]=second[1];
	_secondPoint[2]=second[2];
}

/*
	SETS ROTATION AXIS
*/
void Transformer3D::SetRotationAxis(double* axis)
{
    _rotationAxis[0]=axis[0];
	_rotationAxis[1]=axis[1];
	_rotationAxis[2]=axis[2];
}

/*
	SETS ROTATION AXIS
*/
void Transformer3D::SetSecondRotationAxis(double axis[3])
{
    _secondRotationAxis[0]=axis[0];
	_secondRotationAxis[1]=axis[1];
	_secondRotationAxis[2]=axis[2];
}

/*
	SETS ROTATION ANGLE
*/
void Transformer3D::SetAngle(double angle)
{
    _angle=angle;
}

/*
	SETS SECOND ROTATION ANGLE
*/
void Transformer3D::SetSecondAngle(double angle)
{
    _secondAngle=angle;
}


/*
	GETS THE RESULTANT TRANSFORM
*/
vtkTransform *Transformer3D::GetResult()
{
	return _transform;
}

/*
	GETS THE FIRST RESULTANT TRANSFORM (NTU: Useful for calculating the new position of the points before applying the second rotation)
*/
vtkTransform *Transformer3D::GetFirstResult()
{
	_transform->Identity();

	_transform->Translate(-_firstPoint[0], -_firstPoint[1], -_firstPoint[2]);

	_transform->RotateWXYZ(_angle, _rotationAxis[0], _rotationAxis[1], _rotationAxis[2]);

	_transform->Translate(_secondPoint[0], _secondPoint[1], _secondPoint[2]);

	
	_transform->Update();	
	return _transform;
}

/*
	MAKES THE TRANSFORMATIONS
*/
void Transformer3D::Run()
{	
	//Cleans the transformation matrix
	_transform->Identity();

	//Make all transformations in postmultiply mode
	_transform->PostMultiply();

	//Acomodate in 0,0,0 according to the first point of the second vector
	_transform->Translate(-_secondPoint[0], -_secondPoint[1], -_secondPoint[2]);

	_transform->RotateWXYZ(_angle, _rotationAxis[0], _rotationAxis[1], _rotationAxis[2]);

	_transform->RotateWXYZ(_secondAngle, _secondRotationAxis[0], _secondRotationAxis[1], _secondRotationAxis[2]);

	//Acommodate according to the first point of the first vector
	_transform->Translate(_firstPoint[0], _firstPoint[1], _firstPoint[2]);	
	
	_transform->Inverse();
	_transform->Update();
}
