/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Santé)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------      */                                                                    

#include "Transparency.h"


/*
* Constructor
*/
//------------------------------------------------------------
Transparency::Transparency()
{
	_image1=NULL;
	_image2=NULL;
	_percent=0;
	_newImage = NULL;
	_processed=false;
}

/*
* Destructor
*/
//------------------------------------------------------------
Transparency::~Transparency()
{
	if (_newImage != NULL ) { _newImage->Delete(); }
}

//------------------------------------------------------------
void Transparency::calculateImage()
{

	if(_image1!=NULL && _image2!=NULL)
	{

		// IMAGE 1
			
		// Information from image1
		_image1->GetSpacing(spcImg1);
		_image1->GetExtent(extImg1);
		_type = _image1->GetScalarType();
		
		// The Z dimension is 1 for a 2D image
		dimImg1[0] = extImg1[1] - extImg1[0]+1;
		dimImg1[1] = extImg1[3] - extImg1[2]+1;
		dimImg1[2] = 1;

		//IMAGE 2

		// Information from image2
		_image2->GetSpacing(spcImg2);
		_image2->GetExtent(extImg2);
		
		// The Z dimension is 1 for a 2D image
		dimImg2[0] = extImg2[1] - extImg2[0]+1;
		dimImg2[1] = extImg2[3] - extImg2[2]+1;
		dimImg2[2] = 1;

		long numPixelsImg1 = dimImg1[0]*dimImg1[1];
		long numPixelsImg2 = dimImg2[0]*dimImg2[1];
		
		double factorX = 1;
		double factorY = 1;
		vtkImageResample *resample = vtkImageResample::New();
		vtkImageData *result;
		if(numPixelsImg1<numPixelsImg2)
		{
			resample->SetInput(_image1);
			factorX = (double)extImg2[1]/extImg1[1];
			factorY = (double)extImg2[3]/extImg1[3];
			resample->SetAxisMagnificationFactor(0,factorX);
			resample->SetAxisMagnificationFactor(1,factorY);
			resample->SetInformationInput(_image2);

			initialize(dimImg2, spcImg2);
			result = resample->GetOutput();
			result->Update();
			createImage(result,_image2,dimImg2[0],dimImg2[1]);		
			
		} //if
		else if (numPixelsImg1>numPixelsImg2)
		{
			resample->SetInput(_image2);
			factorX = (double)extImg1[1]/extImg2[1];
			factorY = (double)extImg1[3]/extImg2[3];
			resample->SetAxisMagnificationFactor(0,factorX);
			resample->SetAxisMagnificationFactor(1,factorY);
			resample->SetInformationInput(_image1);

			initialize(dimImg1, spcImg1);
			result = resample->GetOutput();
			
			result->Update();

			createImage(_image1,result,dimImg1[0],dimImg1[1]);
		} // else if
		else
		{
			//If both images have the same number of pixels, the resultant image will have the 
			//properties of the first image.
			resample->SetInput(_image2);
			factorX = (double)extImg1[1]/extImg2[1];
			factorY = (double)extImg1[3]/extImg2[3];
			resample->SetAxisMagnificationFactor(0,factorX);
			resample->SetAxisMagnificationFactor(1,factorY);
			resample->SetInformationInput(_image1);

			initialize(dimImg1, spcImg1);

			result = resample->GetOutput();
			result->Update();
			
			createImage(_image1,result,dimImg1[0],dimImg1[1]);

		} //else
		
		resample->Delete();		
		_processed=true;
	}
}

void Transparency::initialize(int dimensions[], double spacing[])
{
	// Setting the new image
	_newImage = vtkImageData::New();
	_newImage->SetScalarType(_type);
	_newImage->SetSpacing(spacing);
	_newImage->SetDimensions(dimensions);
	_newImage->AllocateScalars();
	_newImage->Update();

}

void Transparency::createImage(vtkImageData *img1, vtkImageData *img2, int sizeX, int sizeY)
{
//	int i, j;  EED ???

	if(_type == VTK_CHAR)
	{
		//POINTERS: 
		char* dataImagePointer1 = NULL;
		char* dataImagePointer2 = NULL;
		char* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_SIGNED_CHAR)
	{
		//POINTERS: 
		signed char* dataImagePointer1 = NULL;
		signed char* dataImagePointer2 = NULL;
		signed char* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_UNSIGNED_CHAR)
	{
		//POINTERS: 
		unsigned char* dataImagePointer1 = NULL;
		unsigned char* dataImagePointer2 = NULL;
		unsigned char* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_SHORT)
	{
		//POINTERS: 
		short* dataImagePointer1 = NULL;
		short* dataImagePointer2 = NULL;
		short* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_UNSIGNED_SHORT)
	{
		//POINTERS: 
		unsigned short* dataImagePointer1 = NULL;
		unsigned short* dataImagePointer2 = NULL;
		unsigned short* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_INT)
	{
		//POINTERS: 
		int* dataImagePointer1 = NULL;
		int* dataImagePointer2 = NULL;
		int* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_UNSIGNED_INT)
	{
		//POINTERS: 
		unsigned int* dataImagePointer1 = NULL;
		unsigned int* dataImagePointer2 = NULL;
		unsigned int* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_LONG)
	{
		//POINTERS: 
		long* dataImagePointer1 = NULL;
		long* dataImagePointer2 = NULL;
		long* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_UNSIGNED_LONG)
	{
		//POINTERS: 
		unsigned long* dataImagePointer1 = NULL;
		unsigned long* dataImagePointer2 = NULL;
		unsigned long* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);		
	}
	else if(_type == VTK_FLOAT)
	{
		//POINTERS: 
		float* dataImagePointer1 = NULL;
		float* dataImagePointer2 = NULL;
		float* dataImageResultPointer = NULL;
	
		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}
	else if(_type == VTK_DOUBLE)
	{
		//POINTERS: 
		double* dataImagePointer1 = NULL;
		double* dataImagePointer2 = NULL;
		double* dataImageResultPointer = NULL;

		createImageByType(dataImagePointer1, dataImagePointer2, dataImageResultPointer, img1, img2, sizeX, sizeY);
	}

	_newImage->Update();
}

template <class T>
void Transparency::createImageByType(T* dataImagePointer1, T* dataImagePointer2, T* dataImageResultPointer, vtkImageData *img1, vtkImageData *img2, int sizeX, int sizeY)
{
	int i, j;
	
	dataImagePointer1 = (T*) img1->GetScalarPointer(0,0,0);
	dataImagePointer2 = (T*) img2->GetScalarPointer(0,0,0);
	dataImageResultPointer = (T*) _newImage->GetScalarPointer(0,0,0);
	for(i = 0; i < sizeX; i++)
	{
		for(j = 0; j < sizeY; j++)
		{
			dataImagePointer1 = (T*)img1->GetScalarPointer(i,j,0);
			dataImagePointer2 = (T*)img2->GetScalarPointer(i,j,0);
			dataImageResultPointer = (T*)_newImage->GetScalarPointer(i,j,0);
			
			*dataImageResultPointer = (T)*dataImagePointer1*_percent + (T)*dataImagePointer2*(1-_percent);				
		}
	}
}

/*
* Get new image
*/
//------------------------------------------------------------
vtkImageData* Transparency::getImage()
{
	if(_processed){
		return _newImage;
	}
	else{
		return NULL;
	}
}

//------------------------------------------------------------
void Transparency::setImage1(vtkImageData *image)
{
	_image1=image;
}

//------------------------------------------------------------
void Transparency::setImage2(vtkImageData *image)
{
	_image2=image;
}

//------------------------------------------------------------
void Transparency::setPercent(int percent)
{
    _percent=(double)percent/100.0;	
}

