/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la SantÈ)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#ifdef WIN32
#define _CRT_SECURE_NO_DEPRECATE
#endif

#include <stdio.h>
#include "bbtkXML.h"
#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>

//==========================================================================
class bbfyException
{
public: 
  bbfyException(const std::string& message) : mMessage(message) {}

  std::string mMessage;
};
//==========================================================================

const std::string itkImageToImageFilterString = "ITK_ImageToImageFilter";
const std::string vtkImageAlgorithmString     = "VTK_ImageAlgorithm";
const std::string vtkPolyDataAlgorithmString  = "VTK_PolyDataAlgorithm";

//==========================================================================
class bbfy
{
public:
  bbfy(const std::string& filename, 
       const std::string& package = "PACKAGE_NAME",
       const std::string& output_path = "",
       bool verbose = false);
  
  void CreateBlackBox();
  void ParseXML();
  void CreateHeader();
  void CreateCode();

  void WriteGenericITKFilterHeader();

  void BeginNamespace();
  void EndNamespace();

private:
  //
  std::string mFilename;
  std::string mOutputPath;
  bool mVerbose;

  //
  std::string mName;
  typedef enum 
    {
      STD,
      itkImageToImageFilter,
      vtkImageAlgorithm,
      vtkPolyDataAlgorithm,
    }
    BoxType;

  BoxType mType;
  bool mIsWidget;
  std::string mParentBlackBox;
  std::string mItkParent;
  std::string mVtkObject;
  bool mGeneric;
  std::string mAuthor;
  std::string mDescription;
  std::string mCategory;
  std::string mPackage;
  // bool mIsInNamespace;
  std::string mNamespace;
  // int mNbTemplateParam;
  std::vector<std::string> mTemplateParam;
  std::string mTemplateDeclaration;
  std::string mTemplateImplementation;

  std::vector<std::string> mInclude;
  std::vector<std::string> mTypedef;

  std::string mUserSetDefaultValues;
  std::string mUserInitializeProcessing;
  std::string mUserFinalizeProcessing;

  typedef struct
  {
    std::string name;
    std::string type;
    std::string nature;
    std::string descr;
    std::string special;
    std::string generic_type;
  }
    IO;
  
  std::vector<IO> mInput;
  std::vector<IO> mOutput;

  std::string mProcess;
  std::string mCreateWidget;

  //
  std::ofstream mFile;
  std::string mHName;
  std::string mCxxName;

  void AlertString();

};
//==========================================================================



//==========================================================================
bbfy::bbfy(const std::string& filename, 
	   const std::string& package,
	   const std::string& output_path,
	   bool verbose)
{
  mIsWidget = false;
  
  mFilename = filename;
  mPackage = package;
  mNamespace = "bb" + mPackage;

  mOutputPath = output_path;
  mVerbose = verbose;

  CreateBlackBox();
}
//==========================================================================

//==========================================================================
void bbfy::CreateBlackBox()
{
  // Parse XML input file
  ParseXML();
  // Create output files
  CreateHeader();
  CreateCode();
}
//==========================================================================

void bbfy::AlertString()
{
	
      mFile << "//===== \n";
      mFile << "// Before editing this file, make sure it's a file of your own ";
      mFile << "(i.e.: it wasn't generated from xml description; if so : your modifications will be lost)\n";
      mFile << "//===== \n";
	  
}

//==========================================================================
void bbfy::ParseXML()
{



  XMLResults* res = new XMLResults;
  XMLNode BB = XMLNode::parseFile(mFilename.c_str(),"blackbox",res);

  if ( res->error != eXMLErrorNone ) 
    {
      std::ostringstream str;
      str << XMLNode::getError(res->error);
      str << " [line " << res->nLine << ", col "<<res->nColumn<<"]"; 
      delete res;
      throw bbfyException(str.str());
    }
  delete res;

  // Name
  if (!BB.isAttributeSet("name")) 
    {
      throw bbfyException("Error : <blackbox> tag : no 'name' attribute found (mandatory)");
    }
  mName = BB.getAttribute("name");

  if (mVerbose) std::cout << "* Creating BlackBox '"<<mName<<"'"<<std::endl;

  // Type 
  mGeneric = false;
  mType = STD;


  if (BB.isAttributeSet("type")) 
    {
      std::string bbtype = BB.getAttribute("type");
      if (bbtype=="standard")
	{
	  mGeneric = false;
	  mType = STD;
	}
      else if (bbtype==itkImageToImageFilterString)
	{
	  mType = itkImageToImageFilter;
	  // Looks for <itkparent> tag
	  if (!BB.nChildNode("itkparent")) 
	    {
	      throw bbfyException("Error : blackbox type '"+itkImageToImageFilterString+"' but no <itkparent> tag found (mandatory)");
	    }
	  bbtk::GetTextOrClear(BB.getChildNode("itkparent"),mItkParent);
	  // 
	  mGeneric = false;
	  if (BB.isAttributeSet("generic")) mGeneric=true;
	}
      else if (bbtype == vtkImageAlgorithmString)
	{
	  mType = vtkImageAlgorithm;
	  // Looks for <vtkobject> tag
	  if (!BB.nChildNode("vtkobject")) 
	    {
	      throw bbfyException("Error : blackbox type '"
				  +vtkImageAlgorithmString
				  +"' but no <vtkobject> tag found (mandatory)");
	    }
	  bbtk::GetTextOrClear(BB.getChildNode("vtkobject"),mVtkObject);
	  // 
	}
    else if (bbtype == vtkPolyDataAlgorithmString )
	{
	  mType = vtkPolyDataAlgorithm;
	  // Looks for <vtkobject> tag
	  if (!BB.nChildNode("vtkobject")) 
	    {
	      throw bbfyException("Error : blackbox type '"
				  +vtkPolyDataAlgorithmString
				  +"' but no <vtkobject> tag found (mandatory)");
	    }
	  bbtk::GetTextOrClear(BB.getChildNode("vtkobject"),mVtkObject);
	  // 
	}
     else 
	{
	  std::string mess("Error : blackbox type '");
	  mess += bbtype;
	  mess += "' unknown. Known types :";
	  mess += "'" + itkImageToImageFilterString + "' ";
	  mess += "'" + vtkImageAlgorithmString + "' ";
	  mess += "'" + vtkPolyDataAlgorithmString + "' ";
	  throw bbfyException(mess);
	}
    }

  // Is a widget box ?
  if (BB.isAttributeSet("widget")) 
    {
      mIsWidget = true;
      mParentBlackBox = "bbtk::WxBlackBox";
      mInclude.push_back("bbtkWxBlackBox.h");
    }
  else 
    {
      mIsWidget = false;
      mParentBlackBox = "bbtk::AtomicBlackBox";
      mInclude.push_back("bbtkAtomicBlackBox.h");
    }

  // Author
  int i,j;
  for (i=0,j=0; i<BB.nChildNode("author"); i++) 
    {
      std::string val;
      bbtk::GetTextOrClear(BB.getChildNode("author",&j),val);
      mAuthor += val;
    }

  // Description
  for (i=0,j=0; i<BB.nChildNode("description"); i++) 
    {
      std::string val;
      bbtk::GetTextOrClear(BB.getChildNode("description",&j),val);
      mDescription += val;
    }
  
  // Category
  for (i=0,j=0; i<BB.nChildNode("category"); i++) 
    {
      std::string val;
      bbtk::GetTextOrClear(BB.getChildNode("category",&j),val);
      mCategory += val;
    }

  // Namespace
  if (BB.nChildNode("namespace"))
    {
      bbtk::GetTextOrClear(BB.getChildNode("namespace"),mNamespace);
    }

  // UserSetDefaultValues body
  if (BB.nChildNode("defaultValues"))
    {
      bbtk::GetTextOrClear(BB.getChildNode("defaultValues"),
			   mUserSetDefaultValues);
    }
    
  // UserInitializeProcessing body
  if (BB.nChildNode("initializeProcessing"))
    {
      bbtk::GetTextOrClear(BB.getChildNode("initializeProcessing"),
			   mUserInitializeProcessing);
    }
    
 // UserFinalizeProcessing body
  if (BB.nChildNode("finalizeProcessing"))
    {
      bbtk::GetTextOrClear(BB.getChildNode("finalizeProcessing"),
			   mUserFinalizeProcessing);
    }



     // Template parameters
  //  mNbTemplateParam = BB.nChildNode("template");

  if ( BB.nChildNode("template") > 0)
    {
      mTemplateDeclaration = "<";
      mTemplateImplementation = "<";
      
      for (i=0,j=0; i<BB.nChildNode("template")-1; i++) 
	{
	  mTemplateDeclaration += "class ";
	  std::string val;
	  bbtk::GetTextOrClear(BB.getChildNode("template",&j),val);
	  mTemplateDeclaration += val;
	  mTemplateDeclaration +=  ",";
	  mTemplateImplementation += val;
	  mTemplateImplementation +=  ",";
	  mTemplateParam.push_back(val);
	}
      mTemplateDeclaration += "class ";
      std::string val;
      bbtk::GetTextOrClear(BB.getChildNode("template",&j),val);
      mTemplateDeclaration += val;
      mTemplateDeclaration +=  ">";
      mTemplateImplementation += val;
      mTemplateImplementation +=  ">";
      mTemplateParam.push_back(val);
    }

  // Includes 
  for (i=0,j=0; i<BB.nChildNode("include"); i++) 
    {
      std::string val;
      bbtk::GetTextOrClear(BB.getChildNode("include",&j),val);
      mInclude.push_back(val);
    }
  // Typedef
  for (i=0,j=0; i<BB.nChildNode("typedef"); i++) 
    {
      std::string val;
      bbtk::GetTextOrClear(BB.getChildNode("typedef",&j),val);
      mTypedef.push_back(val);
    }
  
  // Inputs
  for (i=0,j=0; i<BB.nChildNode("input"); i++) 
    {
      IO io;
      XMLNode n = BB.getChildNode("input",&j); 
      if (!n.isAttributeSet("name"))
	{
	  throw bbfyException("Error : <input> attribute 'name' not found (mandatory)");
	}
      io.name = n.getAttribute("name");
      if (!n.isAttributeSet("type"))
	{
	  throw bbfyException("Error : <input name=\""+io.name+"\"> attribute 'type' not found (mandatory)");
	}
      io.type = n.getAttribute("type"); 
      if (!n.isAttributeSet("description"))
	{
	  throw bbfyException("Error : <input name=\""+io.name+"\"> attribute 'description' not found (mandatory)");
	}
      io.descr = n.getAttribute("description"); 

      if (n.isAttributeSet("special")) 
	{
	  io.special =  n.getAttribute("special");  
	}

      if (n.isAttributeSet("nature")) 
	{
	  io.nature =  n.getAttribute("nature");  
	}

      if (n.isAttributeSet("generic_type")) 
	{
	  io.generic_type =  n.getAttribute("generic_type");  
	}

      mInput.push_back(io);
    }
  
  // Outputs
  for (i=0,j=0; i<BB.nChildNode("output"); i++) 
    {
      IO io;
      XMLNode n = BB.getChildNode("output",&j); 
      if (!n.isAttributeSet("name"))
	{
	  throw bbfyException("Error : <output> attribute 'name' not found (mandatory)");
	}
      io.name = n.getAttribute("name"); 
      if (!n.isAttributeSet("type"))
	{
	  throw bbfyException("Error : <output name=\""+io.name+"\"> attribute 'type' not found (mandatory)");
	}
      io.type = n.getAttribute("type"); 
      if (!n.isAttributeSet("description"))
	{
	  throw bbfyException("Error : <output name=\""+io.name+"\"> attribute 'description' not found (mandatory)");
	}
      io.descr = n.getAttribute("description"); 

      if (n.isAttributeSet("special")) 
	{
	  io.special =  n.getAttribute("special");  
	}

      if (n.isAttributeSet("nature")) 
	{
	  io.nature =  n.getAttribute("nature");  
	}

      if (n.isAttributeSet("generic_type")) 
	{
	  io.generic_type =  n.getAttribute("generic_type");  
	}

      mOutput.push_back(io);
    }


  // Process
  // process tag given ?
   if (BB.nChildNode("process"))
     {
		 bbtk::GetTextOrClear(BB.getChildNode("process"),mProcess);
     }
     
  // CreateWidget
  // createwidget tag given ?
   if (BB.nChildNode("createwidget"))
     {
       bbtk::GetTextOrClear(BB.getChildNode("createwidget"),mCreateWidget);
     }







   // OBSOLETE/UNSUPPORTED TAGS
  // WARN IF OBSOLETE TAGS PROVIDED
  if (BB.nChildNode("constructor"))
    {
      std::cout << "WARNING !!! The tag <constructor> is obsolete !!"<<std::endl;
    }
  if (BB.nChildNode("destructor"))
    {
      std::cout << "WARNING !!! The tag <destructor> is obsolete !!"<<std::endl;
    }
  if (BB.nChildNode("copy_constructor"))
    {
      std::cout << "WARNING !!! The tag <copy_constructor> is obsolete !!"<<std::endl;
    }



}
//==========================================================================


//==========================================================================
void bbfy::BeginNamespace()
{
  //  if (mIsInNamespace)
  // {
  mFile << "namespace "<<mNamespace <<"\n{\n\n";
  //  }
}
//==========================================================================

//==========================================================================
void bbfy::EndNamespace()
{
  // if (mIsInNamespace)
  //  {
  mFile << "}\n// EO namespace "<<mNamespace<<"\n\n";
  //  }
}
//==========================================================================


//==========================================================================
void bbfy::CreateHeader()
{

  mHName = "bb";
  mHName += mPackage;
  mHName += mName;
  mHName += ".h";
  if (mVerbose) std::cout << " - Creating header '"<<mHName<<"'"<<std::endl;
  std::string fullname = mOutputPath + mHName;
  mFile.open(fullname.c_str());
  if (!mFile.good())
    {
      std::string mess("Error : could not open file \"");
      mess += fullname + "\"";
      throw bbfyException(mess);
    }

	AlertString();

  // If is widget 
  if (mIsWidget)
    {
      mFile << "#ifdef _USE_WXWIDGETS_\n";
    }

  // Prevent multiple inclusions
  std::string included("__bb");
  included += mPackage + mName + "_h_INCLUDED__";
  mFile << "#ifndef " << included <<"\n";
  mFile << "#define " << included <<"\n";

  // Includes 
  mFile << "#include \"bb" << mPackage << "_EXPORT.h\"\n";
  std::vector<std::string>::iterator i;
  for (i=mInclude.begin(); i!=mInclude.end(); ++i) 
    {
      mFile << "#include \"" << *i <<"\"\n";
    }
  if (mGeneric) mFile << "#include \"bbitkImage.h\"\n";
  mFile << "\n";

  if (mType == itkImageToImageFilter )
    {
      mFile << "#include \"bbtkItkBlackBoxMacros.h\"\n";
    }
  else if ( (mType == vtkImageAlgorithm) ||
	    (mType == vtkPolyDataAlgorithm) )
    {
      mFile << "#include \"bbtkVtkBlackBoxMacros.h\"\n";
    }
  // Namespace
  BeginNamespace();

  // Interface

  // If it is a template class
  if (mTemplateParam.size() > 0)
    {
      mFile << "template " << mTemplateDeclaration <<"\n";
    }
  
  // Class declaration and parents
  mFile << "class bb"<<mPackage<<"_EXPORT "<<mName<<"\n";
  mFile << " : \n";

  /*
  if (mBB.nChildNode("inherits"))
    {
      mFile << ",\n";
      for (i=0,j=0; i<mBB.nChildNode("inherits")-1; i++) 
	{
	  mFile << "   public " 
		<< mBB.getChildNode("inherits",&j).getText()
		<< ",\n";
	}
      mFile << "   public " 
	    << mBB.getChildNode("Inherits",&j).getText()
	    <<"\n";
    }
  */

  if (mType == itkImageToImageFilter )
    {
      mFile << "   public " << mItkParent <<",\n";
    }

  mFile << "   public "<<mParentBlackBox << "\n";

  mFile << "{\n";

  // Interface

  // ITK 
  if (mType == itkImageToImageFilter)
    {
      mFile << "  BBTK_ITK_BLACK_BOX_INTERFACE("
	    << mName << ","
	    << mParentBlackBox << ","
	    << mItkParent 
	    << ");\n";
    }
  // VTK
  else if ( (mType == vtkImageAlgorithm) ||
       (mType == vtkPolyDataAlgorithm) )
    {
      mFile << "  BBTK_VTK_BLACK_BOX_INTERFACE("
	    << mName << ","
	    << mParentBlackBox << ","
	    << mVtkObject
	    << ");\n";
    }
	
  // Default
  else 
    {
      mFile << "  BBTK_BLACK_BOX_INTERFACE("
	    << mName << ","
	    << mParentBlackBox << ");\n";
    }

  for (i=mTypedef.begin(); i!=mTypedef.end(); ++i) 
    {
      mFile << *i <<"\n";
    }


	AlertString();

  // Inputs
  std::vector<IO>::iterator ioi;
  for (ioi=mInput.begin(); ioi!=mInput.end(); ++ioi) 
    {
      if (ioi->special=="") 
	{
	  mFile << "  BBTK_DECLARE_INPUT(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	}
      else if (ioi->special=="itk input")
	{
	  mFile << "  BBTK_DECLARE_ITK_INPUT(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	}
      else if (ioi->special=="vtk input")
	{
	  if (mType == vtkImageAlgorithm) {
	  mFile << "  BBTK_DECLARE_VTK_IMAGE_ALGORITHM_INPUT(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	  } 
	  else if (mType == vtkPolyDataAlgorithm) {
	  mFile << "  BBTK_DECLARE_POLY_DATA_ALGORITHM_INPUT(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	  }
	}
      else if (ioi->special=="itk parameter")
	{
	  mFile << "  BBTK_DECLARE_ITK_PARAM(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	}
      else if (ioi->special=="vtk parameter")
	{
	  mFile << "  BBTK_DECLARE_VTK_PARAM(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	}
      else 
	{
	  std::string mess("Error : input '");
	  mess += ioi->name;
	  mess += "', 'special' attribute '";
	  mess += ioi->special;
	  mess += "' unknown";
	  throw bbfyException(mess);
	}
    }

  
  // Outputs
  for (ioi=mOutput.begin(); ioi!=mOutput.end(); ++ioi) 
    {
      if (ioi->special=="") 
	{
	  mFile << "  BBTK_DECLARE_OUTPUT(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	}
      else if (ioi->special=="itk output")
	{
	  mFile << "  BBTK_DECLARE_ITK_OUTPUT(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	}  
      else if (ioi->special=="vtk output")
	{
	  mFile << "  BBTK_DECLARE_VTK_OUTPUT(" 
		<< ioi->name
		<< ","
		<< ioi->type
		<< ");\n";
	}  
      else 
	{
	  std::string mess("Error : output '");
	  mess += ioi->name;
	  mess += "', 'special' attribute '";
	  mess += ioi->special;
	  mess += "' unknown";
	  throw bbfyException(mess);
	}
    }
  
  // Process
  if ((mType == STD)||(mProcess.size()))
    {
      mFile << "  BBTK_PROCESS(Process);\n" ;
      mFile << "  void Process();\n";
    }
  else if (mType == itkImageToImageFilter)
    {   
      mFile << "  BBTK_ITK_PROCESS();\n" ;
    }
  else if ((mType == vtkImageAlgorithm) ||
	   (mType == vtkPolyDataAlgorithm) )

    {   
      mFile << "  BBTK_VTK_PROCESS();\n" ;
    }

  // CreateWidget
  if (mIsWidget) 
    {
       mFile << "  BBTK_CREATE_WIDGET(CreateWidget);\n" ;
       mFile << "  void CreateWidget(wxWindow*);\n";
    }

	AlertString();

  // EO black box declaration
  mFile << "};\n\n";

  // BO black box description
  if (mTemplateParam.size()==0)
    {
      mFile << "BBTK_BEGIN_DESCRIBE_BLACK_BOX("
	    << mName << ","
	    << mParentBlackBox << ");\n";
      mFile << "BBTK_NAME(\"" << mName <<"\");\n";
    }
  else if (mTemplateParam.size()==1)
    {
      mFile << "BBTK_BEGIN_DESCRIBE_TEMPLATE_BLACK_BOX("
	    << mName //<< ","
	//<< mParentBlackBox //<< ","
	//   << mTemplateParam[0] 
	    << ");\n";
      mFile << "BBTK_NAME(\"" << mName 
	    << "<\"+bbtk::TypeName<" << mTemplateParam[0]
	    <<">()+\">\");\n";
    }
 else 
    {
      throw bbfyException("template bb with more than 1 templ param not impl");
    } 


  // Author
  mFile << "BBTK_AUTHOR(\""<<mAuthor<< "\");\n";

  // Description
  mFile << "BBTK_DESCRIPTION(\""<<mDescription<< "\");\n"; 
  
  // Category
  mFile << "BBTK_CATEGORY(\""<<mCategory<< "\");\n"; 

  for (i=mTypedef.begin(); i!=mTypedef.end(); ++i) 
    {
      mFile << *i <<"\n";
    }
  
  // Inputs
  for (ioi=mInput.begin(); ioi!=mInput.end(); ++ioi) 
    {
      if (mTemplateParam.size()>0)
	{
	  mFile << "BBTK_TEMPLATE_INPUT(";
	} 
      else 
	{
	  mFile << "BBTK_INPUT(";
	} 
      mFile << mName << "," << ioi->name << ",\""
	    << ioi->descr << "\"," <<  ioi->type << ",\"" 
	    << ioi->nature<<"\");\n";
    }
  
  // Outputs
  for (ioi=mOutput.begin(); ioi!=mOutput.end(); ++ioi) 
    {
      if (mTemplateParam.size()>0)
	{
	  mFile << "BBTK_TEMPLATE_OUTPUT(";
	} 
      else 
	{
	  mFile << "BBTK_OUTPUT(";
	} 
      mFile << mName << "," << ioi->name << ",\""
	    << ioi->descr << "\"," <<  ioi->type << ",\"" 
	    << ioi->nature<<"\");\n";
    }
  
  // EO black box description
  if (mTemplateParam.size()==0)
    {
      mFile << "BBTK_END_DESCRIBE_BLACK_BOX("
	    << mName << ");\n";
    }
  else if (mTemplateParam.size()==1)
    {
      mFile << "BBTK_END_DESCRIBE_TEMPLATE_BLACK_BOX("
	    << mName //<< ","
	// << mTemplateParam[0] 
	    << ");\n";
    }
  else 
    {
      throw bbfyException("template bb with more than 1 templ param not impl");
     
    } 
  
  // Untemplatization of itk filters
  if ( mGeneric )
    {
      WriteGenericITKFilterHeader();
    }

	AlertString();

  // EO namespace
  EndNamespace();
  
  // Prevent multiple inclusions
  mFile << "#endif // " << included <<"\n";
  // If is widget 
  if (mIsWidget)
    {
      mFile << "#endif // _USE_WXWIDGETS_\n";
    }

  // EOF
  mFile << "\n";

  mFile.close();
}
//==========================================================================



//==========================================================================
void bbfy::WriteGenericITKFilterHeader()
{
  mFile << "\n//===================================================\n";
  mFile << "// Generic \"untemplatized\" filter\n";
  mFile << "//===================================================\n";

  // Class declaration and parents
  mFile << "class /*BBTK_EXPORT*/ "<<mName<<"Generic\n";
  mFile << " : \n";
  mFile << "   public bbtk::AtomicBlackBox\n";
  mFile << "{\n";

  // Interface
  mFile << "  BBTK_BLACK_BOX_INTERFACE("
	<< mName << "Generic,bbtk::AtomicBlackBox);\n";

  // Inputs
  std::vector<IO>::iterator ioi;
  for (ioi=mInput.begin(); ioi!=mInput.end(); ++ioi) 
    {
      mFile << "  BBTK_DECLARE_INPUT(" 
	    << ioi->name
	    << ","
	    << ioi->generic_type
	    << ");\n";
    }
  
  // Outputs
  for (ioi=mOutput.begin(); ioi!=mOutput.end(); ++ioi) 
    {
      mFile << "  BBTK_DECLARE_OUTPUT(" 
	    << ioi->name
	    << ","
	    << ioi->generic_type
	    << ");\n";
    }
    
  // Process
  mFile << "  BBTK_PROCESS(ProcessSwitch);\n";
  mFile << "  private :\n";
  mFile << "    inline void ProcessSwitch();\n";
  mFile << "    template <class T, unsigned int D> void Process();\n";
  // EO black box declaration
  mFile << "};\n\n";



  // BO black box description
  mFile << "BBTK_BEGIN_DESCRIBE_BLACK_BOX("
	<< mName << "Generic,bbtk::AtomicBlackBox);\n";
  mFile << "BBTK_NAME(\"" << mName <<"\");\n";

  // Author
  mFile << "BBTK_AUTHOR(\""<<mAuthor<< "\");\n";

  // Description
  mFile << "BBTK_DESCRIPTION(\""<<mDescription<< "\");\n"; 
  
  // Inputs
  for (ioi=mInput.begin(); ioi!=mInput.end(); ++ioi) 
    {
      mFile << "BBTK_INPUT(";
      mFile << mName << "Generic," << ioi->name << ",\""
	    << ioi->descr << "\"," <<  ioi->generic_type <<");\n";
    }
  
  // Outputs
  for (ioi=mOutput.begin(); ioi!=mOutput.end(); ++ioi) 
    {
      mFile << "BBTK_OUTPUT(";
      mFile << mName << "Generic," << ioi->name << ",\""
	    << ioi->descr << "\"," <<  ioi->generic_type <<");\n";
    }
  
  // EO black box description
  mFile << "BBTK_END_DESCRIBE_BLACK_BOX("
	<< mName << "Generic);\n";


  //=================================================================
  // ProcessSwitch implementation
  mFile << "void "<< mName <<"Generic::ProcessSwitch()\n"
	<< "{\n"
	<< "CALL_FOR_ALL_TYPES_AND_DIM(bbGetInputIn()->GetType(),\n"
	<< "                           bbGetInputIn()->GetDimension(),\n"
	<< "                           Process);\n"
	<< "}\n";
  //=================================================================


  //=================================================================
  // Template process implementation
  mFile << "template <class T, unsigned int D>\n"
	<< "void "<<mName<<"Generic::Process()\n"
	<< "{\n"
	<< "  bbtkDebugMessageInc(\"Kernel\",9,\n"
	<< "      \""<<mName 
	<< "Generic::Process<\"<<TypeName<T>()<<\",\"<<D<<\">()\"<<std::endl);\n"
    
	<< "  typedef itk::Image<T,D> ImageType;\n"
	<< "  typedef "<<mName<<"<ImageType> FilterType;\n"
    
	<< "  FilterType* f = new FilterType(\"Temp\");\n"
    
	<< "  f->bbSetInputIn( this->bbGetInputIn()->GetImage<T,D>() );\n";
  
  for (ioi=mInput.begin(); ioi!=mInput.end(); ++ioi) 
    {
      if (ioi->name == "In") continue;
      mFile << "  f->bbSetInput"<<ioi->name<<" ( this->bbGetInput" 
	    << ioi->name << "() );\n";
    }
  
  mFile << "  f->bbUpdate();\n"
	<< "  this->bbSetOutputOut( new itkImage( f->bbGetOutputOut() ) );\n"
	<< "  f->UnRegister();\n"
	<< "  bbtkDebugDecTab(\"Kernel\",9);\n"
	<< "}\n\n";
  //=================================================================


}
//==========================================================================


//==========================================================================
void bbfy::CreateCode()
{
  mCxxName = "bb";
  mCxxName += mPackage;
  mCxxName += mName;
  mCxxName += ".cxx";
  if (mVerbose) std::cout << " - Creating code   '"<<mCxxName<<"'"<<std::endl;
  std::string fullname = mOutputPath + mCxxName;
  mFile.open(fullname.c_str());
  if (!mFile.good()) 
    {
      std::string mess("Error : could not open file \"");
      mess += fullname;
      mess += "\"";
      throw bbfyException(mess);
    }

	AlertString();


  // Includes
  // Header of the class
  mFile << "#include \"" << mHName << "\"\n";

  // Include Package header
  mFile << "#include \"bb"<<mPackage << "Package.h\"\n";

  // BO namespace
  BeginNamespace();
 
  
  // Template class ?
  if (mTemplateParam.size()>0) 
    {
      // Implementation
      mFile << "BBTK_BLACK_BOX_TEMPLATE_IMPLEMENTATION("
	    << mName << ","  
	    << mParentBlackBox << ");\n";
     
      if (mGeneric) 
	{	
	  // Implementation
	  mFile << "BBTK_BLACK_BOX_IMPLEMENTATION("
		<< mName << "Generic,bbtk::AtomicBlackBox);\n";
	  // Package
	  mFile << "BBTK_ADD_BLACK_BOX_TO_PACKAGE("
		<< mPackage << ","
		<< mName << "Generic)\n";
	}
    }
  else 
    {
      // Non template class
      // Package
      mFile << "BBTK_ADD_BLACK_BOX_TO_PACKAGE("
	    << mPackage << ","
	    << mName << ")\n";

      // Implementation
      mFile << "BBTK_BLACK_BOX_IMPLEMENTATION("
	    << mName << ","  
	    << mParentBlackBox << ");\n"; 
    }
  // Process
  if ((mType == STD)||(mProcess.size()))
    {
	AlertString();
      mFile << "void "<<mName<<"::Process()\n{\n";
      mFile << mProcess << "\n";
      mFile << "}\n";
    }
  // CreateWidget
  if (mIsWidget)
    {
	AlertString();
      mFile << "void "<<mName<<"::CreateWidget(wxWindow* parent)\n{\n";
      mFile << mCreateWidget << "\n";
      mFile << "}\n";
    }

		
  // User Set Default Values  
	AlertString();
  mFile <<"void "<<mName<<"::bbUserSetDefaultValues()"<<std::endl;
  mFile << "{"<<std::endl;  
	if ( (mType == vtkImageAlgorithm) || (mType == vtkPolyDataAlgorithm) )
	{
		mFile << "   BBTK_VTK_SET_DEFAULT_VALUES();\n";
	}
	mFile << mUserSetDefaultValues << std::endl;
  mFile << "}" << std::endl;

  // User Initialize Processing
	AlertString();
  mFile <<"void "<<mName<<"::bbUserInitializeProcessing()"
	<<std::endl;
  mFile << "{"<<std::endl;
	if ( (mType == vtkImageAlgorithm) || (mType == vtkPolyDataAlgorithm) )
	{
		mFile <<  "  BBTK_VTK_INITIALIZE_PROCESSING();\n";
	}
	mFile << mUserInitializeProcessing << std::endl;
  mFile << "}" << std::endl;

	// User Finalize Processing
	AlertString();
  mFile <<"void "<<mName<<"::bbUserFinalizeProcessing()"<<std::endl;
  mFile << "{"<<std::endl;
	if ( (mType == vtkImageAlgorithm) || (mType == vtkPolyDataAlgorithm) )
	{
		mFile << "   BBTK_VTK_FINALIZE_PROCESSING();\n";
	}
	mFile << mUserFinalizeProcessing << std::endl;
  mFile << "}" << std::endl;


  // EO namespace
  EndNamespace();

  mFile << "\n";
  
  // EOF
  mFile.close();
  
}
//==========================================================================
 




//==========================================================================
int main(int argc, char **argv)
{
  
  if (argc<2 || argc>5) 
    {
      std::cerr << "usage : "<< argv[0] <<" xml_file [package_name] [output_path] [-q]" << std::endl;
      return 1;
    }

  try 
    {
      std::string package("PACKAGE_NAME");
      std::string output_path("");
      bool verbose = true;
      if (argc>2) package = argv[2];
      if (argc>3) output_path = argv[3];
      if (argc>4) verbose = false;
      
      bbfy B(argv[1],package,output_path,verbose);
    }
  catch (bbfyException e)
    {
      std::cerr << argv[0] << "  " << argv[1] << std::endl
		<< e.mMessage << std::endl;
      return 1;
    }
  return 0;
}
//==========================================================================


