/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


#ifdef _USE_WXWIDGETS_

//==========================================================================
// WITH WX
//==========================================================================
#include "bbtkObject.h"
#include "bbtkInterpreter.h"
#include "bbtkWxBlackBox.h"
#include "bbtkWxGUIConsole.h"

#include <wx/cmdline.h>
#include <vector>
#include <map>

//==========================================================================
// Command line options definition
static const wxCmdLineEntryDesc cmdLineDesc[] =
{
  { wxCMD_LINE_SWITCH, _T("h"), _T("help"),   _T("print this help or help on the application defined in input bbs file if any") },
  { wxCMD_LINE_SWITCH, _T("g"), _T("graphical-dialog"),   _T("prompt the input parameter values using graphical dialog") },
  { wxCMD_LINE_SWITCH, _T("t"), _T("text-dialog"),   _T("prompt the input parameter values in text mode") },
  { wxCMD_LINE_SWITCH, _T("c"), _T("console"), _T("open bbi console") },
  { wxCMD_LINE_SWITCH, _T("N"), _T("no-console"),   _T("never open bbi console even on error") },
  { wxCMD_LINE_SWITCH, _T("q"), _T("quiet"),   _T("be quiet (='message max 0')") },
  { wxCMD_LINE_SWITCH, _T("d"), _T("debug"), _T("turn all messages on (='message all 9')") },
  { wxCMD_LINE_SWITCH, _T("D"), _T("Debug"), _T("memory debug on exit (='debug -D')") },
  { wxCMD_LINE_PARAM,  NULL, NULL, _T("file [file [...]]"), wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_OPTIONAL | wxCMD_LINE_PARAM_MULTIPLE },
  { wxCMD_LINE_NONE }
};
//==========================================================================

//==========================================================================
// Processes the command line parsing result
struct WxProcessCmdLine
{
  WxProcessCmdLine() {}
  void Process(wxCmdLineParser& parser);

  //  int argc;
  std::vector<std::string> argv;
  bool console;
  bool debug;
  bool quiet;
  bool help;
  bool graphical_dialog;
  bool text_dialog;
  bool no_console;
  bool proceed;
  std::map<std::string,std::string> param_map;
  std::vector<std::string> input_file;
  
};
//==========================================================================

//==========================================================================
void WxProcessCmdLine::Process(wxCmdLineParser& parser)
{
  proceed = true;
  if (parser.Found(_T("D"))) 
    {
      bbtk::StaticInitTime::PrintObjectListInfo = true;
    }
  
  debug = ( parser.Found(_T("d")) );  
  quiet = ( parser.Found(_T("q")) );
  help = ( parser.Found(_T("h")) );
  graphical_dialog = ( parser.Found(_T("g")) );
  text_dialog = ( parser.Found(_T("t")) );
  no_console = ( parser.Found(_T("N")) );
  
  if (quiet) bbtk::MessageManager::SetMessageLevel("max",0);
  if (debug) bbtk::MessageManager::SetMessageLevel("all",9);

  // parse the arguments and consider those which contain a "=" 
  // as set input commands, other as files
  int pargc = parser.GetParamCount();
  for (int i=0; i<pargc; ++i) 
    {
      std::string s = bbtk::wx2std(parser.GetParam(i));
      std::string::size_type pos = s.find_first_of("=");
      if (std::string::npos != pos) 
	{
	  std::string left = s.substr(0,pos);
	  std::string right = s.substr(pos+1,s.size());
	  param_map[left]=right;
	}
      else 
	{
	  input_file.push_back(s);
	}
    }

  bool usage = (help && (input_file.size()==0));
  if (usage) {
    std::cout << "BBI (The Black Box Interpreter) - bbtk "
	      << bbtk::GetVersion() << " - (c) Creatis 2007-2008"
	      << std::endl;
    parser.Usage();
    proceed = false;
  }

  console = ( parser.Found(_T("c")) || 
	      ((input_file.size() == 0) && 
	       (!no_console) &&
	       (!usage) ) );
  
}
//==========================================================================


//==========================================================================
class wxBBIApp : public wxApp
{
public:
  bool OnInit( );
  int  OnExit() { 
    //    std::cout << "wxBBIApp::OnExit()"<<std::endl;
    // bbtk::Object::PrintObjectListInfo();
    return true; }
  void OnInitCmdLine(wxCmdLineParser& parser);
  bool OnCmdLineParsed(wxCmdLineParser& parser);

  WxProcessCmdLine cmd;
};
//==========================================================================


//==========================================================================
void wxBBIApp::OnInitCmdLine(wxCmdLineParser& parser)
{
  parser.SetDesc(cmdLineDesc);
}
//==========================================================================

//==========================================================================
bool wxBBIApp::OnCmdLineParsed(wxCmdLineParser& parser)
{
  cmd.Process(parser);
  // if (!cmd.proceed) return false;
  return true;
}
//==========================================================================






//==========================================================================
// The `main program' equivalent, creating the windows and returning the
// main frame
bool wxBBIApp::OnInit( )
{
  //      std::cout << "OnInit"<<std::endl;
  wxApp::OnInit();
#ifdef __WXGTK__
  //See http://www.wxwindows.org/faqgtk.htm#locale
  setlocale(LC_NUMERIC, "C");
#endif
  

  if (cmd.quiet) bbtk::MessageManager::SetMessageLevel("max",0);
  if (cmd.debug) bbtk::MessageManager::SetMessageLevel("all",9);
  

  bbtk::WxGUIConsole *I = new bbtk::WxGUIConsole(0,_T("bbi"),wxSize(800,600));
  SetTopWindow(I);  
  if (cmd.console) I->Show(true);


  I->SetInputs(cmd.param_map);

  bool help_on_script = cmd.help && (cmd.input_file.size() > 0);
  if (help_on_script)     I->SetNoExecMode(true);
  if (cmd.graphical_dialog)     I->SetDialogMode(bbtk::VirtualExec::GraphicalDialog);
  if (cmd.text_dialog)     I->SetDialogMode(bbtk::VirtualExec::TextDialog);

  std::vector<std::string>::const_iterator i;
  bool error = false;

  for (i=cmd.input_file.begin(); i!=cmd.input_file.end(); ++i) 
    {
      error = ! I->InterpretFile(*i);
      if (error) break;
    }
  bool show_on_error = error && ! cmd.no_console;
  if (show_on_error) I->Show();

  I->SetNoExecMode(false);

  if (help_on_script) 
    {
      std::string package; 
      I->GetInterpreter()->GetExecuter()->GetFactory()->PrintHelpDescriptor("workspace",package,false);
    }

  /*
  std::cout << "soe="<<show_on_error <<std::endl;
  std::cout << "con="<<console<<std::endl;
  std::cout << "iws="<<bbtk::Wx::IsSomeWindowShown()<<std::endl;
  */
  if (!(show_on_error || cmd.console || bbtk::Wx::IsSomeWindowAlive() ))
    {
      I->Close();
      //      std::cout << "I->Close"<<std::endl;
    }
  else 
    {
      //      std::cout << "!I->Close"<<std::endl;
    }
	
  //	std::cout << "EO OnInit"<<std::endl;

  return true;

}
//==========================================================================


#if defined(_WIN32) 
//==========================================================================
// WINDOWS
//==========================================================================
IMPLEMENT_APP(wxBBIApp);

//  How to have a Console and wxWidgets
//  http://www.wxwidgets.org/wiki/index.php/MSVC_Setup_Guide
//   In Visual C++ 6 (7 should be similar), to create an application that is both a console application 
//  (cout's to the console are visible) and has a wxWidgets GUI, 
//  you need to use the linker option "/subsystem:console" and the following code:
int main(int argc, char* argv[])
{
    return WinMain(::GetModuleHandle(NULL), NULL, ::GetCommandLine(), SW_SHOWNORMAL);
}

#else 
//==========================================================================
// OTHER ( Linux... )
//==========================================================================

IMPLEMENT_APP_NO_MAIN(wxBBIApp);


int main(int argc, char* argv[])
{	
  wxMessageOutput::Set( new wxMessageOutputBest );

  wxCmdLineParser parser(cmdLineDesc,argc,argv);
  int val = parser.Parse(false);
  if (val>0) 
    {  
      parser.Usage();
      return 0;
    }
  WxProcessCmdLine cmdline;
  cmdline.Process(parser);

  if (!cmdline.proceed) return 0;

  if (cmdline.no_console) 
    {
      //      std::cout << "main NC"<<std::endl;
      // Interpreter 
      bbtk::Interpreter::Pointer I = bbtk::Interpreter::New();
      I->SetInputs(cmdline.param_map);
      bool help_on_script = cmdline.help && (cmdline.input_file.size() > 0);
      if (help_on_script) I->SetNoExecMode(true);
      if (cmdline.text_dialog) I->SetDialogMode(bbtk::VirtualExec::TextDialog);
      std::vector<std::string>::const_iterator i;
      bool error = false;
      for (i=cmdline.input_file.begin(); 
	   i!=cmdline.input_file.end(); ++i) 
	{
	  error = ! I->InterpretFile(*i);
	  if (error) break;
	}
      if (help_on_script) 
	{
	  I->SetNoExecMode(false);
	  std::string package; 
	  I->GetExecuter()->GetFactory()->PrintHelpDescriptor("workspace",
							      package,
							      false);
	}
      if (cmdline.input_file.size()==0)
	I->CommandLineInterpreter();

      //
    }
  else 
    {
      wxEntry(argc, argv);
    }

}


#endif // defined(_WIN32) 
//==========================================================================








#else
//==========================================================================
// WITHOUT WX
//==========================================================================

#include "bbtkInterpreter.h"

// Processes the command line parsing result
struct ProcessCmdLine
{
  ProcessCmdLine() {}
  void Process(int argc_, char *argv_[]);
  bool Found(std::string value);

  int  argc;
  std::vector<std::string> argv;
  bool console;
  bool debug;
  bool quiet;
  bool help;
  bool graphical_dialog;
  bool text_dialog;
  bool no_console;
  bool proceed;
  std::map<std::string,std::string> param_map;
  std::vector<std::string> input_file;
  
};

bool ProcessCmdLine::Found(std::string value)
{
	bool result=false;
  	for (int i=1; i<argc; ++i) 
    	{
		if (value==argv[i]) 
	    	{
			result = true;
		} 
	} // for
	return result;
}

void ProcessCmdLine::Process(int argc_, char *argv_[])
{
   argc = argc_;
   for (int i=0; i<argc; ++i) 
    {
	argv.push_back(argv_[i]);
    } // for

//EED  proceed = true;
  if (Found("D")) 
    {
      bbtk::StaticInitTime::PrintObjectListInfo = true;
    }
  
  debug 		= ( Found("d") );  
  quiet			= ( Found("q") );
  help			= ( Found("h") );
  graphical_dialog 	= ( Found("g") );
  text_dialog 		= ( Found("t") );
  no_console 		= ( Found("N") );
  
  if (quiet) bbtk::MessageManager::SetMessageLevel("max",0);
  if (debug) bbtk::MessageManager::SetMessageLevel("all",9);

  // parse the arguments and consider those which contain a "=" 
  // as set input commands, other as files
//EED  int pargc = parser.GetParamCount();
  for (int i=1; i<argc; ++i) 
    {
      std::string s = argv[i];
      std::string::size_type pos = s.find_first_of("=");
      if (std::string::npos != pos) 
	{
	  std::string left = s.substr(0,pos);
	  std::string right = s.substr(pos+1,s.size());
	  param_map[left]=right;
	}
      else 
	{
	  input_file.push_back(s);
	}
    }// for

  bool usage = (help && (input_file.size()==0));
  if (usage) {
    std::cout << "BBI (The Black Box Interpreter) - bbtk "
	      << bbtk::GetVersion() << " - (c) Creatis 2007-2008"
	      << std::endl;
//EED    parser.Usage();
    proceed = false;
  }

  console = ( Found("c") || 
	      ((input_file.size() == 0) && 
	       (!no_console) &&
	       (!usage) ) );
  
}
//==========================================================================



int main(int argc, char* argv[])
{  
//EED
//  if (argc>2) 
//    {
//      std::cout << "usage : "<<argv[0]<<" [filename]"<<std::endl;
//      return 0;
//    }

  std::cout << "BBI (Black Box Interpreter) - bbtk "
            << bbtk::GetVersion()<< " - (c) Creatis 2007"
            << std::endl;

  bbtk::Interpreter::Pointer I = bbtk::Interpreter::New();
  if (argc==1) 
  {
      	I->CommandLineInterpreter();
  }  else     {
	 
	ProcessCmdLine cmd;
	cmd.Process(argc,argv);
      	I->SetInputs(cmd.param_map);

      	std::string f(argv[1]);
      	I->InterpretFile(f);
  }
  
  //  bbtk::Wx::LoopUntilAllWindowsClose(); 
     
  return 0;

}

// EOF
#endif //#ifdef _USE_WXWIDGETS_



