/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la SantÈ)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkAny.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.9 $
=========================================================================*/
                                                                     


/**
 *\file
 *\brief  Defines the class any which can store any type of data (adapted from boost::any).
 */
#ifndef __BBTKANY_H_INCLUDED__
#define __BBTKANY_H_INCLUDED__

/*
#include <string>
#include <iostream>
#include <typeinfo>
#include <set>
#include <map>
#include <sstream>
#include <exception>

#include <cxxabi.h>
*/
#include <sstream>
#include "bbtkRTTI.h"
#include "bbtkMessageManager.h"
#include "bbtkException.h"

#include <boost/type_traits/is_pointer.hpp>

namespace bbtk 
{





  //=========================================================
  /// Abstract class used by the any class to store values
  class anyplaceholder
  {
  public: // structors
    
    virtual ~anyplaceholder() {}
    
  public: // queries
    
    /// returns the type of the held value
    virtual const std::type_info & type() const = 0;
    /// returns the type of the pointed held value
    virtual const std::type_info & pointed_type() const = 0;
    
    /// returns true iff the stored value is a pointer
    virtual bool is_pointer() const = 0;
    
    /// If the held value is a pointer then 
    /// returns its value
    virtual void* get_pointer() const = 0;
 
    /// 
    virtual void* get_pointer_to( const std::type_info& ) const = 0;
   
    virtual anyplaceholder * clone() const = 0;
    
  };
  //=========================================================
  
  //=========================================================
  /// Concrete template class used by the any class to store values
  /// which stores a value of type ValueType
  template<typename ValueType>
  class anyholder : public anyplaceholder
  {
  public: // structors
    
    anyholder(const ValueType & value)
      : held(value)
    {}
    
  public: // queries
    
    virtual const std::type_info & type() const { return typeid(ValueType);}
    virtual bool is_pointer() const { return false; }
    virtual const std::type_info & pointed_type() const { return typeid(void); }
    virtual void* get_pointer() const { return 0; }
    virtual void* get_pointer_to( const std::type_info& ) const { return 0; }
    virtual anyplaceholder * clone() const { return new anyholder(held); }
    
    public: // representation
    
    ValueType held;
    
  };
  //=========================================================
  
  //=========================================================
  /// specialization of anyholder for pointer types
  template<typename ValueType>
  class anyholder<ValueType*> : public anyplaceholder
  {
  public: // structors

    anyholder(ValueType* const & value)
      : held(value)
    { }

  public: // queries

    virtual const std::type_info & type() const
    {
      return typeid(ValueType*);
    }

    virtual bool is_pointer() const { return true; }
    virtual const std::type_info & pointed_type() const { return typeid(ValueType); }
    virtual void* get_pointer() const { 
      return (void*)held; 
    }
    virtual void* get_pointer_to( const std::type_info& t) const 
    { 
      return run_time_up_or_down_cast(t,typeid(ValueType),held);
    }

    virtual anyplaceholder * clone() const { return new anyholder(held); }

  public: // representation

    ValueType* held;

  };
  //=========================================================


  /** A magic class which can store any type of data which 
   * is allowed by the template template parameter TypeTrait.
   *
   * The only requirement on TypeTrait<T> is to have the member :
   *    static const bool value;
   * which is true iff the type T is an allowed type 
   * 
   * TypeTraits compliant objects are usually template structs 
   * for which the initialisation of value is set to false by default 
   * and set to true for the allowed types (template specialisation)
   * Example :
   *   template <typename T> struct mytypes { static const bool value; };
   *   template <typename T> const bool mytypes<T>::value = false;
   *   template <> const bool mytypes<int>::value = true;
   *   template <> const bool mytypes<float>::value = true; 
   * etc.
   * You can use any boost type_trait, like is_pointer, is_floating_point, etc.
   *
   * The class any is a generalisation of the boost::any class 
   * (see http://www.boost.org/doc/html/any.html).
   * The boost::any class itself is reproduced by any<thing>,
   * where thing is a TypeTrait whose value is true for all types.
   **/
  template <template <class> class TypeTraits >
  class any
  {
  public:
    typedef any< TypeTraits > self;
    // structors

    /// Default constructor
    any()
      : content(0)
    {
    }

    /// Constructor with a value of template type
    template<typename ValueType>
    any(const ValueType & value)
      : content(0)
    {
      bbtkDebugMessage("Data",1,
		       bbtk::HumanTypeName<self>()<<" construction with <"
		       <<bbtk::HumanTypeName<ValueType>()<<">"<<std::endl);
      //      ValueType v = value;
      //      int** i = (int**)(&v);
      //      std::cout << "v="<<*i<<std::endl;
      
      if (accepts<ValueType>()) 
	{ 
	  content = new anyholder<ValueType>(value);
	}
      else 
	{
	  bbtkError(bbtk::HumanTypeName<self>()
		    <<" constructor : data of type <"
		    <<bbtk::HumanTypeName<ValueType>()
		    <<"> are not accepted by traits <"
		    <<bbtk::HumanTypeName<TypeTraits<void> >()<<"> ");
	}
    }
  
    /// Copy constructor
    any(const any & other)
      : content(other.content ? other.content->clone() : 0)
    {
      bbtkDebugMessage("Data",1,
		       HumanTypeName<self>()
		       <<" copy construction with new content : "
		       <<HumanTypeName(type())
		       <<std::endl);
    }
  
    /// Destructor
    ~any()
    {
      delete content;
    }


    /// Swaps the content of this with another any
    any & swap(any & rhs)
    {
      std::swap(content, rhs.content);
      return *this;
    }

    /// Affectation of a value of template type
    template<typename ValueType>
    any & operator=(const ValueType & rhs)
    {
      bbtkDebugMessage("Data",1,
		       HumanTypeName<self>()
		       <<" operator= with data of type <"
		       <<HumanTypeName<ValueType>()
		       <<">"<<std::endl);
      if (accepts<ValueType>()) 
	{
	  any(rhs).swap(*this);
	  return *this;
	}
      else 
	{
	  bbtkError(HumanTypeName<self>()
		    <<" operator= : data of type <"
		    <<HumanTypeName<ValueType>()
		    <<"> are not accepted by traits <"
		    <<HumanTypeName<TypeTraits<void> >()<<"> ");
	}
    
    }
  
    /// Affectation of another any
    any & operator=(const any & rhs)
    {
	bbtkDebugMessage("Data",1,
			 HumanTypeName<self >()
			 <<" operator=(const any&) with content : "
			 <<HumanTypeName(rhs.type())<<std::endl);

      any(rhs).swap(*this);
      return *this;
    }
  

    /// Is it empty (no value held) ?
    bool empty() const
    {
      return !content;
    }
  
    /// Returns the type_info of the held value
    const std::type_info & type() const
    {
      return content ? content->type() : typeid(void);
    }

    /// Returns the type_info of the pointed held value
    const std::type_info & pointed_type() const
    {
      return content ? content->pointed_type() : typeid(void);
    }


    /// Returns true iff the contained type is Type
    template<typename Type>
    inline bool contains()
    {
      return ( type() == typeid(Type) );
    }

    /// Returns true iff the contained type is a pointer
    inline bool contains_pointer()
    {
      return content ? content->is_pointer() : false;
    }

    /// Returns true iff the contained type is t
    inline bool contains(TypeInfo t)
    {
      return ( (bool)((type() == t)!=0) );
     }

    /// Returns true iff any of type ValueType can be held 
    template<typename Type>
    inline bool accepts()
    {
      return TypeTraits<Type>::value; //is_valid();
    }

    /// Returns a const reference on the held value iff its type matches 
    /// the template parameter  
    template<typename ValueType>
    inline const ValueType& get() const
    {
	bbtkDebugMessage("Data",1,
			 HumanTypeName<self >()
			 <<" get<"<<HumanTypeName<ValueType>()
			 <<"> with content : "
			 <<HumanTypeName(type())<<std::endl);

      if ( type() == typeid(ValueType) )
	return static_cast< anyholder<ValueType> *>(content)->held;

      bbtkError(HumanTypeName<self >()
		<<" get with type <"
		<<bbtk::HumanTypeName<ValueType>()
		<<"> does not match content type <"
		<<bbtk::HumanTypeName<>(type())<<">");
    }

    template<typename ValueType>
    inline const ValueType* getP() const
    {
      if ( type() == typeid(ValueType) )
	return &static_cast< anyholder<ValueType> *>(content)->held;
 
      bbtkError(HumanTypeName<self >()
		<<" getP with type <"
		<<bbtk::HumanTypeName<ValueType>()
		<<"> does not match content type <"
		<<bbtk::HumanTypeName<>(type())<<">");
    }

    /// Returns a const reference on the held value 
    /// ** EVEN IF ITS TYPE DOES NOT MATCH THE TEMPLATE PARAMETER **
    /// Hence must be used when one knows that the type is good
    /// Otherwise can lead to unpredictible results 
    template<typename ValueType>
    inline const ValueType& unsafe_get() const
    {
	bbtkDebugMessage("Data",1,
			 HumanTypeName<self>()
			 <<"::unsafe_get<"
			 <<HumanTypeName<ValueType>()<<"> with content : "
			 <<HumanTypeName(this->type())
			 <<std::endl);

	//	PrintValueIfIsPointer<ValueType>(static_cast< anyholder<ValueType> * >(content)->held);
	//	int** i = (int**)(&static_cast< anyholder<ValueType> * >(content)->held);
	//	std::cout << "v="<<*i<<std::endl;

	if (content) 
	  return static_cast< anyholder<ValueType> * >(content)->held;
	
	bbtkError(HumanTypeName<self >()
		  <<"::usafe_get<"
		  <<bbtk::HumanTypeName<ValueType>()
		  <<"> : void content");
    }


    inline void* get_pointer() const 
    {
      void* p = content->get_pointer();
      bbtkDebugMessage("Data",1,
		       HumanTypeName<self>()
		       <<"::get_pointer() with content <"
		       <<HumanTypeName(this->type())
		       <<"> : result = "
		       << p
		       <<std::endl);
      return p;
    }
    
    inline void* get_pointer_to(const std::type_info& t) const 
    {
      void* p = content->get_pointer_to(t);
      bbtkDebugMessage("Data",1,
		       HumanTypeName<self>()
		       <<"::get_pointer_to("<<HumanTypeName(t)
		       <<") with content <"
		       <<HumanTypeName(this->type())
		       <<"> : result = "
		       << p
		       <<std::endl);
      return p;
    }
    
  private: 
    /// content
    anyplaceholder * content;
    
  };

  

  /// The TypeTraits which validates any type 
  template <typename T> struct thing { static const bool value = true; };

  BBTK_DEFINE_HUMAN_READABLE_TYPE_NAME(bbtk::any<bbtk::thing>,"anything");


  /// The TypeTraits which validates all integer types
  template <typename T> struct integer { static const bool value; };
  template <class T> const bool integer<T>::value = false;

  /// The TypeTraits which validates all floating point number types
  template <typename T> struct floating_point { static const bool value; };
  template <class T> const bool floating_point<T>::value = false;


  /// The TypeTraits which validates all numerical types
  template <typename T> struct number { static const bool value; };
  template <class T> const bool number<T>::value = 
	     integer<T>::value || floating_point<T>::value ;
  
  
  /// Macro to begin the enumeration of types included into a 
  /// any_type_collection previously declared with 
  /// DECLARE_any_type_collection. 
  /// Must be placed in the implementation part of the code.
#define BBTK_DECLARE_TYPE_TRAITS(NAME)					\
  template <typename T> struct NAME { static const bool value; };	\
    template <class T> const bool NAME<T>::value = false;
  
#define BBTK_ADD_TO_TYPE_TRAITS(NAME,TYPE)	\
  template <> bool NAME<TYPE>::value = true;


}





#endif

