/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkBlackBox.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.56 $
=========================================================================*/



/**
 *  \file 
 *  \brief Class bbtk::BlackBox : abstract black-box interface. 
 */
#include "bbtkBlackBox.h"
#include "bbtkPackage.h"
#include "bbtkMessageManager.h"
#include "bbtkFactory.h"
#include "bbtkBlackBoxOutputConnector.h"

#include "bbtkConfigurationFile.h"
#include "bbtkWxBlackBox.h"
#include "bbtkWx.h"

#include <fstream>
//#include <vector>


namespace bbtk
{
  static bool bbmgSomeBoxExecuting = false;
  static bool bbmgFreezeExecution = false;
  static std::set<BlackBox::WeakPointer> bbmgExecutionList;

  //=========================================================================

  BlackBox::Deleter::Deleter()
  {
  }
  //=========================================================================
  
  //=========================================================================
  int BlackBox::Deleter::Delete(Object* p)
  {
    BlackBox* b = dynamic_cast<BlackBox*>(p);
    if (!b)
      {
	bbtkInternalError("BlackBox::Deleter::Delete("<<p->GetObjectName()
			  <<"["<<p<<"]) : "
			  <<"dynamic cast to BlackBox* failed !");
      }
    std::string name = p->GetObjectName();//b->bbGetNameWithParent();
    bbtkDebugMessage("object",2,"##> BlackBox::Deleter(\""<<name<<"\")"<<std::endl);


    BlackBoxDescriptor::WeakPointer desc = b->bbGetDescriptor();
    bbtkDebugMessage("object",2,"##> BlackBox::Deleter(\""<<name<<"\") : deleting black box"<<std::endl);
    
    int refs = b->bbDelete();

    bbtkDebugMessage("object",2,"##> BlackBox::Deleter(\""<<name<<"\") : releasing descriptor"<<std::endl);
    
    if (!desc.expired()) 
      {
	Package::WeakPointer pack = desc.lock()->GetPackage();
	if (!pack.expired()) 
	  {
	    Package::ReleaseBlackBoxDescriptor(pack,desc);
	  }
	else 
	  {
	    bbtkDebugMessage("object",2,"##> BlackBox::Deleter(\""<<name<<"\") : descriptor package expired (was not held by a package and the box was the last instance)"<<std::endl);
	  }
      }
    else
      {
	bbtkDebugMessage("object",2,"##> BlackBox::Deleter(\""<<name<<"\") : descriptor expired : nothing to do (was not held by a package or the box is a complex black box prototype)"<<std::endl);
      }
    bbtkDebugMessage("object",2,"<## BlackBox::Deleter(\""<<name<<"\")"<<std::endl);
    return refs;
  }
  //=========================================================================

  //=========================================================================
  BlackBox::BlackBox(const std::string &name) 
    : 
    //    bbmStatus(MODIFIED), 
    bbmInitialized(false),
    bbmExecuting(false),
    bbmName(name),
    bbmBoxProcessMode("Pipeline"),
	bbLetRecursiveExecuteManualMode(false),
    bbmParent()
    
  {
	  //JCP 02-11-09
	 // bbmBoxProcessMode = "Pipeline";	
//std::cout<<"JCP BlackBox::BlackBox(const std::string &name) name=" <<name
//		  <<"bbmBoxProcessMode="<<bbmBoxProcessMode<<std::endl;
    bbtkBlackBoxDebugMessage("object",4,"==> BlackBox::BlackBox(\""
		     <<name<<"\")"<<std::endl);
    bbtkBlackBoxDebugMessage("object",4,"<== BlackBox::BlackBox(\""
		     <<name<<"\")"<<std::endl);
  }
  //=========================================================================

  //=========================================================================
  BlackBox::BlackBox(const BlackBox&)
  {}

  //=========================================================================
  BlackBox::BlackBox(BlackBox& from, const std::string &name) 
    :
    //    bbmStatus(from.bbmStatus), 
    bbmInitialized(false),
    bbmExecuting(false),
    bbmName(name), 
    bbmBoxProcessMode(from.bbmBoxProcessMode),
	bbLetRecursiveExecuteManualMode(false),
    bbmParent()
    
  {
	  //JCP 02-11-09
	  //bbmBoxProcessMode = from.bbmBoxProcessMode;
	  //std::cout<<"JCP BlackBox::BlackBox(const std::string &name) name=" <<name
		//  <<"bbmBoxProcessMode="<<bbmBoxProcessMode<<std::endl;
    bbtkBlackBoxDebugMessage("object",4,"==> BlackBox::BlackBox("
		     <<from.bbGetFullName()<<",\""
		     <<name<<"\")"<<std::endl);
    bbtkBlackBoxDebugMessage("object",4,"<== BlackBox::BlackBox("
		     <<from.bbGetFullName()<<",\""
		     <<name<<"\")"<<std::endl);
  }
  //=========================================================================


  //=========================================================================
  BlackBox::~BlackBox()
  {
    bbtkBlackBoxDebugMessage("object",4,"==> BlackBox::~BlackBox() ["<<bbmName
		     <<"]"<<std::endl);
    this->bbDesallocateConnectors();
    bbtkBlackBoxDebugMessage("object",4,"<== BlackBox::~BlackBox() ["<<bbmName
		     <<"]"<<std::endl);
  }
  //=========================================================================



  //=========================================================================
  std::string BlackBox::bbGetFullName() const
  { 
    return this->bbGetNameWithParent()+"<"+this->bbGetDescriptor()->GetTypeName()+">";
  }
  //=========================================================================
     


  //=========================================================================
  std::string BlackBox::bbGetNameWithParent() const
  {
    if (bbmParent.lock()) 
      {
	return bbmParent.lock()->bbGetNameWithParent() + ":" + bbmName;
      }
    else 
      {
	return bbmName;
      }
  } 
  //=========================================================================

  //=========================================================================
  void BlackBox::bbGetHelp(bool full) const
  {
    bbGetDescriptor()->GetHelp(full); 
  }
  //=========================================================================


  //=========================================================================
  bool BlackBox::bbHasInput(const std::string& name) const
  {
    bbtkBlackBoxDebugMessage("kernel",8,
			"BlackBox::bbHasInput(\""
			<<name<<"\")"
			<<std::endl);
    bool r = ( bbGetDescriptor()->GetInputDescriptorMap().find(name)
	       != bbGetDescriptor()->GetInputDescriptorMap().end());
    bbtkDebugDecTab("kernel",8);
    return r;
  }
  //=========================================================================


  //=========================================================================  
  bool BlackBox::bbHasOutput(const std::string& name) const
  {
    bbtkBlackBoxDebugMessage("kernel",8,"BlackBox::bbHasOutput(\""
			     <<name<<"\")"
			     <<std::endl);
    bool r = ( bbGetDescriptor()->GetOutputDescriptorMap().find(name)
	       != bbGetDescriptor()->GetOutputDescriptorMap().end());
    bbtkDebugDecTab("kernel",8);
    return r;
  }
  //=========================================================================


  //=========================================================================  
  TypeInfo BlackBox::bbGetOutputType( const std::string &name ) const 
  {
    bbtkBlackBoxDebugMessage("kernel",8,
			     "BlackBox::bbGetOutputType(\""
			     <<name<<"\")"
			     <<std::endl);
    TypeInfo r = bbGetDescriptor()->GetOutputDescriptor(name)->GetTypeInfo();
    bbtkDebugDecTab("kernel",8); 
    return r;
  }
  //=========================================================================

  //=========================================================================
  TypeInfo BlackBox::bbGetInputType( const std::string &name ) const
  {
    bbtkBlackBoxDebugMessage("kernel",8,
			     "BlackBox::bbGetInputType(\""
			     <<name<<"\")"
			     <<std::endl);
    TypeInfo r = bbGetDescriptor()->GetInputDescriptor(name)->GetTypeInfo();
    bbtkDebugDecTab("kernel",8);
    return r;
  }
  //=========================================================================


  //=========================================================================
  void BlackBox::bbAllocateConnectors()
  {  
    bbtkBlackBoxDebugMessage("kernel",8,
			"BlackBox::bbAllocateConnectors()"
			<<std::endl);					

    MakeBlackBoxPointer(this,true);

    const BlackBoxDescriptor::InputDescriptorMapType& imap 
      = bbGetDescriptor()->GetInputDescriptorMap(); 
    BlackBoxDescriptor::InputDescriptorMapType::const_iterator i;	
    for ( i = imap.begin(); i != imap.end(); ++i )			
      {									
	bbtkBlackBoxDebugMessage("kernel",8,"* Allocate \""<<i->first<<"\""<<std::endl);
	bbGetInputConnectorMap()[i->second->GetName()] 
	  = new BlackBoxInputConnector(GetThisPointer<BlackBox>());
      }									
    const BlackBoxDescriptor::OutputDescriptorMapType& omap 
      = bbGetDescriptor()->GetOutputDescriptorMap();		       
    BlackBoxDescriptor::OutputDescriptorMapType::const_iterator o; 
    for ( o = omap.begin(); o != omap.end(); ++o )
      {							
	bbtkBlackBoxDebugMessage("kernel",8,"* Allocate \""<<o->first<<"\""<<std::endl);
	bbGetOutputConnectorMap()[o->second->GetName()] 
	  = new BlackBoxOutputConnector(GetThisPointer<BlackBox>());
      }
  }
  //=========================================================================


  //=========================================================================
  void BlackBox::bbDesallocateConnectors()
  {
    bbtkBlackBoxDebugMessage("kernel",8,
			"BlackBox::bbDesallocateConnectors()"
			<<std::endl);					

    InputConnectorMapType::const_iterator i;
    for ( i = bbGetInputConnectorMap().begin();
	  i != bbGetInputConnectorMap().end(); ++i )		       
      {									
	bbtkBlackBoxDebugMessage("kernel",8,"* Delete \""<<i->first<<"\""<<std::endl);
	delete (i->second);
      }									
    OutputConnectorMapType::const_iterator o;	
    for ( o = bbGetOutputConnectorMap().begin(); 
	  o != bbGetOutputConnectorMap().end(); ++o )			
      {									
	bbtkBlackBoxDebugMessage("kernel",8,"* Delete \""<<o->first<<"\""<<std::endl);	       
	delete (o->second);
      }									
   
    bbtkDebugDecTab("kernel",8);
  }
  //=========================================================================


  //=========================================================================
  void BlackBox::bbCopyIOValues(BlackBox& from)
  {
    bbtkBlackBoxDebugMessage("kernel",1,
			     "BlackBox::bbCopyIOValues("
			     <<from.bbGetFullName()<<")"
			     <<std::endl);
    // copies the input values
    const BlackBoxDescriptor::InputDescriptorMapType& imap 
      = bbGetDescriptor()->GetInputDescriptorMap(); 
    BlackBoxDescriptor::InputDescriptorMapType::const_iterator i;	
    for ( i = imap.begin(); i != imap.end(); ++i )			
      {		
	if (! i->second->GetCopyConstruct() ) continue;
	std::string input = i->second->GetName();
	bbtkBlackBoxDebugMessage("kernel",2,"* Copying input "<<input<<std::endl);
	this->bbSetInput(input, from.bbGetInput(input) );
      }									
    // copies the output values
    const BlackBoxDescriptor::OutputDescriptorMapType& omap 
      = bbGetDescriptor()->GetOutputDescriptorMap();		       
    BlackBoxDescriptor::OutputDescriptorMapType::const_iterator o; 
    for ( o = omap.begin(); o != omap.end(); ++o )
      {							
	if (! o->second->GetCopyConstruct() ) continue;
	std::string output = o->second->GetName();
	bbtkBlackBoxDebugMessage("kernel",2,"* Copying output "<<output<<std::endl);
	this->bbSetOutput(output, from.bbGetOutput(output) );
      }

    bbtkDebugDecTab("kernel",9);
  }
  //=========================================================================



  //=========================================================================
  bool BlackBox::bbCanReact() const 
  { 
    return ( bbGlobalGetSomeBoxExecuting() 
#ifdef USE_WXWIDGETS
	     || Wx::IsSomeWindowAlive() 
#endif
	     );
  }
  //=========================================================================



  //=========================================================================
  BlackBox::BoxProcessModeValue BlackBox::bbGetBoxProcessModeValue() const
  {
    const std::string& p = bbmBoxProcessMode;
    if ( (p == "0") ||
	 (p == "P") || (p == "p") ||
	 (p == "Pipeline") || (p == "pipeline") ) return bbPipeline;
    if ( (p == "1") ||
	 (p == "A") || (p == "a") ||
	 (p == "Always") || (p == "always") ) return bbAlways;
    if ( (p == "2") ||
	 (p == "R") || (p == "r") ||
	 (p == "Reactive") || (p == "reactive") ) 
		return bbReactive;
    /*
    if ( (p == "3") ||
	 (p == "F") || (p == "f") ||
	 (p == "Flash") || (p == "flash") ) return Flash;
    */

	  if ( (p == "3") ||
	   (p == "M") || (p == "m") ||
	   (p == "Manual") || (p == "manual") ) return bbManual;
	  
	  bbtkError(bbGetFullName()<<" : BoxProcessMode value '"<<p
	      <<"' unknown. Possible values : "
	      <<"'0'/'P'/'p'/'Pipeline'/'pipeline' | "
	      <<"'1'/'A'/'a'/'Always'/'always' | "
	      <<"'2'/'R'/'r'/'Reactive'/'reactive'"
	      //      <<"'3'/'F'/'f'/'Flash'/'flash'"
	      <<"'3'/'M'/'m'/'Manual'/'manual'"
		  <<std::endl);
  }
  //=========================================================================
  
  //=========================================================================
  bool  BlackBox::bbBoxProcessModeIsReactive() const
  {
    return (bbGetBoxProcessModeValue() == bbReactive);
  }
  //=========================================================================

  //=========================================================================
  bool  BlackBox::bbBoxProcessModeIsAlways() const
  {
    return (bbGetBoxProcessModeValue() == bbAlways);
  }
  //=========================================================================


	//=========================================================================
	bool  BlackBox::bbBoxProcessModeIsManual() const
	{
		return (bbGetBoxProcessModeValue() == bbManual);
	}
	//=========================================================================
	

  //=========================================================================
  void BlackBox::bbAddOutputObserver(const std::string& output, 
				   OutputChangeCallbackType f)
  {
    bbGetOutputConnector(output).AddChangeObserver(f);
  }  
  //=========================================================================

  //=========================================================================
  void BlackBox::bbRemoveOutputObserver(const std::string& output_name, 
				      OutputChangeCallbackType f)
  {
    bbtkError("BlackBox::RemoveChangeObserver NOT IMPLEMENTED");
  }
  //=========================================================================


  //=========================================================================
  void BlackBox::bbSetStatusAndPropagate(BlackBoxInputConnector* c,
					 IOStatus s)
  {
    bbtkBlackBoxDebugMessage("change",5,
			     "=> BlackBox::bbSetStatusAndPropagate(input,"
			     <<GetIOStatusString(s)<<")"
			     <<std::endl);

    if (s==UPTODATE) bbtkError("bbSetStatusAndPropagate with status UPTODATE!");
    c->SetStatus(s);

    // Flash reaction
    /*
    if (bbGetBoxProcessModeValue() == Flash)
      {
	this->bbExecute();
      }
    */

    OutputConnectorMapType::const_iterator o;	
    for ( o = bbGetOutputConnectorMap().begin(); 
	  o != bbGetOutputConnectorMap().end(); ++o )			
      {									
	if (o->second->GetStatus()==UPTODATE) 
	  {
	    o->second->SetStatus(OUTOFDATE);
	    o->second->SignalChange(GetThisPointer<BlackBox>(),o->first); 
	  }
	}								
    
    if (  ( bbBoxProcessModeIsReactive()
	   || (c==bbGetInputConnectorMap().find("BoxExecute")->second))
	   && (bbCanReact() ) )
      {
	bbtkBlackBoxDebugMessage("change",2,
			 "-> Execution triggered by Reactive mode or BoxExecute input change"<<std::endl);
        bbGlobalAddToExecutionList( GetThisPointer<BlackBox>() );
      }    
    bbtkBlackBoxDebugMessage("change",5,
			     "<= BlackBox::bbSetStatusAndPropagate(input)"
			     <<std::endl);
  }
  //=========================================================================


  //=========================================================================  
  void BlackBox::bbSignalOutputModification(bool reaction)
  {
    bbtkBlackBoxDebugMessage("change",5,
			     "=> BlackBox::bbSignalOutputModification("
			     <<reaction<<")"
			     <<"]"<<std::endl);

    OutputConnectorMapType::iterator i;
    for ( i  = bbGetOutputConnectorMap().begin(); 
	  i != bbGetOutputConnectorMap().end(); ++i) 
      {
	//	std::cout << "Stat = "
	//<<GetIOStatusString(i->second->GetStatus())
	//		  <<std::endl;
	// LG : CANNOT SIGNAL ONLY WHEN UPTODATE 
	// See bbtkSampleOutputObserver
	//	if (i->second->GetStatus()==UPTODATE) 
	//	  {
	    i->second->SignalChange(GetThisPointer<BlackBox>(),i->first); 
	    //	  }
      } // for

    if (reaction) bbGlobalProcessExecutionList();

    bbtkBlackBoxDebugMessage("change",5,
			     "<= BlackBox::bbSignalOutputModification()"
			     <<std::endl);
  }  
  //=========================================================================   


  //=========================================================================  
  void BlackBox::bbSignalOutputModification(const std::string& output,
					    bool reaction)
  {
    bbtkBlackBoxDebugMessage("change",5,
			     "=> BlackBox::bbSignalOutputModification("
			     <<output<<","<<reaction<<")"
			     <<std::endl);
    
    OutputConnectorMapType::iterator i = 
      bbGetOutputConnectorMap().find(output);


    if ( i == bbGetOutputConnectorMap().end() ) 
	{
	  bbtkError("BlackBox["<<bbGetFullName()<<"]::bbSignalOutputModification("<<output<<") : unknown output");
	}

    //    if (i->second->GetStatus()==UPTODATE) 
    //      {
	i->second->SignalChange(GetThisPointer<BlackBox>(),i->first); 
	// Has to notify the output "BoxChange" also
	if (output != "BoxChange") 
	  {
	    i = bbGetOutputConnectorMap().find("BoxChange");
	    if ( i != bbGetOutputConnectorMap().end() ) 
	      {
		i->second->SignalChange(GetThisPointer<BlackBox>(),i->first); 
	      }
	  }
	if (reaction) bbGlobalProcessExecutionList();
	//      }

	bbtkBlackBoxDebugMessage("change",5,
			     "<= BlackBox::bbSignalOutputModification("
			     <<output<<")"
			     <<std::endl);
  }  
  //=========================================================================   
  //=========================================================================  
  void BlackBox::bbSignalOutputModification(const std::vector<std::string>& output,
	bool reaction)
  {
    bbtkBlackBoxDebugMessage("change",5,
			"=> BlackBox::bbSignalOutputModification(vector of outputs)"
<<std::endl);
    OutputConnectorMapType::iterator i;
    std::vector<std::string>::const_iterator o;
    bool changed = false;
    for (o=output.begin();o!=output.end();++o) 
      {
	// the output "BoxChange" must be signaled **AFTER** all others
	if (*o == "BoxChange") continue;
	// Look for the connector
	i = bbGetOutputConnectorMap().find(*o);
	if ( i == bbGetOutputConnectorMap().end() ) 
	  {
	    bbtkError("BlackBox["<<bbGetFullName()<<"]::bbSignalOutputModification("<<*o<<") : unknown output");
	  }

	//	if (i->second->GetStatus()==UPTODATE)
	//	  {
	    i->second->SignalChange(GetThisPointer<BlackBox>(),i->first); 
	    changed = true;
	    //  }
      }
    // Has to notify the output "BoxChange" also
    i = bbGetOutputConnectorMap().find("BoxChange");
    if ( changed && (i != bbGetOutputConnectorMap().end())) 
      {
	// if (i->second->GetStatus()==UPTODATE) 
	//	  {
	    i->second->SignalChange(GetThisPointer<BlackBox>(),i->first); 
	    if (reaction) bbGlobalProcessExecutionList();
	    //  }
      }

    bbtkBlackBoxDebugMessage("change",5,
			     "<= BlackBox::bbSignalOutputModification(vector of outputs)"
			     <<std::endl);
  }  
  //=========================================================================   







  //=========================================================================
  /// Main processing method of the box.
  void BlackBox::bbExecute(bool force)
  {
    bbtkBlackBoxDebugMessage("process",2,
			     "=> BlackBox::bbExecute("<<(int)force<<")"
			     <<std::endl);
 
    // If already executing : return
    /*
    if (bbGetExecuting()) 
      {
	bbtkBlackBoxDebugMessage("process",2,
			 " -> already executing : abort"<<std::endl);
	return;
      }
    */

    // If execution frozen : return
    if (bbGlobalGetFreezeExecution()) 
      {
	bbtkBlackBoxDebugMessage("process",2,
			 " -> FreezeExecution global flag is 'true' : abort execution"<<std::endl);
      }

    BBTK_BUSY_CURSOR;

    // If force is true then update is triggered even if the box is UPTODATE
    //    if (force) bbSetModifiedStatus();

	if ( bbBoxProcessModeIsManual() ) 
	{
		bbLetRecursiveExecuteManualMode = true;
    }
	  
	  
    // Calls the main recursive execution method 
    bbRecursiveExecute(Connection::Pointer());

	  
	if ( bbBoxProcessModeIsManual() ) 
	{
		bbLetRecursiveExecuteManualMode = false;
	}
	  
	  
    bbtkBlackBoxDebugMessage("process",2,
			     "<= BlackBox::bbExecute()"
			     <<std::endl);
  }
  //=========================================================================

  //=========================================================================
  void BlackBox::bbInitializeProcessing()
  {
    if (!bbmInitialized) 
      {
	bbtkBlackBoxDebugMessage("process",2,"** Initialize processing"
				 <<std::endl);
	this->bbRecursiveInitializeProcessing();
	bbmInitialized = true;
      }
  }
  //=========================================================================

  //=========================================================================
  void BlackBox::bbFinalizeProcessing()
  {
    if (bbmInitialized) 
      {
	bbtkBlackBoxDebugMessage("process",2,"** Finalize processing"
				 <<std::endl);
	this->bbRecursiveFinalizeProcessing();
	bbmInitialized = false;
      }
  }
  //=========================================================================

  
  //=========================================================================
  void BlackBox::bbRecursiveExecute( Connection::Pointer caller )
  {
    bbtkBlackBoxDebugMessage("process",3,
			"=> BlackBox::bbRecursiveExecute("
			<<(caller?caller->GetFullName():"0")<<")"
			<<std::endl);

    // If already executing : return
    if (bbGetExecuting()) 
      {
	bbtkBlackBoxDebugMessage("process",3,
			 " -> already executing : abort"<<std::endl);
	return; 
      }
    
    // If not initialized do it
    bbInitializeProcessing();

    bbSetExecuting(true);
    bool wasExecuting = bbGlobalGetSomeBoxExecuting();
    bbGlobalSetSomeBoxExecuting(true);
    
    // Creates the window if the black box has one
    this->bbCreateWindow();
    
    // Updates its inputs
	  
//	  IOStatus s;
	  IOStatus s=UPTODATE;
//	  IOStatus s=OUTOFDATE;
//	  IOStatus s=MODIFIED;
	  
	  
	  if ( ( bbBoxProcessModeIsManual()==false )  || 
		   ( (bbBoxProcessModeIsManual()==true) && (bbLetRecursiveExecuteManualMode==true) ) )
	  {
    		  s = bbUpdateInputs();
	  }	  
		  
    if ( (s != UPTODATE) ||  bbBoxProcessModeIsAlways() )
      {
	  // Displays the window (WxBlackbox)
 	  //	bbShowWindow(caller);

	  // Actual processing (virtual)
		  if ( ( bbBoxProcessModeIsManual()==false )  || 
			   ( (bbBoxProcessModeIsManual()==true)&&(bbLetRecursiveExecuteManualMode==true) ) 
			 )
			{
				this->bbProcess();		
		    } // Manual analysis
		  
//EED ups			if ((bbBoxProcessModeIsManual()==true) && (bbLetRecursiveExecuteManualMode==false))
//EED ups			{
//EED ups			  bbSignalOutputModification(true);
//EED ups			}
		  
		  
		  // Update the I/O statuses
		  bbComputePostProcessStatus();
      }
    else 
      {
	// Test output status...
	OutputConnectorMapType::iterator o;
	for ( o = bbGetOutputConnectorMap().begin(); 
	      o!= bbGetOutputConnectorMap().end(); ++o) 
	  {
	    if (o->second->GetStatus() != UPTODATE)
	      {
		bbtkWarning("BlackBox::bbRecursiveExecute: "
			    <<"all inputs are Up-to-date but output '"
			    <<o->first<<"' is Out-of-date ???");
	      }
	  }
	
        bbtkBlackBoxDebugMessage("process",3," -> Up-to-date : nothing to do"
			 <<std::endl);
      }

    // Shows the window if the black box has one
    this->bbShowWindow(); 

	  
    bbtkBlackBoxDebugMessage("process",3,
            "<= BlackBox::bbRecursiveExecute()"
            <<std::endl);

    bbSetExecuting(false);
    bbGlobalSetSomeBoxExecuting(wasExecuting);

    return;
  }
  //=========================================================================
  
   



  //=========================================================================
  IOStatus BlackBox::bbUpdateInputs()
  {
    bbtkBlackBoxDebugMessage("process",4,
			"=> BlackBox::bbUpdateInputs()"
			<<std::endl);	

    IOStatus s = UPTODATE;

    InputConnectorMapType::iterator i;
    for ( i = bbGetInputConnectorMap().begin(); 
	  i!= bbGetInputConnectorMap().end(); ++i) 
      {
	//	if (i->first=="WinHide") continue;
	// If input type is Void : no recurse
	//if (  bbGetDescriptor()->GetInputDescriptor(i->first)->GetTypeInfo() 
	//      == typeid(Void) ) 
	//  continue;
	bbtkBlackBoxDebugMessage("change",2,
			    "Input '"<<i->first
			    <<"': status before update = '"
			    <<GetIOStatusString(i->second->GetStatus())
			    <<"'"<<std::endl);
	i->second->RecursiveExecute();
	IOStatus t = i->second->GetStatus();
	if (t > s) s = t;
	bbtkBlackBoxDebugMessage("change",2,
				 "Input '"<<i->first
				 <<"': status before process = '"
				 <<GetIOStatusString(i->second->GetStatus())
				 <<"'"<<std::endl);
      }
    
    bbtkBlackBoxDebugMessage("process",4,
			"<= BlackBox::bbUpdateInputs()"
			<<std::endl);
    return s;
  }
  //=========================================================================

  //==================================================================
   void BlackBox::bbComputePostProcessStatus()
  {
    bbtkBlackBoxDebugMessage("process",4,
			"=> BlackBox::bbComputePostProcessStatus()"
			<<std::endl);	

    IOStatus new_output_status = UPTODATE;
    if (bbBoxProcessModeIsAlways()) new_output_status = OUTOFDATE;

    // Update the input statuses
    InputConnectorMapType::iterator i;
    for ( i = bbGetInputConnectorMap().begin(); 
	  i!= bbGetInputConnectorMap().end(); ++i) 
      {
	IOStatus t = i->second->GetStatus();
	if (t == OUTOFDATE) new_output_status = OUTOFDATE;
	// A previously MODIFIED status turns to UPTODATE
	if (t==MODIFIED) i->second->SetStatus(UPTODATE);
	bbtkBlackBoxDebugMessage("change",2,
			 "Input '"<<i->first<<"' : "
			 << GetIOStatusString(t) << " -> "
			 << GetIOStatusString(i->second->GetStatus())
			 << std::endl);
      }
    bbtkBlackBoxDebugMessage("change",2,
			     "New output status : "
			     << GetIOStatusString(new_output_status)
			     <<std::endl);
    // Update the output statuses
    OutputConnectorMapType::iterator o;
    for ( o = bbGetOutputConnectorMap().begin(); 
	  o!= bbGetOutputConnectorMap().end(); ++o) 
      {
		
//EED		  if  ( ( bbBoxProcessModeIsManual()==false )  || 
//EED			    ( (bbBoxProcessModeIsManual()==true) && (bbLetRecursiveExecuteManualMode==true) ) 
//EED			  )
//EED		  {
			  o->second->SetStatus(new_output_status);
//EED		  }  else  {
//EED			  if  (( (bbBoxProcessModeIsManual()==true) && (bbLetRecursiveExecuteManualMode==false) )  ) 
//EED			  {
//EED				  o->second->SetStatus(UPTODATE);
//EED			  }
//EED		  } // Manual analysis
			  
      }

    bbtkBlackBoxDebugMessage("process",4,
			"<= BlackBox::bbComputePostProcessStatus()"
			<<std::endl);
  }
  //==================================================================

  //=========================================================================
  void BlackBox::bbConnectInput( const std::string& name, Connection* c)
  {
    bbtkBlackBoxDebugMessage("connection",2,
			"==> BlackBox::bbConnectInput(\""
			<<name<<"\","<<c->GetFullName()<<")"
			<<std::endl);       

    InputConnectorMapType::iterator i = bbGetInputConnectorMap().find(name);
    if (i==bbGetInputConnectorMap().end())
      {
	bbtkError("no input called '"<<name<<"'");
      }
    i->second->SetConnection(c);
    // The input *MUST* be set OUTOFDATE to update its input on next execution
    bbSetStatusAndPropagate(i->second,OUTOFDATE);

    bbtkBlackBoxDebugMessage("connection",2,
			"<== BlackBox::bbConnectInput(\""
			<<name<<"\","<<c->GetFullName()<<")"
			<<std::endl);
  }
  //=========================================================================


  //=========================================================================  
  void BlackBox::bbConnectOutput( const std::string& name, Connection* c)
  {
    bbtkBlackBoxDebugMessage("connection",2,
			     "==> BlackBox::bbConnectOutput(\""<<name<<"\","
			     <<c->GetFullName()<<")"
			     <<std::endl);       

    OutputConnectorMapType::iterator i = bbGetOutputConnectorMap().find(name);
    if (i==bbGetOutputConnectorMap().end())
      {
	bbtkError("no output called '"<<name<<"'");
      }
    i->second->SetConnection(c);

    bbtkBlackBoxDebugMessage("connection",2,
			     "<== BlackBox::bbConnectOutput(\""<<name<<"\","
			     <<c->GetFullName()<<")"
			     <<std::endl);
  }
  //=========================================================================


  //=========================================================================
   void BlackBox::bbDisconnectInput( const std::string& name, Connection* c)
  {

    bbtkBlackBoxDebugMessage("connection",2,
		     "==> BlackBox::bbDisconnectInput(\""<<name
		     <<"\","<<c->GetFullName()<<")"
		     <<std::endl);
    if (!c) 
      {

	bbtkBlackBoxDebugMessage("connection",2,"c==0"<<std::endl);	
	return;
      }

    InputConnectorMapType::iterator i = bbGetInputConnectorMap().find(name);
    if (i==bbGetInputConnectorMap().end())
      {
	bbtkError("no input called '"<<name<<"'");
      }
    i->second->UnsetConnection(c);

    bbtkBlackBoxDebugMessage("connection",2,
		     "<== BlackBox::bbDisconnectInput(\""<<name
		     <<"\","<<c->GetFullName()<<")"
		     <<std::endl);      

  }
  //=========================================================================


  //=========================================================================
   void BlackBox::bbDisconnectOutput( const std::string& name, Connection* c)
  {
    bbtkBlackBoxDebugMessage("connection",2,
		     "==> BlackBox::bbDisconnectOutput(\""<<name
		     <<"\","<<c->GetFullName()<<")"
		     <<std::endl);       
    if (!c) 
      {

	bbtkBlackBoxDebugMessage("connection",2,"c==0"<<std::endl);	
	return;
      }

    OutputConnectorMapType::iterator i = bbGetOutputConnectorMap().find(name);
    if (i==bbGetOutputConnectorMap().end())
      {
	bbtkError("no output called '"<<name<<"'");
      }
    i->second->UnsetConnection(c);

    bbtkBlackBoxDebugMessage("connection",2,
		     "<== BlackBox::bbDisconnectOutput(\""<<name
		     <<"\","<<c->GetFullName()<<")"
		     <<std::endl);       
  } 
  //=========================================================================
 






















  //=========================================================================
  void BlackBox::bbWriteDotInputOutputName(FILE *ff,bool inputoutput,int detail, int level)
  {
    fprintf(ff,"%s%p",bbGetTypeName().c_str(),this);
  }
  //=========================================================================


  //=========================================================================
  std::string BlackBox::bbGetOutputAsString( const std::string &output ) 
  {
    std::string v;
    // Looks for the adaptor
    if (bbGetOutputType(output).name() != typeid(std::string).name() ) 
      {
	// Look for factory 
	Package::Pointer p = bbGetDescriptor()->GetPackage();
	if ((p != 0) && ( ! p->GetFactorySet().empty() ) )
	  {
	    Factory::Pointer f = p->GetFactorySet().begin()->lock();
	    BlackBox::Pointer a;
	    try
	      {
		a = f->NewAdaptor(  
				  bbGetOutputType(output),
				  typeid(std::string),
				  "");
	      } catch (bbtk::Exception e) 
	      {
	      }
	    if (a){
	      //			bbUpdate();
	      a->bbSetInput("In",bbGetOutput(output));
	      a->bbExecute();
	      v = a->bbGetOutput("Out").unsafe_get<std::string>() ;
	    } else {
	      v="? (no adaptor found)";
	    }
	  }
	else 
	  {
	    v="? (no factory found)";
	  }
      } 
    else 
      {
	//         bbUpdate();
	v = bbGetOutput(output).unsafe_get<std::string>() ;
      }
    return v;
  }
  //=========================================================================

  //=========================================================================
  std::string BlackBox::bbGetInputAsString( const std::string &input ) 
  {
    std::string v;
    // Looks for the adaptor
    if (bbGetInputType(input) != typeid(std::string)) 
      {
	// Look for factory 
	Package::Pointer p = bbGetDescriptor()->GetPackage();
	if ((p != 0) && ( ! p->GetFactorySet().empty() ) )
	  {
	    Factory::Pointer f = p->GetFactorySet().begin()->lock();
	    BlackBox::Pointer a;
	    try
	      {
		a = f->NewAdaptor(  
			       bbGetInputType(input),
			       typeid(std::string),
			       "");
	      }catch (bbtk::Exception e) 
	      {
	      }
	    if (a)
	      {
		//			bbUpdate();
		a->bbSetInput("In",bbGetInput(input));
		a->bbExecute();
		v = a->bbGetOutput("Out").unsafe_get<std::string>() ;
	      } 
	    else 
	      {
		v="? (no adaptor found)";
	      }
	  } 
	else 
	  {
	    v="? (no factory found)";
	  }
      }
    else 
      {
	v = bbGetInput(input).unsafe_get<std::string>() ;
      }
    return v;
  }
  //=======================================================================

  //=======================================================================
  // Replaces substrings "<" by "["
  void SubsBrackets ( std::string& s )
  {
    //   std::cout << "BEFORE=["<<s<<"]"<<std::endl;
    std::string ss("<");
    std::string::size_type pos = 0;
    pos = s.find(ss,0);
    std::string cr("[");
    while ( pos != std::string::npos )
      {
	//	std::cout << "*** find one "<<std::endl;
	s.replace(pos,1,cr.c_str(),1);
	pos = s.find(ss, pos);
      } 
    ss = ">";
    pos = 0;
    pos = s.find(ss,0);
    cr = "]";
    while ( pos != std::string::npos )
      {
	//	std::cout << "*** find one "<<std::endl;
	s.replace(pos,1,cr.c_str(),1);
	pos = s.find(ss, pos);
      } 
    ss = ",";
    pos = 0;
    pos = s.find(ss,0);
    cr = "-";
    while ( pos != std::string::npos )
      {
	//	std::cout << "*** find one "<<std::endl;
	s.replace(pos,1,cr.c_str(),1);
	pos = s.find(ss, pos);
      }     //    std::cout << "AFTER=["<<s<<"]"<<std::endl;
  }
  //=======================================================================

  //=========================================================================
  /// Write Graphviz-dot description in file
  void BlackBox::bbWriteDotFileBlackBox(FILE *ff,
					BlackBox::Pointer parentblackbox, 
					int detail, int level,
					bool instanceOrtype,
					bool relative_link )

  { 
    InputConnectorMapType::iterator i;
    // label
    std::string labelStr;
    std::string valueStr("");

	if (detail==0) {
		labelStr = bbGetName() ; 
//EED 18 Fev 2008
		labelStr = labelStr + "\\n[" +this->bbGetDescriptor()->GetPackage()->GetName()+"::"+ bbGetTypeName() + "]";
	} else {
		labelStr = bbGetName();
		labelStr = labelStr + "   [" +this->bbGetDescriptor()->GetPackage()->GetName()+"::"+ bbGetTypeName() + "]  ";
    }

    SubsBrackets(labelStr);
    if (detail==1)
      {
	labelStr = labelStr + " | {{ "; 
	std::string tempStrTypeName;
	bool tmp; 
	tmp=false;
	for ( i = mInputConnectorMap.begin(); i != mInputConnectorMap.end(); ++i ) 
	  {
	    if (tmp==true)
	      {
		labelStr=labelStr+" | ";
	      }
	    tmp=true;
	    if (instanceOrtype==true)
	      {
		valueStr = this->bbGetInputAsString(i->first) + " = ";
	      } 
	    const BlackBoxInputDescriptor* id = bbGetDescriptor()->GetInputDescriptor(i->first);
	    tempStrTypeName=id->GetTypeName();
	    SubsBrackets(tempStrTypeName);
	    std::string Name(i->first);
	    SubsBrackets(Name);
	    labelStr=labelStr + "<"+i->first.c_str()+"> "  + valueStr +  Name.c_str() + "  [" + tempStrTypeName.c_str() + "]";
	  }
	labelStr=labelStr+ " } | {";
	tmp = false;
	OutputConnectorMapType::iterator ii;
	for ( ii = mOutputConnectorMap.begin(); ii != mOutputConnectorMap.end(); ++ii ) 
	{
	   if (tmp==true)
	   {
		   labelStr=labelStr+" | ";
	   }
	   tmp = true;
	   if (instanceOrtype==true)
	   {
		   valueStr = this->bbGetOutputAsString(ii->first) + " = ";
	   }
	   const BlackBoxOutputDescriptor* id = bbGetDescriptor()->GetOutputDescriptor(ii->first); 
	   tempStrTypeName=id->GetTypeName();
	   SubsBrackets(tempStrTypeName);
	   std::string Name(ii->first);
	   SubsBrackets(Name);
	   labelStr=labelStr+"<"+ii->first.c_str()+"> " + valueStr + Name.c_str() + "  ["+tempStrTypeName+"]";
	}
	labelStr = labelStr+ "      } }" ;
} // detail

    fprintf(ff,"  " );
    bbWriteDotInputOutputName(ff,true,detail,level);
    std::string tmp ( bbGetTypeName() );
    SubsBrackets(tmp);
    std::string url;
    if (relative_link) 
      url = this->bbGetDescriptor()->GetPackage()->GetDocRelativeURL() + "#" + tmp;
    else 
      url = this->bbGetDescriptor()->GetPackage()->GetDocURL() + "#" + tmp;
  
    fprintf( ff , " [shape=record, URL=\"%s\",label=\"%s\"]%s\n",url.c_str(),labelStr.c_str(),";" );
    //    std::cout  << labelStr << std::endl;

    // Relation Input
    if (GetThisPointer<BlackBox>()!=parentblackbox){
      for ( i = mInputConnectorMap.begin(); i != mInputConnectorMap.end(); ++i ) 
	{
	  if (i->second)
	    {
	      Connection* con = i->second->GetConnection();
	      if (con!=NULL){
		BlackBox::Pointer a=con->GetOriginalBlackBoxFrom();
		BlackBox::Pointer b=con->GetOriginalBlackBoxTo();
		fprintf(ff,"  ");
		a->bbWriteDotInputOutputName(ff,false,detail,level);
		if (detail==1)
		  {
		    fprintf(ff,":%s",con->GetOriginalBlackBoxFromOutput().c_str());
		  }
		fprintf(ff,"->");
		b->bbWriteDotInputOutputName(ff,true,detail,level);
		if (detail==1)
		  {
		    fprintf(ff,":%s",con->GetOriginalBlackBoxToInput().c_str());
		  }
		fprintf(ff,"%s\n",";");
	      }  // if con
	    } // if second
	} // for
    } // if parentblackbox
  }
  //=========================================================================




  //=========================================================================
  void BlackBox::bbPrintHelp(BlackBox::Pointer parentblackbox, 
				 int detail, int level
				 /*,Factory *factory*/ )
  {
     
    if (this->bbGetDescriptor()->GetPackage()) 
      {
	     bbtkBlackBoxMessage("help",1,"Black Box '"<<bbGetName()<<"' <"<<
		    this->bbGetDescriptor()->GetPackage()->GetName()
		    <<"::"<<this->bbGetDescriptor()->GetTypeName()<<">"<<std::endl);
      }
    else 
      {
	     bbtkBlackBoxMessage("help",1,"Black Box <::"<<this->bbGetDescriptor()->GetTypeName()<<">"<<std::endl);
      }
    /*
    if (bbIsUpToDate())
      {
	     bbtkBlackBoxMessage("help",1,"Up-to-date ["<<mMaxInputChangeTime<<","
		    <<mMinOutputChangeTime<<"]"<<std::endl);
      }
    else 
      {
	     bbtkBlackBoxMessage("help",1,"Out-of-date ["<<mMaxInputChangeTime<<","
		    <<mMinOutputChangeTime<<"]"<<std::endl);
      }
    */
    //    bbtkBlackBoxMessage("help",1," "<<GetDescription()<<std::endl);
    //    bbtkBlackBoxMessage("help",1," By : "<<GetAuthor()<<std::endl);

    std::vector<std::string> iname;
    std::vector<std::string> ivalue;
    std::vector<std::string> iconn;
    std::vector<std::string> istatus;

    InputConnectorMapType::iterator i;
    unsigned int namelmax = 0;
    unsigned int valuelmax = 0;
    //   unsigned int connlmax = 0;
    for ( i = mInputConnectorMap.begin(); i != mInputConnectorMap.end(); ++i ) 
    {
	   iname.push_back(i->first);
	   if (iname.back().size()>namelmax) namelmax = iname.back().size();
	   ivalue.push_back(bbGetInputAsString(i->first));
	   if (ivalue.back().size()>valuelmax) valuelmax = ivalue.back().size();
	   std::string s("");
	   Connection* con = i->second->GetConnection();
	   if (con!=0){
	      s = con->GetOriginalBlackBoxFrom()->bbGetName();
	      s += ".";
	      s += con->GetOriginalBlackBoxFromOutput();
	   }  // if con
	   iconn.push_back(s);
	   istatus.push_back(GetIOStatusString(i->second->GetStatus()));
    }
    OutputConnectorMapType::iterator o;
    std::vector<std::string> oname;
    std::vector<std::string> ovalue;
    std::vector<std::vector<std::string> > oconn;
    std::vector<std::string> ostatus;
    for ( o = mOutputConnectorMap.begin(); o != mOutputConnectorMap.end(); ++o ) 
    {
	   oname.push_back(o->first);
	   if (oname.back().size()>namelmax)
          namelmax = oname.back().size();
	   ovalue.push_back(bbGetOutputAsString(o->first));
	   if (ovalue.back().size()>valuelmax) 
          valuelmax = ovalue.back().size();
	   std::vector<std::string> ss;
	   const std::vector<Connection*>& con 
	                            = o->second->GetConnectionVector();
	   std::vector<Connection*>::const_iterator c;
	   for (c=con.begin();c!=con.end();++c) 
	   {
	       std::string s;
	       s = (*c)->GetOriginalBlackBoxTo()->bbGetName();
	       s += ".";
	       s += (*c)->GetOriginalBlackBoxToInput();
	       ss.push_back(s);
	    }  // if con
	    oconn.push_back(ss);
	    ostatus.push_back(GetIOStatusString(o->second->GetStatus()));
    }

    if (iname.size()) 
      bbtkBlackBoxMessage("help",1," * Inputs : "<<std::endl);
    else 
      bbtkBlackBoxMessage("help",1," * No inputs"<<std::endl);

    std::vector<std::string>::iterator i1,i2,i3,i4;
    for (i1=iname.begin(),i2=ivalue.begin(),i3=iconn.begin(),i4=istatus.begin();
	     i1!=iname.end(),i2!=ivalue.end(),i3!=iconn.end(),i4!=istatus.end();
	   ++i1,++i2,++i3,++i4)
     {
	    std::string name(*i1);
	    name += "'";
	    name.append(1+namelmax-name.size(),' ');
	    std::string value(*i2);
	    value += "'";
	    value.append(1+valuelmax-value.size(),' ');
	    if (i3->size()) 
	      bbtkBlackBoxMessage("help",1,"    '"<<name<<" = '"<<value<<" <-- '" <<*i3<<"'");
	    else 
	      bbtkBlackBoxMessage("help",1,"    '"<<name<<" = '"<<value);
	    bbtkBlackBoxMessage("help",1," ["<<*i4<<"]"<<std::endl);
     }

    if (oname.size()) 
       bbtkBlackBoxMessage("help",1," * Outputs : "<<std::endl);
    else 
       bbtkBlackBoxMessage("help",1," * No outputs"<<std::endl);

    std::vector<std::vector<std::string> >::iterator i5;

    for (i1=oname.begin(),i2=ovalue.begin(),i5=oconn.begin(),i4=ostatus.begin();
	     i1!=oname.end(),i2!=ovalue.end(),i5!=oconn.end(),i4!=ostatus.end();
	   ++i1,++i2,++i4,++i5)
    {
	    std::string name(*i1);
	    name += "'";
	    name.append(1+namelmax-name.size(),' ');
	    std::string value(*i2);
	    value += "'";
	    value.append(1+valuelmax-value.size(),' ');
	    if (!(*i5).size())
	      bbtkBlackBoxMessage("help",1,"    '"<<name<<" = '"<<value);
	    else 
	    {
	       std::string pref = "    '"+name+" = '"+value;
	       for (i3=i5->begin();i3!=i5->end();++i3)
	       {
		      bbtkBlackBoxMessage("help",1,pref<<" --> '"<<*i3<<"'");
		      pref.replace(0,pref.size(),pref.size(),' ');
	       }
	    }
	    bbtkBlackBoxMessage("help",1," ["<<*i4<<"]"<<std::endl);
    }

   }
  //=========================================================================


















  static bool bbmgGlobalProcessingExecutionList = false;

  //=========================================================================
   void BlackBox::bbGlobalProcessExecutionList()
   {   
     bbtkDebugMessage("process",3,
		      "=> BlackBox::bbGlobalProcessExecutionList()"
		      <<std::endl);    
     if (bbmgGlobalProcessingExecutionList) 
       {
	 bbtkDebugMessage("process",3,"BlackBox::bbGlobalProcessExecutionList() reentered !");
	 return;
       }
     bbmgGlobalProcessingExecutionList = true;

     std::set<BlackBox::WeakPointer>::iterator i; 
     while (bbmgExecutionList.size()>0)
       {
	 i = bbmgExecutionList.begin();
	 BlackBox::WeakPointer p = *i;
	 bbmgExecutionList.erase(i);
	 if (p.lock())
	   {
	     bbtkDebugMessage("process",4,
			      " -> Executing '"<<
			      p.lock()->bbGetName()<<"'"<<std::endl);
	     p.lock()->bbExecute(true);
	   }
	 else 
	   {
	     bbtkGlobalError("Strange error in BlackBox::bbGlobalProcessExecutionList() : Weak bb pointer in bbmgExecutionList is no more valid...");
	   }
       }
     
     bbmgExecutionList.clear();
     bbtkDebugMessage("process",3,
			 "<= BlackBox::bbGlobalProcessExecutionList()"
			 <<std::endl);     
     
     bbmgGlobalProcessingExecutionList = false;
     
   }
  //=========================================================================

    bool BlackBox::bbGlobalGetSomeBoxExecuting()
	{ 
		return bbmgSomeBoxExecuting; 
	}

    void BlackBox::bbGlobalSetSomeBoxExecuting(bool b) 
	{ 
		bbmgSomeBoxExecuting = b; 
	}

    void BlackBox::bbGlobalSetFreezeExecution(bool b) 
	{ 
		bbmgFreezeExecution = b;
	}

    bool BlackBox::bbGlobalGetFreezeExecution() 
	{ 
		return bbmgFreezeExecution; 
	}

  void BlackBox::bbGlobalAddToExecutionList( BlackBox::Pointer b )
  {  
    bbtkDebugMessage("process",3,"* bbGlobalAddToExecutionList("<<b->bbGetName()<<")"<<std::endl);
    if (bbmgGlobalProcessingExecutionList) 
      {
	bbtkDebugMessage("process",3,"bbGlobalAddToExecutionList called inside bbGlobalProcessExecutionList !");
      }
    bbmgExecutionList.insert(b); 
  } 


   //=========================================================================

  //=========================================================================
  void BlackBox::Check(bool recursive)
  {
    bbtkBlackBoxMessage("debug",1,"*** Checking"
			<<" ... OK"<<std::endl);
  }
  //=========================================================================




}  // EO namespace bbtk

// EOF

