/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkBlackBoxDescriptor.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.24 $
=========================================================================*/



/**
 *  \file 
 *  \brief Class bbtk::BlackBoxDescriptor : (abstract) describes a BlackBox (name, description, author) and is able to create an instance of it.
 */
#include "bbtkBlackBoxDescriptor.h"
#include "bbtkMessageManager.h"
#include "bbtkPackage.h"
#include "bbtkUtilities.h"
#include "bbtkAtomicBlackBoxDescriptor.h"
#include "bbtkWxBlackBox.h"

namespace bbtk
{

  typedef Package::Pointer PackagePointer;

#define bbtkDMessage(key,level,mess) \
  bbtkMessage(key,level,"["<<GetFullTypeName()<<"] "<<mess)
#define bbtkDDebugMessage(key,level,mess)	\
  bbtkDebugMessage(key,level,"["<<GetFullTypeName()<<"] "<<mess)


  //=========================================================================
  /// Default ctor
  BlackBoxDescriptor::BlackBoxDescriptor()  
    : mTypeName("Unknown"), 
      mDescription(""), 
      mAuthor(""),
      mCategory(""),
      mKind(STANDARD),
	  mTypeOfScript(TS_BINARY),
      mPackage()
  {
    bbtkDDebugMessage("object",4,
		     "==> BlackBoxDescriptor()"<<std::endl);
    bbtkDDebugMessage("object",4,
		     "<== BlackBoxDescriptor()"<<std::endl);
  }
  //=========================================================================

  //=========================================================================
  /// Default ctor
  BlackBoxDescriptor::BlackBoxDescriptor(const BlackBoxDescriptor&)  
  {
	  mScriptFileName="";
  }
  //=========================================================================
  /// Dtor
  BlackBoxDescriptor::~BlackBoxDescriptor()
  {
    bbtkDDebugMessage("object",4,
		     "==> ~BlackBoxDescriptor()"
		     <<std::endl);

    // deletes all I/O descriptors
    InputDescriptorMapType::iterator i;
    for (i=mInput.begin(); i!=mInput.end(); ++i) delete i->second;
    OutputDescriptorMapType::iterator o;
    for (o=mOutput.begin(); o!=mOutput.end(); ++o) delete o->second;

    bbtkDDebugMessage("object",4,
		      "<== ~BlackBoxDescriptor()"
		      <<std::endl);
  }
  //=========================================================================

  /*
  //=======================================================================
  /// Release
  void BlackBoxDescriptor::Release(BlackBoxDescriptor::WeakPointer desc)
  {
    bbtkMessage("object",2,"==> BlackBoxDescriptor::Release('"
		<<desc.lock()->GetTypeName()<<"')"<<std::endl);
    long c = desc.use_count();
    bbtkMessage("object",3," - ref count = "<<c<<std::endl);
    // If only one ref 
    if ((c == 1) && (desc.lock()->mPackage))
      {
	bbtkMessage("object",2," --> No more instance alive = releasing from package"<<std::endl);
	
	Package::WeakPointer pack = desc.lock()->mPackage;
	Package::ReleaseBlackBoxDescriptor(pack,desc);
      }
    else 
      {
	bbtkMessage("object",2," --> Still some instances alive = Keeping it alive"<<std::endl);
      }    
    bbtkMessage("object",2,"<== BlackBoxDescriptor::Release('"
		<<desc.lock()->GetTypeName()<<"')"<<std::endl);
 
  }
  //=========================================================================
  */

  /*
  //=========================================================================
  /// Dtor
  void BlackBoxDescriptor::UnReference()
  {
    bbtkDDebugMessageInc("kernel",1,
			"BlackBoxDescriptor::UnReference() ["
			<<mTypeName<<"] #"<<mRefCount-1<<std::endl);
    mRefCount--;
    if (mRefCount<=0) 
      {
	bbtkDDebugMessage("kernel",1,"--> Destructing BlackBoxDescriptor ["<<mTypeName<<"]"<<std::endl);
	delete this;
      }
  }
  //=========================================================================
  */
  //=========================================================================
  /// Check
  void BlackBoxDescriptor::Check(bool) const
  {
    
  }
  //=========================================================================

  //=========================================================================
  /// Adds the string to the BlackBox description
  void BlackBoxDescriptor::AddToDescription( const std::string& s, bool clear)
  {
    bbtkDDebugMessage("kernel",9,"BlackBoxDescriptor::AddToDescription(\""
		      <<s<<"\")"<<std::endl);
    if (clear) mDescription = s; 
    else mDescription += s;
   }
  //=========================================================================

  //=========================================================================   
  /// Adds the string to the BlackBox author list
  void BlackBoxDescriptor::AddToAuthor( const std::string& s, bool clear)
  {
    bbtkDDebugMessage("kernel",9,"BlackBoxDescriptor::AddToAuthor(\""
		      <<s<<"\")"
		      <<std::endl);
    if (clear) mAuthor = s;
    else mAuthor += s;
  }
  //=========================================================================

  //=========================================================================  
  /// Adds the string to the BlackBox category list
  void BlackBoxDescriptor::AddToCategory( const std::string& s, bool clear)
  {
    bbtkDDebugMessage("kernel",9,"BlackBoxDescriptor::AddToCategory(\""
		      <<s<<"\")"
		      <<std::endl);  
    if (clear) mCategory = s;
    else mCategory += s;
    mCategory += ";";
  }  
  //=========================================================================
  
  //=========================================================================
  const BlackBoxDescriptor::InputDescriptor* 
  BlackBoxDescriptor::GetInputDescriptor(const std::string & name) const
  {
    bbtkDDebugMessage("kernel",9,"BlackBoxDescriptor::GetInputDescriptor('"
		      <<name<<"')"<<std::endl);

    InputDescriptorMapType::const_iterator i;
    i = mInput.find(name);
    if ( i == mInput.end() ) 
    {
	    bbtkError("input '"<<name<<"' does not exist");
    }

    return i->second;
  }
  //=========================================================================

  //=========================================================================
  const BlackBoxDescriptor::OutputDescriptor* 
  BlackBoxDescriptor::GetOutputDescriptor(const std::string & name) const
  {
    bbtkDDebugMessage("kernel",9,"BlackBoxDescriptor::GetOutputDescriptor('"
		      <<name<<"')"<<std::endl);

    OutputDescriptorMapType::const_iterator i;
    i = mOutput.find(name);
    if ( i == mOutput.end() ) 
    {
      bbtkError("output '"<<name<<"' does not exist");
    }

    return i->second;
  }
  //=========================================================================

  //=========================================================================
  void BlackBoxDescriptor::GetHelp(bool full) const
  {
    bbtkDDebugMessage("kernel",9,"BlackBoxDescriptor::GetHelp()"
		      <<std::endl);
    
    bbtkMessage("help",1,"Black Box <"<<GetFullTypeName()<<">"<<std::endl);
    bbtkMessage("help",1," "              <<GetDescription()<<std::endl);
    bbtkMessage("help",1," By : "         <<GetAuthor()     <<std::endl);
    bbtkMessage("help",1," Categories : " <<GetCategory()    <<std::endl);
    if (mInput.size()) 
      bbtkMessage("help",1," * Inputs : "<<std::endl);
    else 
      bbtkMessage("help",1," * No inputs"<<std::endl);
    InputDescriptorMapType::const_iterator i;
    unsigned int namelmax = 0;
    unsigned int typelmax = 0;
    unsigned int natlmax = 0;
    for ( i = mInput.begin();  i != mInput.end(); ++i ) 
    {
	   if (i->second->GetName().size()>namelmax) 
	     namelmax = i->second->GetName().size();
	   if (i->second->GetHumanTypeName().size()>typelmax) 
	     typelmax = i->second->GetHumanTypeName().size();
	   if (i->second->GetNature().size()>natlmax) 
	     natlmax = i->second->GetNature().size();
    }
    OutputDescriptorMapType::const_iterator o;
    for ( o = mOutput.begin();  o != mOutput.end(); ++o ) 
    {
	   if (o->second->GetName().size()>namelmax) 
	     namelmax = o->second->GetName().size();
	   if (o->second->GetHumanTypeName().size()>typelmax) 
	     typelmax = o->second->GetHumanTypeName().size();
	   if (o->second->GetNature().size()>natlmax) 
	     natlmax = o->second->GetNature().size();
    }
    //
    for ( i = mInput.begin();  i != mInput.end(); ++i ) 
    {
	   std::string name(i->second->GetName());
	   name += "'";
	   name.append(1+namelmax-name.size(),' ');
	   std::string type(i->second->GetHumanTypeName());
	   type += ">";
	   type.append(1+typelmax-type.size(),' ');
	   std::string nature(i->second->GetNature());
	   nature += "]";
	   nature.append(1+natlmax-nature.size(),' ');
	   bbtkMessage("help",1,
		       "    '"<<name
		       <<" <"<<type
		       <<" ["<<nature
		       <<" : "<<i->second->GetDescription()<<std::endl);
    }
    if (mOutput.size()) 
      bbtkMessage("help",1," * Outputs : "<<std::endl);
    else 
      bbtkMessage("help",1," * No outputs"<<std::endl);
    for ( o = mOutput.begin();  o != mOutput.end(); ++o ) 
    {
	   std::string name(o->second->GetName());
	   name += "'";
	   name.append(1+namelmax-name.size(),' ');
	   std::string type(o->second->GetHumanTypeName());
	   type += ">";
	   type.append(1+typelmax-type.size(),' ');
	   std::string nature(o->second->GetNature());
	   nature += "]";
	   nature.append(1+natlmax-nature.size(),' ');
	   bbtkMessage("help",1,
		    "    '"<<name
		       <<" <"<<type
		       <<" ["<<nature
		       <<" : "<<o->second->GetDescription()<<std::endl);
      }
   
  
  }
  //=========================================================================
   
  //=========================================================================
  /// Returns the full name of the **TYPE** of the black box (+package name)
  std::string BlackBoxDescriptor::GetFullTypeName() const
  {
    if (GetPackage()!=0) return GetPackage()->GetName() + "::" + mTypeName;
    return "::" + mTypeName;
  }
  //=========================================================================
 	 void BlackBoxDescriptor::GetBoxesInside (NodeTreeC& tree, int cont){
		
	}
  //=========================================================================
  void BlackBoxDescriptor::InsertHtmlHelp ( std::ofstream& s, 
					    int detail, int level,
					    const std::string& output_dir,
					    bool relative_link )
  {
    bbtkDDebugMessage("kernel",9,"BlackBoxDescriptor::InsertHtmlHelp()"
		      <<std::endl);
    
    //-------------
    // General info 
    std::string name = GetTypeName();
    Utilities::html_format(name);

    (s) << "<p><hr>\n";
    (s) << "<a name=\""<<name<<"\"></a>\n";
    (s) << //"Top:&nbsp;
      "<a rel=\"top\" accesskey=\"t\" href=\"#Top\">Top</a>\n";
    // (s) << "Previous:&nbsp;<a rel="previous" accesskey="p" href="#dir">(dir)</a>,
    // (s) << "Up:&nbsp;<a rel="up" accesskey="u" href="#dir">(dir)</a>
    (s) << "<h2 class=\"section\">"<<name<<"</h2>\n";


    std::string descr = GetDescription();
    //Utilities::html_format(descr);
    std::string author = GetAuthor();
    Utilities::html_format(author);

    std::vector<std::string> categories;
    // Split the category string 
    std::string delimiters = ";,";
    Utilities::SplitString(GetCategory(),
			   delimiters,categories);

    
    (s) << "<p><TABLE cellspacing=0  cellpadding=3>\n";
    (s) << "<TR><TD style='vertical-align: top;'><b> Description </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'> " 
      << descr << "</TD></TR>\n";
    (s) << "<TR><TD style='vertical-align: top;'><b> Author(s) </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  " 
      << author << "</TD></TR>\n";
    (s) << "<TR><TD style='vertical-align: top;'><b> Category(s) </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  ";
    std::vector<std::string>::iterator ci;
    for (ci=categories.begin(); ci!=categories.end(); ++ci)
      {
        s << "<a href=\"../index-category.html#"<< *ci <<"\">" << *ci 
	  << "</a>&nbsp;\n";
      }
    s << "</TD></TR>\n";      

    (s) << "<TR><TD style='vertical-align: top;'><b> To use it </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  include " 
	<< GetPackage()->GetName() << "</TD></TR>\n";
    (s) << "</TABLE>\n";

    //-------------
    // Graph
    //i->second->InsertHTMLGraph( &s , detail,level,dir);
    
    //-------------
    // Inputs
    std::string titlecol("#BBBBFF");
    std::string usercol("#FFFFFF");
    std::string ubbcol("#DDFFFF");
    std::string wxbbcol("#EEFFFF");


    //	(s) << "<h3 class=\"subsection\">Inputs</h3>\n";
    (s) << "<p><TABLE border=1 cellspacing=0 cellpadding=3>\n";
    (s) << "<TR><TD colspan=3 align=center bgcolor=\""<<titlecol
      <<"\">Inputs</TD></TR>\n";

    std::vector<std::string> user_defined;
    std::vector<std::string> ubb_defined;
    std::vector<std::string> wxbb_defined;

    const BlackBoxDescriptor::InputDescriptorMapType& imap = 
      GetInputDescriptorMap();
    InputDescriptorMapType::const_iterator in;
    for ( in = imap.begin();  in != imap.end(); ++in ) 
    {
      // Skips system-defined inputs
      std::string col(usercol);
      int iotype = 0;
      if (in->second->GetCreatorTypeInfo() == 
	  typeid(AtomicBlackBoxDescriptor))
	{
	  col = ubbcol; 
	  iotype = 1;
	}
#ifdef USE_WXWIDGETS
      else if (in->second->GetCreatorTypeInfo() == 
	       typeid(WxBlackBoxDescriptor))
	{
	  col = wxbbcol; 
	  iotype = 2;
	}
#endif

      std::string name(in->second->GetName());
      Utilities::html_format(name);
      
      std::string type("<");
      type += in->second->GetTypeName();    
      type += ">";
      Utilities::html_format(type);
      
      std::string descr(in->second->GetDescription());
      //Utilities::html_format(descr);

/*EED 10/11/2009
      std::string out = 
	"<TR><TD style='vertical-align: top;' bgcolor=\"" + col
	+"\"><B><PRE> "+name+" </PRE></B></TD>"
	+ "<TD style='vertical-align: top;' bgcolor=\""+col
	+"\"><I><PRE> "+type+" </PRE></I></TD>"
	+ "<TD style='vertical-align: top;' bgcolor=\""+col
	+"\">"+descr+"</TD></TR>\n";
*/      

      std::string out = 
	"<TR><TD style='vertical-align: top;' bgcolor=\"" + col
	+"\"><B><PRE> "+name+" </PRE></B></TD>"
	+ "<TD style='vertical-align: top;' bgcolor=\""+col
	+"\"><I><PRE> "+descr+" </PRE></I></TD>"
	+ "<TD style='vertical-align: top;' bgcolor=\""+col
	+"\">"+type+"</TD></TR>\n";

      if (iotype==0) user_defined.push_back(out);
      else if (iotype==1) ubb_defined.push_back(out);
      else if (iotype==2) wxbb_defined.push_back(out);
      
    }

    std::vector<std::string>::iterator hi;
    for (hi=user_defined.begin();hi!=user_defined.end();++hi) s << *hi;
    for (hi=wxbb_defined.begin();hi!=wxbb_defined.end();++hi) s << *hi;
    for (hi=ubb_defined.begin();hi!=ubb_defined.end();++hi) s << *hi;

    user_defined.clear();
    ubb_defined.clear();
    wxbb_defined.clear();

    //-------------
    // Outputs
    //	(s) << "<h3 class=\"subsection\">Outputs</h3>\n";
    //	(s) << "<TABLE border=1 cellspacing=0>\n";
    (s) << "<TR><TD colspan=3 align=center bgcolor=\""<<titlecol
      <<"\">Outputs</TD></TR>\n";
    
    const BlackBoxDescriptor::OutputDescriptorMapType& omap = 
      GetOutputDescriptorMap();
    
    BlackBoxDescriptor::OutputDescriptorMapType::const_iterator o;
    
    for ( o = omap.begin();  o != omap.end(); ++o ) 
      {
	std::string col(usercol);
	int iotype = 0;
	if (o->second->GetCreatorTypeInfo() == 
	    typeid(AtomicBlackBoxDescriptor))
	  {
	    col = ubbcol; 
	    iotype = 1;
	  }
#ifdef USE_WXWIDGETS
	else if (o->second->GetCreatorTypeInfo() == 
		 typeid(WxBlackBoxDescriptor))
	  {
	    col = wxbbcol; 
	    iotype = 2;
	  }
#endif
	
	std::string name(o->second->GetName());
	Utilities::html_format(name);
	
	std::string type("<");
	type += o->second->GetTypeName();    
	type += ">";
	Utilities::html_format(type);
	
	std::string descr(o->second->GetDescription());
	//Utilities::html_format(descr);
	
/*EED 10/11/2009
	std::string out = 
	  "<TR><TD style='vertical-align: top;' bgcolor=\"" + col
	  +"\"><B><PRE> "+name+" </PRE></B></TD>"
	  + "<TD style='vertical-align: top;' bgcolor=\""+col
	  +"\"><I><PRE> "+type+" </PRE></I></TD>"
	  + "<TD style='vertical-align: top;' bgcolor=\""+col
	  +"\">"+descr+"</TD></TR>\n";
*/
	std::string out = 
	  "<TR><TD style='vertical-align: top;' bgcolor=\"" + col
	  +"\"><B><PRE> "+name+" </PRE></B></TD>"
	  + "<TD style='vertical-align: top;' bgcolor=\""+col
	  +"\"><I><PRE> "+descr+" </PRE></I></TD>"
	  + "<TD style='vertical-align: top;' bgcolor=\""+col
	  +"\">"+type+"</TD></TR>\n";

	if (iotype==0) user_defined.push_back(out);
	else if (iotype==1) ubb_defined.push_back(out);
	else if (iotype==2) wxbb_defined.push_back(out);
	
      }
    
    for (hi=user_defined.begin();hi!=user_defined.end();++hi) s << *hi;
    for (hi=wxbb_defined.begin();hi!=wxbb_defined.end();++hi) s << *hi;
    for (hi=ubb_defined.begin();hi!=ubb_defined.end();++hi) s << *hi;

    (s) << "</TABLE>\n";
    
    //------------
    // End


   }
  //=========================================================================
 
	
	//==========================================================================
	void BlackBoxDescriptor::SetTypeOfScript_Application()
	{
		SetTypeOfScript(TS_SCRIPT_APPLICATION);
	}	 
	//==========================================================================
	
		
	//==========================================================================
	bool BlackBoxDescriptor::IsTypeOfScript_Application()
	{
		return (GetTypeOfScript()==TS_SCRIPT_APPLICATION);
	}
	//==========================================================================

	
	//==========================================================================
	bool BlackBoxDescriptor::IsTypeOfScript_ComplexBox()
	{
		return (GetTypeOfScript()==TS_SCRIPT_COMPLEXBOX);
	}
	//==========================================================================
	
}
