/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkComplexBlackBoxDescriptor.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.23 $
=========================================================================*/

                                                                    
  

/**
 *  \file 
 *  \brief Class bbtk::ComplexBlackBoxDescriptor : describes a ComplexBlackBox (constituents, connections) and is able to create an instance of it.
 */
#include "bbtkComplexBlackBoxDescriptor.h"
#include "bbtkComplexBlackBox.h"
//#include "bbtkFactory.h"
#include "bbtkMessageManager.h"
#include "bbtkUtilities.h"

#define bbtkDMessage(key,level,mess) \
  bbtkMessage(key,level,"["<<GetFullTypeName()<<"] "<<mess)
#define bbtkDDebugMessage(key,level,mess)	\
  bbtkDebugMessage(key,level,"["<<GetFullTypeName()<<"] "<<mess)

namespace bbtk
{
   //=======================================================================
  /// 
  ComplexBlackBoxDescriptor::Pointer 
  ComplexBlackBoxDescriptor::New(const std::string& name)
  {
    bbtkDebugMessage("object",1,"##> ComplexBlackBoxDescriptor::New(\""<<name<<"\")"<<std::endl);
    ComplexBlackBoxDescriptor::Pointer p = 
      MakePointer(new ComplexBlackBoxDescriptor(name));
    bbtkDebugMessage("object",1,"<## ComplexBlackBoxDescriptor::New(\""<<name<<"\")"<<std::endl);
    return p;
  }
  //=======================================================================

  //=======================================================================
  /// Default ctor
  ComplexBlackBoxDescriptor::ComplexBlackBoxDescriptor(const std::string& name)
  {
    bbtkDDebugMessage("object",2,"==> ComplexBlackBoxDescriptor(\""<<name<<"\")"<<std::endl);
	SetTypeOfScript(TS_SCRIPT_COMPLEXBOX);
    SetTypeName(name);
    AddToCategory("complex box");
    mPrototype = ComplexBlackBox::New(name+std::string("Prototype"),
				      MakePointer(this,true));
    mPrototype->SetAsPrototype();
    bbtkDDebugMessage("object",2,"<== ComplexBlackBoxDescriptor(\""<<name<<"\")"<<std::endl);
  }
  //=======================================================================



  //=======================================================================
  /// Default dtor
  ComplexBlackBoxDescriptor::~ComplexBlackBoxDescriptor()
  {
    bbtkDDebugMessage("object",2,"==> ~ComplexBlackBoxDescriptor()"<<std::endl);
    mPrototype.reset();
    bbtkDDebugMessage("object",2,"<== ~ComplexBlackBoxDescriptor()"<<std::endl);
  }
  //=======================================================================

  //=========================================================================
  /// Check
  void ComplexBlackBoxDescriptor::Check(bool recursive) const
  {
    mPrototype->Check(recursive);
  }
  //=========================================================================

  //=======================================================================
  /// Creates an instance of name <name> of the ComplexBlackBox of which this is the descriptor 
  BlackBox::Pointer 
  ComplexBlackBoxDescriptor::NewBlackBox(const std::string& name)
  {
    bbtkDDebugMessage("kernel",5,
		      "ComplexBlackBoxDescriptor::NewBlackBox(\""
		      <<name<<"\")"
		      <<std::endl);
    
    return mPrototype->bbClone(name);


  }
  //=======================================================================

  /*
  //=======================================================================
  /// Release
  void ComplexBlackBoxDescriptor::Release(bool release_package)
  {
  }
  //=======================================================================
  */

  //=======================================================================
  /// Adds a black box to the complex box
  void ComplexBlackBoxDescriptor::Add ( const std::string& type,
					const std::string& name
					)
  {
    bbtkDDebugMessage("kernel",5,
			"ComplexBlackBoxDescriptor::Add(\""
			<<type<<"\",\""<<name<<"\")"
			<<std::endl);
    
    // 
    if (!GetFactory()) 
      { 
	bbtkError("ComplexBlackBoxDescriptor::Add : no factory set");
      }
    
    // Verify that a box with the same name does not exist already
    if ( mPrototype->bbUnsafeGetBlackBox( name ) ) 
      {
	bbtkError("a black box \""<<name<<"\" already exists");
      }
    // ok : create new one
    mPrototype->bbAddBlackBox ( GetFactory()->NewBlackBox(type,name) );


  }
  //=======================================================================
  
  //=======================================================================
  /// Removes a black box from the complex box
  void ComplexBlackBoxDescriptor::Remove( const std::string& name, 
					  bool remove_connections)
  {    
    mPrototype->bbRemoveBlackBox(name,remove_connections);
  }
  //=======================================================================


  //=======================================================================
  /// Adds a black box to the execution list 
  void ComplexBlackBoxDescriptor::AddToExecutionList ( const std::string& box)
  {
    bbtkDDebugMessage("kernel",5,
		      "ComplexBlackBoxDescriptor::AddToExecutionList(\""
		      <<box<<"\""
		      <<std::endl);
    // Verify that the box exists
    BlackBox::Pointer b = mPrototype->bbUnsafeGetBlackBox( box ); 
    if ( !b ) 
      {
	bbtkError("the black box \""<<box<<"\" does not exist");
      }
    // ok 
    mPrototype->bbAddToExecutionList ( box  );


    }


  //=======================================================================
  /// Connects two black boxes of the complex box
  void ComplexBlackBoxDescriptor::Connect ( const std::string& from,
					    const std::string& output,
					    const std::string& to,
					    const std::string& input
					    )
  {
    bbtkDDebugMessage("kernel",5,
			"ComplexBlackBoxDescriptor::Connect(\""
			<<from<<"\",\""<<output<<"\",\""
			<<to<<"\",\""<<input
			<<"\")"
			<<std::endl);
  // 
    if (!GetFactory()) 
      { 
	bbtkError("ComplexBlackBoxDescriptor::Connect : no factory set");
      }
    

  // Verify that a box with the same name does not exist already
    BlackBox::Pointer bbfrom = mPrototype->bbGetBlackBox( from );
    if ( !bbfrom ) 
      {
	bbtkError("the black box \""<<from<<"\" does not exist");
      }
    BlackBox::Pointer bbto = mPrototype->bbGetBlackBox( to );
    if ( !bbto ) 
      {
	bbtkError("the black box \""<<to<<"\" does not exist");
      }
    
    Connection::Pointer c 
      = GetFactory()->NewConnection( bbfrom, output, bbto, input );

    mPrototype->bbAddConnection(c);


  }
  //=======================================================================


  //=======================================================================
  /// Defines an input of the complex box
  void ComplexBlackBoxDescriptor::DefineInput ( const std::string& name,
						const std::string& box,
						const std::string& input,
						const std::string& help)
  {
    bbtkDDebugMessage("kernel",5,
			"ComplexBlackBoxDescriptor::DefineInput(\""
			<<name<<"\",\""<<box<<"\",\""
			<<input<<"\",\""<<help
			<<"\")"
			<<std::endl);

    BlackBox::Pointer bb = mPrototype->bbGetBlackBox( box );
    if ( !bb ) 
      {
	bbtkError("the black box \""<<box<<"\" does not exist");
      }

    if (!bb->bbHasInput(input) )
      {
	bbtkError("the black box \""<<box<<"\" does not have input \""
		  <<input<<"\"");
      }
    
    const BlackBoxInputDescriptor* d = 
      bb->bbGetDescriptor()->GetInputDescriptor(input);
    AddInputDescriptor ( new ComplexBlackBoxInputDescriptor 
			 ( typeid(ComplexBlackBoxDescriptor),
			   name,
			   help,
			   d->GetNature(),
			   box,
			   input,
			   d->GetTypeInfo()));
    
    

  }
  //=======================================================================

  //=======================================================================
  /// Defines an output of the complex box
  void ComplexBlackBoxDescriptor::DefineOutput ( const std::string& name,
						 const std::string& box,
						 const std::string& output,
						 const std::string& help)
  {
    bbtkDDebugMessage("kernel",5,
			"ComplexBlackBoxDescriptor::DefineOutput(\""
			<<name<<"\",\""<<box<<"\",\""
			<<output<<"\",\""<<help
			<<"\")"
			<<std::endl);

    BlackBox::Pointer bb = mPrototype->bbGetBlackBox( box );
    if ( !bb ) 
      {
	bbtkError("the black box \""<<box<<"\" does not exist");
      }

    if (!bb->bbHasOutput(output) )
      {
	bbtkError("the black box \""<<box<<"\" does not have output \""
		  <<output<<"\"");
      }
    
    const BlackBoxOutputDescriptor* d = 
      bb->bbGetDescriptor()->GetOutputDescriptor(output);
    AddOutputDescriptor ( new ComplexBlackBoxOutputDescriptor 
			  ( typeid(ComplexBlackBoxDescriptor),
			    name,
			    help,
			    d->GetNature(),
			    box,
			    output,
			    d->GetTypeInfo()));
    
    

  }
  //=======================================================================

  //=======================================================================
  void ComplexBlackBoxDescriptor::PrintBlackBoxes()
  {
    mPrototype->bbPrintBlackBoxes(); 
  }
  //=======================================================================


  //=======================================================================
  void ComplexBlackBoxDescriptor::InsertHTMLGraph( std::ofstream& s , 
						   int detail, int level, 
						   const std::string& output_dir, bool relative_link )   
  {
    this->mPrototype->bbInsertHTMLGraph( s, 
					 detail, level, 
					 false, 
					 output_dir,
					 relative_link );
  }
  //=======================================================================

  //=========================================================================
  void ComplexBlackBoxDescriptor::InsertHtmlHelp ( std::ofstream& s, 
						   int detail, int level,
						   const std::string& output_dir, bool relative_link)
  {
    bbtkDDebugMessage("kernel",9,
		      "ComplexBlackBoxDescriptor::InsertHtmlHelp()"
		       <<std::endl);
    
    //-------------
    // General info 
    std::string name = GetTypeName();
    Utilities::html_format(name);

    //   std::ofstream* s = &s1;

    (s) << "<p><hr>\n";
    (s) << "<a name=\""<<name<<"\"></a>\n";
    (s) << //"Top:&nbsp;
      "<a rel=\"top\" accesskey=\"t\" href=\"#Top\">Top</a>\n";
    // (s) << "Previous:&nbsp;<a rel="previous" accesskey="p" href="#dir">(dir)</a>,
    // (s) << "Up:&nbsp;<a rel="up" accesskey="u" href="#dir">(dir)</a>
    (s) << "<h2 class=\"section\">"<<name<<"</h2>\n";


    std::string descr = GetDescription();
    //Utilities::html_format(descr);
    
    std::string author = GetAuthor();
    Utilities::html_format(author);
    
    std::vector<std::string> categories;
    // Split the category string 
    std::string delimiters = ";,";
    Utilities::SplitString(GetCategory(),
			   delimiters,categories);

        
    (s) << "<p><TABLE cellspacing=0  cellpadding=3>\n";
    (s) << "<TR><TD style='vertical-align: top;'><b> Description </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'> " 
        << descr << "</TD></TR>\n";

    (s) << "<TR><TD style='vertical-align: top;'><b> Author(s) </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  " 
        << author << "</TD></TR>\n";

    (s) << "<TR><TD style='vertical-align: top;'><b> Category(s) </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  ";
    std::vector<std::string>::iterator ci;
    for (ci=categories.begin(); ci!=categories.end(); ++ci)
      {
        s << "<a href=\"../index-category.html#"<< *ci <<"\">" << *ci 
	  << "</a>&nbsp;\n";
      }
    s << "</TD></TR>\n";      
    std::string inc = GetScriptFileName();
    if (inc.size()>0) 
      {
	s << "<TR><TD style='vertical-align: top;'><b> To use it </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'> include ";
	// s << inc << "&nbsp;&nbsp;<a href=\"../../../bbs/"<<inc<<"\">[source]</a>";
	// LG TODO : USE PACKAGE BBS PATH
	s << inc << "&nbsp;&nbsp;<a href=\""<<inc<<"\">[source]</a>";
	s << "</TD></TR>\n";
	
      }
    
    const ComplexBlackBox::BlackBoxMapType& B = mPrototype->bbGetBlackBoxMap();
	
    if (B.size()) 
    {
	   (s) << "<TR><TD style='vertical-align: top;'><b> Uses </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  ";

	   std::set<BlackBoxDescriptor::Pointer> pdeps;
	   ComplexBlackBox::BlackBoxMapType::const_iterator b;
	   for ( b = B.begin(); b != B.end(); ++b ) 
	   {
	     BlackBoxDescriptor::Pointer d = b->second->bbGetDescriptor();
	      if (pdeps.find(d) != pdeps.end()) 
            continue;
	      pdeps.insert(d);

	      Package::Pointer p = d->GetPackage();
	    
	      std::string name = b->second->bbGetTypeName();
				
	      std::string url;
	      if (relative_link) 
	         url = p->GetDocRelativeURL();
	      else 
	         url = p->GetDocURL();
		   
	      s << "<a href=\"" <<url<<"#"<<name<<"\">" 
	        << p->GetName()<<"::"<<name<<"</a>\n";
	    }	// for
	
	    (s) << "</TD></TR>\n";

     } // If B.size

     (s) << "</TABLE>\n";

 
   //-------------
    // Graph
    InsertHTMLGraph( s , detail,level, output_dir, relative_link);
    
    //-------------
    // Inputs
    std::string col("#CCCCFF");
    
    //	(s) << "<h3 class=\"subsection\">Inputs</h3>\n";
    (s) << "<p><TABLE border=1 cellspacing=0 cellpadding=3>\n";
    (s) << "<TR><TD colspan=3 align=center bgcolor=\""<<col
      <<"\">Inputs</TD></TR>\n";
    const BlackBoxDescriptor::InputDescriptorMapType& imap = 
      GetInputDescriptorMap();
    
    InputDescriptorMapType::const_iterator in;
    
    for ( in = imap.begin();  in != imap.end(); ++in ) 
      {
	std::string name(in->second->GetName());
	Utilities::html_format(name);
	
	std::string type("<");
	type += in->second->GetTypeName();    
	type += ">";
	Utilities::html_format(type);
	
	std::string descr(in->second->GetDescription());
	//Utilities::html_format(descr);

/*EED 10/11/2009
	(s) << "<TR><TD style='vertical-align: top;'><B><PRE> "<<name<<" </PRE></B></TD>"
	  << "<TD style='vertical-align: top;'><I><PRE> "<<type<<" </PRE></I></TD>"
	  << "<TD style='vertical-align: top;'>"<<descr<<"</TD></TR>\n";
*/

	(s) << "<TR><TD style='vertical-align: top;'><B><PRE> "<<name<<" </PRE></B></TD>"
	  << "<TD style='vertical-align: top;'><I><PRE> "<<descr<<" </PRE></I></TD>"
	  << "<TD style='vertical-align: top;'>"<<type<<"</TD></TR>\n";
	
      }
    //	(s) << "</TABLE>\n";
    
    
    //-------------
    // Outputs
    //	(s) << "<h3 class=\"subsection\">Outputs</h3>\n";
    //	(s) << "<TABLE border=1 cellspacing=0>\n";
    (s) << "<TR><TD colspan=3 align=center bgcolor=\""<<col
      <<"\">Outputs</TD></TR>\n";
    
    const BlackBoxDescriptor::OutputDescriptorMapType& omap = 
      GetOutputDescriptorMap();
    
    BlackBoxDescriptor::OutputDescriptorMapType::const_iterator o;
    
    for ( o = omap.begin();  o != omap.end(); ++o ) 
      {
	std::string name(o->second->GetName());
	Utilities::html_format(name);
	
	std::string type("<");
	type += o->second->GetTypeName();    
	type += ">";
	Utilities::html_format(type);
	
	std::string descr(o->second->GetDescription());
	//Utilities::html_format(descr);
	
/*EED 10/11/2009
	(s) << "<TR><TD style='vertical-align: top;'><B><PRE> "<<name<<" </PRE></B></TD>"
	  << "<TD style='vertical-align: top;'><I><PRE> "<<type<<" </PRE></I></TD>"
	  << "<TD style='vertical-align: top;'>"<<descr<<"</TD></TR>\n";
*/	
	(s) << "<TR><TD style='vertical-align: top;'><B><PRE> "<<name<<" </PRE></B></TD>"
	  << "<TD style='vertical-align: top;'><I><PRE> "<<descr<<" </PRE></I></TD>"
	  << "<TD style='vertical-align: top;'>"<<type<<"</TD></TR>\n";

      }
    (s) << "</TABLE>\n";

    //------------
    // End


   }
  //=========================================================================
 

  //=======================================================================
  void ComplexBlackBoxDescriptor::GetHelp(bool full) const
  {
    if (full) bbtkMessage("help",1,"Complex Black Box <"<<
			  GetPackage()->GetName()<<"::"
			  <<GetTypeName()<<">"<<std::endl);
    bbtkMessage("help",1," "                << GetDescription() <<std::endl);
    bbtkMessage("help",1," By : "           << GetAuthor()      <<std::endl);
    bbtkMessage("help",1," Category(s) : "  << GetCategory()     <<std::endl);    
    if (mInput.size()) 
      bbtkMessage("help",1," * Inputs : "<<std::endl);
    else 
      bbtkMessage("help",1," * No inputs"<<std::endl);
    InputDescriptorMapType::const_iterator i;
    unsigned int namelmax = 0;
    unsigned int typelmax = 0;
    unsigned int natlmax = 0;
    for ( i = mInput.begin();  i != mInput.end(); ++i ) 
    {
	   if (i->second->GetName().size()>namelmax) 
	     namelmax = i->second->GetName().size();
	   if (i->second->GetTypeName().size()>typelmax) 
	     typelmax = i->second->GetTypeName().size();
	   if (i->second->GetNature().size()>natlmax) 
	     natlmax = i->second->GetNature().size();
    }
    OutputDescriptorMapType::const_iterator o;
    if (full) 
    {
	   for ( o = mOutput.begin();  o != mOutput.end(); ++o ) 
	   {
	     if (o->second->GetName().size()>namelmax) 
	       namelmax = o->second->GetName().size();
	    if (o->second->GetTypeName().size()>typelmax) 
	       typelmax = o->second->GetTypeName().size();
	   if (o->second->GetNature().size()>natlmax) 
	     natlmax = o->second->GetNature().size();
	   }
    }
    //

    for ( i = mInput.begin();  i != mInput.end(); ++i ) 
    {
	   std::string name(i->second->GetName());
	   name += "'";
	   name.append(1+namelmax-name.size(),' ');
	   std::string type(i->second->GetTypeName());
	   type += ">";
	   type.append(1+typelmax-type.size(),' ');
	   std::string nature(i->second->GetNature());
	   nature += "]";
	   nature.append(1+natlmax-nature.size(),' ');
	   bbtkMessage("help",1,
		       "    '"<<name
		       <<" <"<<type
		       <<" ["<<nature
		       <<" : "<<i->second->GetDescription()<<std::endl);
    }
    if (full) 
    {
	   if (mOutput.size()) 
	     bbtkMessage("help",1," * Outputs : "<<std::endl);
	   else 
	     bbtkMessage("help",1," * No outputs"<<std::endl);
	   for ( o = mOutput.begin();  o != mOutput.end(); ++o ) 
	   {
	     std::string name(o->second->GetName());
	     name += "'";
	     name.append(1+namelmax-name.size(),' ');
	     std::string type(o->second->GetTypeName());
	     type += ">";
	     type.append(1+typelmax-type.size(),' ');
	     std::string nature(o->second->GetNature());
	     nature += "]";
	     nature.append(1+natlmax-nature.size(),' ');
	     bbtkMessage("help",1,
		       "    '"<<name
			 <<" <"<<type
			 <<" ["<<nature
			 <<" : "<<o->second->GetDescription()<<std::endl);
	   }
    }
    if (full) 
    {
	   const ComplexBlackBox::BlackBoxMapType& B = mPrototype->bbGetBlackBoxMap();
	
	   if (B.size()) 
	     bbtkMessage("help",1," * Boxes : "<<std::endl);
	   else 
	     bbtkMessage("help",1," * No boxes"<<std::endl);
	
	   ComplexBlackBox::BlackBoxMapType::const_iterator b;
	   for ( b = B.begin(); b != B.end(); ++b ) 
	   {
	     bbtkMessage("help",1,"    '"<<b->second->bbGetName()<<
			 "' <"
			 << b->second->bbGetDescriptor()->GetPackage()->GetName() 
			 <<"::"
			 <<b->second->bbGetTypeName()<<">"<<std::endl);
	   }
    }

  }   
  //=======================================================================

  //==========================================================================
  std::string ComplexBlackBoxDescriptor::GetObjectName() const
  {
    return std::string("ComplexBlackBoxDescriptor '")+GetFullTypeName()
      +std::string("'");
  }
  //==========================================================================
  //=======================================================================
  std::string ComplexBlackBoxDescriptor::GetObjectInfo() const
  {
    std::string i;
    return i;     
  }
  //=======================================================================
 //==========================================================================
size_t  ComplexBlackBoxDescriptor::GetObjectSize() const 
{
  size_t s = Superclass::GetObjectSize();
  s += ComplexBlackBoxDescriptor::GetObjectInternalSize();
  return s;
  }
  //==========================================================================
  //==========================================================================
size_t  ComplexBlackBoxDescriptor::GetObjectInternalSize() const 
{
  size_t s = sizeof(ComplexBlackBoxDescriptor);
  return s;
  }
  //==========================================================================
  //==========================================================================
  size_t  ComplexBlackBoxDescriptor::GetObjectRecursiveSize() const 
  {
    size_t s = Superclass::GetObjectRecursiveSize();
    s += ComplexBlackBoxDescriptor::GetObjectInternalSize();
    s += mPrototype->GetObjectRecursiveSize();
    return s;
  }
  //==========================================================================
void ComplexBlackBoxDescriptor::GetBoxesInside (NodeTreeC& tree, int cont) 
	{
		std::cout<<"getBoxesInside recursivo "<<std::endl;
		cont++;
		std::string name = GetTypeName();
		std::cout<<"despues de getTypeName : "<<name<<std::endl;
    std::string descr = GetDescription();
    std::string author = GetAuthor();
	
   	//list.push_back(name);
		tree.setData(name);
   	//list.push_back(descr);
   	//list.push_back(author);
		int k = 0;
		
    const ComplexBlackBox::BlackBoxMapType& B = mPrototype->bbGetBlackBoxMap();

		std::cout<<"----B size "<< B.size() << std::endl;
		std::cout<<"----antes if "<<std::endl;
    if (B.size()) 
    {
	   std::set<BlackBoxDescriptor::Pointer> pdeps;
	   ComplexBlackBox::BlackBoxMapType::const_iterator b;
			std::cout<<"----antes for "<<std::endl;
	   for ( b = B.begin(); b != B.end(); ++b ) 
	   {
				BlackBoxDescriptor::Pointer d = b->second->bbGetDescriptor();
				//std::cout<<"nombre objeto d "<< d->GetObjectName() <<std::endl;
				
	      //if (pdeps.find(d) != pdeps.end()) 
        //    continue;
	      //pdeps.insert(d);
				std::cout<<"----1~ "<<std::endl;
	      Package::Pointer p = d->GetPackage();
	      std::string nameBox = b->second->bbGetTypeName();
				std::string packageBox = p->GetName();
				tree.insertChild(nameBox);
				std::cout<<"----2~ "<<std::endl;
				//list.push_back(nameBox);
				//list.push_back(packageBox);
				std::cout<<"-----cont "<<cont<<std::endl;
				//tree.treeTour(cont);
				d->GetBoxesInside (tree.childs[k], cont);
				std::cout<<"----3~ "<<std::endl;
				//std::cout<<"--despues de la recursion "<<k<<std::endl;
				k++;
	    }	// for
			std::cout<<"----despues for "<<std::endl;
     } // If B.size
			std::cout<<"----despues if "<<std::endl;
			std::cout<<"end getBoxesInside recursivo"<<std::endl;
	}

//===================================================

	 //==========================================================================
}
