/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkConfigurationFile.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.38 $
=========================================================================*/



/**
 *\file
 *\brief Class bbtk::ConfigurationFile
 */

#include "bbtkConfigurationFile.h"
#include "bbtkMessageManager.h"
#include "bbtkXML.h"
#include "bbtkUtilities.h"

#include "creaSystem.h"

#if defined(WIN32)
# include <direct.h> // for getcwd
# include <windows.h>
#endif


namespace bbtk
{
  //====================================================================
  /// Constructor
  ConfigurationFile::ConfigurationFile()
  {
    mFile_separator = VALID_FILE_SEPARATOR;

    // ==> Set system paths
    mBin_path = Utilities::GetExecutablePath();

/*	EED 23 Mars 2009
#ifdef MACOSX
    mInstall_path = mBin_path + "/../../../..";
#else
    mInstall_path = mBin_path + "/..";
#endif
*/
#ifdef MACOSX
	  std::string macPath("Contents/MacOS");
	  int sbp = mBin_path.length();
	  int smp = macPath.length();
	  if (mBin_path.compare( sbp-smp, smp, macPath )==0 )
	  {
		  mBin_path = mBin_path + "/../../..";
	  }
#endif

///\TODO : better use ??
	  mInstall_path = mBin_path + "/..";

    // The relative path to the doc folder (=BBTK_DOC_REL_PATH)
    //    mDoc_rel_path = BBTK_STRINGIFY_SYMBOL(BBTK_DOC_REL_PATH);
    // The path to the doc folder (=mInstall_path+"/"+mDoc_rel_path)
    //    mDoc_path = mInstall_path + "/" + mDoc_rel_path;
    // Have to create bbtk user's dirs if needed

    // LG 27/01/09 : doc is now user dependent
    //               and located in user's .bbtk dir /doc
    // All dirs under .bbtk user's dir are set and created by:
    InitializeDotBbtkStructure();

    // The relative path to the doc folder (=BBTK_BBS_REL_PATH)
    mBbs_rel_path = BBTK_STRINGIFY_SYMBOL(BBTK_BBS_REL_PATH);
    // The path to the bbs folder (=mInstall_path+"/"+mBbs_rel_path)
    mBbs_path = mInstall_path + "/" + mBbs_rel_path;

    mData_path = mInstall_path + "/" + BBTK_STRINGIFY_SYMBOL(BBTK_DATA_REL_PATH);

    Utilities::replace( mBin_path,
			INVALID_FILE_SEPARATOR,
			VALID_FILE_SEPARATOR);
    Utilities::replace( mInstall_path,
			INVALID_FILE_SEPARATOR,
			VALID_FILE_SEPARATOR);
    Utilities::replace( mBbs_path,
			INVALID_FILE_SEPARATOR,
			VALID_FILE_SEPARATOR);
    Utilities::replace( mData_path,
			INVALID_FILE_SEPARATOR,
			VALID_FILE_SEPARATOR);

    bbtkMessage("config",1," ==> bin    : '"<<mBin_path<<"'"<<std::endl);
    bbtkMessage("config",1," ==> prefix : '"<<mInstall_path<<"'"<<std::endl);
    bbtkMessage("config",1," ==> doc    : '"<<mDoc_path<<"'"<<std::endl);
    bbtkMessage("config",1," ==> bbs    : '"<<mBbs_path<<"'"<<std::endl);
    bbtkMessage("config",1," ==> data   : '"<<mData_path<<"'"<<std::endl);

    // bbs_paths
    // always add "." (current working directory) at the begining
    mBbs_paths.push_back( "." );
    // add system bbs path
    mBbs_paths.push_back(mBbs_path);
    // add toolsbbtk/appli
//EED    std::string toolsappli_rel_path(mFile_separator);
    std::string toolsappli_rel_path("/");
//EED    toolsappli_rel_path +=  "toolsbbtk" + mFile_separator + "appli";
    toolsappli_rel_path +=  "toolsbbtk/appli";
    //
    //-----------------------------------------------------------
    // LG : REMOVE BUGGY PATH WITH include *:
    //
    //    mBbs_paths.push_back(mBbs_path + toolsappli_rel_path);
    //-----------------------------------------------------------
    int iStrVec,sizeStrVec;

    sizeStrVec = mBbs_paths.size();
    for (iStrVec=0;iStrVec<sizeStrVec;iStrVec++){
      Utilities::replace( mBbs_paths[iStrVec] , INVALID_FILE_SEPARATOR , VALID_FILE_SEPARATOR);
    }


    // always add "." (current working directory) at the begining
    mPackage_paths.push_back(".");
    // add system bin path (for build tree / standalone folder install)
    mPackage_paths.push_back(mBin_path);
    // add system lib path (for install tree)
    //EED    mPackage_paths.push_back(mInstall_path + mFile_separator + "lib");
    
    //mPackage_paths.push_back(mInstall_path + "/lib"); // JPR
    //mPackage_paths.push_back(mInstall_path + "/lib64");// JPR
    
    ///\TODO : better use BBTK_PACKAGE_LIB_PATH  // JPR
    
    mPackage_paths.push_back(mInstall_path + "/lib/creatools");
    mPackage_paths.push_back(mInstall_path + "/lib64/creatools");
#ifdef WIN32
    // add bin/Debug bin/Release paths (for build/install tree)
    //EED    mPackage_paths.push_back(mBin_path + mFile_separator + "Debug");
    mPackage_paths.push_back(mBin_path + "/Debug");
    //EED    mPackage_paths.push_back(mBin_path + mFile_separator + "Release");
    mPackage_paths.push_back(mBin_path + "/Release");
#endif

    sizeStrVec = mPackage_paths.size();
    for (iStrVec=0;iStrVec<sizeStrVec;iStrVec++){
      Utilities::replace( mPackage_paths[iStrVec] , INVALID_FILE_SEPARATOR , VALID_FILE_SEPARATOR);
    }

    GetHelp(2);

    // ==> First we look for bbtk_config.xml in "."
    char buf[2048];
    const char *currentDir = getcwd(buf, 2048);

    if( !currentDir )
      {
	std::cerr << "Path was too long to fit on 2048 bytes ?!?" << std::endl;
	// \todo : what else?
	// How abort a constructor and warn the caller function?
	// LG : throw an exception
      }

    //    std::string configXmlFullPathName = currentDir + mFile_separator + "bbtk_config.xml";
    std::string configXmlFullPathName = currentDir ;
    configXmlFullPathName += "/bbtk_config.xml";
    Utilities::replace( configXmlFullPathName , INVALID_FILE_SEPARATOR , VALID_FILE_SEPARATOR);

    if ( Utilities::FileExists( configXmlFullPathName ))
      {
	bbtkMessage("config",1, "ConfigurationFile : [" << configXmlFullPathName <<
		    "] found in current directory" << std::endl);
	//Read(configXmlFullPathName.c_str());
	// traiter le fichier local
      }

    // ==> Then we look for bbtk_config.xml in ".bbtk"
    else
      {
	configXmlFullPathName = Utilities::MakeUserSettingsFullFileName("bbtk_config.xml");
	if (!Utilities::FileExists( configXmlFullPathName ))
	  {
	    // ==> Nothing found, we create bbtk_config.xml in ".bbtk"
	    InstallPath ();
	  }
      }

    // In any case, deal with bbtk_config.xml!
    Read(configXmlFullPathName.c_str());
  }
  //=========================================================================

  //=========================================================================
  /// Destructor
  ConfigurationFile::~ConfigurationFile()
  {
  }
  //=========================================================================

  //=========================================================================
  void ConfigurationFile::InitializeDotBbtkStructure()
  {
    mDot_bbtk_path = Utilities::GetUserSettingsDir();
    mDot_bbtk_is_new = false;
    if (!Utilities::FileExists(mDot_bbtk_path)) mDot_bbtk_is_new = true;
    Utilities::CreateDirectoryIfNeeded(mDot_bbtk_path);

    mDoc_path = Utilities::MakeUserSettingsFullFileName("doc");
    Utilities::CreateDirectoryIfNeeded(mDoc_path);

    std::string bbdoc_path =
      Utilities::MakeUserSettingsFullFileName("doc/bbdoc");
    Utilities::CreateDirectoryIfNeeded(bbdoc_path);

    // Create help_contents.html if does not exist
    std::string filename = Utilities::MakeUserSettingsFullFileName("doc/help_contents.html");
    if (!Utilities::FileExists(filename))
      {
			bbtkDebugMessage("config",1,
			 "* Creating [" << filename << "]" << std::endl);
			// The path to the doc folder (=mInstall_path+"/"+mDoc_rel_path)
			std::string doc_path = mInstall_path + "/"
			  + BBTK_STRINGIFY_SYMBOL(BBTK_DOC_REL_PATH) + "/";
			Utilities::MakeValidFileName(doc_path);

			std::ofstream f;
			f.open(filename.c_str(), std::ios::out );
			f << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD "
			  << "HTML 4.01 Transitional//EN\">"
			  << "<html><head><title>Help Contents - bbtk "
			  << GetVersion() << "</title>"
			  << "<meta http-equiv=\"Content-Type\" content=\"text/html; "
			  << "charset=iso-8859-1\"></head><H1>Help Contents</H1>"
			  << "<a href=\"bbdoc/index-category.html#demo\" "
			  << "target=\"information\">Demos</a><br>"
			  << "<a href=\"bbdoc/index-category.html#example\" "
			  << "target=\"information\">Examples</a>"
			  << "<H2>Guides</H2>"
			  << "<a href=\""
			  << doc_path
			  << "bbtkUsersGuide/bbtkUsersGuide.pdf"
			  << "\" target=\"information\">User's Guide</a><br>"
			  << "<a href=\""
			  << doc_path
			  << "bbtkDevelopersGuide/bbtkDevelopersGuide.pdf"
			  << "\" target=\"information\">Developer's Guide</a><br>"
			  << "<a href=\""
			  << doc_path
			  << "bbtkPackageDevelopersGuide/bbtkPackageDevelopersGuide.pdf"
			  << "\" target=\"information\">Package Developer's Guide</a><br>"
			  << "<a href=\""
			  << doc_path
			  << "doxygen/bbtk/main.html"
			  << "\" target=\"information\">bbtk library doxygen doc</a><br>"
			  << "<H2>Boxes</H2>"
			  << "<a target=\"information\" href=\"bbdoc/index-alpha.html\">"
			  << "Alphabetical list</a><br>"
			  << "<a target=\"information\" href=\"bbdoc/index-package.html\">"
			  << "List by package</a><br>"
			  << "<a target=\"information\" href=\"bbdoc/index-category.html\">"
			  << "List by category</a><br>"
			  << "<a target=\"information\" href=\"bbdoc/index-adaptors.html\">"
			  << "List of adaptors</a><br>"
			  << "</body>"
		  << "</html>";
      } // if

    mTemp_path = Utilities::MakeUserSettingsFullFileName("tmp");
    Utilities::CreateDirectoryIfNeeded(mTemp_path);

  }
  //=========================================================================

  //=========================================================================
  void ConfigurationFile::CreateConfigXML( char *rootDirectory )
  {
    FILE *fp;
    char configXml[250];
    sprintf (configXml , "%s/bbtk_config.xml", rootDirectory);
    bbtkDebugMessage("config",1, "in CreateConfigXML[" << configXml << "]" << std::endl);
    fp = fopen (configXml, "w");
    fprintf(fp, "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n");
    fprintf(fp, "<config>\n");
    fprintf(fp, "   <bbs_path>     </bbs_path>\n");
    fprintf(fp, "   <package_path> </package_path>\n");
    fprintf(fp, "</config>\n");
    fclose(fp);
  }
  //=========================================================================




  //=========================================================================
  void ConfigurationFile::InstallPath ()
  {

    /*--------------------------------------------------
      New policy for bbtk_config.xml :

      if bbtk_config.xml found in current directory (user is an aware user!)
      use it!

      else if bbtk_config.xml found in HOME/.bbtk (user already worked with it)
      use it!

      else if bbtk_config.xml.tmp found in /usr/local/bin or c:\\Program Files\\BBTK\\bin
      copy it as .bbtk/bbtk_config.xml

      else (nothing installed)
      create a minimum version in HOME/.bbtk
      ----------------------------------------------------*/


    // -----------------------------------------------------------------
#if defined(__GNUC__)

    // ------------------ create some usefull strings ----------------
    // installed bbtk_path
    char bbtk_path[1000];
    strcpy(bbtk_path, Utilities::GetExecutablePath().c_str());  // JPR
    //strcpy(bbtk_path, "/usr/local/bin");

    // rootDirectory
    char rootDirectory[200];
    sprintf( rootDirectory,  "%s/.bbtk", getenv("HOME"));

    // configPath
    char configPath[200];
    sprintf(configPath, "%s/bbtk_config.xml",rootDirectory);

    // configXmlTmp
    char configXmlTmp[250];
    sprintf(configXmlTmp, "%s/bbtk_config.xml.tmp", bbtk_path);

    // copyFile
    char copyFile[250];

    if (!Utilities::FileExists(configXmlTmp)) // bbtk_config.xml.tmp not found (not installed)
      {
	// if "bbtk_path/bbtk_config.xml.tmp" doesn't exist, hard-create a minimum version in .bbtk
	CreateConfigXML( rootDirectory );// create .bbtk
      }
    else
      {
	sprintf(copyFile,"cp %s  %s/bbtk_config.xml ",configXmlTmp,rootDirectory );
	if (!Utilities::FileExists(configPath))
	  {
	    system(copyFile);
	  }
      }
    return;

    // ------------------------------------------------------------------
#elif defined(WIN32)


    // installed bbtk_path
    char bbtk_path[100];
    strcpy(bbtk_path, "\"c:\\Program Files\\BBTK\\bin\"");
    char bbtk_path2[100];
    strcpy(bbtk_path2, "c:\\Program Files\\BBTK\\bin");

    // rootDirectory
    char rootDirectory[200];
    sprintf(rootDirectory, "%s\\.bbtk",getenv("USERPROFILE"));
    //  std::cout << "[" << rootDirectory << "]" << std::endl;

    // configPath
    char configPath[200];
    sprintf(configPath, "%s\\bbtk_config.xml",rootDirectory);

    // makeDir
    char makeDir[250];
    sprintf( makeDir, "mkdir \"%s\" ", rootDirectory);

    // configXmlTmp
    char configXmlTmp[250];
    sprintf(configXmlTmp, "%s\\bbtk_config.xml.tmp", bbtk_path2);

    // copyFile
    char copyFile[250];

    if (!Utilities::FileExists(configXmlTmp)) // bbtk_config.xml.tmp not found
      {
	// if "bbtk_path/bbtk_config.xml.tmp" doesn't exist, hard-create a minimum version in .bbtk
	CreateConfigXML( rootDirectory );// create .bbtk
	return;
      }

    sprintf(copyFile,"copy %s\\bbtk_config.xml.tmp \"%s\"\\bbtk_config.xml ",bbtk_path,rootDirectory );

    int attribs = GetFileAttributes (rootDirectory);
    bbtkMessage("config",1,std::hex << attribs << " " << FILE_ATTRIBUTE_DIRECTORY << std::endl);
    if ( attribs != 0xFFFFFFFF)
      {
	if ((attribs & FILE_ATTRIBUTE_DIRECTORY) == FILE_ATTRIBUTE_DIRECTORY ) /// \TODO : check !
	  {
	    if ( GetFileAttributes( configPath ) == 0xFFFFFFFF)
	      {
		system(copyFile);
	      }
	  }
      }
    else
      {
	system(makeDir);
	system(copyFile);
      }
    return;
    // ------------------------------------------------------------------
#else
/// \todo  ConfigurationFile::InstallPath() : exit when for not WIN32 and not__GNUC__

  return;
#endif

}
  //=========================================================================



  //=========================================================================
  // Gets the list of directories holding bb scripts, packages, dll, ... from the xml file
  //      bbtk_config.xml

  void ConfigurationFile::Read(const std::string& filename)
  {

    bbtkDebugMessage("config",1,"ConfigurationFile::Read(" <<filename << ")" << std::endl);

    mConfig_xml_full_path = filename;
    XMLResults* res = new XMLResults;
    XMLNode BB = XMLNode::parseFile((XMLCSTR)filename.c_str(),(XMLCSTR)"config",res);

    if ( res->error != eXMLErrorNone )
      {
	std::string mess = GetErrorMessage(res,filename);
	delete res;
	bbtkDebugMessage("config",1,mess<< std::endl);
 	bbtkError(mess);
      }
    delete res;

    bbtkDebugMessage("config",1,"OK" << std::endl);

    int i,j;

    // Description
    for (i=0,j=0; i<BB.nChildNode((XMLCSTR)"description"); i++)
      {
	std::string val;
	GetTextOrClear(BB.getChildNode((XMLCSTR)"description",&j),val);
	mDescription += val;
      }

    // Url
    if( BB.nChildNode((XMLCSTR)"url") )
      GetTextOrClear(BB.getChildNode((XMLCSTR)"url"),mUrl);

    // Data_Path
    if( BB.nChildNode((XMLCSTR)"data_path") )
      GetTextOrClear(BB.getChildNode((XMLCSTR)"data_path"),mData_path);

    // install_path
    //  if( BB.nChildNode((XMLCSTR)"install_path") )
    //   GetTextOrClear(BB.getChildNode((XMLCSTR)"install_path"),mInstall_path);

    // add user bbs paths
    for (i=0,j=0; i<BB.nChildNode((XMLCSTR)"bbs_path"); i++)
      {
	std::string val;
	GetTextOrClear(BB.getChildNode((XMLCSTR)"bbs_path",&j),val);
	mBbs_paths.push_back(val);
      }

    // package_paths

    // add user package path
    for (i=0,j=0; i<BB.nChildNode((XMLCSTR)"package_path"); i++)
      {
	std::string val;
	GetTextOrClear(BB.getChildNode((XMLCSTR)"package_path",&j),val);
	mPackage_paths.push_back(val);
      }

    // default_temp_dir
    /*
    if( BB.nChildNode((XMLCSTR)"default_temp_dir") )
      GetTextOrClear(BB.getChildNode((XMLCSTR)"default_temp_dir"),mDefault_temp_dir);

    if ( mDefault_temp_dir == "$") // no value found in config_xml
      {
	size_t pos = mConfig_xml_full_path.find("bbtk_config.xml");
	mDefault_temp_dir = mConfig_xml_full_path.substr (0,pos);
      }
    */

    GetHelp(2);
  }
  //=========================================================================

  //=========================================================================
  bool ConfigurationFile::AddPackagePathsAndWrite( const std::string& path )
  {
    bbtkDebugMessageInc("config",9,
			"ConfigurationFile::AddPackagePathsAndWrite("
			<<path<<")"<<std::endl);

     XMLResults* res = new XMLResults;
    XMLNode BB =
      XMLNode::parseFile((XMLCSTR)Get_config_xml_full_path().c_str(),
			 (XMLCSTR)"config",res);

    if ( res->error != eXMLErrorNone )
      {
	std::string mess = GetErrorMessage(res,Get_config_xml_full_path());
	delete res;
	bbtkDebugMessage("config",1,mess<< std::endl);
 	bbtkError(mess);
      }
   delete res;

#ifdef _WIN32
    std::string bbs_path = path + "/bbs";
#else
    std::string bbs_path = path + "/share/bbtk/bbs" ;
#endif
    XMLNode BBSPATH = BB.addChild((XMLCSTR)"bbs_path");
    BBSPATH.addText((XMLCSTR)bbs_path.c_str());
    Utilities::replace(bbs_path, INVALID_FILE_SEPARATOR, VALID_FILE_SEPARATOR);
    mBbs_paths.push_back(bbs_path);

#ifdef _WIN32
    std::string pack_path = path + "/bin";
#else
    std::string pack_path = path ;
#endif
    XMLNode PACKPATH = BB.addChild((XMLCSTR)"package_path");
    PACKPATH.addText((XMLCSTR)pack_path.c_str());
    Utilities::replace(pack_path,INVALID_FILE_SEPARATOR,VALID_FILE_SEPARATOR);
    mPackage_paths.push_back(pack_path);

#ifdef _WIN32
    pack_path = path + "/RelWithDebInfo";
    PACKPATH = BB.addChild((XMLCSTR)"package_path");
    PACKPATH.addText((XMLCSTR)pack_path.c_str());
    Utilities::replace(pack_path,INVALID_FILE_SEPARATOR,VALID_FILE_SEPARATOR);
    mPackage_paths.push_back(pack_path);
    pack_path = path + "/Debug";
    PACKPATH = BB.addChild((XMLCSTR)"package_path");
    PACKPATH.addText((XMLCSTR)pack_path.c_str());
    Utilities::replace(pack_path,INVALID_FILE_SEPARATOR,VALID_FILE_SEPARATOR);
    mPackage_paths.push_back(pack_path);
    pack_path = path + "/Release";
    PACKPATH = BB.addChild((XMLCSTR)"package_path");
    PACKPATH.addText((XMLCSTR)pack_path.c_str());
    Utilities::replace(pack_path,INVALID_FILE_SEPARATOR,VALID_FILE_SEPARATOR);
    mPackage_paths.push_back(pack_path);
#endif


    XMLError err = BB.writeToFile((XMLCSTR)Get_config_xml_full_path().c_str());
    if ( err != eXMLErrorNone )
      {
	std::string mess = GetErrorMessage(res,Get_config_xml_full_path());
	bbtkDebugMessage("config",1,mess<< std::endl);
 	bbtkError(mess);
      }

    return true;
  }
  //=========================================================================

  //=========================================================================
  void ConfigurationFile::GetHelp(int level) const
  {
    bbtkDebugMessageInc("config",9,"ConfigurationFile::GetHelp("<<level
			<<")"<<std::endl);

    const std::string config_xml_full_path      = Get_config_xml_full_path();
    const std::string description               = Get_description();
    const std::string url                       = Get_doc_path();
    const std::string data_path                 = Get_data_path();
    const std::string default_temp_dir          = Get_default_temp_dir();
    const std::string file_separator            = Get_file_separator();
    const std::vector<std::string>bbs_paths     = Get_bbs_paths();
    const std::vector<std::string>package_paths = Get_package_paths();

    bbtkMessage("help",level, "============="   << std::endl);
    bbtkMessage("help",level, "Configuration"   << std::endl);
    bbtkMessage("help",level, "============="   << std::endl);
    bbtkMessage("help",level, "bbtk_config.xml    : [" << config_xml_full_path  << "]" << std::endl);
    bbtkMessage("help",level, "Documentation Path : [" << url             << "]" << std::endl);
    bbtkMessage("help",level, "Data Path          : [" << data_path       << "]" << std::endl);
    bbtkMessage("help",level, "Temp Directory     : [" << default_temp_dir << "]" << std::endl);
    bbtkMessage("help",level, "File Separator     : [" << file_separator  << "]" << std::endl);

    std::vector<std::string>::const_iterator i;

    bbtkMessage("help",level, "BBS Paths   " << std::endl);
    for (i = bbs_paths.begin(); i!=bbs_paths.end(); ++i )
    {
      bbtkMessage("help",level,"--- ["<<*i<<"]"<<std::endl);
    }

    bbtkMessage("help",level, "PACKAGE Paths : " << std::endl);
    for (i = package_paths.begin(); i!=package_paths.end(); ++i )
    {
      bbtkMessage("help",level,"--- ["<<*i<<"]"<<std::endl);
    }

    bbtkDebugDecTab("config",9);
  }
  //=========================================================================


} // namespace bbtk
