/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkConnection.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.22 $
=========================================================================*/

                                                                        

/**
 *\file
 *\brief Class bbtk::Connection
 */

#include "bbtkConnection.h"
#include "bbtkFactory.h"
#include "bbtkBlackBox.h"
#include "bbtkMessageManager.h"
#include "bbtkBlackBoxOutputConnector.h"

#define bbtkCMessage(key,level,mess) \
  bbtkMessage(key,level,"["<<GetFullName()<<"] "<<mess)
#define bbtkCDebugMessage(key,level,mess)	\
  bbtkDebugMessage(key,level,"["<<GetFullName()<<"] "<<mess)

namespace bbtk
{
  const std::string IOSTATUS_STRING[3] = 
    {"Up-to-date","Modified","Out-of-date"}; 
  
  const std::string& GetIOStatusString(IOStatus s)
  { return IOSTATUS_STRING[s]; }

  //==================================================================
  Connection::Pointer Connection::New(BlackBox::Pointer from, 
				      const std::string& output,
				      BlackBox::Pointer to, 
				      const std::string& input ,
				      const Factory::Pointer f  )
  {
    bbtkDebugMessage("object",1,"##> Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);
    Connection::Pointer p = 
      MakePointer(new Connection(from,output,to,input,f));
    bbtkDebugMessage("object",1,"<## Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);
    return p;
  }
  //==================================================================

  //==================================================================
  /// Ctor with the black box from and to and their input and output.
/// Check the input and output compatibility
Connection::Connection(BlackBox::Pointer from, const std::string& output,
		       BlackBox::Pointer to, const std::string& input ,
		       const Factory::Pointer f  )
    : mAdaptor(),
      mFactory(f),
      mFromAny(false),
      mToAny(false)
  {
    bbtkDebugMessage("object",2,"==> Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    

    bbtkDebugMessage("connection",1,"==> Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    

    

    if (! from->bbHasOutput(output) )
      {
	bbtkError("The box \""<<from->bbGetTypeName()<<
		  "\" has no output \""<<output<<"\"");
      }
    if (! to->bbHasInput(input) )
      {
	bbtkError("The box \""<<to->bbGetTypeName()<<
		  "\" has no input \""<<input<<"\"");
      } 

    if (to->bbGetInputConnectorMap().find(input)->second->IsConnected())
      {
	bbtkError("The input \""<<input<<"\" of the box \""<<to->bbGetName()
		  <<"\" is already connected");
      }
    
    //  std::string t1 ( from->bbGetOutputType(output).name() );
    //   std::string t2 ( to->bbGetInputType(input).name() );
    // if  //( t1 != t2 ) 
    if ( from->bbGetOutputType(output) !=
	 to->bbGetInputType(input) )
      {
	if ( from->bbGetOutputType(output) == typeid(Data) )
	  {
	    bbtkWarning("Connection: '"<<from->bbGetName()<<"."<<output
			<<"' is of type <"
			<<HumanTypeName<Data>()
			<<"> : type compatibility with '"
			<<to->bbGetName()<<"."<<input
			<<"' will be resolved at run time"
			);
	    mFromAny = true;
	  }
	else if (  to->bbGetInputType(input) == typeid(Data) )
	  {   
	    bbtkDebugMessage("kernel",8," -> '"<<input<<"' type is "
			     <<TypeName<Data>()<<" : can receive any data"
			     <<std::endl);
	    mToAny = true;
	  }
	else 
	  {
	    //   std::cout << "Adaptive connection "<<std::endl;
	    std::string name;
	    name = from->bbGetName() + "." + output + "-" 
	      + to->bbGetName() + "." + input; 
	    mAdaptor = mFactory.lock()
	      ->NewAdaptor(from->bbGetOutputType(output),
			   to->bbGetInputType(input),
			   name);
	    if (!mAdaptor)  
	      {  
		bbtkError("did not find any <"
			  <<TypeName(from->bbGetOutputType(output))
			  <<"> to <"
			  <<TypeName(to->bbGetInputType(input))
			  <<"> adaptor");
	      } 
	  }
      }


    mFrom = from;
    mOriginalFrom = from;
    mTo = to;
    mOriginalTo = to;
    mInput = mOriginalInput = input;
    mOutput = mOriginalOutput = output;

     // Lock this pointer !!!
    //Pointer p = MakePointer(this,true);
    from->bbConnectOutput(output,this);
    to->bbConnectInput(input,this);
 
    from->bbGetOutputConnector(output).AddChangeObserver(boost::bind(&bbtk::Connection::OnOutputChange,this, _1, _2, _3));

    
    bbtkDebugMessage("connection",1,"<== Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    

    bbtkDebugMessage("object",2,"<== Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    
  }
 //==================================================================
 
  //==================================================================
  Connection::Pointer Connection::New(BlackBox::Pointer from, 
				      const std::string& output,
				      BlackBox::Pointer to, 
				      const std::string& input )
  {
    bbtkDebugMessage("object",1,"##> Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);
    Connection::Pointer p = 
      MakePointer(new Connection(from,output,to,input));
    bbtkDebugMessage("object",1,"<## Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);
    return p;
  }
  //==================================================================

  //==================================================================
  /// Ctor with the black box from and to and their input and output.
/// Check the input and output compatibility
Connection::Connection(BlackBox::Pointer from, const std::string& output,
		       BlackBox::Pointer to, const std::string& input )
  : mAdaptor(),
      mFactory(),
      mFromAny(false),
      mToAny(false)
  {
    bbtkDebugMessage("object",2,"==> Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    

    bbtkDebugMessage("connection",1,"==> Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    

    

    if (! from->bbHasOutput(output) )
      {
	bbtkError("The box \""<<from->bbGetTypeName()<<
		  "\" has no output \""<<output<<"\"");
      }
    if (! to->bbHasInput(input) )
      {
	bbtkError("The box \""<<to->bbGetTypeName()<<
		  "\" has no input \""<<input<<"\"");
      } 

    if (to->bbGetInputConnectorMap().find(input)->second->IsConnected())
      {
	bbtkError("The input \""<<input<<"\" of the box \""<<to->bbGetName()
		  <<"\" is already connected");
      }
    
    //  std::string t1 ( from->bbGetOutputType(output).name() );
    //   std::string t2 ( to->bbGetInputType(input).name() );
    // if  //( t1 != t2 ) 
    if ( from->bbGetOutputType(output) !=
	 to->bbGetInputType(input) )
      {
	if ( from->bbGetOutputType(output) == typeid(Data) )
	  {
	    bbtkWarning("Connection '"
			<<GetFullName()
			<<"' : '"<<from->bbGetName()<<"."<<output
			<<"' is of type <"
			<<HumanTypeName<Data>()
			<<"> : type compatibility with '"
			<<to->bbGetName()<<"."<<input
			<<"' will be resolved at run time"
			);
	    mFromAny = true;
	  }
	else if (  to->bbGetInputType(input) == typeid(Data) )
	  {   
	    bbtkDebugMessage("kernel",8," -> '"<<input<<"' type is "
			     <<TypeName<Data>()<<" : can receive any data"
			     <<std::endl);
	    mToAny = true;
	  }
	else 
	  {
	    bbtkError("Connection created between different types without Factory provided");
	  }
      }


    mFrom = from;
    mOriginalFrom = from;
    mTo = to;
    mOriginalTo = to;
    mInput = mOriginalInput = input;
    mOutput = mOriginalOutput = output;

     // Lock this pointer !!!
    //Pointer p = MakePointer(this,true);
    from->bbConnectOutput(output,this);
    to->bbConnectInput(input,this);

    from->bbGetOutputConnector(output).AddChangeObserver(boost::bind(&bbtk::Connection::OnOutputChange,this, _1, _2, _3));
    
    bbtkDebugMessage("connection",1,"<== Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    

    bbtkDebugMessage("object",2,"==> Connection(\""
		     <<from->bbGetName()<<"\",\""<<output<<"\",\""
		     <<to->bbGetName()<<"\",\""<<input<<"\")"
		     <<std::endl);    
  }
 //==================================================================
 
  //==================================================================
  /// Dtor 
  Connection::~Connection()
  {
    bbtkCDebugMessage("object",4,
		     "==> ~Connection()"
		     <<std::endl);

    if (mAdaptor) mAdaptor.reset();
    if (mFrom!=0) 
      {
	mFrom->bbDisconnectOutput(mOutput,this);
	//				  GetThisPointer<Connection>());
	mFrom.reset();
      }
    else 
      {
	bbtkInternalError("~Connection() : invalid initial box pointer");
      }
    if (mTo!=0) 
      {
	mTo->bbDisconnectInput(mInput,this);//   GetThisPointer<Connection>());
	mTo.reset();
      }
    else 
      {
	bbtkInternalError("~Connection() : invalid final box pointer");
      }


    bbtkCDebugMessage("object",4,
		      "<== ~Connection()"
		      <<std::endl);
  }
  //==================================================================
  
  //==================================================================
  /// Recursive execution
  void Connection::RecursiveExecute()
  {
    bbtkCDebugMessage("process",4,
		      "===> Connection::RecursiveExecute()"
		      <<std::endl);

    /*
    // If box from already executing : nothing to do
    if (mFrom->bbGetExecuting()) 
      {
	bbtkDebugMessage("process",3,
			 " -> "<<mFrom->bbGetName()
			 <<" already executing : abort"<<std::endl);
	return; 

      }
    */

    mFrom->bbRecursiveExecute(GetThisPointer<Connection>());

    TransferData();

    // Transfer status
    IOStatus s = MODIFIED;
    if ( mFrom->bbGetOutputConnector(mOutput).GetStatus() == OUTOFDATE) 
      {
	s = OUTOFDATE;
      }
    mTo->bbGetInputConnector(mInput).SetStatus(s);
    
    bbtkCDebugMessage("process",4,
		     " --> '"<<mTo->bbGetName()<<"."<<mInput
		     <<" ["<<&mTo->bbGetInputConnector(mInput)<<"] "
		     <<"' new status '"
		     <<GetIOStatusString(s)
		     <<"'"
		     << std::endl);

    bbtkCDebugMessage("process",4,
		     "<=== Connection::RecursiveExecute()"
		      <<std::endl);
    return; 
  }
  //==================================================================



  //==================================================================
  /// Transfers the data from the source output to the target input
  /// doing necessary conversions (adaptation or pointer cast)
  void Connection::TransferData()
  {
    bbtkCDebugMessage("data",3,
		      "Connection::TransferData()"
		      <<std::endl);
    
    
    // If an adaptor was created we need to adapt the data
    if (mAdaptor) 
      {
	mAdaptor->bbSetInput("In",mFrom->bbGetOutput(mOutput),false);
	mAdaptor->bbExecute();
	// LG : Connection Update does not set mTo as modified
	mTo->bbSetInput(mInput, mAdaptor->bbGetOutput("Out"),false);
	
      }
    // If no adaptor but source type is an any and target is not an any
    else if ( mFromAny && (! mToAny) )
      {
	bbtkCDebugMessage("data",3,
			 " * Source type is an "
			 <<HumanTypeName<Data>()
			 <<" which contains a <"
			 <<HumanTypeName(mFrom->bbGetOutput(mOutput).type())
			 <<">"<<std::endl);
	bbtkCDebugMessage("data",3,
			 " * Target type is <"
			 <<HumanTypeName(mTo->bbGetInputType(mInput))
			 <<">"<<std::endl);
	
	// 0) If from any contents void : nothing to do 
	if (mFrom->bbGetOutput(mOutput).type() == typeid(void)) 
	  {
	    bbtkCDebugMessage("data",3,
			     " -> Source is void : nothing to transfer!"<<std::endl);
	  }
	// 1) Test strict type matching between any content and target
	else if (mFrom->bbGetOutput(mOutput)
	    .contains( mTo->bbGetInputType(mInput) ) )
	  {
	    bbtkCDebugMessage("data",3,
			     " -> Equal types : transfer ok"<<std::endl);
	    mTo->bbSetInput( mInput, 
			     mFrom->bbGetOutput(mOutput),
			     false);
	  }
	else 
	  {
	    // 2) Look for an adaptor
	    bbtk::BlackBox::Pointer adaptor;
	    try 
	      {
		adaptor = mFactory.lock()
		  ->NewAdaptor(mFrom->bbGetOutput(mOutput).type(),
			       mTo->bbGetInputType(mInput),
			       "");
	      }
	    catch (...)
	      {
	      }
	    if (adaptor)  
	      {
		bbtkCDebugMessage("data",3," -> Adaptor found : using it"
				 <<std::endl);
		  adaptor->bbSetInput("In",mFrom->bbGetOutput(mOutput),false);
		adaptor->bbExecute();
		// LG : Connection Update does not set mTo as modified
		mTo->bbSetInput(mInput, adaptor->bbGetOutput("Out"),false);
		//	adaptor->bbDelete();
	      }
	    // 3) If no adaptor found but the any content is a pointer
	    //    and target type is also a pointer : we try run-time cast
	    else if ( (mFrom->bbGetOutput(mOutput).contains_pointer()) &&
		      (mTo->bbGetDescriptor()->GetInputDescriptor(mInput)
		       ->IsPointerType()) )
	      {
		bbtkCDebugMessage("data",3,
				 " -> No adaptor found but source and target types are both pointers : trying up or down cast"<<std::endl);
		
		void* nptr = 
		  mFrom->bbGetOutput(mOutput)
		  .get_pointer_to(mTo->bbGetInput(mInput).pointed_type());
		if (!nptr)  
		  {
		    bbtkError("Connection '"
			      <<GetFullName()
			      <<"' : <"
			      <<HumanTypeName(mFrom->bbGetOutput(mOutput).type())
			      <<"> to <"
			      <<HumanTypeName(mTo->bbGetInputType(mInput))
			      <<"> : no adaptor available and run-time up and down cast failed");
		  }
		mTo->bbBruteForceSetInputPointer(mInput, nptr, false);
	      }
	    // 4) Nothing worked : error
	    else 
	      {
		bbtkError("Connection '"<<GetFullName()<<"' "
			  <<"no adaptor found to convert <"
			  <<HumanTypeName(mFrom->bbGetOutput(mOutput).type())
			  <<"> to <"
			  <<HumanTypeName(mTo->bbGetInputType(mInput))<<">");
	      }
	  }
      }
    // EO : mFromAny && ! mToAny
    // Default case : types are the same; we use simple get-set
    else 
      {
	// LG : Connection Update does not set mTo as modified
	mTo->bbSetInput(mInput, mFrom->bbGetOutput(mOutput),false);
      }

  }
  //==================================================================

  //==================================================================
  /// From.Output change propagation
  void Connection::OnOutputChange(bbtk::BlackBox::Pointer, const std::string&, 
				  IOStatus status)
  {
    bbtkCDebugMessage("change",2,
		     "==> Connection::OnOutputChange("
		     <<GetIOStatusString(status)<<")"
		     <<std::endl);
    if (mAdaptor) 
      {
	    BlackBoxInputConnector* ac = mAdaptor->bbGetInputConnectorMap().find("In")->second;
	    mAdaptor->bbSetStatusAndPropagate(ac,status);
      }
    
    mTo->bbSetStatusAndPropagate( mTo->bbGetInputConnectorMap().find(mInput)->second, status);
    
 
  }
  //==================================================================

  
  //==================================================================
  std::string Connection::GetFullName() const {
    if (mFrom && mTo) 
      {
	std::string res = mFrom->bbGetName()+"."+mOutput+"--"
	  +mTo->bbGetName()+"."+mInput;
	if ((!mOriginalFrom.expired()) && (!mOriginalTo.expired()) &&
	    ((mFrom!=mOriginalFrom.lock())||(mTo!=mOriginalTo.lock())))
	  {
	    res += "("+mOriginalFrom.lock()->bbGetName()
	      +"."+mOriginalOutput+"--"
	      + mOriginalTo.lock()->bbGetName()+"."+mOriginalInput+")";
	  }
	return res;
      }
    return "***Invalid Connection***";
  }
  //==================================================================

  //==================================================================
  void Connection::Check() const
  {
    bbtkMessage("debug",1,"** Checking Connection "<<(void*)this<<" ["
		<<GetFullName()<<"]"
		<<std::endl);
    if (mFrom==0) 
      {
	bbtkMessage("debug",2," - From = 0"<<std::endl);
      }
    else
      {
	bbtkMessage("debug",2," - From : "<<mFrom->bbGetName()<<std::endl);
	if (!mFrom->bbHasOutput(mOutput))
	  {
	    bbtkError("** Checking Connection "<<(void*)this
		       <<" ["<<GetFullName()<<"] : "
		      << mFrom->bbGetName()<<" does not have output '"
		      <<mOutput<<"'");
	  }	
	bbtkMessage("debug",2," - From : Output '"<<mOutput<<"' exists"<<std::endl);
	BlackBox::OutputConnectorMapType::const_iterator i 
	  = mFrom->bbGetOutputConnectorMap().find(mOutput);
	if (i== mFrom->bbGetOutputConnectorMap().end())
	  {
	     bbtkError("** Checking Connection "<<(void*)this
		       <<" ["<<GetFullName()<<"] : "
		       <<mFrom->bbGetName()<<" output '"
		       <<mOutput<<"' is not in OutputConnectorMap");
	  }
	bbtkMessage("debug",2," - From : Output '"<<mOutput
		    <<"' is in OutputConnectorMap"<<std::endl);

        std::vector< Connection* >::const_iterator j;
	/*
	for (j  = i->second->GetConnectionVector().begin();
	     j != i->second->GetConnectionVector().end();
	     ++j)
	  {
	    if ((*j)==this) break;
	  }
	*/
	j = find(i->second->GetConnectionVector().begin(),
		 i->second->GetConnectionVector().end(),
		 this);
       
	if (j==i->second->GetConnectionVector().end())
	  {
	    bbtkError("** Checking Connection "<<(void*)this
		      <<" ["<<GetFullName()<<"] : "
		      <<" OutputConnector '"
		      <<mOutput<<"' of "<<mFrom->bbGetName()
		      <<" does not point to this connection");
	    
	  }
	bbtkMessage("debug",2," - From : This connection is in OutputConnector connection vector"<<std::endl);
	bbtkMessage("debug",2," * Box from : Check successfull"<<std::endl);

      }

    if (mTo==0) 
      {
	bbtkMessage("debug",2," - To   = 0"<<std::endl);
      }
    else
      {
	bbtkMessage("debug",2," - To   : "<<mTo->bbGetName()<<std::endl);
	//	std::cout << mTo << std::endl;
	//	std::cout << mTo->bbGetDescriptor() << std::endl;
	//	std::cout << mTo->bbGetDescriptor()->GetTypeName() << std::endl;
	//	mTo->bbGetName();
	bbtkMessage("debug",2," - To   : "<<mTo->bbGetName()<<std::endl);
	if (!mTo->bbHasInput(mInput))
	  {
	    bbtkError("** Checking Connection "<<(void*)this
		      <<" ["<<GetFullName()<<"] : "
		      <<mTo->bbGetName()<<" does not have input '"
		      <<mInput<<"'");
	  }	
	bbtkMessage("debug",2," - To   : Input '"<<mInput<<"' exists"<<std::endl);
	BlackBox::InputConnectorMapType::const_iterator i 
	  = mTo->bbGetInputConnectorMap().find(mInput);
	if (i== mTo->bbGetInputConnectorMap().end())
	  {
	     bbtkError("** Checking Connection "<<(void*)this
		       <<" ["<<GetFullName()<<"] : "
		       <<mTo->bbGetName()<<" input '"
		       <<mInput<<"' is not in InputConnectorMap");
	  }
	bbtkMessage("debug",2," - To   : Input '"<<mInput
		    <<"' is in InputConnectorMap"<<std::endl);

	if (i->second->GetConnection()==0)
	  {
	    bbtkError("** Checking Connection "<<(void*)this
		      <<" ["<<GetFullName()<<"] : "
		      <<" InputConnector '"
		      <<mInput<<"' of "<<mTo->bbGetName()
		      <<" does not point to this connection");
    
	  }
	bbtkMessage("debug",2," - To   : This connection is in InputConnector connection vector"<<std::endl);
	bbtkMessage("debug",2," * Box to   : Check successfull"<<std::endl);

      }
  }
  //==================================================================
 //==========================================================================
  std::string Connection::GetObjectName() const
  {
    std::string s("Connection '");
    s += GetFullName();
    s += "'";
    return s;
  }
  //==========================================================================
  
  //==========================================================================
  std::string  Connection::GetObjectInfo() const 
  {
    std::stringstream i;
    return i.str();
  }
  //==========================================================================

 //==========================================================================
size_t  Connection::GetObjectSize() const 
{
  size_t s = Superclass::GetObjectSize();
  s += Connection::GetObjectInternalSize();
  return s;
  }
  //==========================================================================
  //==========================================================================
size_t  Connection::GetObjectInternalSize() const 
{
  size_t s = sizeof(Connection);
  return s;
  }
  //==========================================================================
  //==========================================================================
  size_t  Connection::GetObjectRecursiveSize() const 
  {
    size_t s = Superclass::GetObjectRecursiveSize();
    s += Connection::GetObjectInternalSize();
    return s;
  }
  //==========================================================================

}// namespace bbtk





