/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkData.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.5 $
=========================================================================*/

                                                                        
    

/**
 *\file
 *\brief  Defines Data and DataInfo 
 * 
 * Data is bbtk general type exchanged between black boxes (adapted from boost::any).
 * DataInfo is the bbtk type of object storing informations on a data type : includes C++ type info (std::type_info) and a string describing the "nature" of the data (what is the content of the structure). 
 */

#ifndef __bbtkData_h__
#define __bbtkData_h__


#include "bbtkAny.h"
#include "bbtkRTTI.h"

namespace bbtk
{
  
  /// The generic type of "data" exchanged between black boxes
  typedef any<thing> Data;

  /// Object storing informations on a data type : includes C++ type info (std::type_info) and a string describing the "nature" of the data (what is the content of the structure)
  class DataInfo
  {
  public:
    DataInfo( TypeInfo type, const std::string& nature="") 
      : mType(type), mNature(nature)
    {}

    ~DataInfo() {}

    TypeInfo GetType() const { return mType; }
    const std::string& GetNature() const { return mNature; }

    /// Equality
    bool operator== ( const DataInfo& k ) const
      {
	return ( (mType == k.mType)  &&
		 (mNature == k.mNature) );
      }
    /// Comparison
    bool operator< ( const DataInfo& k ) const
      {
	return ( ( mType.before(k.mType) ) ||
		 ( ( mType == k.mType ) &&
		   ( mNature.compare(k.mNature) < 0 ) ) );
      }


  private:
    DataInfo() : mType(typeid(void)), mNature("") {}
    TypeInfo mType;
    std::string mNature;

  };

  
  /// dump in a stream
  inline std::ostream& operator<<(std::ostream& s, const DataInfo& d) 
  {
    s << "<" << TypeName(d.GetType()) << "("<< d.GetNature()<<")>";
    return s;
  }



}// namespace bbtk

#endif

