/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkExecuter.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.34 $
=========================================================================*/

                                                                      

/**
 *  \file 
 *  \brief class Executer: level 0 of script execution (code)
 */

#include "bbtkExecuter.h"
#include "bbtkMessageManager.h"
#include "bbtkFactory.h"
#include "bbtkUtilities.h"
//#include "bbtkWx.h"
#include <fstream>

#ifdef USE_WXWIDGETS
#include <wx/textdlg.h>
#endif

#include "bbtkWxBlackBox.h"

#include "bbtkConfigurationFile.h"

namespace bbtk
{
  //=======================================================================
  Executer::Pointer Executer::New()
  {
    bbtkDebugMessage("object",9,"Executer::New()"<<std::endl);
    return MakePointer(new Executer());
  }
  //=======================================================================

  //=======================================================================
  Executer::Executer()
    : 
    mFactory(),
    mRootPackage(),
    mRootCBB(),
    mNoExecMode(false),
    mDialogMode(NoDialog),
    mNoErrorMode(false)
  {
    bbtkDebugMessage("object",2,"==> Executer()" <<std::endl);
    mFactory = Factory::New();
    // The smart pointer on this is not made yet (is made by New) 
    // -> create it to pass it to the factory
    // We have to "lock" the smart pointer because the factory
    // only keeps a weak pointer on the executer
    // -> this would auto-destroy !!
    mFactory->SetExecuter(MakePointer(this,true));
    Reset();
    bbtkDebugMessage("object",2,"<== Executer()" <<std::endl);
  }
  //=======================================================================

  //=======================================================================
  Executer::~Executer()
  {
     bbtkDebugMessage("object",2,"==> ~Executer()" <<std::endl);
     mOpenDefinition.clear();
     mOpenPackage.clear();
     mFactory->Reset();
     mFactory.reset();
     bbtkDebugMessage("object",2,"<== ~Executer()" <<std::endl);
  }
  //=======================================================================

  //=======================================================================
   /// Loads a package
    void Executer::LoadPackage(const std::string &name )
   {
     GetFactory()->LoadPackage(name);
   }
  //=======================================================================

  //=======================================================================
    /// Unloads a package
    void Executer::UnLoadPackage(const std::string &name )
    {
      GetFactory()->UnLoadPackage(name);
    }
  //=======================================================================

  //=======================================================================
  void Executer::Reset()
  {
    bbtkDebugMessage("kernel",9,"==> Executer::Reset()" <<std::endl);

    //    GetFactory()->Check();
 
    mOpenDefinition.clear();
    mOpenPackage.clear();

    //  Wx::DestroyTopWindow();

    GetFactory()->Reset();
#if(USE_WXWIDGETS)
    Wx::ProcessPendingEvents();
#endif
 
    // Create user package
    Package::Pointer p =
      Package::New("user","internal","User defined black boxes","");
    // Insert the user package in the factory
    GetFactory()->InsertPackage(p);
    // And in the list of open packages
    mOpenPackage.push_back(p);
    mRootPackage = p;

    // Create user workspace
    ComplexBlackBoxDescriptor::Pointer r = 
      ComplexBlackBoxDescriptor::New("workspace"); 
    //    mRootCBB->Reference();
    r->SetFactory(GetFactory());
    r->AddToAuthor("bbtk");
    r->AddToDescription("User's workspace");
    mOpenDefinition.push_back(CBBDefinition(r,"user"));
    // Register it into the user package
    p->Register(r);
    mRootCBB = r;

    //    Object::PrintObjectListInfo();
    //  GetFactory()->CheckPackages();
    bbtkDebugMessage("kernel",9,"<== Executer::Reset()" <<std::endl);
  }
  //=======================================================================


  //=======================================================================
  /// changes the workspace name
  void Executer::SetWorkspaceName( const std::string& n )
  {
    GetUserPackage()->ChangeDescriptorName( GetWorkspace()->GetTypeName(), n );
  }
  //=======================================================================

  //=======================================================================
  void Executer::BeginPackage (const std::string &name)
  {
     bbtkDebugMessage("kernel",9,"==> Executer::BeginPackage(\""<<name<<"\")"
                        <<std::endl);
     Package::Pointer p;
     try 
      {
         p = GetFactory()->GetPackage(name);
      }
    catch (Exception e)
      {
	p = Package::New(name,"","","");
	GetFactory()->InsertPackage(p);
      }
     mOpenPackage.push_back(p);

     bbtkDebugMessage("kernel",9,"<== Executer::BeginPackage(\""<<name<<"\")"
		      <<std::endl);
  }
  //=======================================================================

  //=======================================================================
  void Executer::EndPackage()
  {
    if (mOpenPackage.size()>1) mOpenPackage.pop_back();
  }
  //=======================================================================

  //=======================================================================
  void Executer::Define (const std::string &name,
                         const std::string &pack,
                         const std::string &scriptfilename)
  {
    bbtkDebugMessage("kernel",9,"==> Executer::Define(\""<<name<<
		     ","<<pack<<"\")"
		     <<std::endl);

    ComplexBlackBoxDescriptor::Pointer b 
      = ComplexBlackBoxDescriptor::New(name);
    b->SetFactory(GetFactory());
    b->SetScriptFileName(scriptfilename);
    mOpenDefinition.push_back( CBBDefinition( b, pack ) );
    
    bbtkDebugMessage("kernel",9,"<== Executer::Define(\""<<name<<
		     ","<<pack<<"\")"
		     <<std::endl);
  }
  //=======================================================================

  //=======================================================================
  /// Sets the file name to use for the current definition
  /// (Used to set it after the Define command)
  void Executer::SetCurrentFileName (const std::string &name )
  {
    mOpenDefinition.back().box->SetScriptFileName(name);
  }
  //=======================================================================

	void Executer::SetTypeOfScript_Application ()
	{
		mOpenDefinition.back().box->SetTypeOfScript_Application();
	}
	
	
	
  //=======================================================================
  void Executer::Clear()
  {
    bbtkDebugMessage("kernel",9,"==> Executer::Clear()" <<std::endl);
    GetCurrentDescriptor()->GetPrototype()->Clear();
    bbtkDebugMessage("kernel",9,"<== Executer::Clear()" <<std::endl);
  }
  //=======================================================================

  //=======================================================================
  void Executer::EndDefine ()
  {
    bbtkDebugMessage("kernel",9,"==> Executer::EndDefine(\""
		     <<GetCurrentDescriptor()->GetTypeName()<<"\")" 
		     <<std::endl);
    // Does current package exist ?
    Package::Pointer p;
    std::string pname(mOpenDefinition.back().package);
    if (pname.size()>0)
      {
	try
	  {
	    p = GetFactory()->GetPackage(pname);
	  }
	catch (Exception e)
	  {
	    p = Package::New(pname,"","","");
	    GetFactory()->InsertPackage(p);
	  }
      }
    else
      {
	p = mOpenPackage.back().lock();
      }
    // Register the descriptor in the current package
    p->Register(GetCurrentDescriptor());
    
    bbtkDebugMessage("kernel",9,"<== Executer::EndDefine(\""
		     <<GetCurrentDescriptor()->GetTypeName()<<"\")" 
		     <<std::endl);
    mOpenDefinition.pop_back();
  }
  //======================================================================= 

  //=======================================================================  
  void Executer::Kind(const std::string& kind)
  {
    if (kind=="ADAPTOR")
      {
	GetCurrentDescriptor()->AddToCategory("adaptor");
	GetCurrentDescriptor()->SetKind(bbtk::BlackBoxDescriptor::ADAPTOR);
      }
    else if (kind=="DEFAULT_ADAPTOR")
      {
	GetCurrentDescriptor()->AddToCategory("adaptor");
	GetCurrentDescriptor()->SetKind(bbtk::BlackBoxDescriptor::DEFAULT_ADAPTOR);
      }
    if (kind=="GUI")
      {
	GetCurrentDescriptor()->AddToCategory("gui");
	GetCurrentDescriptor()->SetKind(bbtk::BlackBoxDescriptor::GUI);
      }
    else if (kind=="DEFAULT_GUI")
      {
	GetCurrentDescriptor()->AddToCategory("gui");
	GetCurrentDescriptor()->SetKind(bbtk::BlackBoxDescriptor::DEFAULT_GUI);
      }
    else
      {
	bbtkError("Unknown box kind : '"<<kind<<"'. "
		  <<"Valid kinds are 'ADAPTOR','DEFAULT_ADAPTOR',"
		  <<"'GUI','DEFAULT_GUI'");
      }
  }
  //=======================================================================

  //=======================================================================
  void Executer::Create ( const std::string& nodeType, 
                          const std::string& nodeName)
  {
     GetCurrentDescriptor()->Add(nodeType,nodeName);
  }
  //=======================================================================

  //=======================================================================
  void Executer::Destroy(const std::string &boxName)
  {
    GetCurrentDescriptor()->Remove(boxName,true);
  }
  //=======================================================================

  //=======================================================================
  void Executer::Connect (const std::string &nodeFrom,
                          const std::string &outputLabel,
                          const std::string &nodeTo, 
                          const std::string &inputLabel)
  {
    GetCurrentDescriptor()->Connect(nodeFrom, outputLabel, nodeTo, inputLabel);
  }
  //=======================================================================

  //=======================================================================
  void Executer::Execute (const std::string &nodeName) 
  {
    // if in root
    if (GetCurrentDescriptor()==GetWorkspace()) 
     {
        if (!mNoExecMode) 
        {
           GetCurrentDescriptor()->GetPrototype()->bbGetBlackBox(nodeName)->bbExecute(true);
        }
     }
     else 
     {
        GetCurrentDescriptor()->AddToExecutionList(nodeName) ;
     }
  }
  //=======================================================================

  //=======================================================================
  void Executer::DefineInput ( const std::string &name,
                               const std::string &box,
                               const std::string &input,
                               const std::string& help)
  {
    // If the input is defined in the Root box
    if (GetCurrentDescriptor()==GetWorkspace()) 
      {
      // If the dialog mode is set to NoDialog
      // and the user passed the name in the Inputs map 
      // then the associated value is set to the box.input
      // This is the way command line parameters are passed to the Root box
         if (mDialogMode == NoDialog) 
         {
         // find if name is in mInputs
            std::map<std::string,std::string>::iterator i;
            i = mInputs.find(name);
            if (i!=mInputs.end()) {
               Set(box,input,(*i).second);
            }
         }
        // If the dialog mode is set to TextDialog
        // The user is prompted for the value
        else if (mDialogMode == TextDialog) 
        {
           std::cout << name << "=";
           std::string ans;
           std::cin >> ans;
           Set(box,input,ans);
        }
#ifdef USE_WXWIDGETS
       // If the dialog mode is set to GraphicalDialog
       // A dialog box is pop up
       else if (mDialogMode == GraphicalDialog) 
       {
          std::string mess("Enter the value of '");
          mess += name;
          mess += "' (";
          mess += help;
          mess += ")";
          std::string title(name);
          title += " ?";
          std::string ans = wx2std ( wxGetTextFromUser( std2wx (mess), std2wx(title)));
          Set(box,input,ans); 
       }
#endif
    }

    GetCurrentDescriptor()->DefineInput(name,box,input,help);

  }
  //=======================================================================

  //=======================================================================
   void Executer::DefineOutput ( const std::string &name,
                                 const std::string &box,
                                 const std::string &output,
                                 const std::string& help)
  {
    GetCurrentDescriptor()->DefineOutput(name,box,output,help);
  }
  //=======================================================================

  //=======================================================================
  void Executer::Set (const std::string &box,
                      const std::string &input,
                      const std::string &value)
  {
    BlackBox::Pointer b = GetCurrentDescriptor()->GetPrototype()->bbGetBlackBox(box);
    // Looks for the adaptor

    if ( ( b->bbGetInputType(input) != typeid(bbtk::any<bbtk::thing>) )&&
	 ( b->bbGetInputType(input) != typeid(std::string) ) )
      {
	BlackBox::Pointer a =
	   GetFactory()->NewAdaptor(typeid(std::string),
				    b->bbGetInputType(input),
				    "tmp");
         if (!a) 
	   {
	     bbtkError("No <"<<
		       TypeName(b->bbGetInputType(input))
		       <<"> to <std::string> found");
	   }
         std::string v(value);
         a->bbSetInput("In",v);
         a->bbExecute();
         b->bbSetInput(input,a->bbGetOutput("Out"));
	 //         a->Delete();
      }
    else 
      {
      std::string v(value);
      b->bbSetInput(input,v);
      }
  }
  //=======================================================================

  //=======================================================================
  std::string Executer::Get(const std::string &box,
                            const std::string &output)
  {
    BlackBox::Pointer b = GetCurrentDescriptor()->GetPrototype()->bbGetBlackBox(box);
    // Looks for the adaptor
    if (b->bbGetOutputType(output) != typeid(std::string)) 
      {
	BlackBox::Pointer a =
          GetFactory()->NewAdaptor(
				   b->bbGetOutputType(output),
				   typeid(std::string),
				   "tmp");
	if (!a) 
	  {
	    bbtkError("No <"<<
		      TypeName(b->bbGetOutputType(output))
		      <<"> to <std::string> found");
	  }
        b->bbExecute();
	
        a->bbSetInput("In",b->bbGetOutput(output));
        a->bbExecute();
        std::string r = a->bbGetOutput("Out").unsafe_get<std::string>();
	//std::string v = *((std::string*)a->bbGetOutput("Out")) ;
	//   std::cout << a->bbGetOutput("Out").unsafe_get<std::string>() 
	//             << std::endl;
	//std::string v(value);
	//b->bbSetInput(input,a->bbGetOutput("Out"));
	//        a->bbDelete();
        return r;
      }
    else
      {
	b->bbExecute();
	return b->bbGetOutput(output).unsafe_get<std::string>();
	// std::string v = *((std::string*)b->bbGetOutput(output)) ;
	// std::cout << b->bbGetOutput("Out").unsafe_get<std::string>() 
	//   << std::endl;
	// b->bbSetInput(input,&v);
      }
  }
  //=======================================================================

  //=======================================================================
  void Executer::Author(const std::string &authorName)
  {
    GetCurrentDescriptor()->AddToAuthor(authorName,GetCurrentDescriptor()==GetWorkspace());
  }
  //=======================================================================

  //=======================================================================
  void Executer::Category(const std::string &category)
  {
    GetCurrentDescriptor()->AddToCategory(category,GetCurrentDescriptor()==GetWorkspace());
  }
  //=======================================================================

  //=======================================================================
  void Executer::Description(const std::string &d)
  {
    GetCurrentDescriptor()->AddToDescription(d,GetCurrentDescriptor()==GetWorkspace());
  }
  //=======================================================================



  //=======================================================================
  /// prints the list of the boxes of the current descriptor
  void Executer::PrintHelpListBoxes()
  {
    bbtkMessage("help",1,"The black box descriptor \""
		<<GetCurrentDescriptor()->GetTypeName()<<"\" contains : "<<std::endl);
    GetCurrentDescriptor()->PrintBlackBoxes();
 }
  //=======================================================================

  //=======================================================================
  std::string Executer::ShowGraph(const std::string &nameblackbox, 
				  const std::string &detailStr, 
				  const std::string &levelStr,
				  const std::string &output_html,
				  const std::string &custom_header,
				  const std::string &custom_title,
				  bool system_display )
  {
    int detail	=	atoi(detailStr.c_str());
    int level 	=	atoi(levelStr.c_str());
	  bool relative_link = true;

    Package::Pointer p;
    try
    {
       p = GetFactory()->GetPackage(nameblackbox);
    }
    catch (Exception e)
    {
      p = GetUserPackage();
    }
	  
      std::string doc_path = bbtk::ConfigurationFile::GetInstance().Get_doc_path();
      doc_path += bbtk::ConfigurationFile::GetInstance().Get_file_separator();
      doc_path += "bbdoc";
      doc_path += bbtk::ConfigurationFile::GetInstance().Get_file_separator();
	  
	  std::string pack_name(p->GetName());
	  std::string pack_path = doc_path + pack_name;
	  // Creating directory
	  if ( ! bbtk::Utilities::FileExists(pack_path) )
	  {
		  std::string command("mkdir \"" +pack_path+ "\"");
		  system( command.c_str() );
	  }
	  std::string pack_index(pack_path);
	  pack_index += bbtk::ConfigurationFile::GetInstance().Get_file_separator();
	  pack_index += "index.html"; 
	  
	  
    // Generating documentation-help of workspace
	p->SetDocURL(pack_index);
    p->SetDocRelativeURL("index.html");
	p->CreateHtmlPage(pack_index,"bbtk","user package",custom_header,custom_title,detail,level,relative_link);
	  
    /*
    try 
    {
       ShowGraphTypes(nameblackbox);
    }
    catch (bbtk::Exception a)
    {
       std::cout <<"EXC"<<std::endl;
       page = ShowGraphInstances(nameblackbox,detail,level,system_display);
    }
    */
    return pack_index;
  }
  //=======================================================================

  //=======================================================================
  /// Generate a png file with the actual pipeline (Graphviz-dot needed)
  std::string Executer::ShowGraphInstances(const std::string &nameblackbox, int detail, int level,
                                           bool system_display)
  {

    BlackBox::Pointer blackbox;
    if (nameblackbox==".")
    {
       blackbox = GetCurrentDescriptor()->GetPrototype();
    }
    else
    {
       blackbox = GetCurrentDescriptor()->GetPrototype()->bbFindBlackBox(nameblackbox);
    }
    
    std::string page;

    if (blackbox)
      {      
	// Don't pollute the file store with  "temp_dir" directories ...
	std::string default_doc_dir = ConfigurationFile::GetInstance().Get_default_temp_dir();
	char c = default_doc_dir.c_str()[strlen(default_doc_dir.c_str())-1];

	std::string directory = default_doc_dir; 
	if (c != '/' && c !='\\') directory = directory + "/";

	directory = directory +  "temp_dir";
	//std::string directory("temp_dir");
	std::string filename(directory + "/" + "bbtk_graph_pipeline");
	std::string filename_html(filename+".html");
	std::string command0("mkdir \""+directory + "\"");

#if defined(_WIN32)
	std::string command2("start ");
#else 
	std::string command2("gnome-open ");
#endif

	command2=command2+filename_html;
	page = filename_html;
	// 1. Generate Html Diagram
	std::ofstream s;
	s.open(filename_html.c_str());
	if (s.good()) 
	  {
	    s << "<html><head><title>BBtk graph diagram</title><meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\"></head>\n";
	    s << "<body bgcolor=\"#FFFFFF\" text=\"#000000\"> \n\n";
	    if ( blackbox->bbGetName()=="workspacePrototype" )
	      {
		s << "<center>Current workspace</center>";
	      } else {
	      s << "<center>" << blackbox->bbGetName()<< "</center>";
	    } 
	    blackbox->bbInsertHTMLGraph( s, detail, level, true, directory, false );
	    s << "</body></html>\n";
	  }
	s.close();
	
	// 2. Starting Browser
	if (system_display) system( command2.c_str() );      
      } 
    else 
      {
	bbtkMessage("help",1,"No black box: \""
		    <<nameblackbox<<"\" " <<std::endl);
      }
    return page;
  }
  //=======================================================================

  //=======================================================================
  void Executer::PrintHelpBlackBox(const std::string &nameblackbox, 
			       const std::string &detailStr, 
			       const std::string &levelStr)
  {
    bool found=false;
    
    int detail = atoi(detailStr.c_str());
    int level  = atoi(levelStr.c_str());
    BlackBox::Pointer blackbox;
    if (nameblackbox.compare(".")==0)
      {
	blackbox=GetCurrentDescriptor()->GetPrototype();
      } 
    else 
      {
	blackbox = GetCurrentDescriptor()->GetPrototype()->bbFindBlackBox(nameblackbox);
      }
    
    if (blackbox)
      {
	found=true;
	blackbox->bbPrintHelp(blackbox,detail,level); //,mFactory);
      }
    
    if (!found) 
      {
	bbtkError("box with name '"  <<nameblackbox<<"' unknown");
      }
  }
  //=======================================================================

  //=======================================================================
  /// sets the level of message
  void Executer::SetMessageLevel(const std::string &kind,
				 int level)
  {
    bbtk::MessageManager::SetMessageLevel(kind,level);
  }
  //=======================================================================

  //=======================================================================
  /// Prints help on the messages
  void  Executer::HelpMessages()
  {
    bbtk::MessageManager::PrintInfo();
  }
  //=======================================================================

  //=======================================================================
  ///
  void Executer::Print(const std::string &str)
  {  
    if (GetNoExecMode() &&  (GetCurrentDescriptor()==GetWorkspace()) ) return;
    if (GetCurrentDescriptor()!=GetWorkspace()) return;

    bbtkDebugMessage("kernel",9,"Executer::Print(\""<<str<<"\")"<<std::endl);

 // TO DO :
 // InterpretLine ("load std")
 // InterpretLine("new ConcatStrings _C_ ") -> trouver un nom unique : # commande 
 // InterpretLine("new Print _P_") 
 // InterpretLine("connect _C_.Out _P_.In")
 // int num = 1
 

    std::vector<std::string> chains;
    std::string delimiters("$");

    // Skip delimiters at beginning.
    std::string::size_type lastPos = str.find_first_not_of(delimiters, 0);
    bool is_text = true;
    if (lastPos>0) is_text = false;

    // Find first delimiter.
    std::string::size_type pos     = str.find_first_of(delimiters, lastPos);

    while (std::string::npos != pos || std::string::npos != lastPos)
    {
       if (is_text) 
       {
          // Found a text token, add it to the vector.
          chains.push_back(str.substr(lastPos, pos - lastPos));
 // std::string token = str.substr(lastPos, pos - lastPos)
 // InterpretLine("set _C_.In%num% %token%")
 
       }
       else 
       {

       // is an output (between $$) : decode 
         std::string tok,box,output;
         tok = str.substr(lastPos, pos - lastPos);
         Utilities::SplitAroundFirstDot(tok,box,output);
         chains.push_back( Get(box,output) );

// InterpretLine("connect %tok% _C_.In%num%") 

       }
        // Skip delimiters.  Note the "not_of"
       lastPos = str.find_first_not_of(delimiters, pos);
        // Find next delimiter
       pos = str.find_first_of(delimiters, lastPos);
    //
       is_text = !is_text;
// num ++;
     }
// InterpretLine("exec _P_")
// if (IS_IN_WORKSPACE) InterpretLine("delete _C_; delete _P_");

    std::vector<std::string>::iterator i;
    for (i= chains.begin(); i!=chains.end(); ++i) 
      {
	Utilities::SubsBackslashN(*i);
	bbtkMessage("output",1,*i);
      }
    bbtkMessage("output",1,std::endl);
  }
  //==========================================================================

  //==========================================================================
  std::string Executer::GetObjectName() const
  {
    return std::string("Executer");
  }
  //==========================================================================
  
  //==========================================================================
  std::string  Executer::GetObjectInfo() const 
  {
    std::stringstream i;
    return i.str();
  }
  //==========================================================================
  //==========================================================================
size_t  Executer::GetObjectSize() const 
{
  size_t s = Superclass::GetObjectSize();
  s += Executer::GetObjectInternalSize();
  return s;
  }
  //==========================================================================
  //==========================================================================
size_t  Executer::GetObjectInternalSize() const 
{
  size_t s = sizeof(Executer);
  return s;
  }
  //==========================================================================
  //==========================================================================
  size_t  Executer::GetObjectRecursiveSize() const 
  {
    size_t s = Superclass::GetObjectRecursiveSize();
    s += Executer::GetObjectInternalSize();
    s += mFactory->GetObjectRecursiveSize();
    return s;
  }
  //==========================================================================
}//namespace
