/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkExecuter.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.20 $
=========================================================================*/

                                                                      

/**
 *  \file 
 *  \brief class Executer: level 0 of script execution (header)
 */
/**
 *  \class bbtk::Executer 
 *  \brief class Executer: level 0 of script execution 
 */
 
#ifndef __bbtkExecuter_h__
#define __bbtkExecuter_h__

#include "bbtkVirtualExec.h"

#include "bbtkSystem.h"
#include "bbtkComplexBlackBox.h"
#include "bbtkFactory.h"
#include <string>
#include <deque>

namespace bbtk
{

  class Interpreter;


  class /*BBTK_EXPORT*/ Executer : public VirtualExec
  {
    BBTK_OBJECT_INTERFACE(Executer);
    typedef VirtualExec Superclass;
  public:
    static Pointer New();
    

    /// Gets the factory used by the executer
    Factory::Pointer GetFactory() { return mFactory; }
    /// Gets the factory used by the executer (const)
    //    const Factory::Pointer GetFactory() const { return mFactory; }


    /// Sets the inputs of the workspace : 
    void SetInputs(const std::map<std::string,std::string>& m) { mInputs = m; }

    /// Puts the executer in "no exec" mode, 
    /// which creates but does not execute pipelines 
    void SetNoExecMode(bool b) { mNoExecMode = b; }

    bool GetNoExecMode() const { return mNoExecMode; }

    /// Sets the mode of dialog of the executer for Root inputs 
    void SetDialogMode(DialogModeType t) { mDialogMode = t; }

    void SetNoErrorMode(bool b) { mNoErrorMode = b; }

    bool GetNoErrorMode() const { return mNoErrorMode; }


    //=================================================================
    /// Loads a package
    void LoadPackage(const std::string &name );

    /// Unloads a package
    void UnLoadPackage(const std::string &name );

    /// Starts a package block	
    void BeginPackage (const std::string &name );

    /// Ends a package block
    void EndPackage ();

    /// Starts the definition of a new ComplexBlackBox in package pack
    /// scriptfilename is the file from which the def is read
    void Define (const std::string &name,
		 const std::string& pack,
		 const std::string &scriptfilename);

    /// Sets the file name to use for the current definition
    /// (Used to set it after the Define command)
    void SetCurrentFileName (const std::string &name );

	//Set type of script
	void SetTypeOfScript_Application();
 
	  
	  
    /// End the definition of a ComplexBlackBox
    void EndDefine ();

    /// Sets the kind of the currently defined ComplexBlackBox
    void Kind(const std::string& kind);

    /// Creates a new black box in current complex box
    void Create ( const std::string& boxType, const std::string& boxName);

    /// Destroys a black box
    void Destroy (const std::string &boxName);

    /// Clears the currently defined ComplexBlackBox
    void Clear();

    /// Connects the output boxOutput to the input boxInput
    void Connect (const std::string &boxfrom,
		  const std::string &output,
		  const std::string &boxto,
		  const std::string &input);

    /// Executes the box 
    void Execute (const std::string &box);

    /// Defines an input of the current complex box
    void DefineInput (const std::string &name,
		      const std::string &box,
		      const std::string &input,
		      const std::string &help);

    /// Defines an output of the current complex box
    void DefineOutput (const std::string &name,
		       const std::string &box,
		       const std::string &output,
		       const std::string &help);     

    /// sets the input of the box with the value
    void Set (const std::string &box, 
	      const std::string &input, 
	      const std::string &value);

    /// gets the output of the box
    std::string Get (const std::string &box, 
		     const std::string &output);

    /// changes the workspace name
    void SetWorkspaceName( const std::string& n );

    ///Adds the authorName to the Box author list
    void Author(const std::string &authorName);

    ///Adds the Categories to the Box category list
    void Category(const std::string &category);

    /// The description string which explains what does the ComplexBox
    void Description(const std::string & d);






    /// prints the list off the boxes of the current box
    void PrintHelpListBoxes();

   /// Generate a HTML with a gif file with the current pipeline (Graphviz-dot needed). Returns the file path
    std::string ShowGraph(const std::string &nameblackbox, 
			  const std::string &detailStr, 
			  const std::string &levelStr,
			  const std::string &output_file,
			  const std::string &custom_header,
			  const std::string &custom_title,
			  bool system_display = true);

    /// Generate a HTML with a gif file with the current pipeline (Graphviz-dot needed). Returns the file path
    std::string ShowGraphInstances(const std::string &nameblackbox, int detail, int level, bool system_display=true);
    
    /// Prints help on a black box
    void PrintHelpBlackBox(const std::string &nameblackbox, const std::string &detailStr, const std::string &levelStr);
    





    /// Resets the workspace : destroys all boxes / unloads all packages
    void Reset();

    /// Sets the level of message for kind
    void SetMessageLevel(const std::string &kind,
                         int level);

   /// Prints help on the messages
    void HelpMessages();
    

    void Print(const std::string & message);

  //  static const std::string& GetObjectDescription();
  //  { static std::string s("Executer"); return s; }


    ComplexBlackBoxDescriptor::Pointer GetWorkspace() 
    { return mRootCBB.lock(); }
    Package::Pointer GetUserPackage() 
    { return mRootPackage.lock(); }

    /// Gets the current working black box 
    ComplexBlackBoxDescriptor::Pointer GetCurrentDescriptor() 
    { return mOpenDefinition.back().box; }
    

  protected:

  private:

    
    /// Returns true when we are inside a define/endefine block
    //    bool InDefinitionBlock() { return (mOpenDefinition.size()>1); }

    //==================================================================
    // ATTRIBUTES
     
    /// The factory used
    Factory::Pointer mFactory;
    
 
    /// The Root Package 
    /// Contains the root ComplexBlabkBox
    /// In which ComplexBlackBoxes are put by default
    /// Its name in bbi is 'user'
    Package::WeakPointer mRootPackage;
    
    /// The root ComplexBlackBox
    /// in which operations are done when outside a define/endefine block
    /// Its name in bbi is 'workspace'  
    ComplexBlackBoxDescriptor::WeakPointer mRootCBB;

    /// Struct that stores info on user defined complex black boxes
    struct CBBDefinition
    {
      ComplexBlackBoxDescriptor::Pointer box;
      std::string package;
      CBBDefinition(ComplexBlackBoxDescriptor::Pointer d, 
		    const std::string& p )
        : box(d), package(p) {}
    };

    /// The stack of current working ComplexBlackBox
    /// (is a stack for nested definitions)
    /// only contains the root cbb when outside a define/endefine block
    std::deque<CBBDefinition> mOpenDefinition;

    /// The stack of current working package
    /// (is a stack for nested definitions)
    std::deque<Package::WeakPointer> mOpenPackage;

    /// The input values of the Root ComplexBlackBox
    std::map<std::string,std::string> mInputs;
    
    /// no exec mode flag
    bool mNoExecMode;


    /// Dialog mode
    DialogModeType mDialogMode;

    /// no error mode flag
    bool mNoErrorMode;

  };
}
#endif
