/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkFactory.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.55 $
=========================================================================*/

                                                                       

/**
 *\file
 *\brief  Class bbtk::Factory : can load and unload dynamic libraries containing 
 *        black boxes packages and create instances of the black boxes registered 
 *       in the packages loaded.
 */
#include "bbtkFactory.h"
#include "bbtkMessageManager.h"
#include "bbtkConnection.h"
#include "bbtkConfigurationFile.h"
#include "bbtkUtilities.h"
#include "bbtkConfigurationFile.h"

#include <sys/stat.h> // for struct stat stFileInfo

#if defined(_WIN32)
#include <direct.h> // for getcwd
#endif

#include <cctype>    // std::toupper

#include <time.h>

namespace bbtk
{

  //===================================================================
  /// Default ctor
  Factory::Pointer Factory::New()
  {
    bbtkDebugMessage("kernel",9,"Factory::New()"<<std::endl);
    return MakePointer(new Factory());
  }
  //===================================================================

  //===================================================================
  /// Default ctor
  Factory::Factory()
    : mExecuter()
  {
    bbtkDebugMessage("kernel",7,"Factory()"<<std::endl);
  }
  //===================================================================

  //===================================================================
  /// Dtor
  Factory::~Factory()
  {
    bbtkDebugMessage("kernel",7,"==> ~Factory()"<<std::endl);
    CloseAllPackages();
    bbtkDebugMessage("kernel",7,"<== ~Factory()"<<std::endl);
  }
  //===================================================================



  //===================================================================
  void Factory::Reset()
  {
    bbtkDebugMessage("kernel",7,"==> Factory::Reset()"<<std::endl);
    CloseAllPackages();
    bbtkDebugMessage("kernel",7,"<== Factory::Reset()"<<std::endl);
  }
  //===================================================================


  // ===================================================================
  bool Factory::DoLoadPackage(std::string libname,
                              std::string pkgname,
                              std::string path)
  {
    
    Package::Pointer p = Package::CreateFromDynamicLibrary(libname,
							   pkgname,
							   path);
    if (p!=0)
      {
	//===================================================================
	bbtkMessage("output",2,p->GetName()<<" "
		    <<p->GetVersion()
		    <<" "
		    <<p->GetAuthor() << " Category(s) :"
		    <<p->GetCategory()
		    <<std::endl);
	bbtkMessage("output",2,p->GetDescription()<<std::endl);
	//===================================================================
	p->AddFactory(GetThisPointer<Factory>());
	mPackageMap[pkgname] = p;
	return true;
      }
    return false;
    
  }
  
  //===================================================================
  /// \brief Loads a package.
  ///
  /// The name is the system-independant name of the package (the name of the instance of bbtk::Package).
  /// Tries to open the dynamic library :
  /// - "libbb<name>.so" for linux systems,
  /// - "bb<name>.dll" for windows systems.
  /// If it succeeds, then tries to load the symbols "<name>GetPackage" and "<name>DeletePackage".
  /// "<name>GetPackage" is called to get the pointer on the bbtk::Package of the library
  /// ("<name>DeletePackage" is not used, its presence is just checked before loading the package).
  
  /// now, filename is only the last name (no longer the full name!)
  /// it will be searched within *all* the paths given in bbtk_config.xml
  

  
  void Factory::LoadPackage( const std::string& name )
  {
	  
	  
  // Note : in the following :
  // name : the user supplied name
  //      - abreviated name    e.g.       pkg   pkg.so   libbpkg   libbbpkg.so
  //      - relative full name e.g.       ./libbbpkg.so   ../../libbbpkg.so
  //      - absolute full name e.g.       /home/usrname/proj/lib/libbbpkg.so
  //          same for Windows, with      c:, d: ...
  //
  // lastname : string before the last / (if any), or user supplied name
	  
	 
	  
	if(name != ""){
		bbtkDebugMessageInc("kernel",7,"Factory::LoadPackage(\""<<name<<"\")"<<std::endl);
		bbtkMessage("debug",1,"Factory::LoadPackage(\""<<name<<"\")"<<std::endl);

		std::vector<std::string> package_paths;
		std::string libname;  // full path library name
		std::string pkgname;  // e.g. libbb<pkgname>.so

		std::string upath;

		pkgname = Utilities::ExtractPackageName(name,upath);
				
		bbtkMessage("debug",1,"Package name ["<<pkgname<<"]"<<std::endl);
		bbtkMessage("debug",1,"Package path ["<<upath<<"]"<<std::endl);

		// no loading package if already loaded
		PackageMapType::iterator iUnload;
		iUnload = mPackageMap.find(pkgname);

//EED Debuging message
//     std::cout << "EED Factory::LoadPackage " <<  std::endl;
//		PackageMapType::iterator it;
//		for ( it=mPackageMap.begin() ; it != mPackageMap.end(); it++ )
//			std::cout << (*it).first << " ";
//		std::cout << std::endl;

		
		if (iUnload != mPackageMap.end())
		{
		  bbtkMessage("output",2,"["<< pkgname <<"] already loaded" << std::endl);
		  return;
		}

	// =================================================
	// The following structure was checked to work
	// with any type of relative/absolute path.
	// Please don't modify it without checking
	// *all* the cases. JP
	//==================================================

		bool ok = false;
		bool foundFile = false;

		// If path provided by user will be the first scanned :
		// push it into vector of paths
		if (upath.length()>0)   // ------------------------------------- check user supplied location
		{
		   if (name[0] != '.' && name[0] != '/' && name[1]!= ':')
		   {
			  bbtkError("Use absolute or relative path name! ["<<name<<"] is an illegal name");
			  return;
		   }

		  // std::string path = Utilities::ExpandLibName(upath, false);

		   std::string path = Utilities::ExpandLibName(name,false); // keep last item, here.
		   if (path != "")
		   {
			  std::string p2;
			  Utilities::ExtractPackageName(path,p2);
			  //libname = Utilities::MakeLibnameFromPath(path, pkgname);
			  libname = Utilities::MakeLibnameFromPath(p2, pkgname); // remove last item
			  // Check if library exists
			  if ( !Utilities::FileExists(libname) )
			  {
			  // The following is *NOT* a debug time message :
			  // It's a user intended message.
			  // Please don't remove it.
			bbtkMessage("output",3,"   [" <<libname 
				<<"] : doesn't exist" <<std::endl);
			  }
			  else
			  {
				 ok = DoLoadPackage( libname, pkgname, path);         
			  }
		   }
		   else
		   {
			  bbtkError("Path ["<<upath<<"] doesn't exist");
			  return;
		   }
		}
		else     // ----------------------------------------------------- iterate on the paths  
		{

		std::string path = ".";
		package_paths = ConfigurationFile::GetInstance().Get_package_paths();
		std::vector<std::string>::iterator i;
		for (i=package_paths.begin();i!=package_paths.end();++i)
		{
			foundFile = false;
			path = *i;
//std::cout<<"JCP bbtkFactory.cxx void Factory::LoadPackage = path "<<path<<std::endl;
			// we *really* want '.' to be the current working directory
			if (path == ".")
			{
			  char buf[2048]; // for getcwd
			  char * currentDir = getcwd(buf, 2048);
			  std::string cwd(currentDir);
//std::cout<<"JCP  bbtkFactory.cxx void Factory::LoadPackage = currentDir "<<currentDir<<std::endl;
			  path = currentDir;
			}
//std::cout<<"JCP bbtkFactory.cxx void Factory::LoadPackage = path "<<path<<" pkgnam="<<pkgname<<std::endl;	         
			libname = Utilities::MakeLibnameFromPath(path, pkgname);

			bbtkMessage("debug",2,"-> Trying to load [" << libname << "]" <<std::endl);

		  // Check if library exists           
			if ( !Utilities::FileExists(libname) )
			{
			// The following is *NOT* a debug time message :
			// It's a user intended message.
			// Please don't remove it.
			  bbtkMessage("output",3,
				  "   [" <<libname <<"] : doesn't exist" <<std::endl);
			   continue;  // try next path
			}
			foundFile = true; 

		  // Try to Load the library

			ok = DoLoadPackage( libname, pkgname, path);
			if (ok)
			{
			   bbtkMessage("debug",2,"   OK"<<std::endl);
			}
		break; // we stop iterating even if error : have to signal it to user
		} //------------------ // end for ( package_paths.begin();i!=package_paths.end() )

	}

		if( !ok )  // nothing was loaded
		{
		  if (!foundFile)
		  {
			bbtkError("Could not find package ["<<pkgname<< "]");
		  }
		  else
		  {
	#if defined(__GNUC__)
			bbtkError("Could not load package ["<< pkgname
					  <<"] :" << std::endl 
			  << "  Opening "<<libname<<" failed"
			  << "  Reason: "<< dlerror());
	#elif defined(_WIN32)
			bbtkError("Could not load package ["<<pkgname
			  <<"] :"<< std::endl << "   Error loading " <<libname);

		// look how to get the error message on win
		//<<dlerror());
		// it is the bordel !! (the bloody fucking bordel, you mean?)
		// look : http://msdn2.microsoft.com/en-us/library/ms680582.aspx
	#endif
		  }
		}	
		bbtkMessage("output",2,"[" << libname << "] loaded" << std::endl);
	}

    

  }

  //===================================================================
  /// \brief UnLoads a package.
  ///
  /// The package must have been previously loaded by LoadPackage.
  /// If the entry is found in the map, calls ClosePackage
 void Factory::UnLoadPackage( const std::string& name )
 {
    bbtkDebugMessageInc("kernel",7,"Factory::UnLoadPackage(\""
                       <<name<<"\")"<<std::endl);
  
    PackageMapType::iterator i;
    i = mPackageMap.find(name);
    if (i == mPackageMap.end()) 
    {
      bbtkError("cannot unload package \""<<name
                <<"\" : package not loaded !");
    }
    ClosePackage(i);
    bbtkDebugDecTab("kernel",7);
  }
  //===================================================================


  //===================================================================
  void Factory::CloseAllPackages()
  {
    bbtkDebugMessageInc("kernel",7,"Factory::CloseAllPackages()"<<std::endl);
    
    std::vector< Package::WeakPointer > mAlive;  
    do {
      mAlive.clear();  
      while (mPackageMap.begin() != mPackageMap.end())
	{
	  PackageMapType::iterator i = mPackageMap.begin();
	  Package::WeakPointer p = i->second;
	  ClosePackage(i);
	  if (p.lock()) mAlive.push_back(p);
	}
      std::vector< Package::WeakPointer >::iterator i;
      for (i=mAlive.begin();i!=mAlive.end();++i)
	{
	  // If not dead : reinsert
	  if (i->lock())
	    {
	      bbtkDebugMessage("kernel",7,"Package "<<i->lock()->GetName()
			       <<" still alive"<<std::endl);
	      // InsertPackage(i->lock());
	    }
	}      
    }
    while (mPackageMap.size()>0);

    bbtkDebugDecTab("kernel",7);
  }
  //===================================================================

  //===================================================================
  /// \brief Close the package referenced by the iterator 
 ///
 /// First removes the factory from the set of factories which use the package
 /// If the set is empty then :
 /// If it is a dynamically loaded package :
 /// - Loads and calls the function "<name>DeletePackage" of the dynamic library (responsible for package desallocation)
 /// - Closes the dynamic library
 /// Else :
 /// - deletes the package normally
 /// 
 /// Finally erases the package entry in the packages map
 void Factory::ClosePackage(PackageMapType::iterator& i) 
  {   
     bbtkDebugMessageInc("kernel",7,"Factory::ClosePackage(\""
                         <<i->second->GetName()
                        <<"\")"<<std::endl);

     
     // Removes this from the set of factories which use the package
     i->second->RemoveFactory(GetThisPointer<Factory>());
     Package::WeakPointer p = i->second;
     // remove the entry in the map
     mPackageMap.erase(i);
     // Release the package if not already destroyed
     if (p.lock()) 
       {
	 Package::Release(p);
       }
     bbtkDebugDecTab("kernel",7);
  }
  //===================================================================
  


  //===================================================================  
  void Factory::PrintHelpListPackages(bool details, bool adaptors) const
  {
    bbtkDebugMessageInc("kernel",9,"Factory::PrintPackages"<<std::endl);

    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
    {
      bbtkMessage("help",1, i->first << std::endl);
      if (details) {
         i->second->PrintHelpListDescriptors(false,adaptors);
      }
    }

    bbtkDebugDecTab("kernel",9);
  }
  //===================================================================

  //===================================================================  
  /// Displays help on a package
  void Factory::PrintHelpPackage(const std::string& name, bool adaptors) const
  {
    bbtkDebugMessageInc("kernel",9,"Factory::PrintHelpPackage(\""
			<<name<<"\")"
                        <<std::endl);

    PackageMapType::const_iterator i = mPackageMap.find(name);
    if ( i != mPackageMap.end() ) 
      {
      bbtkMessage("help",1, "Package "<<i->first<<" ");
      
      if (i->second->GetVersion().length()>0)
        bbtkMessageCont("help",1,"v" <<i->second->GetVersion());
        
      if (i->second->GetAuthor().length()>0)
        bbtkMessageCont("help",1,"- "<<i->second->GetAuthor());
        
      if (i->second->GetCategory().length()>0)
        bbtkMessageCont("help",1,"- "<<i->second->GetCategory());        
        
      bbtkMessageCont("help",1,std::endl);
      bbtkIncTab("help",1);
      bbtkMessage("help",1,i->second->GetDescription()<<std::endl);
      if (i->second->GetNumberOfDescriptors()>0) 
        {
          bbtkMessage("help",1, "Black boxes : "<<std::endl);
          i->second->PrintHelpListDescriptors(true,adaptors);
        }
      else 
        {
          bbtkMessage("help",1, "No black boxes"<<std::endl);
        }
      bbtkDecTab("help",1);
      }
    else 
      {
      bbtkDebugDecTab("kernel",9);
      bbtkError("package \""<<name<<"\" unknown");
      }
    
    bbtkDebugDecTab("kernel",9);
  }
  //===================================================================

  //===================================================================
  /// Prints help on the black box of type <name>
  /// Returns the package to which it belongs
  void Factory::PrintHelpDescriptor(const std::string& name, 
			     std::string& package,
			     bool full) const
  {
    bbtkDebugMessageInc("kernel",9,"Factory::PrintHelpDescriptor(\""
			<<name<<"\")"
                        <<std::endl);

    bool found = false;
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
      if (i->second->ContainsDescriptor(name)) 
        {
          i->second->PrintHelpDescriptor(name,full);
	      package = i->second->GetName();
          found = true;
        }
      }
    
    bbtkDebugDecTab("kernel",9);
    if (!found) 
      {
      bbtkError("No package of the factory contains any black box <"
                 <<name<<">");
      }
  }  
  //===================================================================


  //=================================================================== 
  /// Inserts a package in the factory
  void Factory::InsertPackage( Package::Pointer p )
  {
    bbtkDebugMessageInc("kernel",9,"Factory::InsertPackage(\""<<
                        p->GetName()<<"\")"<<std::endl);

    p->AddFactory(GetThisPointer<Factory>());	  
    mPackageMap[p->GetName()] = p;
    bbtkDebugDecTab("kernel",9);
  }
  //===================================================================
  
  //=================================================================== 
  /// Removes a package from the factory (and deletes it)
  void Factory::RemovePackage( Package::Pointer p )
  {
    bbtkDebugMessageInc("kernel",9,"Factory::RemovePackage(\""<<
                        p->GetName()<<"\")"<<std::endl);

    PackageMapType::iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
         if (i->second == p) break;
      };
    
    if (i!=mPackageMap.end())
      {
	ClosePackage(i);
      }
    else 
      {
	bbtkError("Factory::RemovePackage(\""<<
		  p->GetName()<<"\") : package absent from factory");
      }

    bbtkDebugDecTab("kernel",9);
  }
  //===================================================================
  

//	ups3 EED borrame	
	std::string Factory::GetPackageNameOfaBlackBox(std::string boxType)
	{
		std::string result="<void bbtk package name>";
		PackageMapType::const_iterator i;
		for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
		{
			if ( i->second->ifBoxExist( boxType ) == true)
			{
				result = i->first;
			}
		}
		return result;
	}
	
	
  //===================================================================
  /// Creates an instance of a black box of type <type> with name <name>
  BlackBox::Pointer Factory::NewBlackBox(const std::string& type, 
                                 const std::string& name) const
  {
    bbtkDebugMessageInc("kernel",7,"Factory::NewBlackBox(\""
                        <<type<<"\",\""<<name<<"\")"<<std::endl);

    BlackBox::Pointer b; 
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
      b = i->second->NewBlackBox(type,name);
      if (b) break; 
      }
    if (!b) 
      {
       bbtkError("black box type \""<<type<<"\" unknown");
      } 

    bbtkDebugDecTab("kernel",7);
    return b;
  }
  //===================================================================

  //===================================================================
  /// Creates an instance of a black box of type <type> with name <name>
  BlackBox::Pointer Factory::NewAdaptor(const DataInfo& typein,
				const DataInfo& typeout,
				const std::string& name) const
  {
    bbtkDebugMessageInc("kernel",8,"Factory::NewAdaptor("
                        <<typein<<","
                        <<typeout<<",\""
                        <<name<<"\")"<<bbtkendl);


    BlackBox::Pointer b; 
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
      b = i->second->NewAdaptor(typein,typeout,name);
      if (b) break; 
      }
    if (!b) 
      {
	bbtkError("no "<<typein<<" to "<<typeout
		  <<" adaptor available");
      } 
    
    bbtkDebugDecTab("kernel",7);
    return b; 
  }
  //===================================================================


  //===================================================================
  /// Creates an instance of a black box of type <type> with name <name>
  BlackBox::Pointer Factory::NewWidgetAdaptor(const DataInfo& typein,
				      const DataInfo& typeout,
				      const std::string& name) const
  {
    bbtkDebugMessageInc("kernel",8,"Factory::NewWidgetAdaptor(<"
                        <<typein<<">,<"
                        <<typeout<<">,\""
                        <<name<<"\")"<<bbtkendl);


    BlackBox::Pointer b; 
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
	b = i->second->NewWidgetAdaptor(typein,
						 typeout,
						 name);
      if (b) break; 
      }
    if (!b) 
      {
	bbtkError("no "<<typein<<" to "<<typeout
		  <<"> widget adaptor available");
      } 
    
    bbtkDebugDecTab("kernel",7);
    return b; 
  }
  //===================================================================

  //===================================================================
  /// Creates an instance of a black box of type <type> with name <name>
  bool Factory::FindAdaptor(const DataInfo& typein,
				  const DataInfo& typeout,
				  std::string& adaptor) const
  {
    bbtkDebugMessageInc("kernel",8,"Factory::FindAdaptor(<"
                        <<typein<<">,<"
                        <<typeout<<">)"<<bbtkendl);
    
    bool b = false;
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
	b = i->second->FindAdaptor(typein,
					    typeout,
					    adaptor);
	if (b) break; 
      }
    /*
    if (!b) 
      {
	bbtkError("no "<<typein<<" to "<<typeout
		  <<"> widget adaptor available");
      } 
    */

    bbtkDebugDecTab("kernel",7);
    return b; 
  }
  //===================================================================

  //===================================================================
  /// Creates an instance of a black box of type <type> with name <name>
  bool Factory::FindWidgetAdaptor(const DataInfo& typein,
				  const DataInfo& typeout,
				  std::string& adaptor) const
  {
    bbtkDebugMessageInc("kernel",8,"Factory::FindWidgetAdaptor(<"
                        <<typein<<">,<"
                        <<typeout<<">)"<<bbtkendl);
    
    bool b = false;
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
	b = i->second->FindWidgetAdaptor(typein,
						  typeout,
						  adaptor);
	if (b) break; 
      }
    bbtkDebugDecTab("kernel",7);
    return b; 
  }
  //===================================================================

  //===================================================================
  /// Creates an instance of a black box of type <type> with name <name>
  bool Factory::FindWidgetAdaptor2(const DataInfo& typein,
				  const DataInfo& typeout,
				  std::string& widget,
				  std::string& adaptor) const
  {
    bbtkDebugMessageInc("kernel",8,"Factory::FindWidgetAdaptor(<"
                        <<typein<<">,<"
                        <<typeout<<">)"<<bbtkendl);
    
    bool b = false;
    adaptor = widget = "";
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
	b = i->second->FindWidgetAdaptor(typein,
						  typeout,
						  widget);
	if (b) break; 
      }
    if (!b) 
      {
	// Look for a widget adaptor with good nature out
	bbtkMessage("kernel",5,
		    "*** Looking for a two pieces widget adaptor for : "
		    << typein << "->"<<typeout<<std::endl);
	for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
	  {
	    Package::AdaptorMapType::const_iterator j;
	    for (j=i->second->GetAdaptorMap().begin();
		 j!=i->second->GetAdaptorMap().end();
		 ++j)
	      {
		if ( ( j->first.mKind ==  
		       BlackBoxDescriptor::DEFAULT_GUI) &&
		     //(j->first.mTypeIn == typein) &&
		     (j->first.mTypeOut.GetNature() == typeout.GetNature() ) 
		     )
		  {
		    widget = j->second.lock()->GetTypeName();
		    bbtkMessage("kernel",5,
				"===> Found first part : "<<widget
				<< " "<<j->first.mTypeIn<<"->"
				<<j->first.mTypeOut<<std::endl);
		    DataInfo ti( j->first.mTypeOut.GetType(), "");
		    DataInfo to( typeout.GetType(), "");
		    b = FindAdaptor( ti, to, adaptor );
		    if (b) 
		      {
			bbtkMessage("kernel",5,
				    "===> Found second part : "<<adaptor
				    <<std::endl);
			break;
		      }
		    else
		      {
			bbtkMessage("kernel",5,
				    "===> No second part found"<<std::endl);
		      }
		  }
	      }
	    if (b) break;
	  }
      }
    bbtkDebugDecTab("kernel",7);
    return b; 
  }
  //===================================================================

  //===================================================================
  /// Creates an instance of a connection
  Connection::Pointer Factory::NewConnection(BlackBox::Pointer from,
					     const std::string& output,
					     BlackBox::Pointer to,
					     const std::string& input) const
  {
    bbtkDebugMessage("kernel",7,"Factory::NewConnection(\""
                      <<from->bbGetName()<<"\",\""<<output<<"\",\""
                      <<to->bbGetName()<<"\",\""<<input
                      <<"\")"<<std::endl);
    
    return Connection::New(from,output,to,input,
			   GetThisPointer<Factory>());
  }
  //===================================================================



  //===================================================================
  Package::Pointer Factory::GetPackage(const std::string& name) const
  {
    bbtkDebugMessageInc("kernel",9,"Factory::GetPackage(\""<<name<<"\")"
                         <<std::endl);

    PackageMapType::const_iterator i = mPackageMap.find(name);
    if ( i != mPackageMap.end() ) 
    {
      bbtkDebugDecTab("kernel",9); 
      return i->second;
    }
    else 
    {
       bbtkDebugDecTab("kernel",9);
       bbtkError("package \""<<name<<"\" unknown");
    }
    
    bbtkDebugDecTab("kernel",9);  
  }
  //===================================================================
  

  //===================================================================
  void Factory::Check() const
  {
    bbtkMessage("debug",1,"****** Checking Factory "<<(void*)this
		 <<std::endl);
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
      {
	i->second->Check();
      }
    bbtkMessage("debug",1,"****** Checking Factory "<<(void*)this
		<<" ... OK"<<std::endl);
  }
  //===================================================================

  //===================================================================
  void Factory::WriteDotFilePackagesList(FILE *ff)
  {

    bbtkDebugMessageInc("kernel",9,"Factory::WriteDotFilePackagesList()"
                         <<std::endl);

    fprintf( ff , "\n");
    fprintf( ff , "subgraph cluster_FACTORY {\n");
    fprintf( ff , "  label = \"PACKAGES\"%s\n",  ";");
    fprintf( ff , "  style=filled%s\n",";");
    fprintf( ff , "  color=lightgrey%s\n",";");
    fprintf( ff , "  rankdir=TB%s\n",";");

    std::string url;
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
    {
       url=GetPackage(i->first)->GetDocURL();
       fprintf(ff,"  %s [shape=ellipse, URL=\"%s\"]%s\n",i->first.c_str(),url.c_str(),";" );
    }
    fprintf( ff , "}\n\n");
    bbtkDebugDecTab("kernel",9);
  }
  //===================================================================


 void Factory::ShowGraphTypes(const std::string& name) const
 {
   bool found = false;
   PackageMapType::const_iterator i;
   for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
   {
     if (i->second->ContainsDescriptor(name)) 
       {
         std::string separator = ConfigurationFile::GetInstance().Get_file_separator ();
	 
	 // Don't pollute the file store with  "temp_dir" directories ...    
         std::string default_doc_dir = ConfigurationFile::GetInstance().Get_default_temp_dir();
         std::string directory = "\"" + default_doc_dir + separator + "temp_dir"  +separator + "\"";
         std::string filename2 =  default_doc_dir + separator + "temp_dir" + separator + "tmp.html"; 
	 
#if defined(_WIN32)  
	 std::string command("start \"Titre\" /D ");
#else 
	 std::string command("gnome-open ");
#endif
	 command=command + directory +" tmp.html";
	 FILE *ff;
	 ff=fopen(filename2.c_str(),"w");
	 
	 fprintf(ff,"<html><head><title>TMP</title> <script type=\"text/javascript\"> <!--\n");
	 fprintf(ff,"  window.location=\"%s#%s\";\n" , i->second->GetDocURL().c_str(),name.c_str() );
	 fprintf(ff,"//--></script></head><body></body></html>\n");
	 
	 
	 //fprintf(ff, "<a  href=\"%s#%s\">Link</a>\n", i->second->GetDocURL().c_str(),name.c_str() );
	 fclose(ff);
	 system( command.c_str() );      
	 found = true;
       }
   }
   
   bbtkDebugDecTab("kernel",9);
   if (!found) 
   {
     bbtkError("No package of the factory contains any black box <"
	       <<name<<">");
   }
 }
    



  void Factory::CreateHtmlIndex(IndexEntryType type, 
				const std::string& filename)
  {
    bbtkDebugMessageInc("kernel",9,"Factory::CreateHtmlIndex(\""
			<<filename<<"\")"<<bbtkendl);
    
    std::string title;

    typedef std::map<std::string, 
                     std::vector<BlackBoxDescriptor::Pointer> > IndexType;
    IndexType index;
    // Builds the index map
    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
	{
		Package::Pointer pack = i->second;
		if (pack->GetName()=="user") continue;
		Package::DescriptorMapType::const_iterator j;
		for (j = pack->GetDescriptorMap().begin(); 
			j!= pack->GetDescriptorMap().end(); 
			++j)
		{
	    
	    // Skip adaptors 
			if ( type==Adaptors )
			{  
				if (j->second->GetKind() == BlackBoxDescriptor::STANDARD )
				continue;
			}
			else 
				if (j->second->GetKind() != BlackBoxDescriptor::STANDARD )
				continue;

			std::vector<std::string> keys;
			if (type==Packages)
			{
				std::string k("");
				k += pack->GetName();
				keys.push_back(k);
				title = "Boxes by package";
			}
			else if ((type==Initials) || (type==Adaptors))
			{
				std::string init(" ");
				init[0] =  std::toupper(j->second->GetTypeName()[0]);
				keys.push_back(init);
				title = "Alphabetical list";
			}
			else if (type==Categories)
			{
			// Split the category string 
			std::string delimiters = ";,";
			Utilities::SplitString(j->second->GetCategory(),
					   delimiters,keys);
			if (keys.size()==0) 
				keys.push_back(" NONE");
				title = "Boxes by category";
			}

			std::vector<std::string>::const_iterator k;
			for (k=keys.begin(); k!=keys.end(); ++k )
			{
				IndexType::iterator p;
				p = index.find(*k);
				if (p != index.end()) 
				{
					p->second.push_back(j->second);
				}
				else 
				{
					std::vector<BlackBoxDescriptor::Pointer> v;
					v.push_back(j->second);
					index[*k] = v;
				}
			}
		}
	}   
    // Creates the file 
    //---------------------
    // Open output file
    std::ofstream s;
    s.open(filename.c_str());
    if (!s.good()) 
    {
       bbtkError("Factory::CreateHtmlIndex : could not open file '"
		 <<filename<<"'");
    }
    
    //----------------------
    // Html head
    s << "<html lang=\"en\">\n";
    s << "<head>\n";
    s << "<title>"<<title<<"</title>\n";
    s << "<meta http-equiv=\"Content-Type\" content=\"text/html\">\n";
    s << "<meta name=\"description\" content=\""<<title<<"\">\n";
    s << "<meta name=\"generator\" content=\"\">\n";
    s << "<link title=\"Top\" rel=\"top\" href=\"#Top\">\n";
    //<link href="http://www.gnu.org/software/texinfo/" rel="generator-home" title="Texinfo Homepage">
    s << "<meta http-equiv=\"Content-Style-Type\" content=\"text/css\"><style type=\"text/css\"><!--\n";
    s << "pre.display { font-family:inherit }\n";
    s << "pre.format  { font-family:inherit }\n";
    s << "pre.smalldisplay { font-family:inherit; font-size:smaller }\n";
    s << "pre.smallformat  { font-family:inherit; font-size:smaller }\n";
    s << "pre.smallexample { font-size:smaller }\n";
    s << "pre.smalllisp    { font-size:smaller }\n";
    s << "span.sc    { font-variant:small-caps }\n";
    s << "span.roman { font-family:serif; font-weight:normal; } \n";
    s << "span.sansserif { font-family:sans-serif; font-weight:normal; }\n"; 
    s << "--></style>\n";
    s << "</head>\n";
    //----------------------

    //----------------------
    // Html body
    s << "<body>\n";
    s << "<a name=\"Top\"></a>\n"; 
    s << "<h1 class=\"settitle\">"<<title<<"</h1>\n";
    s << "<p>\n";
    IndexType::iterator ii;
    for (ii=index.begin();ii!=index.end();++ii)
      {
		s << "<a href=\"#"<<ii->first<<"\">"<<ii->first<<"</a>&nbsp;&nbsp;";	
      }

    for (ii=index.begin();ii!=index.end();++ii)
      {
	s << "<p><hr>\n";
	s << "<p><a href=\"#Top\">Top</a>";
	if (type==Packages)
	  {
	    s << "<a name=\""<<ii->first<<"\"></a>\n"; 
	    s << "<p><a href=\""<<ii->first<<"/index.html\">"
	      << ii->first<<"</a>\n"; 

	    s << "&nbsp;&nbsp;-&nbsp;&nbsp;\n"; 

	    s << "<a name=\"doxygen\"></a>\n"; 

//EED 26Mars2009
	/*JCP 19 Nov 2009
		std::string bin_path = bbtk::ConfigurationFile::GetInstance().Get_bin_path();
	    s << "<a href=" << bin_path <<"/../share/bbtk/doc/doxygen/" << ii->first << "/main.html>(Doxygen documentation of the source)</a>\n"; 
	JCP 19 Nov 2009*/
		std::string bin_path = bbtk::ConfigurationFile::GetInstance().Get_bin_path();		
	    s << "<a href=" << bin_path <<"/../share/bbtk/doc/doxygen/" << ii->first << "/main.html>(Doxygen documentation of the source)</a>\n"; 
	  }
	else 
	  {
	    s << "<a name=\""<<ii->first<<"\"></a>\n"; 
	    s << "<p><b>"<<ii->first<<"</b>\n";
	  }
	s << "<ul>\n";

	s << "<p><TABLE cellspacing=0  cellpadding=3>\n";

	std::vector<BlackBoxDescriptor::Pointer>::iterator di;
	for (di=ii->second.begin();di!=ii->second.end();++di)
	  {
	    std::string pack = (*di)->GetPackage()->GetName();
	    std::string name = (*di)->GetTypeName();
	    Utilities::html_format(name);
	    std::string descr = (*di)->GetDescription();
	    Utilities::html_format(descr);
	    s << "<TR>";
	    s << "<TD style='vertical-align: top;'>";
	    s << "&nbsp;&nbsp;&nbsp;<a href=\""<<pack
	      <<"/index.html#"<<name<<"\">"
	      <<pack<<"::"<<name<<"</a>";
	    s << "</TD> ";
	    s << " <TD style='vertical-align: top;'>" << descr << " </TD>";
	    s << "</TR>\n";
	  }    
	s << "</TABLE>\n";
	s << "</ul>\n";
	s << "</div>\n";
      }
    //----------------------
    // Footer 
    time_t rawtime;
    tm * ptm;
    time ( &rawtime );
    ptm = gmtime ( &rawtime );

    s << "<p><hr>\n";
    s << "Automatically generated by <b>bbtk</b> on "
      << ptm->tm_mday << "/" << ptm->tm_mon << "/" << ptm->tm_year+1900 
      << " - " << ptm->tm_hour << ":" << ptm->tm_min << " GMT\n";
    s << "</body></html>\n"; 
    s.close();
    //----------------------

    // End
    bbtkDebugDecTab("kernel",9);
  }

 //==========================================================================
  std::string Factory::GetObjectName() const
  {
    return std::string("Factory");
  }
  //==========================================================================
  
  //==========================================================================
  std::string  Factory::GetObjectInfo() const 
  {
    std::stringstream i;
    return i.str();
  }
  //==========================================================================

  //==========================================================================
size_t  Factory::GetObjectSize() const 
{
  size_t s = Superclass::GetObjectSize();
  s += Factory::GetObjectInternalSize();
  return s;
  }
  //==========================================================================
  //==========================================================================
size_t  Factory::GetObjectInternalSize() const 
{
  size_t s = sizeof(Factory);
  return s;
  }
  //==========================================================================
  //==========================================================================
  size_t  Factory::GetObjectRecursiveSize() const 
  {
    size_t s = Superclass::GetObjectRecursiveSize();
    s += Factory::GetObjectInternalSize();

    PackageMapType::const_iterator i;
    for (i = mPackageMap.begin(); i!=mPackageMap.end(); ++i )
    {
      s += i->second->GetObjectRecursiveSize();
    }
    return s;
  }
  //==========================================================================

}
  
