/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbtkInterpreter.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.95 $
=========================================================================*/



/**
 *  \file 
 *  \brief class Interpreter :
 */

#include "bbtkInterpreter.h"
#include "bbtkExecuter.h"
#include "bbtkTranscriptor.h"
#include "bbtkMessageManager.h"
#include "bbtkConfigurationFile.h"
#include "bbtkUtilities.h"
#include "bbtkAtomicBlackBox.h"
#include "bbtkWxBlackBox.h"
#include <sys/stat.h>
#include <algorithm>
#ifdef CMAKE_HAVE_TERMIOS_H
#include <termios.h>
#define BBTK_USE_TERMIOS_BASED_PROMPT
#endif

#include <string>

namespace bbtk
{

 //=======================================================================
  Interpreter::Pointer Interpreter::New(const std::string& cpp_file) 
  {
    bbtkDebugMessage("kernel",9,"Interpreter::New('"<<cpp_file<<"')"<<std::endl);
    return MakePointer(new Interpreter(cpp_file));
  }
 //=======================================================================

 //=======================================================================
  Interpreter::Pointer Interpreter::New(VirtualExec::Pointer e) 
  {
    bbtkDebugMessage("kernel",9,"Interpreter::New(VirtualExec)"<<std::endl);
    return MakePointer(new Interpreter(e));
  }
 //=======================================================================

 //=======================================================================
  Interpreter::Interpreter(const std::string& cpp_file) 
  {
    bbtkDebugMessage("object",2,"==> Interpreter("<<cpp_file<<")"<<std::endl);
    Init(VirtualExec::Pointer(), cpp_file);
    bbtkDebugMessage("object",2,"<== Interpreter("<<cpp_file<<")"<<std::endl);
  }
  //=======================================================================

 //=======================================================================
  Interpreter::Interpreter(VirtualExec::Pointer e) 
  {
    bbtkDebugMessage("object",2,"==> Interpreter(VirtualExec)"<<std::endl);
    Init(e,"");
    bbtkDebugMessage("object",2,"<== Interpreter(VirtualExec)"<<std::endl);
  }
  //=======================================================================

  //=======================================================================
  void Interpreter::Init(VirtualExec::Pointer e, const std::string& cpp_file) 
  {
    if (e)
      {
	mVirtualExecuter = e;
      }
    else if (cpp_file.size()!=0)
      {
	mVirtualExecuter = boost::static_pointer_cast<VirtualExec>(bbtk::Transcriptor::New(cpp_file));
      }
    else 
      {
	bbtk::Executer::Pointer exe = bbtk::Executer::New();
	mRealExecuter = exe;
	mVirtualExecuter = boost::static_pointer_cast<VirtualExec>(exe);
      }

	  
	  
    // Lock this pointer or will auto-destruct !!
    if (!e) mVirtualExecuter->SetInterpreter(MakePointer(this,true));

	  
	  bbtk::InterpreterVirtual::Init();
  } 
  //=======================================================================
  
  
  
  //=======================================================================  
  /**
   *  
   */
  Interpreter::~Interpreter()
  {
    bbtkDebugMessage("object",2,"==> ~Interpreter()" <<std::endl);
    mVirtualExecuter = VirtualExec::Pointer();
    bbtkDebugMessage("object",2,"<== ~Interpreter()" <<std::endl);
  }
  //=======================================================================

/*EED Borrame
  //=======================================================================
  InterpreterException::InterpreterException( const std::string& message,
				      bool in_script_file,
				      const std::string& script_file,
				      int script_line 
				      )
    : Exception("interpreter",0,message),
      mInScriptFile(in_script_file),
      mScriptFile(script_file),
      mScriptLine(script_line)
  {
  }
  //=======================================================================
  //=======================================================================
  InterpreterException::InterpreterException( const Exception& excep,
		      bool in_script_file,
		      const std::string& script_file,
		      int script_line 
		      )
    : Exception(excep),
      mInScriptFile(in_script_file),
      mScriptFile(script_file),
      mScriptLine(script_line)
  {
  }
  //=======================================================================
*/

  //=======================================================================
  void Interpreter::CatchInterpreterException( const InterpreterException& e )
  {
    if (GetExecuter()->GetNoErrorMode()) 
      {
	bbtkWarning("ERROR :"<<e.GetErrorMessage()
		    <<" ("<<e.GetScriptFile()<<":"<<e.GetScriptLine()
		    <<" skipped");

	return;
      }
	  
	  bbtk::InterpreterVirtual::CatchInterpreterException(  e );	  
  }
  //=======================================================================

  //=======================================================================
  void Interpreter::CatchBbtkException( const bbtk::Exception& e )
  {
    if (GetExecuter()->GetNoErrorMode()) 
      {
	std::string file("?");
	int line = 0;
	if (mFileName.size()) {
	  file = mFileName.back();
	  line = mLine.back();
	}    
	bbtkWarning("ERROR '"<<e.GetErrorMessage()
		    <<"' ("<<file<<":"<<line<<") skipped");
	
	return;
      }
	  
	  bbtk::InterpreterVirtual::CatchBbtkException(  e );	  
  }
  //=======================================================================
  
  //=======================================================================
  void Interpreter::CatchStdException( const std::exception& e )
  {  
	if (GetExecuter()->GetNoErrorMode()) 
	{
		std::string file("?");
		int line = 0;
		if (mFileName.size()) 
		{
			file = mFileName.back();
			line = mLine.back();
		}    
		bbtkWarning("ERROR '"<<e.what() <<"' ("<<file<<":"<<line<<") skipped");
		return;
      }
	  bbtk::InterpreterVirtual::CatchStdException(  e );	  
  }
  //=======================================================================

  //=======================================================================
  void Interpreter::CatchUnknownException()
  {
	if (GetExecuter()->GetNoErrorMode()) 
	{
		std::string file("?");
		int line = 0;
		if (mFileName.size()) 
		{
			file = mFileName.back();
			line = mLine.back();
		}  
		bbtkWarning("UNDEFINED ERROR " <<"("<<file<<":"<<line<<") skipped");
		return;
	}

	  bbtk::InterpreterVirtual::CatchUnknownException( );	  
  }
  //=======================================================================

  //=======================================================================
  
#define CATCH_MACRO					\
  catch (InterpreterException e)			\
    {							\
      CatchInterpreterException(e);			\
    }							\
  catch (bbtk::Exception e)				\
    {							\
      CatchBbtkException(e);				\
    }							\
  catch (std::exception& e)				\
    {							\
      CatchStdException(e);				\
    }							\
  catch (...)						\
    {							\
      CatchUnknownException();				\
    }						
  //=======================================================================
   
/*EED Borrame
  //=======================================================================
  Interpreter::ExitStatus Interpreter::InterpretFile( const std::string& filename, bool source )
  {
    bbtkDebugMessage("interpreter",4,"==> Interpreter::InterpretFile(\""<<filename<<"\")"<<std::endl);

    bool exm = mCommandLine;
    mCommandLine = false;

    try 
    {
      mStatus = Interpreter_OK;
      SwitchToFile(filename,source);
      mInsideComment = false;
      InterpretCurrentStreams();
    } 
    CATCH_MACRO;
    
    bbtkDebugMessage("interpreter",4,
		     "<== Interpreter::InterpretFile(\""
		     <<filename<<"\")"<<std::endl);


    mCommandLine = exm;
    
    return mStatus;
  }
  //=======================================================================


  //=======================================================================
  Interpreter::ExitStatus 
  Interpreter::InterpretBuffer( std::stringstream* buffer )
  {
    bbtkDebugMessage("interpreter",4,"==> Interpreter::InterpretBuffer()"<<std::endl);

    bool exm = mCommandLine;
    mCommandLine = false;

    try 
    {
      mStatus = Interpreter_OK;
      SwitchToStream(buffer);
      mInsideComment = false;
      InterpretCurrentStreams();
    }
    CATCH_MACRO;
    
    //    CloseAllFiles();
    bbtkDebugMessage("interpreter",4,"<== Interpreter::InterpretBuffer()"<<std::endl);

    
    mCommandLine = exm;
    return mStatus;
  }
  //=======================================================================

  //=======================================================================
  /// Interprets the currently open streams
  Interpreter::ExitStatus Interpreter::InterpretCurrentStreams()
  {
    bbtkDebugMessage("interpreter",4,
		     "==> Interpreter::InterpretCurrentStreams()"<<std::endl);

    while (mFile.size()>0) 
      {
	while (!mFile.back()->eof()) {
	  mLine.back()++;
	  char buf[500];
	  mFile.back()->getline(buf,500);
	  std::string str(buf);
	  //size 0 JCP 21-09-2009
	  int size=str.length();
	  if(size != 0){
		  if ( str[ size-1 ]==13  )
            {
	      str.erase(size-1,1);
            }
		  try
			{
			  DoInterpretLine(str);
			}
		  CATCH_MACRO;
	  }
	  
	} 
	CloseCurrentFile();
      }
    bbtkDebugMessage("interpreter",4,
		     "<== Interpreter::InterpretCurrentStreams()"<<std::endl);

    return mStatus;
  }
  //=======================================================================

  //=======================================================================
  /// Runs the interpretation of a command
  Interpreter::ExitStatus Interpreter::InterpretLine( const std::string& line )
  {
    bbtkDebugMessage("interpreter",5,"==> Interpreter::InterpretLine('"<<line<<"')"<<std::endl);

    try 
    {
      mStatus = Interpreter_OK;
      mInsideComment = false;
//std::cout<<"JCP bbtkInterpreter.cxx Interpreter::InterpretLine("<<std::endl;
      DoInterpretLine(line );
    }
    CATCH_MACRO;
    
    
   bbtkDebugMessage("interpreter",5,"<== Interpreter::InterpretLine('"<<line<<"')"<<std::endl);
    
    return mStatus;
  }
  //=======================================================================  
*/
	
	
	void Interpreter::commandNew(const std::string &boxType,const  std::string &boxName)  //virtual
    {
		mVirtualExecuter->Create(boxType,boxName);
    }
	
	void Interpreter::commandDelete(const std::string &boxName)  //virtual
    {
		mVirtualExecuter->Destroy(boxName);
    }

	void Interpreter::commandConnection(const std::string &nodeFrom,const  std::string &outputLabel,const  std::string &nodeTo,const  std::string &inputLabel) //virtual
    {
		mVirtualExecuter->Connect(nodeFrom,outputLabel,nodeTo,inputLabel);
    }
		
	void Interpreter::commandPackage(const std::string &packageName) //virtual
    {
		mVirtualExecuter->BeginPackage(packageName);
    }
	
	void Interpreter::commandEndPackage() //virtual
    {
		mVirtualExecuter->EndPackage();
    }
	
	void Interpreter::commandDefine(const std::string &name,const  std::string &pack,const  std::string &scriptfilename) //virtual
    {
		mVirtualExecuter->Define(name,pack,scriptfilename);
    }

	void Interpreter::commandEndDefine() //virtual
    {
		mVirtualExecuter->EndDefine();
    }

	void Interpreter::commandKind(const std::string &kind) //virtual
    {
		mVirtualExecuter->Kind(kind);
    }
	
	void Interpreter::commandPrint(const std::string &value) //virtual
    {
		mVirtualExecuter->Print(value);
    }
	
	
	void Interpreter::commandExec(const std::string &word) //virtual
    {
		if (word=="freeze") 
		{
			mVirtualExecuter->SetNoExecMode(true);
			mThrow = false;
		}
		else if (word=="freeze_no_error") 
		{
			mVirtualExecuter->SetNoExecMode(true);
			mVirtualExecuter->SetNoErrorMode(true);
			mThrow = false;
		}
		else if (word=="unfreeze") 
		{
			mVirtualExecuter->SetNoExecMode(false);
			mVirtualExecuter->SetNoErrorMode(false);
		}
		else
		{
			mVirtualExecuter->Execute(word);
		} // if
    }
	
	
	void Interpreter::commandInput(const std::string &name,const std::string &box,const std::string &input,const std::string  &help) //virtual
    {
		mVirtualExecuter->DefineInput(name,box,input,help);
    }
	
	void Interpreter::commandOutput(const std::string &name,const std::string &box,const std::string &output,const std::string  &help) //virtual
    {
		mVirtualExecuter->DefineOutput(name,box,output,help);
    }
	
	void Interpreter::commandSet(const std::string &box,const std::string &input,const std::string &value) //virtual
    {
		mVirtualExecuter->Set(box,input,value);
    }

	void Interpreter::commandAuthor(const std::string &author) //virtual
    {
		mVirtualExecuter->Author(author);
    }

	void Interpreter::commandCategory(const std::string &categorytype) //virtual
    {
		mVirtualExecuter->Category(categorytype);
    }
	
	void Interpreter::commandDescription(const std::string &description) //virtual
    {
		mVirtualExecuter->Description(description);
    }

	
	void Interpreter::commandClear() //virtual
    {
		mVirtualExecuter->Clear();
    }
	
	void Interpreter::commandInclude(const std::string &word, bool ok) //virtual
    {
		// if 'source' was given (words.size()==3) then tell to set the 
		// source file name of the current complex box with the full file name included
		if (mCommandLine)
		{
			InterpretFile(word, ok ); 
		}
		else
		{
			SwitchToFile(word , ok );
		}
    }
	
	
	void Interpreter::commandLoad(const std::string &packageName) //virtual
    {
		GetExecuter()->LoadPackage(packageName);
    }

	void Interpreter::commandUnload(const std::string &packageName) //virtual
    {
		GetExecuter()->UnLoadPackage(packageName);
    }
	
	void Interpreter::commandBreak() //virtual
    {
	    /*
		 std::cout << "BreakException(" 
		 <<in_script<<","
		 <<file<<","
		 <<line<<")"<<std::endl;
		 */
		bbtkError("break");//,in_script,file,line);
	    //	    throw BreakException(in_script,file,line);
    }
	
	void Interpreter::commandQuit() //virtual
    {
		bbtkError("quit");//,in_script,file,line);
		//throw QuitException(in_script,file,line);
    }
	
	void Interpreter::commandMessage() //virtual
    {
		mVirtualExecuter->HelpMessages();
    }
	
	void Interpreter::commandMessage(const std::string &kind,const std::string &levelstr) //virtual
    {
		int level=0;
		sscanf(levelstr.c_str(),"%d",&level);
		mVirtualExecuter->SetMessageLevel(kind,level);

    }
	
	
/*EED Borrame	
  //=======================================================================  
  void Interpreter::DoInterpretLine( const std::string& line ) //virtual
   {
    bbtkDebugMessage("interpreter",6,"==> Interpreter::DoInterpretLine(\""
		     <<line<<"\")"<<std::endl);
    std::vector<std::string> words;
    SplitLine(line,words);

    // Empty line
    if (words.size()<1) 
    {
       bbtkDebugDecTab("interpreter",9);
       return;
    }

    // Single line comment : # or //
    if ( words[0][0]=='#' || (words[0][0]=='/' && words[0][1]=='/') ) 
    {  
       bbtkDebugDecTab("interpreter",9);
       bbtkMessage("interpreter",9,"Comment"<<std::endl);
       return;
    }

    // Multi line comment ( / * ... * / ) -delimiters on different lines !-   <<<<<<<  / *     * /
    
    if (words[0][0]=='/' && words[0][1]=='*') 
    {  
       bbtkDebugDecTab("interpreter",9);
       bbtkMessage("interpreter",9,"In multiline comment"<<std::endl);
       mInsideComment = true;
       return;
    }

    if (words[0][0]=='*' && words[0][1]=='/') 
    {  
       bbtkDebugDecTab("interpreter",9);
       bbtkMessage("interpreter",9,"Out multiline comment"<<std::endl);
       if ( !mInsideComment ) {
          bbtkDebugDecTab("interpreter",9);
          bbtkMessage("interpreter",9,"Comment mismatch : '* /' with no matching '/ *'"<<std::endl);        <<<<<<<<< * /       / * 
       }
       mInsideComment = false;
       return;
    }

    if (mInsideComment) 
    {  
       bbtkDebugDecTab("interpreter",9);
       bbtkMessage("interpreter",9,"Multiline Comment"<<std::endl);
       return;
    }

    // Command 
    CommandInfoType command;
    InterpretCommand(words,command);
//std::cout<<"JCP bbtkInterpreter command.keyword ="<<command.keyword<<std::endl;
    bbtkDebugMessage("interpreter",9,
                     "Command='"<<command.keyword
                      <<"' code="<<command.code<<std::endl); 
	  
    std::string left,right,left2,right2;
    std::string filename;

//ups1 EED borrame	  
    // message command
    if (command.code==cMessage)
      {
	if (words.size()<3)
	  {
		  commandMessage();
//EED Borrame	    mVirtualExecuter->HelpMessages();
	  }
        else
	  {
		commandMessage(words[1],words[2]);
//EED Borrame		sscanf(words[2].c_str(),"%d",&level);
//EED Borrame	    mVirtualExecuter->SetMessageLevel(words[1],level);
	  }
	return;
      }
    else 
      {
	bbtkMessage("echo",2,line<<std::endl);
      }

    // break and quit commands
    if ((command.code==cBreak) || (command.code==cQuit))
      {
	bool in_script = false;
	std::string file("");
	int line = 0;

	if (mFileName.size()) 
	  {
	    std::ifstream* fs = dynamic_cast<std::ifstream*>(mFile.back());
	    if (fs!=0) in_script = true;	  
	    file = mFileName.back();
	    line = mLine.back();
	  } 
	if (command.code==cBreak)
	  {
//	    std::cout << "BreakException(" 
//		      <<in_script<<","
//		      <<file<<","
//		      <<line<<")"<<std::endl;
		  commandBreak();
//EED Borrame	    bbtkError("break");//,in_script,file,line);
	    //	    throw BreakException(in_script,file,line);
	  }	  
	else 
	  {
		  commandQuit();
//EED Borrame	    bbtkError("quit");//,in_script,file,line);
	      //throw QuitException(in_script,file,line);
	  }
	return;
      }   
//std::cout<<" mVirtualExecuter->Create(words[1],words[2]); "<<line<<std::endl;
    // other cammands

    switch (command.code) 
      {
      case cNew :
		commandNew(words[1],words[2]);
//EED Borrame        mVirtualExecuter->Create(words[1],words[2]);
        break;

      case cDelete :
			  commandDelete(words[1]);
//EED Borrame	mVirtualExecuter->Destroy(words[1]);
        break;

      case cConnect :
        Utilities::SplitAroundFirstDot(words[1],left,right);
        Utilities::SplitAroundFirstDot(words[2],left2,right2);      
        commandConnection(left,right,left2,right2);
//EED Borrame        mVirtualExecuter->Connect(left,right,left2,right2);
        break;

      case cPackage :
			  commandPackage(words[1]);
//EED Borrame                mVirtualExecuter->BeginPackage(words[1]);
        break;

      case cEndPackage :
			  commandEndPackage();
//EED Borrame        mVirtualExecuter->EndPackage();
        break;

      case cDefine :
        if (mFileName.size()>0) 
        {
//???			commandDefine(????);
		   filename = mFileName.back(); //mIncludeFileName.back(); //Utilities::get_file_name(mFileName.back());
        }
        if (words.size()==2) 
        {
			commandDefine(words[1],"",filename);
//EED Borrame           mVirtualExecuter->Define(words[1],"",filename);
        }
        else
        {
			commandDefine(words[1],words[2],filename);
//EED Borrame           mVirtualExecuter->Define(words[1],words[2],filename);
        }
        break;

      case cEndDefine :
		commandEndDefine();
//EED Borrame        mVirtualExecuter->EndDefine();
        break;

      case cKind :
			commandKind(words[1]);
//EED Borrame        mVirtualExecuter->Kind(words[1]);
        break;

      case cPrint :
			  commandPrint(words[1]);
//EED Borrame        mVirtualExecuter->Print(words[1]);
        break;
			  
      case cExec :
			  commandExec(words[1]);
//EED Borrame        if (words[1]=="freeze") 
//EED Borrame	  {
//EED Borrame	    mVirtualExecuter->SetNoExecMode(true);
//EED Borrame	    mThrow = false;
//EED Borrame	  }
//EED Borrame	else if (words[1]=="freeze_no_error") 
//EED Borrame	  {
//EED Borrame	    mVirtualExecuter->SetNoExecMode(true);
//EED Borrame	    mVirtualExecuter->SetNoErrorMode(true);
//EED Borrame	    mThrow = false;
//EED Borrame	  }
//EED Borrame	else if (words[1]=="unfreeze") 
//EED Borrame	  {
//EED Borrame	    mVirtualExecuter->SetNoExecMode(false);
//EED Borrame	    mVirtualExecuter->SetNoErrorMode(false);
//EED Borrame	  }
//EED Borrame	else
//EED Borrame	  {
//EED Borrame	    mVirtualExecuter->Execute(words[1]);
//EED Borrame	  }
			  
        break;

      case cInput :
        Utilities::SplitAroundFirstDot(words[2],left,right);
			  commandInput(words[1],left,right,words[3]);
//EED Borrame        mVirtualExecuter->DefineInput(words[1],left,right,words[3]);
        break;

      case cOutput :
        Utilities::SplitAroundFirstDot(words[2],left,right);
		commandOutput(words[1],left,right,words[3]);
//EED Borrame         mVirtualExecuter->DefineOutput(words[1],left,right,words[3]);
        break;

      case cSet :
        Utilities::SplitAroundFirstDot(words[1],left,right);
		commandSet(left,right,words[2]);
//EED Borrame        mVirtualExecuter->Set(left,right,words[2]);
        break;

      case cAuthor :
		commandAuthor(words[1]);
//EED Borrame        mVirtualExecuter->Author(words[1]);
        break;

      case cNewGUI :
		commandNewGUI(words[1],words[2]);
        break;

      case cCategory :
		commandCategory(words[1]);
//EED Borrame   mVirtualExecuter->Category(words[1]);
        break;

      case cIndex :
        if (words.size()==1)
			commandIndex("tmp_index.html");
        else if (words.size()==2)
			commandIndex(words[1]);
        else if (words.size()==3)
			commandIndex(words[1],words[2]);
        break;

      case cDescription :
		commandDescription(words[1]);
//EED Borrame        mVirtualExecuter->Description(words[1]);
        break;

      case cHelp :
        commandHelp(words);
        break;


      case cGraph :
        commandGraph(words);
        break;

      case cConfig :
        commandConfig();
        break;

      case cReset :  
	commandReset();
        break;
	
      case cClear :  
		commandClear();
//EED Borrame        	mVirtualExecuter->Clear();
        break;

      case cInclude :
			  commandInclude( words[1] , (words.size()==3) );
//EED Borrame		// if 'source' was given (words.size()==3) then tell to set the 
//EED Borrame		// source file name of the current complex box with the full file name included
//EED Borrame  		if (mCommandLine)
//EED Borrame        {
//EED Borrame           InterpretFile(words[1],(words.size()==3)); 
//EED Borrame        } else{
//EED Borrame            SwitchToFile(words[1],(words.size()==3) );
//EED Borrame        }
		break;

      case cLoad:
		commandLoad( words[1] );
//EED Borrame        GetExecuter()->LoadPackage(words[1]);
        break;

      case cUnload:
		commandUnload( words[1] );
//EED Borrame        GetExecuter()->UnLoadPackage(words[1]);
        break;

      case cDebug :
			  if (words.size()==2) commandDebug(words[1]);
				else commandDebug("");
        break;
			  
	// obsolete
    //  case cWorkspace :
    //    if (words.size() == 2) 
    //    {
    //       if (words[1]=="freeze")        mVirtualExecuter->SetNoExecMode(true);
    //       else if (words[1]=="unfreeze") mVirtualExecuter->SetNoExecMode(false);
    //    }
    //    else
    //    {
    //       mVirtualExecuter->SetWorkspaceName(words[2]);
    //    }
    //    break;

		default:
        bbtkInternalError("should not reach here !!!");
   }

    bbtkDebugMessage("interpreter",6,"<== Interpreter::DoInterpretLine(\""
		     <<line<<"\")"<<std::endl);
    
  }
  //=======================================================================  




  //=======================================================================
  void Interpreter::SplitLine ( const std::string& str, std::vector<std::string>& tokens)
{
    bbtkDebugMessage("interpreter",9,"==> Interpreter::SplitLine(\""<<str<<"\")"<<std::endl);

    std::string delimiters = "\"";
    std::vector<std::string> quote;
    Utilities::SplitString(str,delimiters,quote);

    delimiters = " \t";
    std::vector<std::string>::iterator i;
    for (i=quote.begin(); i!=quote.end(); ) 
    {
       Utilities::SplitString(*i,delimiters,tokens);
       ++i;
       if (i!=quote.end()) 
       {
        //    bbtkDebugMessage("interpreter",0,"\""<<*i<<"\""<<std::endl);
          tokens.push_back(*i);
          ++i;
       }
    }

    for (i=tokens.begin(); i!=tokens.end(); ++i) 
    {
      bbtkDebugMessage("interpreter",9,"--["<<*i<<"]"<<std::endl);
    }
    bbtkDebugMessage("interpreter",9,"<== Interpreter::SplitLine(\""<<str<<"\")"<<std::endl);

 }
  //=======================================================================
*/


  //=======================================================================
  void Interpreter::commandReset()  // virtual
  {
    // Cannot close all files if the reset command is read from a file !
    CloseAllFiles();
    mFileNameHistory.clear();
    this->mVirtualExecuter->Reset();
  }
  //=======================================================================

  //=======================================================================
  /**
   *
   */
  /*
  void Interpreter::Print( const std::string& str)
  {
    if (mVirtualExecuter->GetNoExecMode()) return;

    bbtkDebugMessageInc("interpreter",9,"Interpreter::Print(\""<<str<<"\")"<<std::endl);

 // TO DO :
 // InterpretLine ("load std")
 // InterpretLine("new ConcatStrings _C_ ") -> trouver un nom unique : # commande 
 // InterpretLine("new Print _P_") 
 // InterpretLine("connect _C_.Out _P_.In")
 // int num = 1
 

    std::vector<std::string> chains;
    std::string delimiters("$");

    // Skip delimiters at beginning.
    std::string::size_type lastPos = str.find_first_not_of(delimiters, 0);
    bool is_text = true;
    if (lastPos>0) is_text = false;

    // Find first delimiter.
    std::string::size_type pos     = str.find_first_of(delimiters, lastPos);

    while (std::string::npos != pos || std::string::npos != lastPos)
    {
       if (is_text) 
       {
          // Found a text token, add it to the vector.
          chains.push_back(str.substr(lastPos, pos - lastPos));
 // std::string token = str.substr(lastPos, pos - lastPos)
 // InterpretLine("set _C_.In%num% %token%")
 
       }
       else 
       {

       // is an output (between $$) : decode 
         std::string tok,box,output;
         tok = str.substr(lastPos, pos - lastPos);
         Utilities::SplitAroundFirstDot(tok,box,output);
         chains.push_back( mVirtualExecuter->Get(box,output) );

// InterpretLine("connect %tok% _C_.In%num%") 

       }
        // Skip delimiters.  Note the "not_of"
       lastPos = str.find_first_not_of(delimiters, pos);
        // Find next delimiter
       pos = str.find_first_of(delimiters, lastPos);
    //
       is_text = !is_text;
// num ++;
     }
// InterpretLine("exec _P_")
// if (IS_IN_WORKSPACE) InterpretLine("delete _C_; delete _P_");

     std::vector<std::string>::iterator i;
     for (i= chains.begin(); i!=chains.end(); ++i) 
     {

       Utilities::SubsBackslashN(*i);
       std::cout << *i;
     }
     std::cout << std::endl;
     bbtkDebugDecTab("interpreter",9);
 }
*/

  //=======================================================================
  /**
   *
   */
/*EED Borrame
  // =========================================================================
  void Interpreter::SwitchToFile( const std::string& name , bool source )
  {
  // Note : in the following :
  // name : the user supplied name 
  //      - abreviated name    e.g.       scr   scr.bbs
  //      - relative full name e.g.       ./scr.bbs   ../../scr.bbs 
  //      - absolute full name e.g.       /home/usrname/proj/dir/scr.bbs
  //          same for Windows, with      c:, d: ...
  //
  // use ./directory/subdir/scrname.bbs
  //

    bbtkDebugMessage("interpreter",4,"==> Interpreter::SwitchToFile( \""
		     <<name<<"\")"<<std::endl);

    std::vector<std::string> script_paths;
    std::string fullPathScriptName;  // full path script name
    std::string pkgname;             // e.g. <scriptname>.bbs
    std::vector<std::string> Filenames;

    // The following is *NOT* a debug time message :
    // It's a user intended message.
    // Please don't remove it.
    bbtkMessage("interpreter",1,
        "look for : [" << name
        << "]" << std::endl);


    std::string upath;
    pkgname = Utilities::ExtractScriptName(name,upath);

    bbtkMessage("interpreter",3,
		"package name:[" << pkgname
		 << "] path:[" << upath << "]" << std::endl);
    bool fullnameGiven = false; 
    bool foundFile     = false;

    // ==== "*" provided : load all scripts in given path 
    // relative (e.g. std/boxes/ *) or absolute      <<<<<<<< / *
    if (pkgname == "*") 
      {

	std::stringstream* stream = new std::stringstream;
	//if (upath.size()!=0) // avoid troubles for "*"
	
	// ==== no path provided : look in root bbs path
	if (upath.size()==0)
	  {
	    //	    bbtkMessage("interpreter",1,
	    // LG : add all bbs path
	    //  script_paths.push_back(  ConfigurationFile::GetInstance().Get_root_bbs_path() );
	    std::vector<std::string>::const_iterator i;
	    for (i=ConfigurationFile::GetInstance().Get_bbs_paths().begin();
		 i!=ConfigurationFile::GetInstance().Get_bbs_paths().end();
		 i++)
	      {
		script_paths.push_back(*i);
	      }
	  }
	// ==== absolute path provided 
	else if (upath[0]=='/' || upath[1] == ':' ) 
	  {
	    if ( Utilities::IsDirectory( upath ) )
	      {
		script_paths.push_back(upath);
	      }
	    else 
	      {
		bbtkError("'"<<upath<<"' : directory does not exist"); 
	      }
	  }
	// ==== relative path provided : search all bbs path appended with 
	// the relative path provided
	else
	  {    
	    std::vector<std::string>::const_iterator i;
	    for (i=ConfigurationFile::GetInstance().Get_bbs_paths().begin();
		 i!=ConfigurationFile::GetInstance().Get_bbs_paths().end();
		 i++)
	      {
		std::string full_path(*i);
		// we *really* want '.' to be the current working directory
		if (full_path == ".") 
		  {
		    char buf[2048]; // for getcwd
		    char * currentDir = getcwd(buf, 2048);
		    std::string cwd(currentDir);
		    full_path = currentDir;
		  } // if full_path
		
		full_path += ConfigurationFile::GetInstance().Get_file_separator();
		full_path += upath;
		
		if ( Utilities::IsDirectory( full_path ) )
		  {
		    script_paths.push_back(full_path);
		  }
	      } 
	    if (script_paths.empty())
	      {
		bbtkError("no '"<<upath<<"' subdir found in search paths" 
			  << std::endl);
	      }
	  }
	
	
	// === search paths list complete : now explore it
	int nbBssFiles = 0;	
	// ==== relative name, iterate + load all .bbs/.bbp files
	std::vector<std::string>::iterator i;
	for (i=script_paths.begin();i!=script_paths.end();i++)
	  {
	    bbtkMessage("interpreter",1,
			"--> Looking in '" << *i << "'" << std::endl);
	    
	    Filenames.clear();
	    //int nbFiles = 
	    Utilities::Explore(*i, false, Filenames);
	    
	    for (std::vector<std::string>::iterator j = Filenames.begin(); 
		 j!= Filenames.end(); ++j)
	      {
		int lgr = (*j).size();
		if (lgr < 5) continue;  
		// ignore non .bbp file
		if ( (*j).substr(lgr-4, 4) != ".bbp") continue; 
		
		(*stream) << "include \"" << *j << "\"\n";
		bbtkMessage("interpreter",2,"  --> Found '" << *j << "'" << std::endl);
		
		nbBssFiles++;
	      } // for (std::vector...
	  } // for (i=script_...

	// === Result ...
	if (nbBssFiles==0)
	  {
	    bbtkMessage("interpreter",1,
			"  --> No .bbp found"<< std::endl);
	  } 
	else 
	  {
	    bbtkMessage("interpreter",1,
			"  --> "<<nbBssFiles<<" .bbp found"<< std::endl);
	    SwitchToStream(stream);
	  }
	return;
      }  
    //=============== end pkgname=="*" ===========
    
    
    // if name starts with a / or a . or contains : user is assumed to have passed a relative/absolute name
    // (not only a plain script name)
    // we trust him, and try to expland the directory name
    // WARNING : starting from current local directory :  ./whatYouWant  (./ mandatory!)

    if (name[0]=='/' || name[1] == ':' || name[0]=='.')  // absolute path (linux/windows) or relative path
    { 

      // ===========================================================check user supplied location
      fullnameGiven = true;

      fullPathScriptName =  Utilities::ExpandLibName(name, false);

      // allow user to always forget ".bbs"
      int l = fullPathScriptName.size();

      if (l!=0) 
	  {
         if ((fullPathScriptName.substr(l-4, 4) != ".bbs")&&
			 (fullPathScriptName.substr(l-4, 4) != ".bbp"))
         {
			std::string tfullPathScriptName = fullPathScriptName + ".bbs";
			if ( Utilities::FileExists(tfullPathScriptName) )
			{
				fullPathScriptName = tfullPathScriptName;
				foundFile = true;
			}
			else 
			{
				tfullPathScriptName = fullPathScriptName + ".bbp";
				if ( Utilities::FileExists(tfullPathScriptName) )
				{
					fullPathScriptName = tfullPathScriptName;
					foundFile = true;
				}
			}
		 }
		 else 
		 {
			if ( Utilities::FileExists(fullPathScriptName) )
			{
				foundFile = true;
			}
		 }
	  } // endif l != 0
  }
  else
  // =============================== iterate on the paths
  {
      script_paths = ConfigurationFile::GetInstance().Get_bbs_paths();
      std::string path;
      std::vector<std::string>::iterator i;
      for (i=script_paths.begin();i!=script_paths.end();++i)
	  {
		path = *i;
		// we *really* want '.' to be the current working directory
		if (path == ".") 
		{
			char buf[2048]; // for getcwd
			char * currentDir = getcwd(buf, 2048);
			std::string cwd(currentDir);
			path = currentDir;
		}
	  
		std::string tfullPathScriptName = Utilities::MakePkgnameFromPath(path, name, false);
//Addition JCP tfullPathScriptName.size()>=4 
		if(tfullPathScriptName.size()>=4){
			if (tfullPathScriptName.substr(tfullPathScriptName.size()-4, 3)==".bb")
			{
			  fullPathScriptName = tfullPathScriptName;
				if ( ! Utilities::FileExists(fullPathScriptName) )
				{
					// The following is *NOT* a debug time message :
					// It's a user intended message.
					// Please don't remove it.
					bbtkMessage("interpreter",2,
				  "   [" <<fullPathScriptName <<"] : does not exist" 
				  <<std::endl);
					continue;  // try next path
				}
				bbtkMessage("interpreter",2,
					  "   [" <<fullPathScriptName 
					  <<"] : found" <<std::endl);
				foundFile = true;
				break; // a script was found; we stop iterating
			}
			else 
			{
				fullPathScriptName = tfullPathScriptName + ".bbs";
				// Check if library exists
				if ( ! Utilities::FileExists(fullPathScriptName) )
				{
					fullPathScriptName = tfullPathScriptName + ".bbp";
					if ( ! Utilities::FileExists(fullPathScriptName) )
					{
						// The following is *NOT* a debug time message :
						// It's a user intended message.
						// Please don't remove it.
						bbtkMessage("interpreter",2,
						"   [" <<tfullPathScriptName <<".bbs/.bbp] : do not exist" 
						<<std::endl);
						continue;  // try next path
					}
				}
				bbtkMessage("interpreter",2,
				  "   [" <<fullPathScriptName 
				  <<"] : found" <<std::endl);
				foundFile = true;
				break; // a script was found; we stop iterating
			}
		}		
	} //------------------ // end for ( package_paths.begin();i!=package_paths.end() )
  }

    if (!foundFile)
      {
	if (fullnameGiven)
	  if(fullPathScriptName == "")
            bbtkError("Path ["<<upath<<"] doesn't exist");
	  else
            bbtkError("Script ["<<fullPathScriptName<<"] not found");
	else
          bbtkError("No ["<<pkgname<<".bbs/.bbp] script found");
	return;
      }
    else
	{
      LoadScript(fullPathScriptName,name);
	  if (source) SetCurrentFileName(fullPathScriptName);
	}
    
    return;
  }
  //=======================================================================
*/

	//=======================================================================
	void Interpreter::SetCurrentFileName(const std::string &fullPathScriptName)  // virtual 
	{
		GetExecuter()->SetCurrentFileName(fullPathScriptName);
	}	
	//=======================================================================
	
	
	void Interpreter::SetTypeOfScript_Application()
	{
		GetExecuter()->SetTypeOfScript_Application();
	}

	
	
/*EED Borrame	
  //=======================================================================
void Interpreter::SwitchToStream( std::stringstream* stream )
{
  bbtkDebugMessage("interpreter",4,"==> Interpreter::SwitchToStream()"
		   <<std::endl);
   mFile.push_back(stream);
    std::ostringstream buffer_name;
    bufferNb++;
    buffer_name << "buffer_" ;

    if (mFileName.size()>0 )
    {
       buffer_name << mFileName.back() << "_" << mLine.back();
    }
    mFileName.push_back(buffer_name.str());
    mIncludeFileName.push_back(buffer_name.str());
    mLine.push_back(0);
}
  //=======================================================================
*/
	
/*	
  //=======================================================================
  void Interpreter::LoadScript( std::string fullPathScriptName,
				std::string includeScriptName)
  {
    bbtkDebugMessage("interpreter",4,"==> Interpreter::LoadScript("
		     <<fullPathScriptName<<")"
		     <<std::endl);

    Utilities::replace( fullPathScriptName , 
			 INVALID_FILE_SEPARATOR , VALID_FILE_SEPARATOR);
   
     if (find(mFileNameHistory.begin(),
	      mFileNameHistory.end(),
	      fullPathScriptName)!=mFileNameHistory.end())
     {
        return;
     }

    std::ifstream* s;
    s = new std::ifstream;
    s->open(fullPathScriptName.c_str());
    if (!s->good())
    {
        bbtkError("Could not open file ["<<fullPathScriptName<<"]");
        return;
    }

    bbtkMessage("interpreter",1,"   -->[" << fullPathScriptName 
		<< "] found" << std::endl);

    mFile.push_back(s);
    mFileName.push_back(fullPathScriptName);
    mFileNameHistory.push_back(fullPathScriptName);
    mIncludeFileName.push_back(includeScriptName);
    mLine.push_back(0);

    return;
  }
  //=======================================================================
*/

/*EED Borrame
  //=======================================================================
  void Interpreter::CloseCurrentFile()
  {
    bbtkDebugMessage("interpreter",9,"==> Interpreter::CloseCurrentFile()"
                      <<std::endl);

    if (mFile.size()==0)
    {
      bbtkDebugMessage("interpreter",9," -> no file left open"<<std::endl);
      return;
    }

    bbtkDebugMessage("interpreter",9," Closing file '"<<mFileName.back()<<"'"<<std::endl);

    std::ifstream* file = dynamic_cast<std::ifstream*>(mFile.back());
    if (file!=0) file->close();

    delete mFile.back();
    mFile.pop_back();
    mFileName.pop_back();
    mIncludeFileName.pop_back();
    mLine.pop_back();

    bbtkDebugMessage("interpreter",9," Remains "
                     <<mFile.size()
                     <<" open"<<std::endl);
    bbtkDebugMessage("interpreter",9,"<== Interpreter::CloseCurrentFile()"
                     <<std::endl);
  }
  //=======================================================================

 //=======================================================================
  void Interpreter::CloseAllFiles()
  {
    bbtkDebugMessage("interpreter",9,"==> Interpreter::CloseAllFiles()"
                      <<std::endl);

    while (mFile.size() != 0) 
    {
       CloseCurrentFile();
    }
    bbtkDebugMessage("interpreter",9,"<== Interpreter::CloseAllFiles()"
                      <<std::endl);
  }
  //=======================================================================


  //=======================================================================
  void Interpreter::InterpretCommand( const std::vector<std::string>& words,
                                      CommandInfoType& info )
  {
    bbtkDebugMessage("interpreter",9,"==> Interpreter::InterpretCommand(...)"<<std::endl);

    // searches the command keyword
    CommandDictType::iterator c;
    c = mCommandDict.find(words[0]);
    if ( c == mCommandDict.end() ) {
      bbtkError(words[0]<<" : unknown command");
    }

    // tests the number of args 
    if ( ( ((int)words.size())-1 < c->second.argmin ) ||
         ( ((int)words.size())-1 > c->second.argmax ) )
    {
       commandHelp(words[0]);
       bbtkError(words[0]<<" : wrong number of arguments");
    }
//std::cout<<"Interpreter::InterpretCommand( const std::vector<std::string>& words,"<<std::endl;
    info = c->second;

    bbtkDebugMessage("interpreter",9,"<== Interpreter::InterpretCommand(...)"<<std::endl);

  }
  //=======================================================================
 */


  //=======================================================================
  /// Displays help on all the commands
void Interpreter::commandHelp(const std::vector<std::string>& words)
{
    unsigned int nbarg = words.size()-1;

    if (nbarg==0) 
    {
       HelpCommands();
    }
    else if (nbarg==1) 
    {
      if (words[1]=="packages") 
      {
         GetExecuter()->GetFactory()->PrintHelpListPackages(true);
         return;
      }
      try 
      {
          commandHelp(words[1]);
      }
      catch (bbtk::Exception e) 
      {
         try 
         {
            GetExecuter()->GetFactory()->PrintHelpPackage(words[1]);
	    if ( mUser != 0 )
	      {
		std::string url = 
		  ConfigurationFile::GetInstance().Get_doc_path();
		url += "/bbdoc/" + words[1] + "/index.html";
		if (Utilities::FileExists(url)) 
		  {
		    mUser->InterpreterUserViewHtmlPage(url);
		  }
	      }
         }
         catch (bbtk::Exception f) 
         {
	   try 
	     {
	       std::string package;
	       GetExecuter()->GetFactory()->PrintHelpDescriptor(words[1],
								package);
	       if ( mUser != 0 )
		 {
		   std::string url = 
		     ConfigurationFile::GetInstance().Get_doc_path();
		   url += "/bbdoc/" + package + "/index.html";
		   if (Utilities::FileExists(url)) 
		     {
		       url += "#" + words[1];
		       mUser->InterpreterUserViewHtmlPage(url);
		     }
		 }
	     }
	   catch (bbtk::Exception g) 
	     {
	       try
		 {
		   GetExecuter()->PrintHelpBlackBox(words[1],"0","9999");
		 }
	       catch (bbtk::Exception h){
		 bbtkError("\""<<words[1].c_str()
			   <<"\" is not a known command, package, black box type or black box name");
	       }
	     }
         }
      }
    }
    else if (nbarg==2) 
    {
      if (words[2]=="all")
      {
         if ( words[1]=="packages" )
         {
            GetExecuter()->GetFactory()->PrintHelpListPackages(true,true);
            return;
          }
         try 
         {
            GetExecuter()->GetFactory()->PrintHelpPackage(words[1],true);
         }
         catch (bbtk::Exception f) 
         {
         }
     }
     else 
     {
        commandHelp(words[0]);
        bbtkError(words[0]<<" : syntax error");
     }
  }
  else 
  {
     bbtkError("Should not reach here !!!");
  }
}
  //=======================================================================

   //===================================================================    
  /// Displays the Configuration
  void Interpreter::commandConfig() const
  {
    ConfigurationFile::GetInstance().GetHelp(1);
  }  
   //===================================================================    

  //=======================================================================
  /// Displays help on all the commands
  void Interpreter::HelpCommands()
  {
    std::cout << "Available commands :" << std::endl;
    CommandDictType::iterator i;
    for ( i =  mCommandDict.begin();
          i != mCommandDict.end();
        ++i) {
              std::cout << " " << i->first << std::endl;
      //      std::cout << "   usage : " << i->second.syntax << std::endl;
      //     std::cout << "    " << i->second.help << std::endl;

    }
  }
  //=======================================================================


	
  //=======================================================================
  /// Displays help on a particular commands
  void Interpreter::commandHelp(const std::string& s)
  {
    CommandDictType::iterator c;
    c = mCommandDict.find(s);
    if ( c == mCommandDict.end() ) {
      bbtkError(s<<" : Unknown command");
    }	
    //    std::cout << " " << s << " : "<<  std::endl;
    //    CommandParamDictType::iterator i;
    //    for ( i =  c->second.begin();
    //      i != c->second.end();
    //      ++i) {
    std::cout << " usage : " << c->second.syntax << std::endl;
    std::cout << "  " << c->second.help << std::endl;

  }
  //=======================================================================

/*EED Borrame
  //=======================================================================
  /// Fills the vector commands with the commands which 
  /// have the first n chars of buf for prefix
  /// TODO : skip initial spaces in buf and also return the position of first
  /// non blank char in buf
  void Interpreter::FindCommandsWithPrefix( char* buf,
                                            int n,
                                            std::vector<std::string>& commands )
  {
    CommandDictType::const_iterator i;
    for (i=mCommandDict.begin(); i!=mCommandDict.end(); ++i)
    {
      if ((i->first).find(buf,0,n) == 0) 
        commands.push_back(i->first);
    }
  }
  //=======================================================================
*/
 

/*EED Borrame
  //=======================================================================
#ifdef BBTK_USE_TERMIOS_BASED_PROMPT
  
  inline void PrintChar(char c) { write(STDOUT_FILENO,&c,1); }
  inline void BackSpace() { write(STDOUT_FILENO,"\b \b",3); }
  
  // LG : KEYBOARD CODES AS SCANNED ON MY TTY : UNIVERSAL ?
  // IF NOT THE USER SHOULD BE ABLE TO CONFIGURE IT
  // E.G. STORE THIS IN bbtk_config.xml
#define BBTK_UP_ARROW_KBCODE    0x00415B1B
#define BBTK_DOWN_ARROW_KBCODE  0x00425B1B
#define BBTK_RIGHT_ARROW_KBCODE 0x00435B1B
#define BBTK_LEFT_ARROW_KBCODE  0x00445B1B
#define BBTK_BACKSPACE_KBCODE   0x00000008
#define BBTK_DEL_KBCODE         0x0000007F
#define BBTK_SPACE_KBCODE       0x00000020 

  //=======================================================================
  void Interpreter::GetLineFromPrompt(std::string& s)
  {
    int c;
    unsigned int ind=0;

    unsigned int MAX_LINE_SIZE = 160;
    unsigned int MAX_HISTORY_SIZE = 100;

    char* newline = new char[MAX_LINE_SIZE];
    memset(newline,0,MAX_LINE_SIZE);
    char* histline = new char[MAX_LINE_SIZE];
    memset(histline,0,MAX_LINE_SIZE);

    char* line = newline;
    unsigned int hist = mHistory.size();

    write(1,"> ",2);
    while(1)
    {
       c=0;
       read ( STDIN_FILENO, &c, 4) ;

       bbtkDebugMessage("debug",9,"[0x"<<std::hex<<c<<"]\n");

       // Printable character
       if ( (ind<MAX_LINE_SIZE-1) &&
            ( c >= BBTK_SPACE_KBCODE ) && 
            ( c <  BBTK_DEL_KBCODE )) 
       {
          PrintChar(c);
          line[ind++]=c;
       }
      // CR
       else if (c=='\n')
       {
       // delete the unused line
          if (line==newline)
              delete histline;
          else
              delete newline;
   
    // empty lines are not stored in from history
          if (strlen(line)) 
          {
             // if history too long : delete oldest command
             if (mHistory.size()>MAX_HISTORY_SIZE) 
             {
                delete mHistory.front();
                mHistory.pop_front();
             }
             mHistory.push_back(line);
          }
          break;
        }
       // Backspace
        else if ( (ind>0) && 
		  ((c == BBTK_BACKSPACE_KBCODE) ||
		   (c == BBTK_DEL_KBCODE)) )
	  {
	    line[ind--]=' ';
	    BackSpace();
	  }
	// Tab 
	else if (c=='\t')
	  {
	    // TODO : Command completion  
	    std::vector<std::string> commands;
	    FindCommandsWithPrefix( line,ind,commands);
	    if (commands.size()==1) 
	      {
		std::string com = *commands.begin();
		for (; ind<com.size(); ++ind) 
		  {
		    PrintChar(com[ind]); 
		    line[ind]=com[ind];
		  }
		PrintChar(' '); 
		line[ind++]=' ';
	      }
	    else if (commands.size()>1) 
	      {
		std::vector<std::string>::iterator i;
		write(1,"\n",1);
		for (i=commands.begin();i!=commands.end();++i) 
		  {
		    write(STDOUT_FILENO,(*i).c_str(),strlen((*i).c_str()));
		    PrintChar(' ');
		  }
		write(STDOUT_FILENO,"\n> ",3);
		//for (int j=0;j<ind;++j) 
		  //{
		    write(STDOUT_FILENO,line,ind); 
		    //  }
	      }
	  }
	// Arrow up : back in history
	else if (c==BBTK_UP_ARROW_KBCODE)
	  {
	    if (hist) 
	      {
		// erase current line
		while (ind--) BackSpace();
		// 
		hist--;
		// 
		strcpy(histline,mHistory[hist]);
		line = histline;
		ind = strlen(line);
		
		write(STDOUT_FILENO,line,ind);
	      }
	  }
	// Arrow down : down in history
	else if (c==BBTK_DOWN_ARROW_KBCODE)
	  {
	    if (hist<mHistory.size()-1) 
	      {
		// erase current line
		while (ind--) BackSpace();
		// 
		hist++;
		// 
		strcpy(histline,mHistory[hist]);
		line = histline;
		ind = strlen(line);
		
		write(STDOUT_FILENO,line,ind);
	      }
	    // end of history : switch back to newline
	    else if (hist==mHistory.size()-1)
	      {
		// erase current line
		while (ind--) BackSpace();
		// 
		hist++;
		// 
		line = newline;
		ind = strlen(line);
		
		write(STDOUT_FILENO,line,ind);
	      }
	  }
	// Arrow right
	else if (line[ind]!=0 && c==BBTK_RIGHT_ARROW_KBCODE)
	  {
	    PrintChar(line[ind]);
	    ind++;
	  }

	// Arrow left
	else if (ind>0 && c==BBTK_LEFT_ARROW_KBCODE)
	  {
	    PrintChar('\b');
	    ind--;
    
	  }

      }
    write(STDOUT_FILENO,"\n\r",2);
    
    
    s = line;
    
  }
#else

  //=======================================================================
  void Interpreter::GetLineFromPrompt(std::string& s)
  {  
    s.clear();

    putchar('>');
    putchar(' ');

    do 
    {
      char c = getchar();
      if (c=='\n') 
      {
        putchar('\n');
        break;
      }
      if (c=='\t') 
      {
        // putchar('T');
        continue;
      }
      // putchar(c);
      s += c;
    } 
    while (true);  
    
  }
  //=======================================================================  

#endif
*/

	
	
	
	
	
/*EED Borrame
  //=======================================================================
  void Interpreter::CommandLineInterpreter()
  {
    bbtkDebugMessageInc("interpreter",9,
                        "Interpreter::CommandLineInterpreter()"<<std::endl);

#ifdef BBTK_USE_TERMIOS_BASED_PROMPT  
    // Initialise the tty in non canonical mode with no echo
    // oter remembers the previous settings to restore them after 
    struct termios ter,oter;
    tcgetattr(0,&ter);
    oter=ter;
    ter.c_lflag &= ~ECHO;
    ter.c_lflag &= ~ICANON;
    ter.c_cc[VMIN]=1;
    ter.c_cc[VTIME]=0;
    tcsetattr(0,TCSANOW,&ter);
#endif
    
    mCommandLine = true;
    bool again = true;
    // bool insideComment = false; // for multiline comment  
    mInsideComment = false;
    do 
    {
      try
      {
        std::string line;
        GetLineFromPrompt(line);
        DoInterpretLine(line); //, insideComment);
      }
 //     catch (QuitException e)
 //     {
 //	bbtkMessage("interpreter",1,"Interpreter : Quit"<<std::endl);
 //        again = false;
      }
      catch (bbtk::Exception e) 
      {
        e.Print();
      }
        catch (std::exception& e) 
      {
        std::cerr << "* ERROR :: "<<e.what()<<" (not in bbtk)"<<std::endl;
      }
      catch (...)
      {
        std::cerr << "* UNDEFINED ERROR (not a bbtk nor a std exception)"<<std::endl;
      }
    }
    while (again);

#ifdef BBTK_USE_TERMIOS_BASED_PROMPT
    tcsetattr(0,TCSANOW,&oter);
#endif

    std::cout << "Good bye !" << std::endl;

    bbtkDebugDecTab("interpreter",9);
  }
*/
		  
//=======================================================================
void Interpreter::commandGraph(const std::vector<std::string>& words)
{
  std::string page;
    bool system_display = true;

    if ( ( mUser != 0 ) && ( mUser->InterpreterUserHasOwnHtmlPageViewer() ) )
      system_display = false; 
 
    if (words.size()==1) 
    {
      page = mVirtualExecuter->ShowGraph(".","0","0","","","",system_display);
    }
    else if (words.size()==2) 
    {
      page = mVirtualExecuter->ShowGraph(words[1],"0","0","","","",system_display);
    }
    else if (words.size()==3) 
    {
      page = mVirtualExecuter->ShowGraph(words[1],words[2],"0","","","",system_display);
    }
    else if (words.size()==4) 
    {
      page = mVirtualExecuter->ShowGraph(words[1],words[2],words[3],"","","",system_display);
    } 
    else if (words.size()==5) 
    {
      page = mVirtualExecuter->ShowGraph(words[1],words[2],words[3],words[4],"","",system_display);
    } 
    else if (words.size()==6) 
    {
      page = mVirtualExecuter->ShowGraph(words[1],words[2],words[3],words[4],words[5],"",system_display);
    } 
    else if (words.size()==7) 
      {
	page = mVirtualExecuter->ShowGraph(words[1],words[2],words[3],words[4],words[5],words[6],system_display);
      } 
    
    if ( ( mUser != 0 ) && ( mUser->InterpreterUserHasOwnHtmlPageViewer() ) )
      mUser->InterpreterUserViewHtmlPage(page);

  }
//=======================================================================


//=======================================================================
void  Interpreter::commandIndex(const std::string& filename, 
			 const std::string& type)
{
  Factory::IndexEntryType t;
  if (type=="Initials") t = Factory::Initials;
  else if (type=="Categories") t = Factory::Categories;
  else if (type=="Packages") t = Factory::Packages;
  else if (type=="Adaptors") t = Factory::Adaptors;
  
  GetExecuter()->GetFactory()->CreateHtmlIndex(t,filename);
}
//=======================================================================


//=======================================================================
void  Interpreter::commandNewGUI(const std::string& boxname,
			     const std::string& instanceName)
{
  if (mRealExecuter.expired())
    {
      bbtkError("command 'newgui' cannot be compiled yet");
    }

  std::string typeName = instanceName+"Type";
  std::stringstream* s = new std::stringstream;
  // create the complex box
  (*s) << "define "<<typeName<<std::endl;
  //  (*s) << "  description 'Automatically generated user interface for the box "
  //       << boxname << "'" <<std::endl;
  // create the Layout box
  (*s) << "  load wx"<<std::endl;
  (*s) << "  new LayoutLine layout"<<std::endl;
  // create the output 'Widget'
  (*s) << "  output Widget layout.Widget Widget"<<std::endl;
  // the box change output 
  (*s) << "  new MultipleInputs change"<<std::endl;
  (*s) << "  output BoxChange change.Out BoxChange"<<std::endl;

  // Browse the inputs of the box in order to find which ones are not 
  // connected and can be adapted from a widget adaptor
  // vector which stores the list of inputs of the box which must be connected
  std::vector<std::string> in;
 
  Factory::Pointer F = mVirtualExecuter->GetFactory();
  /*
  Package::Pointer user = F->GetPackage("user");
  */
  ComplexBlackBoxDescriptor::Pointer workspace = 
    mRealExecuter.lock()->GetCurrentDescriptor();

  if (workspace==0)
    {
      delete s;
      bbtkError("interpreter::CreateGUI : could not access the executer currently defined complex box");
    }
 

  /*
    (ComplexBlackBoxDescriptor::Pointer)(user->GetBlackBoxMap().find("workspace")->second.get());
  */

  BlackBox::Pointer box = workspace->GetPrototype()->bbGetBlackBox(boxname);
  //  BlackBox::InputConnectorMapType incm = box->bbGetInputConnectorMap();
  // int nb = 0;
  BlackBox::InputConnectorMapType::iterator i;
  for (i=box->bbGetInputConnectorMap().begin();
       i!=box->bbGetInputConnectorMap().end();
       ++i)
    {
      // If the input is connected : continue
      if (i->second->IsConnected()) continue;
      // Get the input descriptor 
      const BlackBoxInputDescriptor* d = box->bbGetDescriptor()->GetInputDescriptor(i->first);
      // If it is a "system" input : skip it
#ifdef USE_WXWIDGETS
      if ( ( d->GetCreatorTypeInfo() == typeid(AtomicBlackBoxDescriptor)) ||
	   ( d->GetCreatorTypeInfo() == typeid(WxBlackBoxDescriptor)) )
	continue;
#else
      if ( ( d->GetCreatorTypeInfo() == typeid(AtomicBlackBoxDescriptor)) )
	continue;
#endif
      bool widok = true;
      std::string widget,adaptor;
      // try to find a widget adaptor
      if (F->FindWidgetAdaptor(DataInfo(d->GetTypeInfo(),""),
			       d->GetDataInfo(),
			       adaptor))
	{
	  // command to create the adaptor
	  (*s) << "  new "<<adaptor<<" "<<i->first<<std::endl;
	  // Sets the label of the widget adaptor to the name of the input
	  (*s) << "  set "<<i->first<<".Label "<<i->first<<std::endl;
	  // Sets the initial value of the widget to the value of the input
	  (*s) << "  set "<<i->first<<".In \" "
	       <<box->bbGetInputAsString(i->first)<<"\""
	       << std::endl;
	  // store the input name
	  in.push_back(i->first);
	  (*s) << "  connect "<<i->first<<".Widget layout.Widget"<<in.size()<<std::endl;
	    //<i->first<<"'"<<std::endl;
	  (*s) << "  connect "<<i->first<<".BoxChange change.In"<<in.size()<<std::endl;
	}
      // try to find a two pieces adaptor
      else if (F->FindWidgetAdaptor2(DataInfo(d->GetTypeInfo(),""),
				     d->GetDataInfo(),
				     widget,adaptor) )
	{
	  // command to create the widget
	  (*s) << "  new "<<widget<<" "<<i->first<<"Widget"<<std::endl;
	  // command to create the adaptor
	  (*s) << "  new "<<adaptor<<" "<<i->first<<std::endl;
	  // connect the two
	  (*s) << "  connect "<<i->first<<"Widget.Out "
	       <<i->first<<".In"<<std::endl;
	  // Sets the label of the widget adaptor to the name of the input
	  (*s) << "  set "<<i->first<<"Widget.Label "<<i->first<<std::endl;
	  // Sets the initial value of the widget to the value of the input
	  (*s) << "  set "<<i->first<<"Widget.In \" "
	       <<box->bbGetInputAsString(i->first)<<"\""<< std::endl;
	  // store the input name
	  in.push_back(i->first);
	  (*s) << "  connect "<<i->first<<"Widget.Widget layout.Widget"<<in.size()<<std::endl;
	    //<i->first<<"'"<<std::endl;
	  (*s) << "  connect "<<i->first<<"Widget.BoxChange change.In"<<in.size()<<std::endl;

	}
      // try to find an adaptor from string 
      // If found then can create a text input which 
      // will be automatically adapted 
      else if (F->FindAdaptor(DataInfo(typeid(std::string),""),
			       d->GetDataInfo(),
			       adaptor))
	{
	  // command to create the adaptor
	  (*s) << "  new InputText "<<i->first<<std::endl;
	  // Sets the label of the widget adaptor to the name of the input
	  (*s) << "  set "<<i->first<<".Title "<<i->first<<std::endl;
	  // Sets the initial value of the widget to the value of the input
	  (*s) << "  set "<<i->first<<".In \" "
	       <<box->bbGetInputAsString(i->first)<<"\""<< std::endl;
	  // store the input name
	  in.push_back(i->first);
	  (*s) << "  connect "<<i->first<<".Widget layout.Widget"<<in.size()<<std::endl;
	    //<i->first<<"'"<<std::endl;
	  (*s) << "  connect "<<i->first<<".BoxChange change.In"<<in.size()<<std::endl;

	}
      else 
	{
	  widok = false;
	}
      if (widok)
	{
	  // command to create the output
	  (*s) << "  output "<<i->first<<" "
	       <<i->first<<".Out "<<i->first<<std::endl;
	    //	       <<" Output of the widget which allows to set "
	  
	}
    }   
  // Inputs for window properties
  (*s) << "  input WinTitle layout.WinTitle Title"<<std::endl;
  (*s) << "  input WinWidth layout.WinWidth Width"<<std::endl;
  (*s) << "  input WinHeight layout.WinHeight Height"<<std::endl;
  (*s) << "  input WinDialog layout.WinDialog Dialog"<<std::endl;
  (*s) << "  input WinHide layout.WinHide Hide"<<std::endl;

  
  
  // Execute the box executes the layout
  (*s) << "  exec layout" << std::endl;
  (*s) << "endefine" << std::endl;
  // (*s) << "help "<< typeName<< std::endl;
  // instanciate the box and connect it
  (*s) << "new "<<typeName<<" "<<instanceName<<std::endl;
  // connections
  std::vector<std::string>::iterator j;
  for (j=in.begin();j!=in.end();++j)
    {
      // connect
      (*s) << "connect "<<instanceName<<"."<<*j<<" "
	   << boxname<<"."<<*j<<std::endl;
    }
  // That's all folks ! now execute the commands :
  SwitchToStream(s);
}
//=======================================================================



 //==========================================================================
  void Interpreter::commandDebug(const std::string& name)
  {
    if ((name.length()==2)&&(name[0]=='-'))
      {
	if (name[1]=='D')
	  {
	    bbtk::StaticInitTime::PrintObjectListInfo = true;
	  }
	if (name[1]=='C')
	  {
	    //	    int o = MessageManager::GetMessageLevel("debug");
	    //	    if (o<2) MessageManager::SetMessageLevel("debug",2);
	    mVirtualExecuter->GetFactory()->Check();
	    //	    MessageManager::SetMessageLevel("debug",o);
	  }
      }
    else 
      {
	Object:: PrintObjectListInfo(name);
      }
  }
 //==========================================================================

  /*
  //==========================================================================
  // Adds a callback when 'break' command issued
  void Interpreter::AddBreakObserver( BreakCallbackType c )
  {
    mBreakSignal.connect(c);
  }
 //==========================================================================
 */

 //==========================================================================
  std::string Interpreter::GetObjectName() const
  {
    return std::string("Interpreter");
  }
  //==========================================================================
  
  //==========================================================================
  std::string  Interpreter::GetObjectInfo() const 
  {
    std::stringstream i;
    return i.str();
  }
  //==========================================================================

  //==========================================================================
size_t  Interpreter::GetObjectSize() const 
{
  size_t s = Superclass::GetObjectSize();
  s += Interpreter::GetObjectInternalSize();
  return s;
  }
  //==========================================================================
  //==========================================================================
size_t  Interpreter::GetObjectInternalSize() const 
{
  size_t s = sizeof(Interpreter);
  return s;
  }
  //==========================================================================
  //==========================================================================
  size_t  Interpreter::GetObjectRecursiveSize() const 
  {
    size_t s = Superclass::GetObjectRecursiveSize();
    s += Interpreter::GetObjectInternalSize();
    s += mVirtualExecuter->GetObjectRecursiveSize();
    return s;
  }
  //==========================================================================
}//namespace


