/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */



/*=========================================================================                                                                               
  Program:   bbtk
  Module:    $RCSfile: bbtkPackage.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:49:01 $
  Version:   $Revision: 1.37 $
=========================================================================*/



/**
 *\file
 *\brief Class bbtk::Package : registers black boxes descriptors and is able to create instances of the black boxes registered.
 */
#include "bbtkPackage.h"
#include "bbtkComplexBlackBoxDescriptor.h"
#include "bbtkMessageManager.h"
#include "bbtkConfigurationFile.h"
#include <fstream>
#include <time.h>
#include "bbtkUtilities.h"

namespace bbtk
{
  


  //==========================================================================
  /// Creates a new package
  Package::Pointer Package::New(const std::string& name,
				const std::string& author,
				const std::string& description,
				const std::string& version) 
  {
    bbtkDebugMessage("object",1,"##> Package::New('"<<name<<"',...)"
		     <<bbtkendl);
    Package::Pointer p = MakePointer(new Package(name,
					       author,
					       description,
					       version));
    bbtkDebugMessage("object",2,"<## Package::New('"<<name<<"',...)"
		     <<bbtkendl);
    return p;
  }
  //==========================================================================

  //==========================================================================
  /// Ctor with the name of the package
  Package::Package(const std::string& name,
		   const std::string& author,
		   const std::string& description,
		   const std::string& version) 
    :
    mDynamicLibraryHandler(0),
    mName(name),
    mAuthor(author),
    mDescription(description),
    mVersion(version)
  {
    bbtkDebugMessage("object",2,"==> Package('"<<name<<"',...)"
		     <<bbtkendl);
    std::string default_doc_dir = ConfigurationFile::GetInstance().Get_default_temp_dir();
    char c = default_doc_dir.c_str()[strlen(default_doc_dir.c_str())-1];
    std::string url = default_doc_dir; 
    if (c != '/' && c !='\\') url = url + "/";
    url = url +  "temp_dir/" + name + "/index.html";    
    
    SetDocURL(url);
    SetDocRelativeURL("Relative url not set");

    /*
    std::string relurl(BBTK_STRINGIFY_SYMBOL(BBTK_DOC_REL_PATH));
    relurl += "/packages/"+name+"/bbdoc/index.html";
    std::string url = ConfigurationFile::GetInstance().Get_url()
      + relurl; 
    SetDocURL(url);
    SetDocRelativeURL(relurl);   
    */

    //    std::cout  << "   url=["<<url<<"]"<<std::endl;
    //    std::cout  << "relurl=["<<relurl<<"]"<<std::endl;
    bbtkDebugMessage("object",2,"<== Package::Package('"<<name<<"',...) OK"
		     <<bbtkendl);

  }
  //==========================================================================



  //==========================================================================
  /// Dtor
  Package::~Package()
  {
    bbtkDebugMessage("object",2,"==> ~Package(\""<<mName<<"\")"<<bbtkendl);
  }
  //==========================================================================


  //==========================================================================
  void PackageReleaseBlackBoxDescriptorInternal(Package::WeakPointer pack,
						const std::string& descname)
  {
    // Try to release descriptor
    std::string packname = pack.lock()->GetName();
    
    bbtkDebugMessage("package",5,"--- Releasing descriptor '"
		     <<packname<<"::"<<descname<<"'"<<bbtkendl);
    
    
    Package::DescriptorMapType::iterator desc = 
      pack.lock()->GetDescriptorMap().find(descname);
    if (desc ==  pack.lock()->GetDescriptorMap().end())
      {
	bbtkDebugMessage("package",5,
			 "    Descriptor has already been released"
			<<bbtkendl);
	return;
		  }
    //    bbtkDebugMessage("package",3,
    //		     "    Trying unreferencing it ... "<<std::endl);
    BlackBoxDescriptor::WeakPointer pdesc = desc->second;
    desc->second.reset();
    // if it is dead : remove it 
    if (pdesc.expired()) 
      {
	bbtkDebugMessage("package",2," ==> '"<<packname<<"::"<<descname<<"' Descriptor expired"<<bbtkendl);
	if (pack.expired()) 
	  {
	    bbtkDebugMessage("package",2,
                             "     ... and caused its package death"
			     <<bbtkendl);
	    return;
	  } // pack.expired
	desc = pack.lock()->GetDescriptorMap().find(descname);
	if (desc !=  pack.lock()->GetDescriptorMap().end())	  pack.lock()->GetDescriptorMap().erase(desc);
      }   else    {  //pdesc.expired
      bbtkDebugMessage("package",5,"    ... Descriptor still alive ("
		       <<pdesc.use_count()<<" refs)"
		       <<bbtkendl);
      pack.lock()->GetDescriptorMap()[descname] = pdesc.lock();
    } // pdesc.expired
  }
  //==========================================================================


  //==========================================================================
  /// Release
  void Package::Release(Package::WeakPointer pack)
  {
    std::string packname = pack.lock()->mName;
    bbtkDebugMessage("package",1,"==> Package::Release('"<<
		     packname<<"')"<<bbtkendl);

    long dyn = pack.lock()->mDynamicLibraryHandler ? 1:0; 
    long ndesc = pack.lock()->GetDescriptorMap().size();
    long nrefs = pack.use_count();

    bbtkDebugMessage("package",5," "<<nrefs<<" refs / "
		     <<ndesc<<" descr / dyn="
		     <<dyn<<std::endl);

    // A package is "free" from any external reference iff :
    // i) It is not dynamically loaded and nrefs == ndesc 
    // (each desc references its package) or
    // ii) It is dynamically loaded and nrefs == ndesc + 1
    // (A dynamic library holds a static pointer on the package it contains
    //  which is allocated when the PACKAGENAMEGetPackage() func is called,
    //  and descallocated (reset) by PACKAGENAMEDeletePackage())
    if (nrefs == ndesc + dyn) 
      {
	bbtkDebugMessage("package",5,
			 " -> No more external ref : checking descriptors"
			 <<bbtkendl);
	// We must take care that removing refs on descriptors 
	// can lead to their deletion which can in turn unref 
	// internal boxes which can release their descriptors hence 
	// call Package::ReleaseBlackBoxDescriptor 
	// As a consequence during descriptors release :
	// 1) The map can change dynamically : we cannot iterate over it 
	//    as any iterator can become invalid
	// 2) The package can auto-destruct : we must test its existence 
	//    after each release
	// We must also take care of not locking the package pointer 
	// or any ref count check in Package::ReleaseBlackBoxDescriptor
	// would be wrong

	// The list of descriptors names at start
	std::vector<std::string> descnamelist;
	DescriptorMapType::iterator i;
	for (i=pack.lock()->mDescriptorMap.begin();
	     i!= pack.lock()->mDescriptorMap.end();
	     ++i)
	  descnamelist.push_back(i->first);

	// Iterator over the initial names
	std::vector<std::string>::iterator descname;
	for (descname=descnamelist.begin();
	     descname!=descnamelist.end();
	     ++descname)
	  {
	    // Is package still alive ?
	    if (pack.expired()) 
	      {
		bbtkDebugMessage("package",1,"--- Package::Release('"<<
				 packname
				 <<"') : package expired during release : bailing out"<<bbtkendl);
		break;
	      }
  
#if defined(MACOSX)
		  BlackBoxDescriptor::Pointer desc = pack.lock()->mDescriptorMap[*descname];
		  if ( (dyn==0) || (boost::dynamic_pointer_cast<ComplexBlackBoxDescriptor>(desc)) )
		  {
			  PackageReleaseBlackBoxDescriptorInternal(pack,*descname);
		  } 
#else
		  PackageReleaseBlackBoxDescriptorInternal(pack,*descname);
#endif

	  }

	//
	UnLoadDynamicLibrary(pack);
	// Unload orphan dl packages 
	Package::UnLoadReleasedDynamicallyLoadedPackages();

      }
    
#ifdef BBTK_COMPILE_DEBUG_MESSAGES

    bbtkDebugMessage("package",2,"<== Package::Release('"<<
		     packname<<"')"<<bbtkendl);

    if (!pack.expired())
      {
	long dyn = pack.lock()->mDynamicLibraryHandler ? 1:0; 
	long ndesc = pack.lock()->GetDescriptorMap().size();
	long nrefs = pack.use_count();
	
	bbtkDebugMessage("package",1," ... Package still alive ("
			 <<nrefs<<" refs / "
			 <<ndesc<<" descr / dyn="
			 <<dyn<<")"<<std::endl);
      }
    else 
      {
	bbtkDebugMessage("package",1," ... Package has been released"
			 <<std::endl);
      }
#endif
  }
  //==========================================================================

  //==========================================================================
  /// "Releases" the package
  /// Signals the package that it can free the given descriptor
  /// if they are no more used and free itself if it is no
  /// more used
  /// Note : Any non-weak pointer on the package must have been freed
  void Package::ReleaseBlackBoxDescriptor(Package::WeakPointer pack,
					  BlackBoxDescriptor::WeakPointer descr)
  {
    std::string packname = pack.lock()->mName;
    std::string dname = descr.lock()->GetTypeName();    
    bbtkDebugMessage("package",3,"==> Package::ReleaseBlackBoxDescriptor('"<<
		     packname<<"','"<<dname<<"') : refs="
		     <<descr.use_count()<<bbtkendl);

    long dyn = pack.lock()->mDynamicLibraryHandler ? 1:0; 
    long ndesc = pack.lock()->GetDescriptorMap().size();
    long nrefs = pack.use_count();

    bbtkDebugMessage("package",5," "<<nrefs<<" refs / "
		     <<ndesc<<" descr / dynamically loaded = "
		     <<dyn<<std::endl);

    // A package is "free" from any external reference iff :
    // i) It is not dynamically loaded and nrefs == ndesc 
    // (each desc references its package) or
    // ii) It is dynamically loaded and nrefs == ndesc + 1
    // (A dynamic library holds a static pointer on the package it contains
    //  which is allocated when the PACKAGENAMEGetPackage() func is called,
    //  and descallocated (reset) by PACKAGENAMEDeletePackage())
    if (nrefs == ndesc + dyn) 
      {
#if defined(MACOSX)
		  //
		  if (dyn==0)
			  PackageReleaseBlackBoxDescriptorInternal(pack,dname);
#else
		  PackageReleaseBlackBoxDescriptorInternal(pack,dname);
#endif	
      }
    
    // If the package is released and dynamically loaded 
    // then put it in the static list mReleasedDynamicallyLoadedPackages
    UnLoadDynamicLibrary(pack,false);
        
    bbtkDebugMessage("package",4,"<== Package::ReleaseBlackBoxDescriptor('"<<
		     packname<<"','"<<dname<<"'): refs="
		     <<descr.use_count()<<bbtkendl);
    /*
    if (!pack.expired())
      {
	long dyn = pack.lock()->mDynamicLibraryHandler ? 1:0; 
	long ndesc = pack.lock()->GetDescriptorMap().size();
	long nrefs = pack.use_count();
	
	bbtkDebugMessage("package",3," ... Package still alive ("
			 <<nrefs<<" refs / "
			 <<ndesc<<" descr / dyn="
			 <<dyn<<")"<<std::endl);
      }
    else 
      {
	bbtkDebugMessage("package",3,"   ... Package has been released"
			 <<std::endl);
      }  
    */
  }
  //==========================================================================    

  //==========================================================================
  /// Opens a dynamic library which contains a bbtk package
  /// Returns the handler 
  /// Load the package management symbols from the lib
  /// returns false if a problem occured hence can be used 
  /// to test that a dyn lib is a valid bbtk package lib
  /// NB : The BBTK version exported from the library 
  ///      is tested against the current bbtk version
  DynamicLibraryHandler Package::OpenDynamicLibrary( const std::string& libname,const std::string& package_name,
													 DLGetPackageFunction& getpack, DLDeletePackageFunction& delpack)
  {
    bbtkDebugMessage("package",3,"==> Package::OpenDynamicLibrary("
		     <<libname<<")"<<std::endl);
#if defined(__GNUC__)

    // Open shared lib
    void *handler;
    handler = dlopen(libname.c_str(), BBTK_RTLD_TIME | BBTK_RTLD_SCOPE );
//EED     handler = dlopen(libname.c_str(), RTLD_LAZY | RTLD_LOCAL );
	  
    if (!handler)
      {
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading could not open shared library [" <<libname<<"] : "
		    <<dlerror() << std::endl);
	return 0;
      }

    bbtkDebugMessage("package",3,"* Shared lib ["<<libname<<"] open"<<std::endl);

    // Loads the Package bbtk version function 
    std::string getvername(package_name);
    getvername += BBTK_STRINGIFY_SYMBOL(BBTK_GET_PACKAGE_BBTK_VERSION_FUNCTION_NAME);
    DLGetPackageBBTKVersionFunction getbbtkversion  = (DLGetPackageBBTKVersionFunction)(dlsym(handler,getvername.c_str()));
    if (!getbbtkversion)
      {
	bbtkDebugMessage("package",3,"***"<<std::endl);
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading shared library ["<<libname
		    <<"] is not a valid bbtk package."
		    <<" Symbol ["<<getvername<<"] :"<<dlerror()<< std::endl);
 	dlclose(handler);
	return 0;
      }

    bbtkDebugMessage("package",3,"* Symbol ["<<getvername
		     <<"] found"<<std::endl);
    // version matches ?

    if (getbbtkversion() != bbtk::GetVersion())
      {
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading: "<<package_name
			<<" - Shared library ["<<libname
		    <<"] was build with bbtk version "
		    <<getbbtkversion()
		    <<" but the current program runs with version "
		    <<bbtk::GetVersion()<<" : cannot load it. You have to recompile your BBTK-Package."<<std::endl);
	dlclose(handler);
	return 0;
	
      }

    bbtkDebugMessage("package",3,"* Package bbtk version '"<<getbbtkversion()<<"' matches"<<std::endl);
	    // Loads the Package get function
    std::string getpackname(package_name);
    getpackname += BBTK_STRINGIFY_SYMBOL(BBTK_GET_PACKAGE_FUNCTION_NAME);
    getpack = (DLGetPackageFunction)(dlsym(handler, getpackname.c_str()));
    if (!getpack)
      {
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading shared library ["<<libname
		    <<"] is not a valid bbtk package."
		    <<" Symbol ["<<getpackname<<"] :"<<dlerror()<< std::endl);
	dlclose(handler);
	return 0;
      }
  
    bbtkDebugMessage("package",3,"* Symbol ["<<getpackname<<"] found"<<std::endl);
    // Loads the Package delete function
					  
    std::string delpackname(package_name);
    delpackname += BBTK_STRINGIFY_SYMBOL(BBTK_DEL_PACKAGE_FUNCTION_NAME);
    delpack = (DLDeletePackageFunction)(dlsym(handler, delpackname.c_str()));
    if (!delpack)
      {
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading shared library ["<<libname
		    <<"] is not a valid bbtk package."
		    <<" Symbol ["<<delpackname<<"] :"<<dlerror()<< std::endl);
	dlclose(handler);
	return 0;
      }
    bbtkDebugMessage("package",3,"* Symbol ["<<delpackname<<"] found"<<std::endl);		  
#elif defined(_WIN32)
    
    HINSTANCE handler;
    
    SetErrorMode(0);
    // Open shared lib
    handler = LoadLibrary(libname.c_str());
    if (!handler)
      {
	bbtkMessage("package",0,
		    "BBTK ..ERROR..  could not open shared library [" <<libname<<"]"
		    << std::endl);
	DWORD dwErrorCode = 0;
	dwErrorCode = GetLastError();
	bbtkMessage("package",2,
		"Windows Error: [" << dwErrorCode <<"]"
		<< std::endl);
	 
	return 0;
      }
    
    // Loads the Package bbtk version function 
    std::string getvername(package_name);
    getvername += BBTK_STRINGIFY_SYMBOL(BBTK_GET_PACKAGE_BBTK_VERSION_FUNCTION_NAME);
    DLGetPackageBBTKVersionFunction getbbtkversion = (DLGetPackageBBTKVersionFunction)(GetProcAddress(handler, getvername.c_str()));
	  
    if (!getbbtkversion)
      {
	FreeLibrary(handler);
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading shared library ["<<libname
		    <<"] is not a valid bbtk package."
		    <<" Symbol ["<<getbbtkversion<<"] not found"<< std::endl);
	return 0;
      }
    
    // version matches ?
    if (getbbtkversion() != bbtk::GetVersion())
      {
		  FreeLibrary(handler);
		  bbtkMessage("package",0,
				"BBTK ..ERROR.. loading: "<<package_name
				<<" - Shared library ["<<libname
				<<"] was build with bbtk version "
				<<getbbtkversion()
				<<" but the current program runs with version "
				<<bbtk::GetVersion()<<" : cannot load it. You have to recompile your BBTK-Package."<<std::endl);
		  return 0;
      }

     // Loads the Package get function
    std::string getpackname(package_name);
    getpackname += BBTK_STRINGIFY_SYMBOL(BBTK_GET_PACKAGE_FUNCTION_NAME);
    getpack = (DLGetPackageFunction)(GetProcAddress(handler, getpackname.c_str()));
    if (!getpack)
      {
	FreeLibrary(handler);
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading shared library ["<<libname
		    <<"] is not a valid bbtk package."
		    <<" Symbol ["<<getpackname<<"] not found"<< std::endl);
	return 0;
      }
    
    // Loads the Package delete function
    std::string delpackname(package_name);
    delpackname += BBTK_STRINGIFY_SYMBOL(BBTK_DEL_PACKAGE_FUNCTION_NAME);
    delpack = (DLDeletePackageFunction)(GetProcAddress(handler, delpackname.c_str()));
    if (!delpack)
      {
	FreeLibrary(handler);
	bbtkMessage("package",0,
		    "BBTK ..ERROR.. loading shared library ["<<libname
		    <<"] is not a valid bbtk package."
		    <<" Symbol ["<<delpackname<<"] not found"<< std::endl);
	return 0;
      }
					 
#else
    bbtkError("neither __GNUC__ nor _WIN32 ?!? How did you compile ?");
#endif
    
    return handler;
  }
  //==========================================================================
  
  //==========================================================================
  /// Loads a package from a dynamic library
  Package::Pointer Package::CreateFromDynamicLibrary(const std::string& libname,
						     const std::string& pkgname,
						     const std::string& path)
  {
    bbtkDebugMessage("package",1,"==> Package::CreateFromDynamicLibrary("
		     <<libname<<")"<<std::endl);

    DLGetPackageFunction gf;
    DLDeletePackageFunction df;
    DynamicLibraryHandler h = Package::OpenDynamicLibrary(libname,
							  pkgname,
							  gf,df);
    if (h==0) return Package::Pointer(); 
    Package::Pointer p = gf();
    p->mDynamicLibraryHandler = h;
    p->mDLDeletePackageFunction = df;
    
    std::string separator =
      ConfigurationFile::GetInstance().Get_file_separator ();
    //BBTK_STRINGIFY_SYMBOL(BBTK_DOC_REL_PATH)
    std::string docreldoc = 
      separator + "bbdoc" + separator + pkgname + separator + "index.html";
    std::string reldoc = 
      ".." + separator + ".." + docreldoc;
    std::string doc = path + separator + ".." + separator
      + BBTK_STRINGIFY_SYMBOL(BBTK_DOC_REL_PATH)
      + docreldoc;
    
    p->SetDocURL(doc);
    p->SetDocRelativeURL(reldoc);
    
    bbtkDebugMessage("package",2,"<== Package::CreateFromDynamicLibrary("
		     <<libname<<") .. OK"<<std::endl);
    return p;
  }
  //==========================================================================




  //==========================================================================
  /// UnLoads the package dynamic library (if any)
  void Package::UnLoadDynamicLibrary(Package::WeakPointer pack, bool doit)
  {
    if (pack.expired() || (!pack.lock()->mDynamicLibraryHandler))
      return;

    std::string packname = pack.lock()->GetName();
    bbtkDebugMessage("package",5,"==> Package::UnLoadDynamicLibrary('"
		     <<packname<<"')"
		     <<std::endl);

    if (!pack.lock()->GetDescriptorMap().empty())
      {

	bbtkDebugMessage("package",5,"   Package not empty ... abort"
			 <<std::endl);
	return;
	/*
	bbtkGlobalError("Package::UnLoadDynamicLibrary('"<<packname<<") : "
			<<"DescriptorMap not empty "
			<<BBTK_INTERNAL_ERROR_MESSAGE);
	*/
      }

    if (pack.use_count()!=1)
      {
	bbtkGlobalError("Package::UnLoadDynamicLibrary('"<<packname<<") : "
			<<"empty dl package with external refs"
			<<BBTK_INTERNAL_ERROR_MESSAGE);
      } 

    if (doit) 
      {
	UnLoad(pack);
	bbtkDebugMessage("package",5,"==> dynamic library for package '"
			 <<packname<<"' closed"
			 <<std::endl);    
      }
    else 
      {
	mReleasedDynamicallyLoadedPackages.insert(pack);
	bbtkDebugMessage("package",1,"==> package '"<<packname
			 <<"' put in the 'to unload' list"
			 <<std::endl);
      }

    bbtkDebugMessage("package",5,"<== Package::UnLoadDynamicLibrary('"
		     <<packname<<"')"
		     <<std::endl); 
  }
  //==========================================================================

  //==========================================================================
  /// UnLoads released packages that were loaded dynamically
  /// see UnLoadDynamicLibrary and ReleaseBlackBoxDescriptor
  void Package::UnLoadReleasedDynamicallyLoadedPackages()
  {
    bbtkDebugMessage("package",5,"==> Package::UnLoadReleasedDynamicallyLoadedPackages()"<<std::endl);

    std::set<Package::WeakPointer>::iterator i;

//JCP- 21-04-09
	if(mReleasedDynamicallyLoadedPackages.size()>0){
		for (i=mReleasedDynamicallyLoadedPackages.begin();
		i!=mReleasedDynamicallyLoadedPackages.end();
		++i)
		{
		if (!i->expired()) UnLoad(*i);
		}
	}
//JCP- 21-04-09
    bbtkDebugMessage("package",5,"<== Package::UnLoadReleasedDynamicallyLoadedPackages()"<<std::endl);
  }
  //==========================================================================

  //==========================================================================
  void Package::UnLoad(Package::WeakPointer pack)
  {
    std::string packname = pack.lock()->GetName();
    bbtkDebugMessage("package",6,"==> Package::UnLoad("<<packname<<")"<<std::endl);

    Package* p = pack.lock().get();
    
    DynamicLibraryHandler h = p->mDynamicLibraryHandler;
    
    // deletes the package
    p->mDLDeletePackageFunction();
    
    // closes the dl handler
#if defined(__GNUC__)  
	  if (dlclose(h)!=0)
	  {
		  printf("EED Package::UnLoad ERROR %s\n", packname.c_str() );
		  bbtkWarning("Failed to close dynamic library for package '"<<packname
					  <<"'"<<std::endl);
	  }  
	  
#elif defined(_WIN32)
    FreeLibrary(h);
#endif

    bbtkDebugMessage("package",1,"==> dynamic library for package '"
		     <<packname<<"' closed"
		     <<std::endl);    
    bbtkDebugMessage("package",6,"   ... dynamic library unloaded"<<std::endl);
  }
  //==========================================================================

	bool Package::ifBoxExist( std::string type)	
    {
		bool ok=false;
		DescriptorMapType::const_iterator i = mDescriptorMap.find(type);
		if (i != mDescriptorMap.end())  
		{
			ok=true;
		}
		return ok;
    }
	
  //==========================================================================
  /// Creates an instance of a black box of type <type> with name <name>
  BlackBox::Pointer Package::NewBlackBox(const std::string& type, 
					 const std::string& name) const
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()<<">::NewBlackBox(\""<<type<<"\",\""<<name<<"\")"<<bbtkendl);
    
    DescriptorMapType::const_iterator i = mDescriptorMap.find(type);
    if (i == mDescriptorMap.end())  
    {
	   bbtkDebugDecTab("kernel",8);
	   return BlackBox::Pointer();
    }
    BlackBox::Pointer bb =i->second->NewBlackBox(name);
    bbtkDebugDecTab("kernel",8);
    return bb;   

  }
  //==========================================================================



  //==========================================================================
  /// Creates an instance of an adaptor of input type <typein> and 
  /// output type <typeout>  with name <name>
  BlackBox::Pointer Package::NewAdaptor(const DataInfo& typein,
				const DataInfo& typeout,
				const std::string& name) const
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()<<
			">::NewAdaptor("
			<<typein<<","
			<<typeout<<",\""
			<<name<<"\")"<<bbtkendl);

    AdaptorKey key(typein,typeout,
		   BlackBoxDescriptor::DEFAULT_ADAPTOR);
    AdaptorMapType::const_iterator i = mAdaptorMap.find(key);
    if (i == mAdaptorMap.end())  
      {
	bbtkDebugDecTab("kernel",8);
	return BlackBox::Pointer();
      }
    BlackBox::Pointer bb =i->second.lock()->NewBlackBox(name);
    bbtkDebugDecTab("kernel",8);
    return bb;   

  }
  //==========================================================================

  //==========================================================================
  /// Creates an instance of an adaptor of input type <typein> and 
  /// output type <typeout>  with name <name>
  BlackBox::Pointer Package::NewWidgetAdaptor(const DataInfo& typein,
				      const DataInfo& typeout,
				      const std::string& name) const
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()<<
			">::NewWidgetAdaptor("
			<<typein<<","
			<<typeout<<",\""
			<<name<<"\")"<<bbtkendl);

    AdaptorKey key(typein,typeout,
		   BlackBoxDescriptor::DEFAULT_GUI);
    AdaptorMapType::const_iterator i = mAdaptorMap.find(key);
    if (i == mAdaptorMap.end())  
      {
	bbtkDebugDecTab("kernel",8);
	return BlackBox::Pointer();
      }
    BlackBox::Pointer bb =i->second.lock()->NewBlackBox(name);
    bbtkDebugDecTab("kernel",8);
    return bb;   

  }
  //==========================================================================



  //==========================================================================
  /// Returns true is the package contains 
  /// an adaptor of input type <typein> and 
  /// output type <typeout>
  /// If successfull then adaptor contains the black box type name
  bool Package::FindWidgetAdaptor(const DataInfo& typein,
				  const DataInfo& typeout,
				  std::string& adaptor) const
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()<<
			">::FindWidgetAdaptor("
			<<typein<<","
			<<typeout<<")"<<bbtkendl);
   
    AdaptorKey key(/*typein*/
		   DataInfo(typeid(void),""),
		   typeout,
		   BlackBoxDescriptor::DEFAULT_GUI);
    // First try to find a single widget adaptor
    AdaptorMapType::const_iterator i = mAdaptorMap.find(key);
    if (i == mAdaptorMap.end())  
      {
	bbtkDebugDecTab("kernel",8);
	return false;
      }
    adaptor = i->second.lock()->GetTypeName();
    bbtkDebugDecTab("kernel",8);
    return true;   

  }
  //==========================================================================



  //==========================================================================
  /// Returns true is the package contains 
  /// an adaptor of input type <typein> and 
  /// output type <typeout>
  /// If successfull then adaptor contains the black box type name
  bool Package::FindAdaptor(const DataInfo& typein,
			    const DataInfo& typeout,
			    std::string& adaptor) const
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()<<
			">::FindAdaptor("
			<<typein<<","
			<<typeout<<")"<<bbtkendl);
    
    AdaptorKey key(typein,typeout,
		   BlackBoxDescriptor::DEFAULT_ADAPTOR);
    AdaptorMapType::const_iterator i = mAdaptorMap.find(key);
    if (i == mAdaptorMap.end())  
      {
	bbtkDebugDecTab("kernel",8);
	return false;
      }
    adaptor = i->second.lock()->GetTypeName();
    bbtkDebugDecTab("kernel",8);
    return true;   

  }
  //==========================================================================


  //==========================================================================
  /// Registers a black box descriptor in the package
  bool Package::Register(BlackBoxDescriptor::Pointer d) 
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()<<">::Register(\""<<d->GetTypeName()<<"\")"<<std::endl);
    
    DescriptorMapType::iterator i = mDescriptorMap.find(d->GetTypeName());
    if (i!=mDescriptorMap.end())
      {
	bbtkWarning("Package<"<<GetName()<<"> : Trying to register box type <"
		    <<d->GetTypeName()<<"> which is already in the package");
	return false;
      }

    mDescriptorMap[d->GetTypeName()] = d;
    //    d->Reference();
    d->SetPackage(GetThisPointer<Package>());
    
    // If it is a default adaptor, also register it in the adaptors map
    if ( d->GetKind() == BlackBoxDescriptor::DEFAULT_ADAPTOR )
      {
	bbtkDebugMessage("kernel",8,"Package<"<<GetName()<<">::Register(\""<<d->GetTypeName()<<"\") : The box is an adaptor, inserting it in adaptors map ..."<<std::endl);   
	
	TypeInfo typein = d->GetInputDescriptor("In")->GetTypeInfo();
	TypeInfo typeout = d->GetOutputDescriptor("Out")->GetTypeInfo();
	DataInfo infoin(typein,d->GetInputDescriptor("In")->GetNature());
	DataInfo infoout(typeout,d->GetOutputDescriptor("Out")->GetNature());
	AdaptorKey key(infoin,infoout,d->GetKind());
	
	AdaptorMapType::const_iterator i;
	i = mAdaptorMap.find(key);	  
	if (i == mAdaptorMap.end())  
	  {
	    mAdaptorMap[key] = d;
	  }
	// If already an adaptor registered : error
	else 
	  {
	    if (i->second.lock()->GetTypeName() != d->GetTypeName()) 
	      {
		bbtkError("Package <"<<GetName()<<
			  "> : trying to register black box <"
			  <<d->GetTypeName()
			  <<"> as default adaptor but there is already a default adaptor registered (<"
			  <<i->second.lock()->GetTypeName()<<">)");
	      }
	  }
      }
    // If it is a default adaptor, also register it in the adaptors map
    else if ( d->GetKind() == BlackBoxDescriptor::DEFAULT_GUI)
      {
	bbtkDebugMessage("kernel",8,"Package<"<<GetName()<<">::Register(\""<<d->GetTypeName()<<"\") : The box is a widget adaptor, inserting it in adaptors map ..."<<std::endl);   
	
	TypeInfo typeout = d->GetOutputDescriptor("Out")->GetTypeInfo();
	DataInfo infoin(typeid(void),"");
	DataInfo infoout(typeout,d->GetOutputDescriptor("Out")->GetNature());
	AdaptorKey key(infoin,infoout,d->GetKind());

	AdaptorMapType::const_iterator i;
	i = mAdaptorMap.find(key);	  
	if (i == mAdaptorMap.end())  
	  {
	    mAdaptorMap[key] = d;
	  }
	// If already an adaptor registered : error
	else 
	  {
	    if (i->second.lock()->GetTypeName() != d->GetTypeName()) 
	      {
		bbtkError("Package <"<<GetName()<<
			  "> : trying to register black box <"
			  <<d->GetTypeName()
			  <<"> as default widget adaptor but there is already a default adaptor registered (<"
			  <<i->second.lock()->GetTypeName()<<">)");
	      }
	  }
      }
    

    bbtkDebugDecTab("kernel",8);
   
    return true;
  }
  //==========================================================================
  
  //===================================================================
  void Package::Check() const
  {
    bbtkMessage("debug",1,"****** Checking Package "<<(void*)this
		<<" ["<<GetName()<<"]"<<std::endl);
    DescriptorMapType::const_iterator i;
    for (i=mDescriptorMap.begin();
	 i!=mDescriptorMap.end();
	 ++i) 
      {
	i->second->Check(true);
      }
    bbtkMessage("debug",1,"****** Checking Package "<<(void*)this
		<<" ["<<GetName()<<"] ... OK"<<std::endl);
  }
  //===================================================================


  //==========================================================================
  /// Changes the name of a black box type
  void Package::ChangeDescriptorName( const std::string& oldname, const std::string& newname )
  { 
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()
			<<">::ChangeDescriptorName(\""<<oldname
			<<"\",\""<<newname<<"\")"<<std::endl);
    // Looking into the bb map
    DescriptorMapType::iterator i = mDescriptorMap.find(oldname);
    if (i == mDescriptorMap.end())  
      {
         bbtkDebugDecTab("kernel",8);
         bbtkError("ChangeDescriptorName : The package <"<<GetName()<<"> does not contains the black box <"<<oldname<<">");
      }

    i->second->SetTypeName(newname);
    mDescriptorMap[newname] = i->second;
    mDescriptorMap.erase(i);

    bbtkDebugDecTab("kernel",8);    
  }
  //==========================================================================



  //==========================================================================
  void Package::PrintHelpListDescriptors(bool description, bool adaptors) const
  {
    unsigned int lmax = 0;
    std::vector<std::string> names;
    std::vector<std::string> kinds;
    std::vector<std::string> descrs;

    DescriptorMapType::const_iterator i;
    for (i=mDescriptorMap.begin();
	 i!=mDescriptorMap.end();
	 ++i) 
      {
	if ( adaptors || 
	     ( i->second->GetKind() == BlackBoxDescriptor::STANDARD) ) 
	  {
	    std::string name("  ");
	    name += i->second->GetTypeName();
	    names.push_back(name);

	    std::string kind;
	    if ( i->second->GetKind() == BlackBoxDescriptor::ADAPTOR )
	      {
		kind = std::string("[A]");
	      }
	    else if ( i->second->GetKind() == 
		      BlackBoxDescriptor::DEFAULT_ADAPTOR )
	      {
		kind = std::string("[DA]");
	      }
	    kinds.push_back(kind);

	    unsigned int l = name.size()+kind.size();
	    if (l>lmax) lmax = l;

	    std::string descr;
	    if (description) 
	      {
		descr += " : ";
		descr += i->second->GetDescription();
	      } 
	    descrs.push_back(descr);
	  }
      } 
    

    std::string offs;
    offs.append(lmax+3,' ');
    std::vector<std::string>::iterator ni,ci,di;
    for (ni = names.begin(), ci = kinds.begin(), di = descrs.begin();
	 ni != names.end(); ++ni, ++ci, ++di)
      {
	std::string space;
	space.append(lmax - ni->size() - ci->size(),' ');
	bbtkMessage("help",1,*ni << space << *ci );
	std::string d(*di);
	unsigned int dmax = 75 - lmax;
	//	while (d.size() > dmax ) 
	//  {
	if (d.size()>dmax) 
	  bbtkMessage("help",1,d.substr(0,dmax) << "..." << std::endl);
	else 
	  bbtkMessage("help",1,d << std::endl);
	//    d = d.substr(dmax,d.size());
	//  }
      }

  }
  //==========================================================================

  //==========================================================================
  /// Displays the list of adaptors of the package
  void Package::PrintHelpListAdaptors(bool description) const
  {
    DescriptorMapType::const_iterator i;
    for (i=mDescriptorMap.begin();
	 i!=mDescriptorMap.end();
	 ++i) 
      {
	if ( i->second->GetKind() != BlackBoxDescriptor::STANDARD ) 
	  {
	    bbtkMessage("help",1,
			"  "<<i->second->GetTypeName());
	    if ( i->second->GetKind() == 
		 BlackBoxDescriptor::DEFAULT_ADAPTOR )
	      {
		bbtkMessage("help",1,
			    " [default]");
	      }  
	    if (description) 
	      {
		bbtkMessage("help",1,
			    " : "<<i->second->GetDescription());

	      } 
	    bbtkMessage("help",1,std::endl);
	  }
      } 
    /*
    AdaptorMapType::const_iterator i;
    for (i=mAdaptorMap.begin();
	 i!=mAdaptorMap.end();
	 ++i) 
      {
	bbtkMessage("help",1,
		    "  "<<i->second->GetTypeName());
	if (detail_level>0) 
	  {
	    bbtkMessage("help",1,
			" : "<<i->second->GetDescription());
  
	  } 
	bbtkMessage("help",1,std::endl);
      }
    */ 
  }
  //==========================================================================

  //==========================================================================
  /// Prints help on a black box descriptor
  void Package::PrintHelpDescriptor(const std::string& name, bool full) const
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()
			<<">::PrintHelpDescriptor(\""
			<<name<<"\")"<<bbtkendl);

    DescriptorMapType::const_iterator i = mDescriptorMap.find(name);
    if (i == mDescriptorMap.end())  
      {
	bbtkDebugDecTab("kernel",8);
	bbtkError("The package <"<<GetName()<<"> does not contains the black box <"<<name<<">");
      }
    //    bbtkMessage("help",1,"["<<GetName()<<"] ");
    i->second->GetHelp(full);
    bbtkDebugDecTab("kernel",8);

  }
  //==========================================================================


  //==========================================================================
  /// Returns true iff the package contains the box of name boxname
  bool Package::ContainsDescriptor(const std::string& name) const 
  {
    bbtkDebugMessageInc("kernel",8,"Package<"<<GetName()
			<<">::ContainsDescriptor(\""
			<<name<<"\")"<<bbtkendl);
    
    DescriptorMapType::const_iterator i = mDescriptorMap.find(name);
    if (i == mDescriptorMap.end())  
    {
      bbtkDebugDecTab("kernel",8);
      return false;
    }
    bbtkDebugDecTab("kernel",8);
    return true;
  }
  //==========================================================================


 
  //==========================================================================
  void Package::CreateHtmlPage(const std::string& filename,
			       const std::string& caller,
			       const std::string& source,
			       const std::string& custom_header,
			       const std::string& custom_title,
			       int detail, 
			       int level,
			       bool relative_link ) const
  {
    bbtkDebugMessageInc("kernel",9,"Package<"<<GetName()<<">::CreateHtmlPage(\""
			<<filename<<"\")"<<bbtkendl);

//std::cout<<"JCP bbtkPackage.cxx void Package::CreateHtmlPage() ln 1133"<<std::endl;

    //---------------------
    // Open output file
    std::ofstream s;
    s.open(filename.c_str());
    if (!s.good()) 
    {
       bbtkError("Package "<<GetName()<<" : CreateHtmlPage : could not open file '"<<filename<<"'");
    }
    
    //----------------------
    // Html head
    std::string title = "BBTK Package "+GetName()+" "+GetVersion(); 

    if (custom_title.length() != 0) title = custom_title;

    s << "<html lang=\"en\">\n";
    s << "<head>\n";
    s << "<title>" << title << "</title>\n";
    s << "<meta http-equiv=\"Content-Type\" content=\"text/html\">\n";
    s << "<meta name=\"description\" content=\""<<title<<"\">\n";
    s << "<meta name=\"generator\" content=\"\">\n";
    s << "<link title=\"Top\" rel=\"top\" href=\"#Top\">\n";
    //<link href="http://www.gnu.org/software/texinfo/" rel="generator-home" title="Texinfo Homepage">
    s << "<meta http-equiv=\"Content-Style-Type\" content=\"text/css\"><style type=\"text/css\"><!--\n";
    s << "pre.display { font-family:inherit }\n";
    s << "pre.format  { font-family:inherit }\n";
    s << "pre.smalldisplay { font-family:inherit; font-size:smaller }\n";
    s << "pre.smallformat  { font-family:inherit; font-size:smaller }\n";
    s << "pre.smallexample { font-size:smaller }\n";
    s << "pre.smalllisp    { font-size:smaller }\n";
    s << "span.sc    { font-variant:small-caps }\n";
    s << "span.roman { font-family:serif; font-weight:normal; } \n";
    s << "span.sansserif { font-family:sans-serif; font-weight:normal; }\n"; 
    s << "--></style>\n";
    s << "</head>\n";
    //----------------------

    //----------------------
    // Html body
    s << "<body>\n";
    s << "<a name=\"Top\"></a>\n"; 
    
    //----------------------
    // Header
    if ( custom_header.length() != 0) 
      {
	if ( custom_header != "none" )
	  { 
	    std::ifstream in;
	    in.open(custom_header.c_str());    
	    if (!in.good()) 
	      {
		bbtkError("Could not open file \""<<custom_header<<"\"");
	      }
	    char buffer[512];
	    while (!in.eof()) 
	      {
		in.getline(buffer,512);
		std::string line(buffer);
		s << line;
	      }
	    in.close();
	    s << "<hr>\n";
	   
	    /*   
	    s << "<object data=\"" << custom_header 
	      << "\" type = \"text/html\"\"style=\"width: 1200px; height: 400px;\"> Warning: "
	      << custom_header <<" could not be embedded.</object>\n";
	    
	    s << "<hr>\n";
	    */
	  }
      }

    else 
      {
	s << "<h1 class=\"settitle\">"<<title<<"</h1>\n";
	s << "<p><TABLE cellspacing=0  cellpadding=3>\n";
	s << "<TR><TD style='vertical-align: top;'><b> Description </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'> " 
	  << GetDescription() << "</TD></TR>\n";
	s << "<TR><TD style='vertical-align: top;'><b> Author(s) </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  " 
	  << GetAuthor() << "</TD></TR>\n";
	s << "<TR><TD style='vertical-align: top;'><b> Author(s) </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'>  " 
	  << GetCategory() << "</TD></TR>\n";
  	s << "<TR><TD style='vertical-align: top;'><b> Version </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'> " 
	  << GetVersion() << "</TD></TR>\n";
	s << "<TR><TD style='vertical-align: top;'><b> bbtk Version </b></TD><TD style='vertical-align: top;'> : </TD><TD style='vertical-align: top;'> " 
	  << bbtk::GetVersion() << "</TD></TR>\n";
	s << "</TABLE>\n";
      }
//std::cout<<"JCP bbtkPackage.cxx void Package::CreateHtmlPage() ln 1225"<<std::endl;
    //-------------------
    // Table of contents
    // Black boxes list
    //  s << "<div class=\"contents\">\n";
    s << "<p><b> Black Boxes : </b>\n";
    s << "<ul>\n";

    s << "<p><TABLE cellspacing=0  cellpadding=3>\n";

    DescriptorMapType::const_iterator i;
//std::cout<<"JCP bbtkPackage.cxx void Package::CreateHtmlPage() ln 1236"<<std::endl;
    for (i=mDescriptorMap.begin(); i!=mDescriptorMap.end(); ++i) {
		if ( i->second->GetKind() != BlackBoxDescriptor::STANDARD) 
			continue;
	
		std::string name = i->second->GetTypeName();
		Utilities::html_format(name);
		std::string descr = i->second->GetDescription();
		//Utilities::html_format(descr);
//std::cout<<"JCP bbtkPackage.cxx void Package::CreateHtmlPage() ln 1246"<<std::endl;
		s << "<TR>";
		s << "<TD style='vertical-align: top;'>";
		s << "&nbsp;&nbsp;&nbsp;<a name=\"toc_"<<name
		<<"\" href=\"#"<<name<<"\">"
		<<name<<"</a>";
		s << "</TD> ";
		s << " <TD style='vertical-align: top;'>" << descr << " </TD>";
		s << "</TR>\n";
		}    
		s << "</TABLE>\n";


		s << "</ul>\n";
		s << "</div>\n";
    
		//-------------------
		// Adaptors list
		if (mAdaptorMap.size()>0) 
		{
			//  s << "<div class=\"contents\">\n";
			s << "<p><b> Adaptors : </b>\n";
			s << "<ul>\n";
//std::cout<<"JCP bbtkPackage.cxx void Package::CreateHtmlPage() ln 1268"<<std::endl;
			//    DescriptorMapType::const_iterator i;
			s << "<p><TABLE cellspacing=0  cellpadding=3>\n";
			for (i=mDescriptorMap.begin(); i!=mDescriptorMap.end();++i) 
			{
				if ( i->second->GetKind() == BlackBoxDescriptor::STANDARD) 
					continue;
    
				std::string name = i->second->GetTypeName();
				Utilities::html_format(name);
				std::string descr = i->second->GetDescription();
		    
				s << "<TR>";
				s << "<TD style='vertical-align: top;'>";
				s << "&nbsp;&nbsp;&nbsp;<a name=\"toc_"<<name
				  <<"\" href=\"#"<<name<<"\">"
				  <<name<<"</a>";
				s << "</TD> ";
				s << " <TD style='vertical-align: top;'>" << descr << " </TD>";
				s << "</TR>\n";
			}    
			s << "</TABLE>\n";

			s << "</ul>\n";
			s << "</div>\n";
		}
    
    
    //  s << "<div class=\"node\">\n";

    //    s << "<p><hr>\n";
    //    s << "<a name=\"Top\"></a>\n";
    //  s << "Top:&nbsp;<a rel=\"top\" accesskey=\"t\" href=\"#Top\">Top</a>\n";
    // s << "Previous:&nbsp;<a rel="previous" accesskey="p" href="#dir">(dir)</a>,
    // s << "Up:&nbsp;<a rel="up" accesskey="u" href="#dir">(dir)</a>
    
    //    s << "</div>\n";

    //----------------------
    // Boxes doc

    //-------------------
    // Computes output directory from filename to pass it to 
    // BlackBoxDescriptor::InsertHtmlHelp
		std::string dir;

		std::string::size_type slash_position = filename.find_last_of("/\\");


		if (slash_position != std::string::npos) {
			if (slash_position == 0)
			 slash_position = 1;  
			dir = filename.substr(0,slash_position);
		}

		for (i=mDescriptorMap.begin();
		 i!=mDescriptorMap.end();
		 ++i) 
		{
			i->second->InsertHtmlHelp(s,detail,level,dir,relative_link);
		}    

    //----------------------
    // Footer 
    time_t rawtime;
    tm * ptm;
    time ( &rawtime );
    ptm = gmtime ( &rawtime );

    s << "<p><hr>\n";
    s << "Automatically generated by <b>"<<caller<<"</b> "//from <b>"
      //      <<source<<"</b>
      <<"on "
      << ptm->tm_mday << "/" << ptm->tm_mon << "/" << ptm->tm_year+1900 
      << " - " << ptm->tm_hour << ":" << ptm->tm_min << " GMT\n";
    s << "</body></html>\n"; 
    s.close();
    //----------------------

    // End
    bbtkDebugDecTab("kernel",9);
  }
  //==========================================================================
  
  //==========================================================================
  std::string  Package::GetObjectName() const 
  { 
    return std::string("Package '")+mName+std::string("'"); 
  }
  //==========================================================================

  //==========================================================================
  std::string Package::GetObjectInfo() const 
  {
    std::stringstream i;
    i << "  - "<<mDescriptorMap.size() << " boxes" << std::endl;
    if (mDynamicLibraryHandler) 
      {
	i<< "  - Loaded from dynamic library"<<std::endl;
      }
    return i.str();
  }
  //==========================================================================


  //==========================================================================
  size_t  Package::GetObjectSize() const 
  {
    size_t s = Superclass::GetObjectSize();
    s += Package::GetObjectInternalSize();
    return s;
  }
  //==========================================================================
  //==========================================================================
  size_t  Package::GetObjectInternalSize() const 
  {
    size_t s = sizeof(Package);
    return s;
  }
  //==========================================================================
  //==========================================================================
  size_t  Package::GetObjectRecursiveSize() const 
  {
    size_t s = Superclass::GetObjectRecursiveSize();
    s += Package::GetObjectInternalSize();
    
    DescriptorMapType::const_iterator i;
    for (i = mDescriptorMap.begin(); i!=mDescriptorMap.end(); ++i )
      {
	s += i->second->GetObjectRecursiveSize();
      }
    return s;
  }
  //==========================================================================
  void  Package::GetBoxesInside(NodeTreeC& tree, int cont) 
  {
		DescriptorMapType::const_iterator i;
		std::cout<<"*********a********"<<std::endl;
		for (i=mDescriptorMap.begin(); i!=mDescriptorMap.end(); ++i) 
		{
			i->second->GetBoxesInside(tree, cont);
			std::cout<<"*****************"<<std::endl;
		}    
  }
	//==========================================================================
  //==========================================================================
  std::set<Package::WeakPointer> 
  Package::mReleasedDynamicallyLoadedPackages;
  //==========================================================================

}

