/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbitkResampleImageFilter.h,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:50:39 $
  Version:   $Revision: 1.13 $
=========================================================================*/




#ifdef _USE_ITK_

#include <math.h>
#include "bbtkAtomicBlackBox.h"
#include "itkResampleImageFilter.h"
#include "bbitkImage.h"
#include "itkNearestNeighborInterpolateImageFunction.h"
#include "itkLinearInterpolateImageFunction.h"
#include "itkBSplineInterpolateImageFunction.h"

// For ITK4
#include <itkAffineTransform.h>

namespace bbitk
{

 
 //===================================================
   class /*BBTK_EXPORT*/ ResampleImageFilter
    : 
    public bbtk::AtomicBlackBox
  {
    BBTK_BLACK_BOX_INTERFACE(ResampleImageFilter, bbtk::AtomicBlackBox);
	BBTK_DECLARE_INPUT(In,anyImagePointer);
	BBTK_DECLARE_INPUT(Spacing,std::vector<double>);
	BBTK_DECLARE_INPUT(Interpolation,std::string);
	BBTK_DECLARE_OUTPUT(Out,anyImagePointer);
    BBTK_PROCESS(ProcessSwitch);
  private :
    inline void ProcessSwitch();
    template <class T> void Process();
    itk::Object* mOutput;
  };
 //===================================================
   
 //===================================================
   BBTK_BEGIN_DESCRIBE_BLACK_BOX(ResampleImageFilter,
				bbtk::AtomicBlackBox);
  BBTK_NAME("ResampleImageFilter");
  BBTK_AUTHOR("laurent.guigues at creatis.insa-lyon.fr");
  BBTK_DESCRIPTION("Resamples an image");
  BBTK_CATEGORY("image;filter");
  BBTK_INPUT(ResampleImageFilter,In,"Input image. Can be any itk::Image<T,D>*",anyImagePointer,"");
  BBTK_INPUT(ResampleImageFilter,Spacing,"Spacing",std::vector<double>,"spacing");
  BBTK_INPUT(ResampleImageFilter,Interpolation,"Interpolation: Linear (default) || BSpline || NearestNeighbor)  ",std::string,"");
  BBTK_OUTPUT(ResampleImageFilter,Out,"Output image. Of the same type and dimension than the input image",anyImagePointer,"");
  BBTK_END_DESCRIBE_BLACK_BOX(ResampleImageFilter);
 //===================================================
 


 //===================================================
   void ResampleImageFilter::ProcessSwitch()
  {
    bbtk::TypeInfo t = bbGetInputIn().type();
    BBTK_TEMPLATE_ITK_IMAGE_SWITCH(t, this->Process);
  }
 //===================================================
 
 //===================================================
   template <class T> 
  void ResampleImageFilter::Process()
  {
    bbtkDebugMessageInc("Core",9,
			"bbitk::ResampleImageFilter::Process<"
			<<bbtk::TypeName<T>()<<">()"<<std::endl);
 
    typedef T ImageType;
    typedef itk::ResampleImageFilter<ImageType,ImageType> FilterType;
    typename FilterType::Pointer filter = FilterType::New();
    const unsigned int Dimension = ImageType::ImageDimension;

    // Input
    T* in = this->bbGetInputIn().get<T*>();
    filter->SetInput( in );

    // Size, Spacing, Origin and DefaultPixelVal
    typename ImageType::SizeType size;
    typename ImageType::SpacingType spacing;
    typename ImageType::PointType origin;
    typename ImageType::RegionType LPR;
    LPR = in->GetLargestPossibleRegion();
    size = LPR.GetSize();
    //    origin = LPR.GetIndex(); //in->GetOrigin();
     for (unsigned int i=0;i<Dimension;++i) 
      {
	origin[i] = LPR.GetIndex()[i]*in->GetSpacing()[i];
	spacing[i] = bbGetInputSpacing()[i];
	double tmp = (LPR.GetSize()[i]*in->GetSpacing()[i]/spacing[i] ) + 0.5;
	size[i] = (long)floor(tmp);
//	size[i] = (long)lrint(LPR.GetSize()[i]*in->GetSpacing()[i]/spacing[i]);
       }
   
    filter->SetOutputOrigin (origin);
    filter->SetSize (size);
    filter->SetOutputSpacing( spacing );

    filter->SetDefaultPixelValue (0);
    filter->SetOutputDirection( in->GetDirection() );



    // Transform
    typedef itk::AffineTransform < double, Dimension> TransformType;
    
    // Instance of the transform object to be passed to the resample filter
    // By default, identity transform is applied
    typename TransformType::Pointer transform =  TransformType::New();
    filter->SetTransform ( transform );

    
 
    if  ( bbGetInputInterpolation() == "NearestNeighbor" ) {
      typedef itk::NearestNeighborInterpolateImageFunction < ImageType, double > InterpolatorType;     
      // We create an interpolator of the found type 
      typename InterpolatorType::Pointer interpolator = InterpolatorType::New();
      filter->SetInterpolator( interpolator );
    }
    else if  ( bbGetInputInterpolation() == "BSpline") { 
      typedef itk::BSplineInterpolateImageFunction < ImageType, double > InterpolatorType; 
      // We create an interpolator of the found type 
      typename InterpolatorType::Pointer interpolator = InterpolatorType::New();
      filter->SetInterpolator(interpolator);    
      // When handling unsigned data, it is possible that the interpolated value is negative
      // if ( (m_InputImage->GetComponentTypeAsString() == "uchar")   || 
      //	   (m_InputImage->GetComponentTypeAsString() == "ushort")  ||
      //	   (m_InputImage->GetComponentTypeAsString() == "uint") ) {   
      //	std::cout << "Warning! you're using unsigned data ! The interpolated value may result negative! "<< std::endl;
      // }
    } //end else if
    // Interpolation 
    else { // if ( bbGetInputInterpolation() == "Linear" ) {
      typedef itk::LinearInterpolateImageFunction < ImageType, double > InterpolatorType;     
      // We create an interpolator of the found type 
      typename InterpolatorType::Pointer interpolator =  InterpolatorType::New();
      filter->SetInterpolator( interpolator );
    }

    filter->Update();
    filter->GetOutput()->Register();
    if (mOutput) mOutput->UnRegister();
    this->bbSetOutputOut( filter->GetOutput() );
    mOutput = filter->GetOutput();

    bbtkDebugDecTab("Core",9);
  }
   //===================================================
 

	//-----------------------------------------------------------------	
	void ResampleImageFilter::bbUserSetDefaultValues()
	{
		std::vector<double> res;
		res.push_back(1);
		res.push_back(1);
		res.push_back(1);
		bbSetInputSpacing(res);
		mOutput = 0;
	}
	
	//-----------------------------------------------------------------	
	void ResampleImageFilter::bbUserInitializeProcessing()
	{
	}
	
	//-----------------------------------------------------------------	
	void ResampleImageFilter::bbUserFinalizeProcessing()
	{
	}	
	

} // EO namespace bbitk

#endif   // _USE_ITK_
