/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================
  Program:   bbtk
  Module:    $RCSfile: bbwxSlider.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/16 08:52:14 $
  Version:   $Revision: 1.27 $
=========================================================================*/

/**
 * \file 
 * \brief Short description in one line
 *
 * Long 
 * description
 *  
 */

#ifdef _USE_WXWIDGETS_

#include "bbwxSlider.h"
#include "bbwxPackage.h"
#include "bbtkUtilities.h"

namespace bbwx
{
  //--------------------------------------------------------------------------
  // The widget created by the box 
  class SliderWidget : public wxPanel 
  {
  public:
    /// Ctor with the two first params the parent window and the creator box
    /// which must be passed to the WxBlackBoxWidget constructor.
    /// The other params initialize the widget 
    SliderWidget(Slider* box, wxWindow *parent,
		 int orientation,
		 bool changeresolution,
		 bool label, 
		 wxString title,
		 int vmin,
		 int vmax,
		 int value,
                 bool track);
    /// Dtor
    ~SliderWidget();
    /// Events callbacks
    /// Called when the slider is moved
    void OnSliderTrack(wxScrollEvent& event);
    /// Called when the slider is released
    void OnSliderRelease(wxScrollEvent& event);
    /// Called when the little slider which controls the resolution
    /// of the main slider is moved (if activated)
    void OnResolutionOfSlider(wxScrollEvent& event);

    // Accessors
    void SetValue(int v) { mwxSlider->SetValue(v); }
    int  GetValue() { return mwxSlider->GetValue(); }
    int GetMin() {return min;}
    int GetMax() {return max;}    
    void SetRange(int min, int max);
    // Update the texts which display the min/max/current values of the slider
    void RefreshLabels();

	void SetReactiveOnTrack(bool ok);

	  
  private:
    Slider*      mBox;
    wxSlider     *mwxSlider;
    wxSlider     *mResolutionOfSlider;
    int          min;
    int          max;
    bool         reactiveOnTrack;
    wxStaticText *label_vertical;
    wxStaticText *label_min;
    wxStaticText *label_max;
  };
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------
  //------------------------------------------------------------------------

  
    
  //-------------------------------------------------------------------------
  SliderWidget::SliderWidget(Slider* box, wxWindow *parent,
			     int orientation,
			     bool changeresolution,
			     bool labels,
			     wxString title,
			     int vmin,
			     int vmax,
			     int value,
                             bool reactiveontrack)
    :  
    wxPanel( parent, -1, wxDefaultPosition, wxDefaultSize, wxTAB_TRAVERSAL),
    mBox(box),
    min(vmin),
    max(vmax),
    reactiveOnTrack(reactiveontrack)
  {
    wxPanel * panel = this;
    
    label_min           = NULL;
    label_max           = NULL;
    label_vertical      = NULL;
    mResolutionOfSlider	= NULL;

    int sizeX,sizeY;
    long wxlabels=0;
    long wxorientation=0;
    if (orientation==0) 
      {
	sizeX			= 2; 
	sizeY			= 40;
	wxorientation		= wxSL_HORIZONTAL;
	if (labels==true)
	  {
	    wxlabels = wxSL_LABELS;
	  }
      } 
    else 
      {
	sizeX		= 20;
	sizeY		= 2;
	wxorientation	= wxSL_VERTICAL;
      }
    
    //---------------------------------------------------------------------
    // 1) Creation of the components of the widget
    // Any top level sub-widget must have the panel returned by panel
    // for parent
    mwxSlider = new wxSlider( panel, 
			      -1, 0 , 0, 500 , 
			      wxDefaultPosition, 
			      wxSize(sizeX,sizeY), 
			      wxorientation  | wxlabels );

    //    mwxSlider->SetInitialSize(wxSize(sizeX,sizeY));

    //	mwxSlider->SetTickFreq(100,0);
    mwxSlider->SetRange(min,max);
    mwxSlider->SetValue(value);

    //    RefreshLabels();

    // Connecting events to callbacks
    Connect( mwxSlider->GetId(), 
	     wxEVT_SCROLL_THUMBRELEASE, 
	     (wxObjectEventFunction) 
	     (void (wxPanel::*)(wxScrollEvent&))
	     &SliderWidget::OnSliderRelease ); 

    Connect( mwxSlider->GetId(),  
	     wxEVT_SCROLL_THUMBTRACK, 
	     (wxObjectEventFunction) 
	     (void (wxPanel::*)(wxScrollEvent&))
	      &SliderWidget::OnSliderTrack ); 
    
    Connect( mwxSlider->GetId(),  
	     wxEVT_SCROLL_CHANGED, 
	     (wxObjectEventFunction) 
	     (void (wxPanel::*)(wxScrollEvent&))
	      &SliderWidget::OnSliderTrack );
	       
    // If asked : creation of the other little slider which allows to change 
    // the resolution of the main slider
    if (changeresolution==true){
      // has panel for parent too
      mResolutionOfSlider = new wxSlider(panel,
					 -1,5,1,10,
					 wxDefaultPosition,
					 wxSize(25,45),
					 wxSL_VERTICAL | 
					 wxSL_AUTOTICKS |
					 wxSL_LEFT  );
   

      mResolutionOfSlider->SetRange(1,8);
      mResolutionOfSlider->SetValue(5);
      // Is wxEVT_COMMAND_SLIDER_UPDATED event 
      // is connected to the method OnResolutionOfSlider
      Connect(mResolutionOfSlider->GetId(),
	      wxEVT_COMMAND_SLIDER_UPDATED,
	      (wxObjectEventFunction) 
	      (void (wxPanel::*)(wxScrollEvent&))
	      &SliderWidget::OnResolutionOfSlider ); 
    }
    //---------------------------------------------------------------------

    //---------------------------------------------------------------------
    // 2) Insertion of the components in the window
    
    // We use a FlexGridSizer
    wxFlexGridSizer *sizer;


    if (orientation==0) 
      {
	// HORIZONTAL
	if (mResolutionOfSlider!=NULL) 
	  {
	    sizer	= new wxFlexGridSizer(2);
	    if (title!=_T(""))
	      {
		sizer->Add( new wxStaticText(panel,-1, title ) ); 
		sizer->Add( new wxStaticText(panel,-1, _T("") ) ); 
	      }
	  } 
	else 
	  {
	    sizer	= new wxFlexGridSizer(1);
	    if (title!=_T(""))
	      {
		sizer->Add( new wxStaticText(panel,-1, title ) ); 
	      }
	  }
	sizer->Add( mwxSlider,1,wxGROW ); 
	sizer->AddGrowableCol(0);
	if (mResolutionOfSlider!=NULL) 
	  {
	    sizer->Add( mResolutionOfSlider ); 
	  }
      } 
    else 
      {
	// VERTICAL 
	sizer	= new wxFlexGridSizer(1);
	if (labels==true)  // with lable
	  {
	    label_vertical = new wxStaticText(panel,-1,_T(""));
	    label_min = new wxStaticText(panel,-1,_T(""));
	    label_max = new wxStaticText(panel,-1,_T(""));
	    if (title!=_T(""))
	      {
		sizer->Add( new wxStaticText(panel,-1, title ) );
		sizer->AddGrowableRow(3);
	      } 
	    else 
	      {
		sizer->AddGrowableRow(2);
	      }
	    sizer->Add( label_vertical );
	    sizer->Add( label_min );
	    sizer->Add( mwxSlider,1,wxGROW );
	    sizer->Add( label_max );
	    if (mResolutionOfSlider!=NULL) 
	      {
		sizer->Add( mResolutionOfSlider ); 
	      }
	  } 
	else 
	  {
	    if (title!=_T(""))
	      {
		sizer->Add( new wxStaticText(panel,-1, title ) );
		sizer->AddGrowableRow(1);
	      } 
	    else 
	      {
		sizer->AddGrowableRow(0);
	      }
	    sizer->Add( mwxSlider,1,wxGROW );
	    if (mResolutionOfSlider!=NULL) 
	      {
		sizer->Add( mResolutionOfSlider ); 
	      }
	  }
      }
    // Initialize the labels 
    RefreshLabels();  
    // Insert the sizer in the main panel and refresh the layout
    panel->SetSizer(sizer);
  }
  //-------------------------------------------------------------------------
  
	
	
  //-------------------------------------------------------------------------
  SliderWidget::~SliderWidget()
  {
  }
  //-------------------------------------------------------------------------


  //-------------------------------------------------------------------------
  void SliderWidget::OnResolutionOfSlider(wxScrollEvent& event)
  {
    int value  = mwxSlider->GetValue();
    int delta  = (int) (pow((double) 4 ,(double) mResolutionOfSlider->GetValue() ));
    int minTmp = value - delta/2;
    int maxTmp = value + delta/2;
    if (minTmp<min)
      {
	minTmp = min;
	//		maxTmp = delta;
      }
    if (maxTmp>max)
      {
	maxTmp = max;
      }
    mwxSlider->SetRange(minTmp,maxTmp);
    RefreshLabels();
  }
  //-------------------------------------------------------------------------



  //-------------------------------------------------------------------------
  void SliderWidget::OnSliderTrack(wxScrollEvent& event)
  {
    if(reactiveOnTrack)
    {
    // When user releases the slider 
    // we update the output of the box
    
    mBox->bbSetOutputOut( mwxSlider->GetValue() );
    mBox->bbSetInputIn( mwxSlider->GetValue() );
    // and signal that the output has changed
    mBox->bbSignalOutputModification(std::string("Out"));    
    }
    RefreshLabels();
  }
  //-------------------------------------------------------------------------


  //-------------------------------------------------------------------------
  void SliderWidget::OnSliderRelease(wxScrollEvent& event)
  {
    // When user releases the slider 
    // we update the output of the box
    mBox->bbSetOutputOut( mwxSlider->GetValue() );
    mBox->bbSetInputIn( mwxSlider->GetValue() );
    // and signal that the output has changed
    mBox->bbSignalOutputModification(std::string("Out"));
  }
  //-------------------------------------------------------------------------
  

  //-------------------------------------------------------------------------
  void SliderWidget::RefreshLabels()
  {
    wxString strValue;
    if (label_vertical!=NULL)
      {
	strValue.Printf( _T("%d"), mwxSlider->GetValue() );
	label_vertical->SetLabel(strValue);
      }
    if (label_min!=NULL)
      {
	strValue.Printf( _T("%d"), mwxSlider->GetMin() );
	label_min->SetLabel(strValue);
      }
    if (label_max!=NULL)
      {
	strValue.Printf( _T("%d"), mwxSlider->GetMax() );
	label_max->SetLabel(strValue);
      }  
  }
  //-------------------------------------------------------------------------
  

  //-------------------------------------------------------------------------
  void SliderWidget::SetRange(int min, int max)
  {
    this->min = min;
    this->max = max;
    mwxSlider->SetRange(min,max);
    RefreshLabels();
  }
	
	
	//-------------------------------------------------------------------------
	void SliderWidget::SetReactiveOnTrack(bool ok)
	{
		reactiveOnTrack = ok;
	}
	
  //-------------------------------------------------------------------------

 


  //--------------------------------------------------------------------------
  //-------------------------------------------------------------------------
  // WxBlackBox implementation
  //--------------------------------------------------------------------------
  //--------------------------------------------------------------------------

  //--------------------------------------------------------------------------
  BBTK_BLACK_BOX_IMPLEMENTATION(Slider,bbtk::WxBlackBox);
  BBTK_ADD_BLACK_BOX_TO_PACKAGE(wx,Slider);
  
	//-----------------------------------------------------------------	
	void Slider::bbUserSetDefaultValues()
	{
		bbSetInputIn(0);
		bbSetInputMin(0);
		bbSetInputMax(500);
		bbSetOutputOut(0);
		bbSetInputOrientation("HORIZONTAL");
		bbSetInputChangeResolution(false);
		bbSetInputLabel(true);
		bbSetInputReactiveOnTrack(false);    
	}
	
	//-----------------------------------------------------------------	
	void Slider::bbUserInitializeProcessing()
	{
	}
	
	//-----------------------------------------------------------------	
	void Slider::bbUserFinalizeProcessing()
	{
	}	
  
  //--------------------------------------------------------------------------
  void Slider::Process() 
  {
    bbtkDebugMessage("process",3,
		     "Slider "<<bbGetName()<<" input="
		     <<bbGetInputIn()<<std::endl);

// desperate try // JPR
      if ( bbGetInputMin() != ((SliderWidget*)bbGetOutputWidget())->GetMin() ||  bbGetInputMax() != ((SliderWidget*)bbGetOutputWidget())->GetMax() )       
      {    
		  ((SliderWidget*)bbGetOutputWidget())->SetRange(bbGetInputMin(),bbGetInputMax()) ;
      }

	  ((SliderWidget*)bbGetOutputWidget())->SetReactiveOnTrack( bbGetInputReactiveOnTrack() ) ;
	  
    bbSetOutputOut( bbGetInputIn() );
    if (bbGetOutputWidget()!=0)
      {
	((SliderWidget*)bbGetOutputWidget())->SetValue(bbGetInputIn());
      }
  }

//--------------------------------------------------------------------------
  void Slider::CreateWidget(wxWindow* parent)
  {

    int orientation=0;
    if (bbtk::Utilities::loosematch(bbGetInputOrientation(),"0|H|HORIZONTAL")==true)  { orientation=0; }
    if (bbtk::Utilities::loosematch(bbGetInputOrientation(),"1|V|VERTICAL")==true)    { orientation=1; }
    
    
    //    std::cout << "bbGetWxParent = "<<bbGetWxParent()<<std::endl;
    SliderWidget *w =  new SliderWidget(this, 
					parent, //bbGetWxParent(),
					orientation , 
					bbGetInputChangeResolution(), 
					bbGetInputLabel(), 
					bbtk::std2wx( bbGetInputTitle() ),
					bbGetInputMin(), 
					bbGetInputMax(),
					bbGetInputIn(),
					bbGetInputReactiveOnTrack()
					); 
    //    std::cout << "w = "<<w<<std::endl;
    //	w->SetName( bbtk::std2wx( bbGetInputWinTitle() ) );
	  
    bbSetOutputWidget( w );
  }
  

} //namespace bbwx

#endif // _USE_WXWIDGETS_ 


