/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: wxvtkImageViewer2.cxx,v $

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "wxvtkImageViewer2.h"

#include "vtkCamera.h"
#include "vtkCommand.h"
#include "vtkImageActor.h"
#include "vtkImageData.h"
#include "vtkImageData.h"
#include "vtkImageMapToWindowLevelColors.h"
#include "vtkInteractorStyleImage.h"
#include "vtkObjectFactory.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRenderer.h"

vtkCxxRevisionMacro(wxvtkImageViewer2, "$Revision: 1.8 $");
vtkStandardNewMacro(wxvtkImageViewer2);

//----------------------------------------------------------------------------
wxvtkImageViewer2::wxvtkImageViewer2()
{
  this->RenderWindow    = NULL;
  this->Renderer        = NULL;
  this->ImageActor      = vtkImageActor::New();
  this->WindowLevel     = vtkImageMapToWindowLevelColors::New();
  this->Interactor      = NULL;
  this->InteractorStyle = NULL;

  this->Slice = 0;
  this->FirstRender = 1;
  this->SliceOrientation = wxvtkImageViewer2::SLICE_ORIENTATION_XY;

  // Setup the pipeline

  vtkRenderWindow *renwin = vtkRenderWindow::New();
  this->SetRenderWindow(renwin);
  renwin->Delete();

  vtkRenderer *ren = vtkRenderer::New();
  this->SetRenderer(ren);
  ren->Delete();

  this->InstallPipeline();
}

//----------------------------------------------------------------------------
wxvtkImageViewer2::~wxvtkImageViewer2()
{
  if (this->WindowLevel)
    {
    this->WindowLevel->Delete();
    this->WindowLevel = NULL;
    }

  if (this->ImageActor)
    {
    this->ImageActor->Delete();
    this->ImageActor = NULL;
    }

  if (this->Renderer)
    {
    this->Renderer->Delete();
    this->Renderer = NULL;
    }

  if (this->RenderWindow)
    {
    this->RenderWindow->Delete();
    this->RenderWindow = NULL;
    }

  if (this->Interactor)
    {
    this->Interactor->Delete();
    this->Interactor = NULL;
    }

  if (this->InteractorStyle)
    {
    this->InteractorStyle->Delete();
    this->InteractorStyle = NULL;
    }
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetupInteractor(vtkRenderWindowInteractor *arg)
{
  if (this->Interactor == arg)
    {
    return;
    }

  this->UnInstallPipeline();

  if (this->Interactor)
    {
    this->Interactor->UnRegister(this);
    }

  this->Interactor = arg;

  if (this->Interactor)
    {
    this->Interactor->Register(this);
    }

  this->InstallPipeline();

  if (this->Renderer)
    {
    this->Renderer->GetActiveCamera()->ParallelProjectionOn();
    }
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetRenderWindow(vtkRenderWindow *arg)
{
  if (this->RenderWindow == arg)
    {
    return;
    }

  this->UnInstallPipeline();

  if (this->RenderWindow)
    {
    this->RenderWindow->UnRegister(this);
    }

  this->RenderWindow = arg;

  if (this->RenderWindow)
    {
    this->RenderWindow->Register(this);
    }

  this->InstallPipeline();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetRenderer(vtkRenderer *arg)
{
  if (this->Renderer == arg)
    {
    return;
    }

  this->UnInstallPipeline();

  if (this->Renderer)
    {
    this->Renderer->UnRegister(this);
    }

  this->Renderer = arg;

  if (this->Renderer)
    {
    this->Renderer->Register(this);
    }

  this->InstallPipeline();
  this->UpdateOrientation();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetSize(int a,int b)
{
  this->RenderWindow->SetSize(a, b);
}

//----------------------------------------------------------------------------
int* wxvtkImageViewer2::GetSize()
{
  return this->RenderWindow->GetSize();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::GetSliceRange(int &min, int &max)
{
  vtkImageData *input = this->GetInput();
  if (input)
    {
    input->UpdateInformation();
    int *w_ext = input->GetWholeExtent();
    min = w_ext[this->SliceOrientation * 2];
    max = w_ext[this->SliceOrientation * 2 + 1];
    }
}

//----------------------------------------------------------------------------
int* wxvtkImageViewer2::GetSliceRange()
{
  vtkImageData *input = this->GetInput();
  if (input)
    {
    input->UpdateInformation();
    return input->GetWholeExtent() + this->SliceOrientation * 2;
    }
  return NULL;
}

//----------------------------------------------------------------------------
int wxvtkImageViewer2::GetSliceMin()
{
  int *range = this->GetSliceRange();
  if (range)
    {
    return range[0];
    }
  return 0;
}

//----------------------------------------------------------------------------
int wxvtkImageViewer2::GetSliceMax()
{
  int *range = this->GetSliceRange();
  if (range)
    {
    return range[1];
    }
  return 0;
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetSlice(int slice)
{
  int *range = this->GetSliceRange();
  if (range)
    {
    if (slice < range[0])
      {
      slice = range[0];
      }
    else if (slice > range[1])
      {
      slice = range[1];
      }
    }

  if (this->Slice == slice)
    {
    return;
    }

  this->Slice = slice;
  this->Modified();

  this->UpdateDisplayExtent();
  this->Render();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetSliceOrientation(int orientation)
{
  if (orientation < wxvtkImageViewer2::SLICE_ORIENTATION_YZ ||
      orientation > wxvtkImageViewer2::SLICE_ORIENTATION_XY)
    {
    vtkErrorMacro("Error - invalid slice orientation " << orientation);
    return;
    }

  if (this->SliceOrientation == orientation)
    {
    return;
    }

  this->SliceOrientation = orientation;

  // Update the viewer

  int *range = this->GetSliceRange();
  if (range)
    {
    this->Slice = static_cast<int>((range[0] + range[1]) * 0.5);
    }

  this->UpdateOrientation();
  this->UpdateDisplayExtent();

  if (this->Renderer && this->GetInput())
    {
    double scale = this->Renderer->GetActiveCamera()->GetParallelScale();
    this->Renderer->ResetCamera();
    this->Renderer->GetActiveCamera()->SetParallelScale(scale);
    }

  this->Render();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::UpdateOrientation()
{
  // Set the camera position

  vtkCamera *cam = this->Renderer ? this->Renderer->GetActiveCamera() : NULL;
  if (cam)
    {

// EED 21 mars FLIP problem  ..PLOP..		
    switch (this->SliceOrientation)
	{
		case wxvtkImageViewer2::SLICE_ORIENTATION_YZ:
			  cam->SetViewUp(0,0,1);
			  cam->SetPosition(1,0,0); // -1 if medical ?
			  cam->SetFocalPoint(0,0,0);
			break;
			
		case wxvtkImageViewer2::SLICE_ORIENTATION_XZ:
			cam->SetViewUp(0,0,1);
			cam->SetPosition(0,-1,0); // 1 if medical ?
			cam->SetFocalPoint(0,0,0);
			break;
			
		case wxvtkImageViewer2::SLICE_ORIENTATION_XY:
			  cam->SetViewUp(0,-1,0);
			  cam->SetPosition(0,0,-1); 
			  cam->SetFocalPoint(0,0,0);
			break;
      }
	
/*		
		switch (this->SliceOrientation)
		{
			case wxvtkImageViewer2::SLICE_ORIENTATION_YZ:
				cam->SetViewUp(0,0,1);
				cam->SetPosition(1,0,0); // -1 if medical ?
				cam->SetFocalPoint(0,0,0);
				break;
 
			case wxvtkImageViewer2::SLICE_ORIENTATION_XZ:
				cam->SetViewUp(0,0,1);
				cam->SetPosition(0,-1,0); // 1 if medical ?
				cam->SetFocalPoint(0,0,0);
				break;
 
			case wxvtkImageViewer2::SLICE_ORIENTATION_XY:
				cam->SetViewUp(0,1,0);
				cam->SetPosition(0,0,1); // -1 if medical ?
				cam->SetFocalPoint(0,0,0);
				break;
 
				
		}		
*/		
		
    }
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::UpdateDisplayExtent()
{
  vtkImageData *input = this->GetInput();
  if (!input || !this->ImageActor)
    {
    return;
    }

  //  std::cout << "--- wxvtkImageViewer2::UpdateDisplayExtent()"<<std::endl;
  input->UpdateInformation();
  int *w_ext = input->GetWholeExtent();

  // 	std::cout << "ext = "
  //	<<w_ext[0]<<" - "<<w_ext[1]<<" ; "
  //		<<w_ext[2]<<" - "<<w_ext[3]<<" ; "
  //		<<w_ext[4]<<" - "<<w_ext[5]
  //		<<std::endl;
      // Is the slice in range ? If not, fix it

  int slice_min = w_ext[this->SliceOrientation * 2];
  int slice_max = w_ext[this->SliceOrientation * 2 + 1];
  if (this->Slice < slice_min || this->Slice > slice_max)
    {
    this->Slice = static_cast<int>((slice_min + slice_max) * 0.5);
    }

  // Set the image actor

  switch (this->SliceOrientation)
    {
    case wxvtkImageViewer2::SLICE_ORIENTATION_XY:
      this->ImageActor->SetDisplayExtent(
        w_ext[0], w_ext[1], w_ext[2], w_ext[3], this->Slice, this->Slice);
      break;

    case wxvtkImageViewer2::SLICE_ORIENTATION_XZ:
      this->ImageActor->SetDisplayExtent(
        w_ext[0], w_ext[1], this->Slice, this->Slice, w_ext[4], w_ext[5]);
      break;

    case wxvtkImageViewer2::SLICE_ORIENTATION_YZ:
      this->ImageActor->SetDisplayExtent(
        this->Slice, this->Slice, w_ext[2], w_ext[3], w_ext[4], w_ext[5]);
      break;
    }

  // Figure out the correct clipping range

  if (this->Renderer)
    {
    if (this->InteractorStyle &&
        this->InteractorStyle->GetAutoAdjustCameraClippingRange())
      {
      this->Renderer->ResetCameraClippingRange();
      }
    else
      {
      vtkCamera *cam = this->Renderer->GetActiveCamera();
      if (cam)
        {
        double bounds[6];
        this->ImageActor->GetBounds(bounds);
        double spos = bounds[this->SliceOrientation * 2];
        double cpos = cam->GetPosition()[this->SliceOrientation];
        double range = fabs(spos - cpos);
        double *spacing = input->GetSpacing();
        double avg_spacing =
          //(spacing[0] + spacing[1] + spacing[2]) / 3.0;
	  spacing[2]; // JPR??
        cam->SetClippingRange(
          range - avg_spacing * 3.0, range + avg_spacing * 3.0);
        }
      }
    }
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetPosition(int a,int b)
{
  this->RenderWindow->SetPosition(a, b);
}

//----------------------------------------------------------------------------
int* wxvtkImageViewer2::GetPosition()
{
  return this->RenderWindow->GetPosition();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetDisplayId(void *a)
{
  this->RenderWindow->SetDisplayId(a);
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetWindowId(void *a)
{
  this->RenderWindow->SetWindowId(a);
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetParentId(void *a)
{
  this->RenderWindow->SetParentId(a);
}

//----------------------------------------------------------------------------
double wxvtkImageViewer2::GetColorWindow()
{
  return this->WindowLevel->GetWindow();
}

//----------------------------------------------------------------------------
double wxvtkImageViewer2::GetColorLevel()
{
  return this->WindowLevel->GetLevel();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetColorWindow(double s)
{
  this->WindowLevel->SetWindow(s);
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetColorLevel(double s)
{
  this->WindowLevel->SetLevel(s);
}

//----------------------------------------------------------------------------
class wxvtkImageViewer2Callback : public vtkCommand
{
public:
  static wxvtkImageViewer2Callback *New() { return new wxvtkImageViewer2Callback; }

  void Execute(vtkObject *caller,
               unsigned long event,
               void *vtkNotUsed(callData))
    {
      if (this->IV->GetInput() == NULL)
        {
        return;
        }

      // Reset

      if (event == vtkCommand::ResetWindowLevelEvent)
        {
        this->IV->GetInput()->UpdateInformation();
        this->IV->GetInput()->SetUpdateExtent
          (this->IV->GetInput()->GetWholeExtent());
        this->IV->GetInput()->Update();
        double *range = this->IV->GetInput()->GetScalarRange();
        this->IV->SetColorWindow(range[1] - range[0]);
        this->IV->SetColorLevel(0.5 * (range[1] + range[0]));
        this->IV->Render();
        return;
        }

      // Start

      if (event == vtkCommand::StartWindowLevelEvent)
        {
        this->InitialWindow = this->IV->GetColorWindow();
        this->InitialLevel = this->IV->GetColorLevel();
        return;
        }

      // Adjust the window level here

      vtkInteractorStyleImage *isi =
        static_cast<vtkInteractorStyleImage *>(caller);

      int *size = this->IV->GetRenderWindow()->GetSize();
      double window = this->InitialWindow;
      double level = this->InitialLevel;

      // Compute normalized delta

      double dx = 4.0 *
        (isi->GetWindowLevelCurrentPosition()[0] -
         isi->GetWindowLevelStartPosition()[0]) / size[0];
      double dy = 4.0 *
        (isi->GetWindowLevelStartPosition()[1] -
         isi->GetWindowLevelCurrentPosition()[1]) / size[1];

      // Scale by current values

      if (fabs(window) > 0.01)
        {
        dx = dx * window;
        }
      else
        {
        dx = dx * (window < 0 ? -0.01 : 0.01);
        }
      if (fabs(level) > 0.01)
        {
        dy = dy * level;
        }
      else
        {
        dy = dy * (level < 0 ? -0.01 : 0.01);
        }

      // Abs so that direction does not flip

      if (window < 0.0)
        {
        dx = -1*dx;
        }
      if (level < 0.0)
        {
        dy = -1*dy;
        }

      // Compute new window level

      double newWindow = dx + window;
      double newLevel;
      newLevel = level - dy;

      // Stay away from zero and really

      if (fabs(newWindow) < 0.01)
        {
        newWindow = 0.01*(newWindow < 0 ? -1 : 1);
        }
      if (fabs(newLevel) < 0.01)
        {
        newLevel = 0.01*(newLevel < 0 ? -1 : 1);
        }

      this->IV->SetColorWindow(newWindow);
      this->IV->SetColorLevel(newLevel);
      this->IV->Render();
    }

  wxvtkImageViewer2 *IV;
  double InitialWindow;
  double InitialLevel;
};

//----------------------------------------------------------------------------
void wxvtkImageViewer2::InstallPipeline()
{
  if (this->RenderWindow && this->Renderer)
    {
    this->RenderWindow->AddRenderer(this->Renderer);
    }

  if (this->Interactor)
    {
    if (!this->InteractorStyle)
      {
      this->InteractorStyle = vtkInteractorStyleImage::New();
      wxvtkImageViewer2Callback *cbk = wxvtkImageViewer2Callback::New();
      cbk->IV = this;
      this->InteractorStyle->AddObserver(
        vtkCommand::WindowLevelEvent, cbk);
      this->InteractorStyle->AddObserver(
        vtkCommand::StartWindowLevelEvent, cbk);
      this->InteractorStyle->AddObserver(
        vtkCommand::ResetWindowLevelEvent, cbk);
      cbk->Delete();
      }

    this->Interactor->SetInteractorStyle(this->InteractorStyle);
    this->Interactor->SetRenderWindow(this->RenderWindow);
    }

  if (this->Renderer && this->ImageActor)
    {
    this->Renderer->AddViewProp(this->ImageActor);
    }

  if (this->ImageActor && this->WindowLevel)
    {
    this->ImageActor->SetInput(this->WindowLevel->GetOutput());
    }
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::UnInstallPipeline()
{
  if (this->ImageActor)
    {
    this->ImageActor->SetInput(NULL);
    }

  if (this->Renderer && this->ImageActor)
    {
    this->Renderer->RemoveViewProp(this->ImageActor);
    }

  if (this->RenderWindow && this->Renderer)
    {
    this->RenderWindow->RemoveRenderer(this->Renderer);
    }

  if (this->Interactor)
    {
    this->Interactor->SetInteractorStyle(NULL);
    this->Interactor->SetRenderWindow(NULL);
    }
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::Render()
{
  if (this->FirstRender)
    {
    // Initialize the size if not set yet

    vtkImageData *input = this->GetInput();
    if (this->RenderWindow->GetSize()[0] == 0 &&
	input)
      {

      input->UpdateInformation();
      int *w_ext = input->GetWholeExtent();
      int xs = 0, ys = 0;

      //	std::cout << "wxvtkImageViewer2::Render ext = "
      //	<<w_ext[0]<<" - "<<w_ext[1]<<" ; "
      //		<<w_ext[2]<<" - "<<w_ext[3]<<" ; "
      //		<<w_ext[4]<<" - "<<w_ext[5]
      //		<<std::endl;

      switch (this->SliceOrientation)
        {
        case wxvtkImageViewer2::SLICE_ORIENTATION_XY:
        default:
          xs = w_ext[1] - w_ext[0] + 1;
          ys = w_ext[3] - w_ext[2] + 1;
	  //	  std::cout << "SLICE_ORIENTATION_XY" << std::endl;
          break;

        case wxvtkImageViewer2::SLICE_ORIENTATION_XZ:
          xs = w_ext[1] - w_ext[0] + 1;
          ys = w_ext[5] - w_ext[4] + 1;
	  //	  std::cout << "SLICE_ORIENTATION_XZ" << std::endl;
          break;

        case wxvtkImageViewer2::SLICE_ORIENTATION_YZ:
          xs = w_ext[3] - w_ext[2] + 1;
          ys = w_ext[5] - w_ext[4] + 1;
	  //	  std::cout << "SLICE_ORIENTATION_YZ" << std::endl;
          break;
        }

      // if it would be smaller than 150 by 100 then limit to 150 by 100
      this->RenderWindow->SetSize(
        xs < 150 ? 150 : xs, ys < 100 ? 100 : ys);

      //      std::cout << "wxvtkImageViewer2::Render() : "<<xs<<"-"<<ys<<std::endl;
      if (this->Renderer)
        {
        this->Renderer->ResetCamera();
        this->Renderer->GetActiveCamera()->SetParallelScale(
          xs < 150 ? 75 : (xs - 1 ) / 2.0);
        }
      this->FirstRender = 0;

      }
    }
  if (this->GetInput())
    {
    this->RenderWindow->Render();
    }
}

//----------------------------------------------------------------------------
const char* wxvtkImageViewer2::GetWindowName()
{
  return this->RenderWindow->GetWindowName();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetOffScreenRendering(int i)
{
  this->RenderWindow->SetOffScreenRendering(i);
}

//----------------------------------------------------------------------------
int wxvtkImageViewer2::GetOffScreenRendering()
{
  return this->RenderWindow->GetOffScreenRendering();
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetInput(vtkImageData *in)
{
  //  std::cout << "### wxvtkImageViewer2::SetInput"<<std::endl;
  this->WindowLevel->SetInput(in);
  this->UpdateDisplayExtent();
  // LG 03/12/08
  //  FirstRender = 1;
}
//----------------------------------------------------------------------------
vtkImageData* wxvtkImageViewer2::GetInput()
{
  return vtkImageData::SafeDownCast(this->WindowLevel->GetInput());
}

//----------------------------------------------------------------------------
void wxvtkImageViewer2::SetInputConnection(vtkAlgorithmOutput* input)
{
  this->WindowLevel->SetInputConnection(input);
  this->UpdateDisplayExtent();
}

//----------------------------------------------------------------------------
#ifndef VTK_LEGACY_REMOVE
int wxvtkImageViewer2::GetWholeZMin()
{
  VTK_LEGACY_REPLACED_BODY(wxvtkImageViewer2::GetWholeZMin, "VTK 5.0",
                           wxvtkImageViewer2::GetSliceMin);
  return this->GetSliceMin();
}
int wxvtkImageViewer2::GetWholeZMax()
{
  VTK_LEGACY_REPLACED_BODY(wxvtkImageViewer2::GetWholeZMax, "VTK 5.0",
                           wxvtkImageViewer2::GetSliceMax);
  return this->GetSliceMax();
}
int wxvtkImageViewer2::GetZSlice()
{
  VTK_LEGACY_REPLACED_BODY(wxvtkImageViewer2::GetZSlice, "VTK 5.0",
                           wxvtkImageViewer2::GetSlice);
  return this->GetSlice();
}
void wxvtkImageViewer2::SetZSlice(int s)
{
  VTK_LEGACY_REPLACED_BODY(wxvtkImageViewer2::SetZSlice, "VTK 5.0",
                           wxvtkImageViewer2::SetSlice);
  this->SetSlice(s);
}
#endif

//----------------------------------------------------------------------------
void wxvtkImageViewer2::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "RenderWindow:\n";
  this->RenderWindow->PrintSelf(os,indent.GetNextIndent());
  os << indent << "Renderer:\n";
  this->Renderer->PrintSelf(os,indent.GetNextIndent());
  os << indent << "ImageActor:\n";
  this->ImageActor->PrintSelf(os,indent.GetNextIndent());
  os << indent << "WindowLevel:\n" << endl;
  this->WindowLevel->PrintSelf(os,indent.GetNextIndent());
  os << indent << "Slice: " << this->Slice << endl;
  os << indent << "SliceOrientation: " << this->SliceOrientation << endl;
  os << indent << "InteractorStyle: " << endl;
  if (this->InteractorStyle)
    {
    os << "\n";
    this->InteractorStyle->PrintSelf(os,indent.GetNextIndent());
    }
  else
    {
    os << "None";
    }
}
