/*
 # ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la SantÈ)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */


#include <bbwxSlider.h>
#include <bbwxLayoutLine.h>

//=========================================================================
// Illustrates the Output Observer mechanism
//=========================================================================

//=========================================================================
// FIRST EXAMPLE USING A FUNCTION AS CALLBACK
void CallbackFunction ( bbtk::BlackBox::Pointer p,  // The black box which changed 
			const std::string& o, // The output which changed
			bbtk::IOStatus s) // The new status of the output
{
  std::cout << "== Callback function called with p="<<p->bbGetName()
	    <<" o="<<o<<" s="<<s/*bbtk::GetIOStatusString(s)*/<<std::endl;

  // Cast the black box pointer into Slider pointer
  bbwx::Slider::Pointer slider = boost::dynamic_pointer_cast<bbwx::Slider>(p);
  if (slider)
    {
      std::cout << "* Current slider value = "<<slider->bbGetOutputOut()
		<<std::endl;
    }
}
//=========================================================================

//=========================================================================
void SliderCallback()
{
  std::cout << "============== Callback function called when slider is released" 
	    << std::endl;
   try
    {
      bbwx::Slider::Pointer slider = bbwx::Slider::New("slider");
      slider->bbSetInputWinDialog(true);  // mandatory
      // Add the observer 
      slider->bbAddOutputObserver( "Out", &CallbackFunction );
      slider->bbExecute();
    }
  catch (bbtk::Exception e)
    {
      e.Print();
    } 
}
//=========================================================================


//=========================================================================
// SECOND EXAMPLE : MEMBER METHOD CALLED WHEN SLIDER IS RELEASED
class ClassWithTwoSliders
{
public:
  // Ctor
  ClassWithTwoSliders();
  // Callback
  void OnSlider( bbtk::BlackBox::Pointer p, 
		 const std::string& o, 
		 bbtk::IOStatus s );
private:
  bbwx::Slider::Pointer mSlider1;
  bbwx::Slider::Pointer mSlider2;
bbwx::LayoutLine::Pointer mLayout;
};
//=========================================================================

//=========================================================================
ClassWithTwoSliders::ClassWithTwoSliders()
{
  std::cout << "============== Callback member called when first slider is released or second slider moves" 
	    << std::endl;
  try
    {
      mSlider1 = bbwx::Slider::New("slider1");
      // Add the observer method
      // As it is not a FREEHAND function we must use the macro 
      // BBTK_MAKE_OUTPUT_OBSERVER which takes the object and the method to call
      mSlider1->bbAddOutputObserver("Out", 
				    BBTK_MAKE_OUTPUT_OBSERVER
				    (this,
				     &ClassWithTwoSliders::OnSlider ));
      
      mSlider2 = bbwx::Slider::New("slider2");
      mSlider2->bbSetInputReactiveOnTrack(true);
      mSlider2->bbAddOutputObserver("Out", 
				    BBTK_MAKE_OUTPUT_OBSERVER
				    (this,
				     &ClassWithTwoSliders::OnSlider ));
      
      mLayout = bbwx::LayoutLine::New("layout");
      mLayout->bbSetInputWinDialog(true);  // mandatory
      
      bbtk::Connection::Pointer c1 = bbtk::Connection::New( mSlider1, 
							    "Widget", 
							    mLayout, 
							    "Widget1");
      bbtk::Connection::Pointer c2 = bbtk::Connection::New( mSlider2, 
							    "Widget", 
							    mLayout, 
							    "Widget2");
      
      mLayout->bbExecute();
    }
  catch (bbtk::Exception e)
    {
      e.Print();
    }   
}
//=========================================================================

//=========================================================================
void ClassWithTwoSliders::OnSlider( bbtk::BlackBox::Pointer p, 
				    const std::string& o, 
				    bbtk::IOStatus s )
{
  std::cout << "== 'OnSlider' called with p="<<p->bbGetName()
	    <<" o="<<o<<" s="<<s/*bbtk::GetIOStatusString(s)*/<<std::endl;
  
  if ((p==mSlider1) && (o=="Out"))
    {
      std::cout << "* First slider released. New value = "
		<<mSlider1->bbGetOutputOut()<<std::endl;
    }						       
  else if ((p==mSlider2) && (o=="Out"))
    {
      std::cout << "* Second slider moved. New value = "
		<<mSlider2->bbGetOutputOut()<<std::endl;
    }
}
//=========================================================================

//=========================================================================
int main(int argv, char* argc[])
{
 // To track all ...
	bbtk::MessageManager::SetMessageLevel("all",9);
  SliderCallback();
  ClassWithTwoSliders C;
}
//=========================================================================
