/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


//----------------------------------------------------------------------------------------------------------------
// Class definition include
//----------------------------------------------------------------------------------------------------------------
#include "CommandsHandler.h"

//----------------------------------------------------------------------------------------------------------------
// Class implementation
//----------------------------------------------------------------------------------------------------------------
/** @file CommandsHandler.cxx */

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------
	
	/*
	* Constructs the CommandsHandler
	*/
	CommandsHandler :: CommandsHandler()
	{
		redo_actions = new CommandsRegisterStructure();
		unDo_actions = new CommandsRegisterStructure();

		posibleREDO = false;
		posibleUNDO = false;

		isLastREDO_executed = true;
		isLastUNDO_executed = false;
	}

	/*
	* Destructs the CommandsHandler
	*/
	CommandsHandler :: ~CommandsHandler()
	{
		clearActions();
		delete redo_actions;
		delete unDo_actions;
	}

//------------------------------------------------------------------------------------------------------------
// Methods
//------------------------------------------------------------------------------------------------------------

	/*
	* Registers in the vectors of doneActions and unDoActions the given commands that all ready corresponds each other to the inverse of the otherone. 
	* If is the first registered action notifies the posibleUNDO avaliability.
	* @param doneAction Is the action to register in the redo_actions vector.
	* @param unDoAction Is the action to register in the unDo_actions vector.
	*/
	void CommandsHandler :: registerCommand(CommandObject* doneAction, CommandObject* unDoAction)
	{
		int actToUNDO = unDo_actions->getActualIndex();	
		redo_actions->setActualIndex( actToUNDO );				
		
		redo_actions->registerCommand(doneAction);
		unDo_actions->registerCommand(unDoAction);
		
		posibleREDO = false;
		posibleUNDO = true;

		isLastREDO_executed = true;
		isLastUNDO_executed = false;
		
	}

	/*
	* Undo a command. Managing the correspondig vectors and the execution of the inverse action to the - ACTUAL DONE- action
	* @return Returns true if the inverse command ( the actual to execute in UNDO actions) is executed. If it is false the state of the vector must not change.
	*/
	bool CommandsHandler :: undo()
	{
		bool executed = posibleUNDO;
		if( posibleUNDO )
		{			
			validateOperationsAvaliability();//para borrar!!!!!!!!-----------------------------*********************---------------------
			executed = theWorksSpaceBoss->executeCommand(getActual_UNDO(), true);
			isLastUNDO_executed = true;
			if (!unDo_actions->hasActualNext() && !redo_actions->hasActualNext())
			{				
				if(unDo_actions->hasActualPrevious())
					executed = unDo_actions->moveBack_Actual() ;
				else
					executed = true;
				if( unDo_actions->hasLastNext() ) 
					executed = executed && unDo_actions->moveBack_Last();	
				isLastREDO_executed = false;
			}
			else if (!unDo_actions->hasActualPrevious() && redo_actions->hasActualPrevious())
			{
				executed = redo_actions->moveBack_Actual();	
				executed = executed && redo_actions->moveBack_Last();	
				executed = executed && unDo_actions->moveBack_Last();				
			}
			else 
			{					
				executed = unDo_actions->moveBack_Last();
				executed = executed && unDo_actions->moveBack_Actual();	
				executed = executed && redo_actions->moveBack_Actual();	
				if( redo_actions->hasLastNext() ) 
					executed = executed && redo_actions->moveBack_Last();	
			}
			validateOperationsAvaliability();			
		}
		return executed;
	}

	/*
	* Redo a command. Managing the correspondig vectors and the execution of the inverse action to the - ACTUAL DONE- action
	* @return Returns true if the actual command to execute ( the actual to execute in REDO actions )has been executed. If it is false the state of the vector must not change.
	*/
	bool CommandsHandler :: redo()
	{
		bool executed = posibleREDO;
		if( posibleREDO )
		{			
			validateOperationsAvaliability();//para borrar!!!!!!!!-----------------------------*********************---------------------
			isLastREDO_executed = true;
			//isLastUNDO_executed = false; // Posible needed
			executed = theWorksSpaceBoss->executeCommand(getActual_REDO(), true);
			if (!unDo_actions->hasActualPrevious() && !redo_actions->hasActualPrevious())
			{								
				executed = redo_actions->moveForward_Actual();
				executed = executed && redo_actions->moveBack_Last();
				isLastUNDO_executed = false;
			}
			else if (!unDo_actions->hasActualPrevious() && redo_actions->hasActualPrevious())
			{
				executed = redo_actions->moveForward_Actual();
				if(unDo_actions->hasLastNext())
					executed = executed && unDo_actions->moveForward_Last();	
				executed = executed && unDo_actions->moveForward_Actual();
				if(redo_actions->hasLastNext())
					executed = executed && redo_actions->moveForward_Last();
			}
			else 
			{	
				executed = unDo_actions->moveForward_Actual();	
				executed = executed && unDo_actions->moveForward_Last();	
				executed = executed && redo_actions->moveForward_Actual();		
				if( redo_actions->hasLastNext() ) 
						executed = executed && redo_actions->moveForward_Last();	
				else
					executed = executed && redo_actions->moveBack_Last();	
			}
			if (!unDo_actions->hasActualPrevious() && redo_actions->hasActualPrevious())
			{
				isLastUNDO_executed = false;
			}
			if (!unDo_actions->hasActualNext() && !redo_actions->hasActualNext())
			{
				isLastUNDO_executed = false;
			}
			validateOperationsAvaliability();
		}		
		return executed;
	}

	/*
	* Notitify if posibleREDO is posible or not.
	* @return Returns the state of posibleUNDO
	*/
	bool CommandsHandler :: isPosibleUNDO()
	{
		return posibleUNDO;
	}

	/*
	* Indicates if posibleUNDO is posible or not.
	* @return Returns the state of posibleREDO
	*/
	bool CommandsHandler :: isPosibleREDO()
	{
		return posibleREDO;
	}

	/*
	* Sets  posibleREDO state.
	* @param UNDOstate The state of posibleUNDO to set
	*/
	void CommandsHandler :: setPosibleUNDO( bool UNDOstate )
	{
		posibleUNDO = UNDOstate;
	}

	/*
	* Sets posibleUNDO state.
	* @param REDOstate The state of posibleREDO to set
	*/
	void CommandsHandler :: setPosibleREDO( bool REDOstate )
	{
		posibleREDO = REDOstate;
	}

   	/*
	* Clear the registered actions in the DO and UNDO vectors.
	*/
	void CommandsHandler :: clearActions()
	{
		redo_actions->clearActions();
		unDo_actions->clearActions();
	}

	/*
	* Returns hoy mane paired commands had been registered, if the done and unDo vectors dont match returns -1 as error code.
	* @return Returns how many paired-commands had been registered
	*/
	int CommandsHandler :: getTotalCommands()
	{
		int value = redo_actions->getTotalCommandsCount();
		return value == (unDo_actions->getTotalCommandsCount())  ? value : -1;
	}


	/*
	* Gets the actual command in the UNDO-list
	* @return  Returns a pointer to the actual undo action
	*/
	CommandObject * CommandsHandler :: getActual_UNDO()
	{
		return unDo_actions->getActual_Pointer();
	}

	/*
	* Gets the actual command in the REDO-list
	* @return  Returns a pointer to the actual do action
	*/
	CommandObject * CommandsHandler :: getActual_REDO()
	{
		return redo_actions->getActual_Pointer();
	}

	/*
	* Gets the command at the given position in the DO (REDO) vector
	* @return The pointer to the referenced object by the position
	*/
	CommandObject * CommandsHandler :: get_DO_CommandAt(int position)
	{
		return redo_actions->getCommandAt(position);
	}

	/*
	* Gets the command at the given position in the UNDO vector
	* @return The pointer to the referenced object by the position
	*/
	CommandObject * CommandsHandler :: get_UNDO_CommandAt(int position)
	{
		return unDo_actions->getCommandAt(position);
	}

		/*
	* Sets the model parent of the action handler
	* @param theModelParent The boss reference
	*/
	void CommandsHandler :: setModelBoss(ICommandsUser * theModelParent)
	{
		theWorksSpaceBoss = theModelParent;
	}

	/*
	* Validates if it is posible of not to do UNDO and/or REDO and sets the corresponding values
	*/
	void CommandsHandler :: validateOperationsAvaliability()
	{
		posibleREDO = (redo_actions->hasActualNext() && unDo_actions->hasActualNext() )? true :
			( (!redo_actions->hasActualNext()&& unDo_actions->hasActualNext() ) ? true : 
			( !unDo_actions->hasActualNext() && !redo_actions->hasActualNext() )? false : true && !isLastREDO_executed );	

		posibleUNDO = (redo_actions->hasActualPrevious() && unDo_actions->hasActualPrevious() )? true :
			( (!unDo_actions->hasActualPrevious()&& redo_actions->hasActualPrevious() ) ? true : 
			( !unDo_actions->hasActualPrevious() && !redo_actions->hasActualPrevious() )? false : true && !isLastUNDO_executed );	

	}
