/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


//----------------------------------------------------------------------------------------------------------------
// Class definition include
//----------------------------------------------------------------------------------------------------------------
#include "OutlineGroup.h"

//----------------------------------------------------------------------------------------------------------------
// Class implementation
//----------------------------------------------------------------------------------------------------------------
/** @file OutlineGroup.cxx */

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------

	/*
	* Constructs an outline group with the given name
	* @param theName Is the name for the group
	* @param theGroupType Is the type for the group corresponding to one of the constants of this class
	*/ 
	OutlineGroup :: OutlineGroup(std::string theName, int theGroupType)
	{
		name = theName;		
		groupType = theGroupType;
		totalCount = 0;
		
		if ( groupType == PROPAGATION || groupType == PLANE_SECTION  )
		{
			visibleGroup = true;			
			selectedGroup = false;
			editingGroup = false;
			staticGroup = false;
			acceptsRepetedOutlines = false;
		}
		else if ( groupType ==  STRIP )
		{	
			visibleGroup = true;			
			selectedGroup = false;
			editingGroup = false;
			staticGroup = false;
			acceptsRepetedOutlines = true;
		}
		else if ( groupType == OVERLAPED )
		{				
			visibleGroup = true;			
			selectedGroup = false;
			editingGroup = false;
			staticGroup = true;
			acceptsRepetedOutlines = false;
		}
		else if ( groupType ==  MANUAL_GROUP )
		{				
			visibleGroup = true;
			selectedGroup = true;
			editingGroup = true;
			staticGroup = false;
			acceptsRepetedOutlines = true;
		}				
	}
	
	/*
	* Destroyes the outline and its dependencies
	*/
	OutlineGroup :: ~ OutlineGroup()
	{
		outlines_keyNames.clear();
	}

//------------------------------------------------------------------------------------------------------------
// Methods
//------------------------------------------------------------------------------------------------------------


	/*
	* Indicates if a given name of an outline is member of the group or not
	* @param aKeyName Is the name of an outline to search for
	*/
	bool OutlineGroup :: isMemberOfGroup(std::string aKeyName)
	{
		for (int a=0; a < outlines_keyNames.size(); a++)
		{
			if ( outlines_keyNames[a].compare(aKeyName) == 0 )
				return true;
		}		
		return false;
	}

	/*
	* Removes an outline with the given name from the group 
	* @param theNameToRemove Is the name to remove from member name list
	*/
	void OutlineGroup :: removeOutline(std::string theNameToRemove, bool allOcurrencies)
	{
		std::vector <std::string>::iterator iter;		
		
		bool ifContinue = true;
		bool deleted = false;
		for ( iter = outlines_keyNames.begin( ); iter != outlines_keyNames.end( ) && ifContinue; iter++ )
		{
			if ( (*iter).compare(theNameToRemove)==0 )
			{
				outlines_keyNames.erase(iter);
				deleted = true;
			}
			ifContinue = allOcurrencies ? acceptsRepetedOutlines : deleted; 
		}
		//delete iter; Se incluye esta linea o no ????????????????????
	}
	
	/*
	* Adds an outline with the given name to the group members list
	* @param theNameNw Is the name to add to the group
	*/
	void OutlineGroup :: addOutline(std::string theNameNw)
	{
		bool ifInclude = acceptsRepetedOutlines ? true : !isMemberOfGroup(theNameNw);
		if ( ifInclude )
			outlines_keyNames.push_back(theNameNw);
	/*
		if( groupType == PROPAGATION )
			addOutline_PropagationType( theNameNw );
		else if ( groupType == PLANE_SECTION )
			addOutline_PlaneSectionType( theNameNw );
		else if ( groupType == OVERLAPED )
			addOutline_OverlapedType( theNameNw );
		else if ( groupType == STRIP )
			addOutline_StripType( theNameNw );
		else if ( groupType == MANUAL_GROUP )
			addOutline_ManualType( theNameNw );
	*/
	}	

	/*
	* Gets the name of the group
	* @return name Is the name of the group
	*/ 
	std::string OutlineGroup :: getName()
	{ 
		return name;
	}

	/*
	* Sets the name of the group as the given one 
	* @param name Is the new name of the group
	*/ 
	void OutlineGroup :: setName(std::string theNwName)
	{ 
		name = theNwName;
	}

	/*
	* Gets the state of visiblility (true:visible) or not of the group
	* @return visibleGroup Is the corresponding state
	*/ 
	bool OutlineGroup ::  getIfVisibleGroup()
	{ 
		return visibleGroup;
	}

	/*
	* Sets state of visible (true) or not of the with the given one 
	* @param theNwVisiblity Is the corresponding state
	*/ 
	void OutlineGroup :: setIfVisibleGroup(bool theNwVisiblity)
	{ 
		visibleGroup = theNwVisiblity;
	}

	/*
	* Gets the state of static (true:static) or not of the group
	* @return staticGroup Is the corresponding state
	*/ 
	bool OutlineGroup ::  getIfStaticGroup()
	{ 
		return staticGroup;
	}

	/*
	* Sets state of static (true) or not of the with the given one 
	* @param theNwStatic Is the corresponding state
	*/ 
	void OutlineGroup :: setIfStaticGroup(bool theNwStatic)
	{ 
		staticGroup = theNwStatic;
	}

	/*
	* Gets the state of selection (true:selected) or not of the group
	* @return selecetedGroup Is the corresponding state
	*/ 
	bool OutlineGroup ::  getIfSelectedGroup()
	{ 
		return selectedGroup;
	}

	/*
	* Sets state of visible (true) or not of the with the given one 
	* @param theNwSelected Is the corresponding state
	*/ 
	void OutlineGroup :: setIfSelectedGroup(bool theNwSelected)
	{ 
		selectedGroup = theNwSelected;
	}

	/*
	* Gets the state of edition (true:editing) or not of the group
	* @return editingGroup Is the corresponding state
	*/ 
	bool OutlineGroup ::  getIfEditingGroup()
	{ 
		return editingGroup;
	}

	/*
	* Sets state of editing (true) or not of the with the given one 
	* @param theNwEditing Is the corresponding state
	*/ 
	void OutlineGroup :: setIfEditingGroup(bool theNwEditing)
	{ 
		editingGroup = theNwEditing;
	}

	/*
	* Gets the total count of outlines in the group
	* @return totalCount Is the corresponding number of elements
	*/ 
	int  OutlineGroup :: getGroupType()
	{
		return groupType;
	}

	/*
	* Sets the group type 
	* @param theType Is the corresponding new type to assign 
	*/ 
	void OutlineGroup :: setGroupType(int  theType)
	{
		groupType = theType;
	}

	/*
	* Sets the total count of outlines in the group
	* @param theNwVisiblity Is the corresponding state
	*/ 
	void OutlineGroup :: setIfEditingGroup(int  theTotal)
	{
		totalCount = theTotal;
	}

	/*
	* Gets the group type 
	* @return type Is the corresponding number of elements
	*/ 
	int  OutlineGroup :: getOutlinesCount()
	{
		return totalCount;
	}

	/*
	* Adds an outline to the group as propagation type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void OutlineGroup :: addOutline_PropagationType(std::string theOutlineKeyName)
	{
		outlines_keyNames.push_back(theOutlineKeyName);
	}

	/*
	* Adds an outline to the group as plane section type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void OutlineGroup :: addOutline_PlaneSectionType(std::string theOutlineKeyName)
	{
		outlines_keyNames.push_back(theOutlineKeyName);
	}


	/*
	* Adds an outline to the group as overlaped type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void OutlineGroup :: addOutline_OverlapedType(std::string theOutlineKeyName)
	{
		bool ifInclude = isMemberOfGroup(theOutlineKeyName);
		if ( ifInclude )
			outlines_keyNames.push_back(theOutlineKeyName);
	}

	/*
	* Adds an outline to the group as strip type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void OutlineGroup :: addOutline_StripType(std::string theOutlineKeyName)
	{
		outlines_keyNames.push_back(theOutlineKeyName);
	}

	/*
	* Adds an outline to the group as manual type
	* @param theOutlineKeyName Is the name used as identifier of the outline	
	*/
	void OutlineGroup :: addOutline_ManualType(std::string theOutlineKeyName)
	{
		outlines_keyNames.push_back(theOutlineKeyName);
	}

	/*
	* Gets the outlines of the group
	* @return Returns the names of the outlines that belong to the group
	*/
	 std::vector< std::string > OutlineGroup :: getGroupOutlinesNames ( )
	 {
		 return outlines_keyNames;
	 }
 
