/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


//----------------------------------------------------------------------------------------------------------------
// Class definition include
//----------------------------------------------------------------------------------------------------------------
#include "OutlineModelManager.h"

//----------------------------------------------------------------------------------------------------------------
// Class implementation
//----------------------------------------------------------------------------------------------------------------
/** @file OutlineModelManager.cxx */

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------
/*
	* Creates the outline manager
	*/
	OutlineModelManager :: OutlineModelManager( SomeEnvironment<ImageSourceThing *> *imSourceEnv, SomeEnvironment<ImageSectionThing *> *imSectionEnv, SomeEnvironment<AxeThing *> *axesEnv, SomeEnvironment<ContourThing *> *contourEnv )
	{
		actualInstant				= NULL;
		imageSourceEnvironment		= imSourceEnv;
		imagesSectionEnvironment	= imSectionEnv;
		axesEnvironment				= axesEnv;
		outlinesEnvironment			= contourEnv;

		workSpace					= new ContourWorkspace(this);//implies a shared workSpace
		keyGenerator				= * new PrefixMaxKeyGenerator();
		counterIdKey				= 0;

		actualInstantWrapping		= new InstantMembersNameList();
		changeSourceImage			= true;

		bool allOK	= keyGenerator.addKeyThing("Axes", "Axe");
		allOK &= keyGenerator.addKeyThing("Image Sources", "Image source");
		allOK &= keyGenerator.addKeyThing("Image Sections", "Image section");
		allOK &= keyGenerator.addKeyThing("Outlines", "Outline");
		//setAutomaticConcepts();
	}

	/*
	* Destroys the outline manager
	*/
	OutlineModelManager :: ~OutlineModelManager()
	{
		clean();
		delete actualInstant;
		delete workSpace;
		delete actualInstantWrapping;
	}

//------------------------------------------------------------------------------------------------------------
// Methods
//------------------------------------------------------------------------------------------------------------


	/*
	* Creates and sets a workSpace object
	* @return Retourns the created workSpace
	*/
	ContourWorkspace * OutlineModelManager :: createWorkSpace()
	{
		workSpace = new ContourWorkspace(this);
		return workSpace;
	}

	/*
	* Sets the workSpace object
	* @param  aWorkSpace The workSpace to set
	*/
	void OutlineModelManager :: setWorkSpace( ContourWorkspace * aWorkSpace )
	{
		workSpace = aWorkSpace;
	}

	/*
	* Executes a command over an outline object
	* @param imaKName Is the key name of the outline
	* @param theCommand Is the command to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/

	bool OutlineModelManager :: executeCommand_OutlineModel(std::string outKName, CommandObject * theCommand, bool fromRegistration)
	{
		ContourThing * theOutline = *outlinesEnvironment->getThingByName( outKName );
		if(!fromRegistration)
		{
			CommandObject * undoCommand = theOutline->getUndoCommandOf( theCommand );
			workSpace->callRegisterCommand(theCommand, undoCommand);
		}
		return theOutline->executeCommand( theCommand );
	}

	/*
	* Executes a command identifying which actions have to realize before executing it.
	* @param theCommand Is the command to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/
	bool OutlineModelManager :: executeCommand(CommandObject * theCommand, bool fromRegistration)
	{
		bool executedCom = true;
		//TODO**********************************************************************************************************************************************
		//Interpreting who is the one that executes the command

		//Modifiying the command in necessary for detaching interpreted information

		//If is an outline, sending the command to execute the specific command

		return executedCom;
	}

	/*
	* Executes a command queue identifying which actions have to realize before executing it, using FIFO for each.
	* @param executionQueue Is the command queue to execute
	* @param fromRegistration Indicates if the execution is directed from the registration, by default comes from the GUI = false.
	*/
	bool OutlineModelManager ::  executeCommandsQueue(std::deque<CommandObject *> executionQueue, bool fromRegistration)
	{
		bool executedComs = executionQueue.size()>0 ? true : false;
		while( executionQueue.size()>0 )
		{
			CommandObject * aCmmand = executionQueue.front();
			executedComs &= executeCommand(aCmmand, fromRegistration);
			executionQueue.pop_front();
		}
		return executedComs;
	}


	/*
	* Sets the concepts of the environments and includes the concepts managed by the program
	* @conceptsScript Is the script for with the concepts descrition
	* @return Returns true if successful insert of the given concepts, false otherwise
	*/
	bool OutlineModelManager :: setUserConcepts(std::string conceptsScript)
	{
		bool allOK = true;
		int endSub;
		std::string::size_type equalIndex = conceptsScript.find("=", 0);
		std::string::size_type nextIndex = 0;
		while( equalIndex != std::string::npos && allOK)
		{
            endSub = equalIndex;
			std::string concept = conceptsScript.substr( 0, endSub );
			conceptsScript.erase( 0, endSub+1 );
			std::cout << "C: " << concept <<std::endl;
			nextIndex = conceptsScript.find(";", 0);
			std::string cSize;
			if(nextIndex != std::string::npos)
			{
				endSub = nextIndex;
				cSize = conceptsScript.substr( 0, endSub );
				conceptsScript.erase( 0, endSub+1 );
			}
			else
			{
				endSub = conceptsScript.size();
				cSize = conceptsScript.substr( 0, endSub );
				conceptsScript.erase( 0, endSub );
			}
			int intReturn = atoi(cSize.c_str());
			std::cout << "SZ: "<<intReturn <<std::endl;
			equalIndex = conceptsScript.find("=", 0);
			if( intReturn>0 )
				allOK &= addUserConcept(concept, intReturn);
			else
				allOK = false;
		}
		return allOK;
	}

	/*
	* Add a concept to all the environments
	* @param theConceptName Is the name of the new concept
	* @param conceptSize Is the size of the concept, that represent the ammount of concept posible instances
	* @return Returns true if successful insert of concept, false otherwise
	*/
	bool OutlineModelManager :: addUserConcept(std::string theConceptName, int conceptSize)
	{
		bool allOK = axesEnvironment->addConcept(theConceptName, conceptSize);
		allOK &= imagesSectionEnvironment->addConcept(theConceptName, conceptSize);
		allOK &= imageSourceEnvironment->addConcept(theConceptName, conceptSize);
		allOK &= outlinesEnvironment->addConcept(theConceptName, conceptSize);
		return allOK;
	}

	/**
	* Gets the outlines at an instant form the outline's environment
	* @param anInstant The instant wondered to get outlines at
	* @param ifAnnotate Indicates if it is needed to annotate the searched outlines
	* @return The vector to the outlines at the given instance
	*/
	std::vector<NameWrapper *> OutlineModelManager :: getActualInstantOutlines ()
	{
		return actualInstantWrapping->getOutlinesNamesVector();
	}

//EED01
	void OutlineModelManager :: ChangeContourOfList(std::string keyName, Instant *instant)
	{
		int ispartofstaticlist = IsPartOfStaticList(keyName);
		if (ispartofstaticlist==-1)
		{
			staticContourLst.push_back(  this->outlinesEnvironment->getThingByName(keyName)  );
			this->outlinesEnvironment->removeThingFromInstant(keyName, instant);
		} else {
			ContourThing **contourthing  = staticContourLst[ispartofstaticlist];
			this->outlinesEnvironment->addInstantToThing	( keyName , instant );
			staticContourLst.erase( staticContourLst.begin()+ispartofstaticlist );
		}
	}


	/**
	* Gets the outlines at an instant form the outline's environment
	* @param anInstant The instant wondered to get outlines at
	* @param ifAnnotate Indicates if it is needed to annotate the searched outlines
	* @return The vector to the outlines at the given instance
	*/
	std::vector<ContourThing**>  OutlineModelManager :: getOutlinesAtInstant(Instant * anInstant, bool ifAnnotate)
	{
		std::vector<std::string> kNamesVector;
		std::vector<ContourThing **> outlinesVector; //= new std::vector<ContourThing **>();
		outlinesEnvironment->getThings( kNamesVector, outlinesVector, anInstant);

//EED01
		int i,sizeStaticContourLst =  staticContourLst.size();
		for ( i=0 ; i<sizeStaticContourLst ; i++ )
		{
			ContourThing **contourthing  = staticContourLst[i];
			outlinesVector.push_back( contourthing );
			kNamesVector.push_back( (*contourthing)->getName() );
		}// for i

		if (ifAnnotate)
		{
			annotateOutlinesWrap(kNamesVector, outlinesVector);
		}
		return outlinesVector;
	}

//EED01
	std::vector<std::string> OutlineModelManager :: GetLstNameThingsStatic()
	{
		std::vector<std::string> result;
		int i,sizeStaticContourLst =  staticContourLst.size();
		for ( i=0 ; i<sizeStaticContourLst ; i++ )
		{
			ContourThing **contourthing  = staticContourLst[i];
			result.push_back( (*contourthing)->getName() );
		}// for i
		return result;
	}

	std::vector<std::string> OutlineModelManager :: GetLstNameThings()
	{
		std::vector<std::string> kNamesVector;
		std::vector<ContourThing **> outlinesVector; //= new std::vector<ContourThing **>();
		std::vector<Instant *> lstInstants = getOutlineInstants();
		int i,sizeLstInstants = lstInstants.size();
		Instant *instant;
		for ( i=0 ; i<sizeLstInstants ; i++ )
		{
			instant = lstInstants[i];
			outlinesEnvironment->getThings( kNamesVector, outlinesVector, instant);
		} // for i
		return kNamesVector;
	}


	/**
	* Gets the outlines at a specific form the outline's environment
	* @param aGroupName The name of the group containing the outlines names to get
	* @return The vector to the outlines at the given group
	*/
	std::vector<ContourThing*> OutlineModelManager :: getOutlinesFromGroup(std::string aGroupName)
	{
		std::vector<ContourThing *> outlinesVector;
		std::map< std::string,OutlineGroup * >::iterator iterP = outlineGroups.find(aGroupName);
		if ( iterP != outlineGroups.end() )
		{
			std::vector<std::string> kNamesVector = iterP->second->getGroupOutlinesNames();
			for(int i=0; i<kNamesVector.size();i++)
			{
				ContourThing * outlineI = getOutlineByKeyName(kNamesVector[i]);
				outlinesVector.push_back(outlineI);
			}
		}
		return outlinesVector;
	}

	/**
	* Gets the outlines at a specific form the outline's environment
	* @param aGroupName The name of the group containing the outlines names to get
	* @return The vector to the outlineGroups at the given group
	*/
	bool OutlineModelManager :: addOutlinesGroup( std::string theOutlineName, OutlineGroup * theGroup )
	{
        outlineGroups.insert(std::pair <std::string, OutlineGroup *> ( theOutlineName, theGroup ));
//		std::map<std::string, OutlineGroup *> :: iterator iter = NULL;
		std::map<std::string, OutlineGroup *> :: iterator iter;
		iter = outlineGroups.find( theOutlineName );
		bool ifAdded = iter->first.compare( theOutlineName ) == 0;
		return ifAdded;
	}
	/*
	* Gets the an outline given the keyName used to be indexed in the outlines's environment
	* @param outKName Is the outline keyName to search
	* @return  The corresponding unique outline with the given key name
	*/
	ContourThing* OutlineModelManager :: getOutlineByKeyName(std::string outKName)
	{
		return *(outlinesEnvironment->getThingByName(outKName));
	}

	/*
	* Gets the an imageSourceThing given the keyName used to be indexed in the ImageSourceThing's environment
	* @param outKName Is the imageSourceThing keyName to search
	* @return  The corresponding unique outline with the given key name
	*/
	ImageSourceThing* OutlineModelManager ::getImageSourceThingByKeyName(std::string iSeourceKName)
	{
		return *(imageSourceEnvironment->getThingByName(iSeourceKName));
	}

	/*
	* Gets the an imageSectionThing given the keyName used to be indexed in the ImageSectionThing's environment
	* @param outISectionName Is the imageSectionThing keyName to search
	* @return  The corresponding unique outline with the given key name
	*/
	ImageSectionThing* OutlineModelManager ::getImageSectionThingByKeyName(std::string iSectionName)
	{
		return *(imagesSectionEnvironment->getThingByName(iSectionName));
	}

	/*
	* Creates an outlineThing with a given name, if no name is given it would have an automatic
	* @param aName The name for the outlineThing
		* @return Returns the key name of the created outline, or ""if it wasn't created
	*/
//	bool OutlineModelManager :: createOutline(std::string aName, ContourThing * &theOutline)
	std::string  OutlineModelManager :: createOutline(manualBaseModel * model, std::vector<int> theInstantData, std::string aName)
	{
		ContourThing * theOutline = new ContourThing ( model );
		theOutline->setName(aName);
		return addOutline( theOutline, new Instant( &theInstantData ) );
	}

	/*
	* Creates an axeThing with a given name
	* @param aDescription The description for the axeThing
	* @return Returns true if the creation of the axe was successful
	*/
	//bool OutlineModelManager :: createAxe(std::string aDescription, AxeThing * &theAxe)
	bool  OutlineModelManager :: createAxe(std::string aDescription, Instant * theInstantData)
	{
		AxeThing * theAxe = new AxeThing ();
		theAxe->setDescription(aDescription);
		return addAxe(theAxe, theInstantData);
	}

	/*
	* Creates an imageSourceThing with a given name
	* @param aSource The name for the imageSourceThing
	* @return Returns true if the creation of the imageSource was successful
	*/
	//bool OutlineModelManager :: createImageSource(std::string aSource, ImageSourceThing * &imageSource)
	bool OutlineModelManager :: createImageSource(std::string aSource, Instant * theInstantData)
	{
//		imageSource = new ImageSourceThing(aSource);
		//return addImageSource(new ImageSourceThing(aSource), theInstantData);
		return NULL;//***********************************************************************************************
	}

	/*
	* Creates an imageSectionThing with a given name
	* @param aSecImageData The name for the imageSectionThing
	* @return Returns true if the creation of the imageSection was successful
	*/
	//bool OutlineModelManager :: createImageSection(std::string aSecImageData, ImageSectionThing * &imageSection)
	bool OutlineModelManager :: createImageSection(std::string aSecImageData, Instant * theInstantData)
	{
		//imageSection = new ImageSectionThing(aSecImageData);
//		return addImageSection(new ImageSectionThing(aSecImageData), theInstantData);
		return NULL;
	}

	/*
	* Adds an outlineThing
	* @param theOutline The outline/contour (thing)
	* @param theInstantData Is the instant for the outline to add
	* @return Returns true if the addition of the outline was successful
	*/
	std::string OutlineModelManager :: addOutline( ContourThing * theOutline, Instant * theInstantData)
	{
		std::string kName;

// EED
//		bool added  = keyGenerator.generateKeyOf("Outlines", outlinesEnvironment->getNumberOfThings(),kName);

		counterIdKey++;
		bool added = keyGenerator.generateKeyOf( "Outlines", counterIdKey , kName );

		if( theOutline->getName().compare("")==0 )
		{
			theOutline->setName(kName);
		}
		added &= outlinesEnvironment->addThingWithInstant(kName,theOutline, theInstantData);
		if( added )
			return kName;
		else
			return "";
	}


	int OutlineModelManager::IsPartOfStaticList(std::string ss)
	{
		int iBack=-1;
		int i, size = staticContourLst.size();
		for( i=0 ; i<size ; i++ )
		{
			ContourThing **contourthing  = staticContourLst[i];
			if ((*contourthing)->getName() == ss)
			{
				iBack=i;
			} // getName() == ss
		}
		return iBack;
	}


	/*
	* Remove an outlineThing
	* @param theOutline The outline/contour (thing)
	*/
	void OutlineModelManager :: removeOutline( std::string ss )
	{
			outlinesEnvironment->removeThing( ss );
	}

//EED01
	void OutlineModelManager :: removeAllOutlines()
	{
		//
		staticContourLst.clear();

		//
		std::vector< ContourThing **>  thingsVector;
		outlinesEnvironment->getThingsOfEnvironment( &thingsVector );
		int i,sizeThingVector = thingsVector.size();
		for(i=0;i<sizeThingVector;i++)
		{
			ContourThing ** contourthing =  thingsVector[i];
		    removeOutline(  (*contourthing)->getName()   );
		}
	}


	/*
	* Adds an axeThing
	* @param thaAxe The axe (thing)
	* @param theInstantData Is the instant for the axe to add
	* @return Returns true if the addition of the axe was successful
	*/
	bool OutlineModelManager :: addAxe( AxeThing * theAxe , Instant * theInstantData)
	{
		std::string kName;
		bool added  = keyGenerator.generateKeyOf("Axes", axesEnvironment->getNumberOfThings(),kName);
		added &= axesEnvironment->addThingWithInstant(kName,theAxe, theInstantData);
		return added;
	}

	/*
	* Adds an imageSourceThing
	* @param imgageSource The image source (thing)
	* @param theInstantData Is the instant for the source to add
	* @return Returns true if the addition of the imageSource was successful
	*/
	bool OutlineModelManager :: addImageSource(ImageSourceThing * imageSource, Instant * theInstantData)
	{
		std::string kName;
		bool added  = keyGenerator.generateKeyOf("Image Sources", imageSourceEnvironment->getNumberOfThings(),kName);
		added &= imageSourceEnvironment->addThingWithInstant(kName,imageSource, theInstantData );
		return added;
	}

	/*
	* Adds an imageSectionThing with a given name, if no name is given it would have an automatic
	* @param aName The imageSection (thing)
	* @param theInstantData Is the instant for the imageSection to add
	* @return Returns true if the addition of the imageSection was successful
	*/
	bool OutlineModelManager :: addImageSection(ImageSectionThing * imageSection, Instant * theInstantData)
	{
		std::string kName;
		bool added  = keyGenerator.generateKeyOf("Image Sections", imagesSectionEnvironment->getNumberOfThings(),kName);
		added &= imagesSectionEnvironment->addThingWithInstant(kName, imageSection, theInstantData);
		return added;
	}

	/*
	* Annotates an outline keyname at the actual instant
	* @param kOutlineName The key name to annotate
	* @param theRealName The real name asigned to the outline
	*/
	void OutlineModelManager :: annotateOutlineWrap(std::string kOutlineName, std::string theRealName)
	{
		actualInstantWrapping -> addOutlineName(kOutlineName, theRealName);
	}

	/*
	* Annotates a set of outline keynames-real names wrapping at the actual instant
	* @param kNamesVector The key names vector to annotate
	* @param theOutlinesVector The outlines pointers vector to get the real names from
	*/
	void OutlineModelManager :: annotateOutlinesWrap(std::vector<std::string> kNamesVector, std::vector<ContourThing **> theOutlinesVector)
	{
		actualInstantWrapping ->clean();
		for(int i=0; i<kNamesVector.size(); i++)
		{
			ContourThing * anOutline = *theOutlinesVector[i];
			actualInstantWrapping -> addOutlineName( kNamesVector[i], anOutline->getName() );
		}
	}

	/*
	* Annotates the actual outline keyName-real name wrapping at the actual instant
	* @param actualKeyOutline The key name to annotate
	* @param theRealName The real name asigned to the actual outline
	*/
	void OutlineModelManager :: annotateActualOutlineWrap(std::string actualKeyOutline, std::string theRealName)
	{
		actualInstantWrapping -> setActualOutline(actualKeyOutline, theRealName);
	}

	/*
	* Annotate the actual axe keyName-real name wrapping at the actual instant
	* @param actualKeyAxe The key name to annotate
	* @param theRealName The real name asigned to the actual axe
	*/
	void OutlineModelManager :: annotateActualAxeWrap(std::string actualKeyAxe, std::string theRealName)
	{
		actualInstantWrapping -> setActualAxeWrap(actualKeyAxe, theRealName);
	}

	/*
	* Annotate the annotateActualSection image keyName-real name wrapping at the actual instant
	* @param actualKeyImage The key name to annotate
	* @param theRealName The real name asigned to the annotateActualSection
	*/
	void OutlineModelManager :: annotateActualSectionImageWrap(std::string actualKeyImage, std::string theRealName)//---BORRAR...
	{
		actualInstantWrapping -> setActualSectionImageNamesWrapp(actualKeyImage, theRealName);
	}

	/*
	* Annotate the annotateActualSource image keyName-real name wrapping at the actual instant
	* @param actualKeyImage The key name to annotate
	* @param theRealName The real name asigned to the annotateActualSource
	*/
	void OutlineModelManager :: annotateActualSourceImageWrap(std::string actualKeyImage, std::string theRealName)//---BORRAR...
	{
		actualInstantWrapping -> setActualSourceImageNamesWrapp(actualKeyImage, theRealName);
	}


	/*
	* Sets the actual instant and manage the search of the corresponding elements with the specified instant in all the enviornments
	* @param newActualInstantData Is the instant data
	*/
	void OutlineModelManager :: setInstant(Instant * newActualInstantData)
	{
		actualInstant = newActualInstantData;
		updateToActualInstant();
	}

	Instant* OutlineModelManager :: getInstant()
	{
		return actualInstant;
	}

	/*
	* Gets the an axe with a given keyName
	* @axeKName The axe keyName for searching in the axes environment
	* @return The corresponding axe
	*/
	AxeThing * OutlineModelManager :: getAxeByKeyName(std::string axeKName)
	{
		return *axesEnvironment->getThingByName(axeKName);
	}

	/*
	* Gets the instants of a specific outline
	* @param thekName Is the name of the outline
	* @return The instants set
	*/
	std::vector<Instant *> OutlineModelManager :: getOutlineInstantsByName(std::string thekName)
	{
		return *outlinesEnvironment->getInstantsOfThing(thekName);
	}

	/*
	* Gets all instants outlines
	* @return The instants set
	*/
	std::vector<Instant *> OutlineModelManager :: getOutlineInstants()
	{
		return *outlinesEnvironment->getExistingInstants();
	}



	/*
	* Includes an instant to the specified axe
	* @param outKName Is outline key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool OutlineModelManager :: includeOutlineInstant(std::string outKName,Instant * anInstantData)
	{
		return outlinesEnvironment->addInstantToThing(outKName,anInstantData);
	}

	/*
	* Includes an instant to the specified axe
	* @param axeKName Is axe key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool OutlineModelManager :: includeAxeInstant(std::string axeKName,Instant * anInstantData)
	{
		return axesEnvironment->addInstantToThing( axeKName,anInstantData );
	}

	/*
	* Includes an instant to the specified image section
	* @param imaKName Is the image section key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool OutlineModelManager :: includeImageSectionInstant(std::string imaKName,Instant * anInstantData)
	{
		return imagesSectionEnvironment->addInstantToThing( imaKName,anInstantData );
	}

	/*
	* Includes an instant to the specified image source
	* @param imaKName Is the image section key name
	* @param anInstantData Is the instant data
	* @return Returns if the insertion was successful or not
	*/
	bool OutlineModelManager :: includeImageSourceInstant(std::string imaKName,Instant * anInstantData)
	{
		return imageSourceEnvironment->addInstantToThing( imaKName,anInstantData );
	}

	/*
	*	Method that retorns the name of each concept and the size of it.
	*	@param conceptNameVect, Vector in which is disposed to be setted the name for each of the included concepts
	*	@param conceptSizeVect, Vector in which is disposed to be setted the size for each of the included concepts
	*/
	void OutlineModelManager :: getConceptsInformation(std::vector<std::string>& conceptNameVect, std::vector<int>& conceptSizeVect)
	{
		if( outlinesEnvironment != NULL )
		{
			//if (outlinesEnvironment->getNumberOfThings()>0)
				outlinesEnvironment->getConceptsInformation(conceptNameVect, conceptSizeVect);
		}
	}


	/*
	* Gets the contourWorspace
	* @return Returns the workspace
	*/
	ContourWorkspace * OutlineModelManager :: getContourWorkspace()
	{
		return workSpace;
	}


	/*
	* Cleans the outline model manager and its dependencies
	*/
	void OutlineModelManager :: clean()
	{
		/*axesEnvironment->clean();
		imagesSectionEnvironment->clean();
		imageSourceEnvironment->clean();
		outlinesEnvironment->clean();*/
		outlineGroups.clear();
		actualInstantWrapping->clean();
		keyGenerator.clear();
		//workSpace->clear();
	}

	/*
	* Update the registered objects in the InstantMemebersNameList, is the one that really changes the instant in the model
	*/
	void OutlineModelManager :: updateToActualInstant()
	{
		Instant * longInstant = actualInstant;
		/*Instant * mediumInstant = new Instant();
		Instant * shortInstant = new Instant();
		std::vector<int>* theInstant = longInstant->getInstant();
		for(int i=1; i<theInstant->size(); i++)
		{
			if (i>=2)
				shortInstant->addConcept( (*theInstant)[i] );
			mediumInstant->addConcept( (*theInstant)[i] );
		}

		//getting the sourceAtInstant
		std::vector<std::string> kSourceVector;
		std::vector<ImageSourceThing **> imSourceVector;
		imageSourceEnvironment->getThings(kSourceVector, imSourceVector, shortInstant);
		annotateActualSourceImageWrap(kSourceVector[0], (**imSourceVector[0]).getSourceImage());

        //getting the aAxeAtInstant
		std::vector<std::string> kAxeVector;
		std::vector<AxeThing **> axesVector;
		axesEnvironment->getThings(kAxeVector, axesVector, mediumInstant);
		if ( !kAxeVector.empty() )
			annotateActualAxeWrap(kAxeVector[0], (**axesVector[0]).getDescription());

		//getting the sectionAtInstant
		std::vector<std::string> kSectionVector;
		std::vector<ImageSectionThing **> imSectionVector;
		imagesSectionEnvironment->getThings(kSectionVector, imSectionVector, longInstant); if ( !kSectionVector.empty() )
	    //annotateActualSectionImageWrap(kSectionVector[0], (**imSectionVector[0]).getImageData());
		*/

		//getting the outlines

		std::vector<ContourThing ** > vect = getOutlinesAtInstant( longInstant );

	}

	/*
	* Sets the automatic managed concepts including them in the environments. That are at the beginning of the instant vector for the corresponding environments.
	* @return Returns true if successful insert of the automatic concepts, false otherwise
	*/
	bool OutlineModelManager :: setAutomaticConcepts()
	{
		axeConcept = "Axe";
		axeDepthConcept = "Axe Depth";
		int axeC_size = 10;
		int axeDepthC_size = INT_MAX;

		bool allOK = axesEnvironment->addConcept(axeConcept, axeC_size);
		allOK &= imagesSectionEnvironment->addConcept(axeConcept, axeC_size);
		allOK &= outlinesEnvironment->addConcept(axeConcept, axeC_size);

		allOK &= imagesSectionEnvironment->addConcept(axeDepthConcept, axeDepthC_size);
		allOK &= outlinesEnvironment->addConcept(axeDepthConcept, axeDepthC_size);
/*
		allOK &= keyGenerator.addKeyThing("Axes", "Axe");
		allOK &= keyGenerator.addKeyThing("Image Sources", "Image source");
		allOK &= keyGenerator.addKeyThing("Image Sections", "Image section");
		allOK &= keyGenerator.addKeyThing("Outlines", "Outline");*/

		return allOK;
	}


	std::string OutlineModelManager :: createCopyContourOf( std::string anExistingKName, std::vector<int> &instantNoTouchData )
	{
		manualBaseModel * manModelContour  = getOutlineByKeyName( anExistingKName )->getModel()->Clone();
		return createOutline( manModelContour, instantNoTouchData );
	}

	void OutlineModelManager::SaveThingName( FILE *pFile, FILE *pFileData, std::string nameThing )
	{
		std::vector< Instant * > *lstInstants;
		lstInstants = outlinesEnvironment->getInstantsOfThing( nameThing );
		Instant *instant = (*lstInstants)[0];
		std::vector<int> *vecInst =instant->getInstant();
		int i,sizeVecInst = vecInst->size();
		fprintf(pFile,"Instant ");
		for ( i=0 ; i<sizeVecInst ; i++ )
		{
			fprintf(pFile,"%d ",(*vecInst)[i]);
		}
		fprintf(pFile,"\n");
		ContourThing *contourthing = *outlinesEnvironment->getThingByName(nameThing);
		contourthing->getModel()->Save(pFile);
		contourthing->getModel()->SaveData(pFileData);
	}

