/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


//----------------------------------------------------------------------------------------------------------------
// Class definition include
//----------------------------------------------------------------------------------------------------------------
#include "OsirixParser.h"
#include <stdio.h>


OsirixParser::OsirixParser(std::string xsdfile, double* spacing, int* extent)
{
	schema=xsdfile;

	OSIRIX_DICT = "dict";
	OSIRIX_KEY = "key";
	OSIRIX_IMAGES = "Images";
	OSIRIX_ARRAY = "array";
	OSIRIX_IMAGEINDEX = "ImageIndex";
	OSIRIX_INTEGER = "integer";
	OSIRIX_NUMBEROFROIS = "NumberOfROIs";
	OSIRIX_ROIS = "ROIs";
	OSIRIX_POINT_MM = "Point_mm";
	OSIRIX_POINT_PX = "Point_px";
	OSIRIX_STRING = "string";
	OSIRIX_NAME = "Name";

	if(spacing != NULL){
		_spacing[0] = spacing[0];
		_spacing[1] = spacing[1];
		_spacing[2] = spacing[2];
	}else{
		_spacing[0] = 1;
		_spacing[1] = 1;
		_spacing[2] = 1;
	}

	if(extent != NULL){
		_extent[0] = extent[0];
		_extent[1] = extent[1];
		_extent[2] = extent[2];
		_extent[3] = extent[3];
		_extent[4] = extent[4];
		_extent[5] = extent[5];
		
	}else{
		_extent[0] = 1;
		_extent[1] = 1;
		_extent[2] = 1;		
		_extent[3] = 1;
		_extent[4] = 1;
		_extent[5] = 1;
	}


	/*HEADER*/
	CREACONTOUR = "--CreaContour--";
	CREACONTOUR_VERSION = "Version 1.0.2";
	CREACONTOUR_IMAGEDIMENSIONS = "ImageDimensions";
	CREACONTOUR_IMAGESPACING = "ImageSpacing";
	CREACONTOUR_NUMBEROFCONTOURS = "NumberOfContours";
	/*FOR EACH CONTOUR*/
	CREACONTOUR_INSTANT = "Instant";
	CREACONTOUR_TYPEMODEL = "TypeModel";
	CREACONTOUR_NUMBEROFCONTROLPOINTS = "NumberOfControlPoints";
	CREACONTOUR_TYPEVIEW = "TypeView";	
	/*STATICCONTOURS*/
	CREACONTOUR_NUMBEROFCONTOURSSTATIC = "NumberOfContoursStatic";

        TEMPIMPORTOSIRIXFILE = "";

#ifdef WIN32

	char currentPath[_MAX_PATH];	
	GetModuleFileName(NULL, currentPath, _MAX_PATH);
	TEMPIMPORTOSIRIXFILE = currentPath;

	TEMPIMPORTOSIRIXFILE = TEMPIMPORTOSIRIXFILE.substr(0,TEMPIMPORTOSIRIXFILE.find_last_of("\\"));

	TEMPIMPORTOSIRIXFILE.append("\\data\\TEMPIMPORTOSIRIXFILE.roi");
	
#else
        char * pPath;
        pPath = getenv ("HOME");

        if(pPath){
            TEMPIMPORTOSIRIXFILE.append(pPath);
        }else{
            TEMPIMPORTOSIRIXFILE.append(".");
        }
        TEMPIMPORTOSIRIXFILE.append("/.creaContourDataTemp/TEMPIMPORTOSIRIXFILE.roi");
#endif
	
}
OsirixParser::~OsirixParser(){

}

int OsirixParser::ParseFile(const char* xmlfile){

	xercesc::DOMDocument *doc;


	try {
            XMLPlatformUtils::Initialize();
        }
        catch (const XMLException& toCatch) {
            char* message = XMLString::transcode(toCatch.getMessage());
            cout << "Error during initialization! :\n"
                 << message << "\n";
            XMLString::release(&message);
            return 1;
        }

        XercesDOMParser* OsirixParser = new XercesDOMParser();


		OsirixParser->setDoNamespaces(true);
		OsirixParser->setDoSchema(true);
		OsirixParser->setValidationScheme(XercesDOMParser::Val_Always );
		OsirixParser->setExternalNoNamespaceSchemaLocation(XMLString::transcode(schema.c_str()));
		OsirixParser->setValidationSchemaFullChecking(true);
		OsirixParser->setValidationConstraintFatal(true);
		OsirixParser->setExitOnFirstFatalError(true);


        ErrorHandler* errHandler = (ErrorHandler*) new OsirixParserErrorHandler();

        OsirixParser->setErrorHandler(errHandler);


        try {
            OsirixParser->parse(xmlfile);


			if(OsirixParser->getErrorCount() > 0){

				errorline = ((OsirixParserErrorHandler*)OsirixParser->getErrorHandler())->getErrorMsg();


				delete OsirixParser;
				delete errHandler;
				return -2;
			}



        }
        catch (const XMLException& toCatch) {
            char* message = XMLString::transcode(toCatch.getMessage());
            cout << "Exception message is: \n"
                 << message << "\n";
            XMLString::release(&message);
			delete OsirixParser;
		    delete errHandler;
            return -1;
        }
        catch (const DOMException& toCatch) {
            char* message = XMLString::transcode(toCatch.msg);
            cout << "Exception message is: \n"
                 << message << "\n";
            XMLString::release(&message);
			delete OsirixParser;
			delete errHandler;
            return -1;
        }
        catch (...) {
            cout << "Unexpected Exception \n" ;
			delete OsirixParser;
			delete errHandler;
            return -1;
        }

		std::cout<<"parsing document..."<<std::endl;


        doc = OsirixParser->getDocument();
//		DOMNodeList* list = doc->getChildNodes();		
		getUserData(doc->getDocumentElement());


		saveCreaContoursFormat();

        delete OsirixParser;
        delete errHandler;



		return 0;

}

void OsirixParser::setErrorLine(DOMNodeList* list){

	DOMNode* node = list->item(0);
	this->errorline =	XMLString::transcode(node->getTextContent());

}
DOMNode* OsirixParser::getLastNode(DOMNodeList* list){


	DOMNode* node;
	/*for(int i = list->getLength()-1; i >= 0 ;i--){

		node = list->item(i);

		if(node->getNodeType() == DOMNode::ELEMENT_NODE){
			i = -1;
		}

	}*/
	node = list->item(list->getLength()-1);



	if(node->getChildNodes()->getLength()>0){
		return getLastNode(node->getChildNodes());
	}
	return node;

}

void OsirixParser::getUserData(DOMElement* element){
	parseOSIRIX_DICT(element->getElementsByTagName(XMLString::transcode(OSIRIX_DICT)));
}

void OsirixParser::parseOSIRIX_DICT(DOMNodeList* list){
	int i, j;
	DOMNode* node, *childnode, *childnode1, *childarray;
	std::string point_mm, point_px, osirixinteger, imageindex, temp;
	DOMNodeList* childlist;
	point_mm = OSIRIX_POINT_MM;
	point_px = OSIRIX_POINT_PX;
	imageindex = OSIRIX_IMAGEINDEX;
	osirixinteger = OSIRIX_INTEGER;
	
	std::string osirixstring = OSIRIX_STRING;
	std::string osirixname = OSIRIX_NAME;
	

	for(i = 0; i < (int)(list->getLength()); i++){
		node = list->item(i);		
		childlist = node->getChildNodes();
		for(j = 0; j < (int)(childlist->getLength());j++){
			childnode = childlist->item(j);			
			temp = XMLString::transcode(childnode->getTextContent());	


			if(point_mm.compare(temp)==0){				
				childarray = childlist->item(j+2);
				//temp = XMLString::transcode(childarray->getNodeName());					
				if(childarray != 0){					
					parseOSIRIX_POINT_MM(childarray->getChildNodes());					
				}				
			}else if(point_px.compare(temp)==0){				
				childarray = childlist->item(j+2);
				//temp = XMLString::transcode(childarray->getNodeName());					
				if(childarray != 0){					
					parseOSIRIX_POINT_PX(childarray->getChildNodes());					
				}				
			}else if(imageindex.compare(temp) == 0){
				childnode1 = childlist->item(j+2);
				if(childnode1 != NULL && osirixinteger.compare(XMLString::transcode(childnode1->getNodeName())) == 0){
					_imageindex = atoi(XMLString::transcode(childnode1->getTextContent()));						
				}				
			} else if(osirixname.compare(temp) == 0) {
			  // keep information about the name of the ROI
			        childnode1 = childlist->item(j+2);
				if(childnode1 != NULL && osirixstring.compare(XMLString::transcode(childnode1->getNodeName())) == 0){
				  char* roiname = XMLString::transcode(childnode1->getTextContent());
				  _roiname = string(roiname);
				}
			}
		}		
	}
}

void OsirixParser::parseOSIRIX_POINT_MM(DOMNodeList* list){
	int i, stringfound0, stringfound1;
	DOMNode* node;
	string temp, osirix_string, numx, numy, numz;
	vector<double>* vectorx;
	vector<double>* vectory;
	vector<double>* vectorz;
	vectorXYZ vectorxyz;
	double x, y, z;	

	vectorx = new vector<double>;
	vectory = new vector<double>;
	vectorz = new vector<double>;

	osirix_string = OSIRIX_STRING;

	for(i = 0; i < (int)(list->getLength()); i++){
		node = list->item(i);
		if(osirix_string.compare(XMLString::transcode(node->getNodeName()))==0){
			temp = XMLString::transcode(node->getTextContent());			

			stringfound0 = temp.find(",",0);
			numx = temp.substr(1, stringfound0-1);

			stringfound1 = temp.find(",",stringfound0+1);
			numy = temp.substr(stringfound0+1, stringfound1-stringfound0-1);

			stringfound0 = temp.find(")",stringfound1+1);
			numz = temp.substr(stringfound1+1, stringfound0-stringfound1-1);

			x = atof(numx.c_str());		
			y = atof(numy.c_str());		
			z = atof(numz.c_str());		

			vectorx->push_back(x);
			vectory->push_back(y);
			vectorz->push_back(z);
		}		
	}
	if(vectorx->size() > 0){
		vectorxyz.push_back(*vectorx);
		vectorxyz.push_back(*vectory);
		vectorxyz.push_back(*vectorz);
		contoursmapMM.insert(pair<int, vectorXYZ>(contoursmapMM.size(), vectorxyz));
		contoursnameMM.insert(pair<int, string>(contoursnameMM.size(), _roiname));
	}
}

void OsirixParser::parseOSIRIX_POINT_PX(DOMNodeList* list){
	int i, stringfound0, stringfound1;
	DOMNode* node;
	string temp, osirix_string, numx, numy;
	vector<double>* vectorx;
	vector<double>* vectory;	
	vector<double>* vectorz;	
	vectorXYZ vectorxyz;
	double x, y, z;	

	vectorx = new vector<double>;
	vectory = new vector<double>;	
	vectorz = new vector<double>;	

	osirix_string = OSIRIX_STRING;

	for(i = 0; i < (int)(list->getLength()); i++){
		node = list->item(i);
		if(osirix_string.compare(XMLString::transcode(node->getNodeName()))==0){
			temp = XMLString::transcode(node->getTextContent());			

			stringfound0 = temp.find(",",0);
			numx = temp.substr(1, stringfound0-1);					

			stringfound1 = temp.find(")",stringfound0+1);
			numy = temp.substr(stringfound0+1, stringfound1-stringfound0-1);

			x = atof(numx.c_str());		
			y = atof(numy.c_str());					

			vectorx->push_back(x);
			vectory->push_back(y);	
			vectorz->push_back(_imageindex);
		}		
	}
	if(vectorx->size() > 0){
		vectorxyz.push_back(*vectorx);
		vectorxyz.push_back(*vectory);	
		vectorxyz.push_back(*vectorz);	
		contoursmapPX.insert(pair<int, vectorXYZ>(contoursmapPX.size(), vectorxyz));
		contoursnamePX.insert(pair<int, string>(contoursnamePX.size(), _roiname));
	}	
}
		


void OsirixParser::getData(DOMNodeList* list, std::vector<std::string>& vect, std::string tagname){

	for(int i = 0; i < (int)(list->getLength()); i++){
		DOMNode* node = list->item(i);
		if(tagname.compare(XMLString::transcode(node->getNodeName()))==0){
			std::cout<<"NODENAME "<<XMLString::transcode(node->getTextContent())<<std::endl;
			vect.push_back(XMLString::transcode(node->getTextContent()));
		}

	}
}

void OsirixParser::saveCreaContoursFormat(){
	FILE *pFile=fopen(TEMPIMPORTOSIRIXFILE.c_str(),"w+");

	if(pFile){
		writeHeader(pFile);
		writeContours(pFile);
		writeContoursStatic(pFile);
		fclose(pFile);
	}
}

void OsirixParser::writeContoursStatic(FILE* pFile){
	fprintf(pFile, CREACONTOUR_IMAGEDIMENSIONS);
	fprintf(pFile, " %d %d %d\n", _extent[1] - _extent[0],_extent[3] - _extent[2], _extent[5] - _extent[4]);
	fprintf(pFile, CREACONTOUR_IMAGESPACING);
	fprintf(pFile, " %f %f %f\n", _spacing[0], _spacing[1], _spacing[2]);
	fprintf(pFile, CREACONTOUR_NUMBEROFCONTOURSSTATIC);
	fprintf(pFile, " 0\n");
}

void OsirixParser::writeContours(FILE* pFile){
	
	map<int, vectorXYZ>::iterator itPX;
	vector<double> vectx, vecty, vectz;
	int i, valuez;
	int dimz = 0, dimy = 0;

	if(_extent != 0){
		dimz = _extent[5] - _extent[4] + 1;
		dimy = _extent[3] - _extent[2] + 1;
	}

	/*for (itMM = contoursmapMM.begin(),  itPX = contoursmapPX.begin(); 
			itMM != contoursmapMM.end(), itPX != contoursmapPX.end(); 
			itMM++, itPX++ ){*/
	for (itPX = contoursmapPX.begin(); itPX != contoursmapPX.end(); itPX++ ){

		vectx = ((*itPX).second)[0];
		vecty = ((*itPX).second)[1];
		vectz = ((*itPX).second)[2];

		if(!vectz.empty()){

			valuez = (int) vectz[0] / _spacing[2];

			fprintf(pFile, CREACONTOUR_INSTANT);
			fprintf(pFile, " 1 %d 1 1 1 1\n",dimz - valuez);
			fprintf(pFile, CREACONTOUR_TYPEMODEL);
			fprintf(pFile, " 1\n");
			fprintf(pFile, CREACONTOUR_NUMBEROFCONTROLPOINTS);
			fprintf(pFile, " %d\n", vectz.size());
			for(i = 0; i < vectx.size(); i++){
			  fprintf(pFile, "%f %f 900.00\n", vectx[i]/ _spacing[0], dimy - vecty[i]/ _spacing[1]);
			  // fprintf(pFile, "%f %f %f\n", vectx[i] , vecty[i], vectz[i]);
			}		
			fprintf(pFile, CREACONTOUR_TYPEVIEW);
			fprintf(pFile, " 1\n");
			
		}
	}
}

void OsirixParser::writeHeader(FILE* pFile){ 
	 

	fprintf(pFile, CREACONTOUR);
	fprintf(pFile, "\n");
	fprintf(pFile, CREACONTOUR_VERSION);
	fprintf(pFile, "\n");
	fprintf(pFile, CREACONTOUR_IMAGEDIMENSIONS);
	fprintf(pFile, " %d %d %d\n", _extent[1] - _extent[0],_extent[3] - _extent[2], _extent[5] - _extent[4]);
	fprintf(pFile, CREACONTOUR_IMAGESPACING);
	fprintf(pFile, " %f %f %f\n", _spacing[0], _spacing[1], _spacing[2]);
	fprintf(pFile, CREACONTOUR_NUMBEROFCONTOURS);
	fprintf(pFile, " %d\n", contoursmapPX.size());



}


std::string OsirixParser::getContoursFileName(){
	return TEMPIMPORTOSIRIXFILE;
}

/**
**	The Error Handler's interface implementation
**/

/**
	** Default Constructor
	**/
OsirixParserErrorHandler::OsirixParserErrorHandler()
: ErrorHandler(){




}
	/**
	** Desctructor by defect
	**/
OsirixParserErrorHandler::~OsirixParserErrorHandler(){
}

void 	OsirixParserErrorHandler::warning (const SAXParseException &exc){
}

void 	OsirixParserErrorHandler::error (const SAXParseException &exc){
	char c[1000];
	errormsg = "Column ";
//	errormsg +=	itoa(exc.getColumnNumber(),c,10);
#ifdef WIN32
	sprintf_s(c,"%d",(int)(exc.getColumnNumber()));
#else
	sprintf(c,"%d",(int)(exc.getColumnNumber()));
#endif
	errormsg +=	std::string(c);

	errormsg += " Line ";
//	errormsg +=	itoa(exc.getLineNumber(),c,10);
#ifdef WIN32
	sprintf_s(c,"%d",(int)(exc.getLineNumber()));
#else
	sprintf(c,"%d",(int)(exc.getLineNumber()));
#endif
	errormsg +=	std::string(c);

	errormsg += " ";
	errormsg += XMLString::transcode(exc.getMessage());

}

void 	OsirixParserErrorHandler::fatalError (const SAXParseException &exc){
	char c[1000];
	errormsg = "Column ";
//	errormsg +=	itoa(exc.getColumnNumber(),c,10);
#ifdef WIN32
	sprintf_s(c,"%d",(int)(exc.getColumnNumber()));
#else
	sprintf(c,"%d",(int)(exc.getColumnNumber()));
#endif
	errormsg +=	std::string(c);

	errormsg += " Line ";
//	errormsg +=	itoa(exc.getLineNumber(),c,10);
#ifdef WIN32
	sprintf_s(c,"%d",(int)(exc.getLineNumber()));
#else
	sprintf(c,"%d",(int)(exc.getLineNumber()));
#endif
	errormsg +=	std::string(c);

	errormsg += " ";
	errormsg += XMLString::transcode(exc.getMessage());
}

void 	OsirixParserErrorHandler::resetErrors (){
}

std::string OsirixParserErrorHandler::getErrorMsg(){
	return this->errormsg;
}
