/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */


//----------------------------------------------------------------------------------------------------------------
// Class definition include
//----------------------------------------------------------------------------------------------------------------
#include "PrefixMaxKeyGenerator.h"

//----------------------------------------------------------------------------------------------------------------
// Class implementation
//----------------------------------------------------------------------------------------------------------------
/** @file PrefixMaxKeyGenerator.cxx */

//------------------------------------------------------------------------------------------------------------
// Constructors & Destructors
//------------------------------------------------------------------------------------------------------------

	/*
	* Creates the prefix+max key generator
	*/
	PrefixMaxKeyGenerator :: PrefixMaxKeyGenerator()
	{
		numberOfKeyThings = 0;		
	}

	/*
	* Destroys the outline manager
	*/
	PrefixMaxKeyGenerator :: ~PrefixMaxKeyGenerator()
	{
		clear();
	}

//------------------------------------------------------------------------------------------------------------
// Public Methods
//------------------------------------------------------------------------------------------------------------

	
	/*
	* Adds a key thing to the keyThings building the respective KeyThing (new keyThing). 
	* @param theName Is the name of the new keyThing to include. If the name is not unique, returns false.
	* @param thePrefix Is the prefix of the new keyThing for the key generation correponding to the new keyThing
	* @param theMax Is the maximum value for the key generation correponding to the new keyThing
	* @return Returns true if the keyThing could be added of not.
	*/
	bool PrefixMaxKeyGenerator :: addKeyThing( std::string theName, std::string thePrefix, int theMax )
	{
		KeyThing * kThing = new KeyThing(thePrefix, theMax);
		int before = keyThings.size();
		keyThings.insert(std::pair < std::string, KeyThing >( theName, *kThing ));
		int after = keyThings.size();		 	
		return after > before;
	}

	/*
	* Remove a key thing  
	* @param theName Is the name of the keyThing to remove. 
	*/
	void PrefixMaxKeyGenerator :: removeKeyThing( std::string theName )
	{
		keyThings.erase(theName);		
	}


	/*
	* Indicates if a key thing existis in the generator
	* @param theName Is the name of the keyThing to search. 
	* @return Returns true if the keyThing exists in the keyThings.
	*/
	bool PrefixMaxKeyGenerator :: existsKeyThing( std::string theName )
	{
		return keyThings.find(theName) != keyThings.end();		
	}

	/*
	* Updates the maximum value of a key thing if necesary (posibleMax>theMaxOfKeyThing). If the key thing doesn't exist nothing is done.
	* @param theName Is the name of the keyThing to update. 	
	* @param posibleMax Is the number that corresponds to a posible max value of the keyThing to update. 
	*/
	void PrefixMaxKeyGenerator :: updateMaxTo( std::string theName, int posibleMax )
	{
		std::map<std::string, KeyThing >::iterator iterP = keyThings.find(theName);
		if ( iterP != keyThings.end() )
		{
			int max = (iterP->second).getValue();
			if ( max < posibleMax )
				(iterP->second).setValue(posibleMax);
		}
	}

	/*
	* Generates a (std::string) key for a given keyThing. If the key thing doesn't exist nothing is done and returns false.
	* @param theName Is the name of the keyThing to search. 
	* @param theInputString Is string to load the generated key formed like <prefixOfTheKeyThing> <maxOfTheKeyThing>
	* @return theKey Returns true if the key was generated successfully. (If theName is an existent keyThing)
	*/
	bool PrefixMaxKeyGenerator :: generateKeyOf( std::string theName, std::string &theInputString )
	{
		theInputString = "";
		std::map<std::string, KeyThing >::iterator iterP = keyThings.find(theName);
		if ( iterP != keyThings.end() )
		{			
			KeyThing kthing = (iterP->second);
			int max = kthing.getValue();			
			std::ostringstream genKey;
			genKey<<kthing.getPrefix() << " " << max;
			theInputString = genKey.str();
			return true;
		}
		return false;
	}

	/*
	* Generates a (std::string) key for a given keyThing and updates the max value of it if necesary. If the key thing doesn't exist nothing is done. 
	* @param theName Is the name of the keyThing to search. 
	* @param posibleMax Is the number that corresponds to a posible max value of the keyThing to update. 
	* @param theInputString Is string to load the generated key formed like <prefixOfTheKeyThing> <maxOfTheKeyThing>
	* @return Returns true if the key was generated successfully. (If theName is an existent keyThing)
	*/
	bool PrefixMaxKeyGenerator :: generateKeyOf( std::string theName,  int posibleMax, std::string &theInputString  )
	{
		theInputString = "";
		std::map<std::string, KeyThing >::iterator iterP = keyThings.find(theName);
		if ( iterP != keyThings.end() )
		{	
			KeyThing kthing = (iterP->second);
			int max = kthing.getValue();

			if ( max < posibleMax )
			{
				kthing.setValue(posibleMax);
			}


			std::ostringstream genKey;
			genKey<<kthing.getPrefix() << " " << kthing.getValue();
			theInputString = genKey.str();
			return true;
		}
		return false;
	}

	/*
	* Gets the prefix of a specific keyThing identified with the name in the parameter, if the key thing doesn't exists return false.
	* @param theName Is the name of the keyThing to search the prefix. 
	* @param theInputString Is string to load the prefix of the searched keyThing.
	* @return isStringOutputReal Returns if the loaded string in theInputString is real (i.e if the -theName- keyThing exists)
	*/
	bool PrefixMaxKeyGenerator :: getPrefixOf( std::string theName, std::string &theInputString )		
	{
		theInputString = "";
		std::map<std::string, KeyThing >::iterator iterP = keyThings.find(theName);
		if ( iterP != keyThings.end() )
		{	
			KeyThing kthing = (iterP->second);
			theInputString = kthing.getPrefix();
			return true;
		}
		return false;
	}

	/*
	* Gets the max value of a specific keyThing identified with the name in the parameter. If the key thing doesn't exists returns -1.
	* @param theName Is the name of the keyThing to search the maximum. 	
	* @return theMax Returns the maxumum value for key generation at the keyThing. Returns -1 if not finded keyThing.
	*/
	int PrefixMaxKeyGenerator :: getCurrentMaxOf( std::string theName )
	{
		std::map<std::string, KeyThing >::iterator iterP = keyThings.find(theName);
		if ( iterP != keyThings.end() )
		{	
			KeyThing kthing = (iterP->second);
			return kthing.getValue();			
		}
		return -1;
	}

	/*
	* Gets the total of keyThings managed
	* @return Retuns the total of keyThing managed
	*/
	int PrefixMaxKeyGenerator :: getTotalThingsNumber()
	{
		return keyThings.size();
	}

	/*
	* Clears the generator deleating the existring keyThings
	*/
	void PrefixMaxKeyGenerator :: clear()
	{		
		std::map<std::string, KeyThing >::iterator iter = keyThings.begin();
		while( iter != keyThings.end() )
		{
			keyThings.erase(iter);			
		}
		keyThings.clear();		
		numberOfKeyThings = 0;
	}
