/*
	# ---------------------------------------------------------------------
	#
	# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
	#                        pour la Sant)
	# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
	# Previous Authors : Laurent Guigues, Jean-Pierre Roux
	# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
	#
	#  This software is governed by the CeCILL-B license under French law and 
	#  abiding by the rules of distribution of free software. You can  use, 
	#  modify and/ or redistribute the software under the terms of the CeCILL-B 
	#  license as circulated by CEA, CNRS and INRIA at the following URL 
	#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
	#  or in the file LICENSE.txt.
	#
	#  As a counterpart to the access to the source code and  rights to copy,
	#  modify and redistribute granted by the license, users are provided only
	#  with a limited warranty  and the software's author,  the holder of the
	#  economic rights,  and the successive licensors  have only  limited
	#  liability. 
	#
	#  The fact that you are presently reading this means that you have had
	#  knowledge of the CeCILL-B license and that you accept its terms.
	# ------------------------------------------------------------------------
*/
#include <boost/program_options.hpp>

using namespace boost;
namespace po = boost::program_options;

#include <creaImageIOGimmick.h>
#include <creaMessageManager.h>

using namespace std;

int main(int ac, char* av[])
{
  
  creaImageIO::Gimmick g;
  int verb,deb;
  std::string handler("Local database");
  
  //Describes first group of options
  po::options_description generic("GENERIC");
  generic.add_options()
    ("help,h", "Print help and exit")
    ("version,V", "Print version and exit");
  
  //Describes second group of options
  po::options_description command("COMMANDS");
  command.add_options()
    ("print,p","Prints the local database tree  (default=off)")
    ("files,f",po::value< vector<string> >(),"Adds the file(s) to the local database")
    ("dir,d",po::value< vector<string> >(),"Adds the directory to the local database")
    ("sync,s",po::value< vector<string> >(),"Synchronizes the local database with the files")
	("copy,c",po::value< vector<string> >(),"Copies the files into a local directory");

  //Describes third group of options
  po::options_description option("OPTIONS");
  option.add_options()
    ("verbose,v",po::value<int>(&verb),"Verbosity level (default=`1')")
    ("debug,D",po::value<int>(&deb),"Debug messages level (default=`0')")
	("recurse,r","Recurse into sub-directories  (default=off)")
  	("repair,R","Repair the database (on synchronization) (default=off)")
	("check,C","Check for attribute differences (on synchronization) (default=off)")
	("handler,H",po::value<string>(&handler),"Handler name (default=`Local database')");
 
  //Adds the groups into a big one
  po::options_description cmdline_options;
  cmdline_options.add(generic).add(command).add(option);
  
  //Adds the corresponding variables
  po::variables_map vm;
  po::store(po::parse_command_line(ac, av, cmdline_options), vm);
  po::notify(vm);    
  
  
  //Does something on each option
  //GENERIC
  if (vm.count("help")) {
    cout << "Usage: gimmick COMMAND [OPTIONS]:\n";
    cout << cmdline_options << "\n";
  }
  if (vm.count("version")) {
    cout << "gimmick 0.1.0\n";
  }
  
  //OPTIONS
  if (vm.count("verbose")) {
    g.SetMessageLevel(verb);
    cout << "Verbose level is now "<<verb<<"\n";
  }
  if (vm.count("debug")) {
    g.SetDebugMessageLevel(deb);
    cout << "Debug level is now "<<deb<<"\n";
  }
  
  //COMMANDS
  if (  vm.count("print")
	||vm.count("file")
	||vm.count("dir")
	||vm.count("sync")
	||vm.count("copy")
	)
  {

    try
      {
	g.Initialize();
	if (vm.count("print")) {
	  g.GetTreeHandler(handler)->LoadChildren(0,0);
	  g.Print(handler);
	}
	if (vm.count("file")) {
	  std::vector<std::string> files=vm["file"].as< vector<string> >();
	  g.AddFiles(handler,files);
	}
	if (vm.count("dir")) {
	  std::vector<std::string> dirs=vm["dir"].as< vector<string> >();
	  bool recu=false;
	  std::stringstream out;
	  out<<vm.count("recurse");
	  if(out.str().compare("1")==0){recu=true;}
	  g.AddDir(handler,dirs.front(), recu);
	}
	if (vm.count("sync")) {
	  bool rep=false;
      bool chk=false;
	  std::vector<std::string> name=vm["sync"].as< vector<string> >();
	  std::stringstream out;
	  out<<vm.count("repair");
	  if(out.str().compare("1")==0){rep=true;}
	  std::stringstream out2;
	  out2<<vm.count("check");
	  if(out2.str().compare("1")==0){chk=true;}
	  cout<<g.Synchronize(name.front(),rep,chk)<<"\n";
	}	
	if (vm.count("copy")) {
	  std::vector<std::string> name=vm["copy"].as< vector<string> >();
	  g.CopyFiles(name,handler);
	}
	g.Finalize();
      }
    catch (crea::Exception e)
      {
	e.Print();
      }
  }
	  
  
  return 0;
}
