/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#include <creaImageIOGimmick.h>

#include <boost/filesystem.hpp>
#include <boost/algorithm/string.hpp>

//#include "io.h"
#ifndef PATH_MAX // If not defined yet : do it 
#  define PATH_MAX 2048
#endif
#include <creaImageIOGimmick.h>

#if defined(_WIN32)
#pragma warning(disable: 4996)
#endif

#ifdef _DEBUG
#define new DEBUG_NEW
#endif


namespace creaImageIO
{
  //==============================================================
  Gimmick::Gimmick()
    : mImageAdder(0)
  {    
  RegisterGimmickMessageTypes();
	mSettings=0;
	mSynchronizer=0;
	mLocalDescpName = "localdatabase_Descriptor.dscp";
	mLocalDBName = "Local database";
  }
  //==============================================================


  //==============================================================
  Gimmick::~Gimmick()
  {

	 if(mSettings!=0)
	  {
	  	mSettings->writeSettingsFile();
		delete mSettings;
	  }
	if(mSynchronizer!=0)
	  {
	  	delete mSynchronizer;
	  }
  }
  //==============================================================
  
  //==============================================================
  void Gimmick::Initialize(const std::string i_namedescp, const std::string i_namedb)
  {
	  mLocalDescpName = i_namedescp;
	  mLocalDBName = i_namedb;
	  Initialize();
  }

  //==============================================================
  void Gimmick::Initialize()
  {
	std::string i_nameDB = mLocalDBName;
    // Create the UserSettings dir if does not exist
    CreateUserSettingsDirectory();
    // Sets the current directory to the home dir
    mCurrentDirectory =  GetHomeDirectory();
    mSynchronizer= new Synchronizer(GetUserSettingsDirectory()+"share/creaImageIO/");

    mSettings  = new Settings(mCurrentDirectory);
	
    std::string dbpath = GetLocalDatabasePath();

    // Create or open local database
    std::string dpath= mCurrentDirectory + "/.creaImageIO/share/creaImageIO/" + mLocalDescpName;
	
    boost::algorithm::replace_all( dpath,
				   INVALID_FILE_SEPARATOR , 
				   VALID_FILE_SEPARATOR);
    mLocalDatabase = createDB(i_nameDB, dpath, dbpath);
    // Add it to the TreeHandlerMap
    mTreeHandlerMap[i_nameDB] = mLocalDatabase;
    
    //Add additional DB from user Settings
    addDBSettings();	
  }

   ///////////////////////////////////////////////////////////////////////
   // add DB to TreeHandler Map						//
   // @param i_name : DB name						//
   // @param i_location : DB location					//
   // return : -							//
  ////////////////////////////////////////////////////////////////////////
 void Gimmick::addDB(const std::string &i_name, 
		     const std::string &i_location)
	{
		if(mTreeHandlerMap.find(i_name) == mTreeHandlerMap.end())
		{
			mTreeHandlerMap[i_name] = new SQLiteTreeHandler(i_location);
			mTreeHandlerMap[i_name]->Open(true);
			mSettings->addDB(i_location);
		}
	}

  ///////////////////////////////////////////////////////////////////////////
  // create a DB from a attributes descriptor file for medical images      //
  // @param i_name : DB name											   //
  // @param i_locDesc : location of descriptor file						   //
  // @param i_locDB : location of DB									   //
  // return : the SQLiteTreeHandler object on DB						   //
	/////////////////////////////////////////////////////////////////////////
	SQLiteTreeHandler* Gimmick::createDB(const std::string &i_name,
					     const std::string &i_locDesc,
					     const std::string &i_locDB)
  {
     SQLiteTreeHandler* sqlTreeH( new SQLiteTreeHandler(i_locDB) );
    // Create or open local database
    if (! boost::filesystem::exists(i_locDB) )
     {
         std::string mess = "Local database '";
         mess += i_locDB;
         mess += "' does not exist : creating it";
         GimmickMessage(1,mess<<std::endl);
         
		 // CREATING DB STRUCTURE
         sqlTreeH->GetTree().GetDescriptor().createDescriptorfromFile(i_locDesc);
         if ( ! sqlTreeH->Create(true) )
	 {
		GimmickError("ERROR CREATING '"<<i_locDB<<"'");
         }
         sqlTreeH->SetAttribute(0,"Name",i_name);
	 }
	 else 
	 {
		/// Open and test it
		
		GimmickDebugMessage(1,"Opening local database '" <<i_locDB<< "' " << std::endl);
        	if ( !sqlTreeH->Open(true) )
		{
			GimmickError("ERROR OPENING '"<<i_locDB<<"'");
		}
	}
	return sqlTreeH;
  }

  //==============================================================
  void Gimmick::Finalize()
  {
	  if(mTreeHandlerMap.size() >0)
	  {
		// delete SQLiteTreeHandler Object
		for( TreeHandlerMapType::const_iterator it = mTreeHandlerMap.begin();
													it!= mTreeHandlerMap.end(); 
							++it)
		{
			delete it->second;
		}
	  }
  }
  //==============================================================

  //================================================================
  // file separator
#if defined(_WIN32)
#define VALID_FILE_SEPARATOR "\\"
#define INVALID_FILE_SEPARATOR "/"
#else
#define INVALID_FILE_SEPARATOR "\\"
#define VALID_FILE_SEPARATOR "/"
#endif
  //================================================================

  //================================================================
  const std::string& Gimmick::GetHomeDirectory()
  {
    if (mHomeDirectory.size()==0) 
      {
#if defined(__GNUC__)
	mHomeDirectory = getenv("HOME");
#elif defined(_WIN32)
	mHomeDirectory = getenv("USERPROFILE");
#endif
      }
    return mHomeDirectory;
  }
  //================================================================
  const std::string& Gimmick::GetUserSettingsDirectory()
  {
    if (mUserSettingsDirectory.size()==0) 
      {
	mUserSettingsDirectory = GetHomeDirectory();
	mUserSettingsDirectory += "/.creaImageIO/";
	boost::algorithm::replace_all( mUserSettingsDirectory, 
				       INVALID_FILE_SEPARATOR , 
				       VALID_FILE_SEPARATOR);
      }
    return mUserSettingsDirectory;
  }
  //================================================================


  //================================================================
  const std::string& Gimmick::GetLocalDatabasePath()
  {
    if (mLocalDatabasePath.size()==0) 
      {
	mLocalDatabasePath = GetUserSettingsDirectory();
	mLocalDatabasePath += "share/creaImageIO/";
	mLocalDatabasePath += mLocalDBName;
	mLocalDatabasePath +=".sqlite3";
	boost::algorithm::replace_all( mLocalDatabasePath,
				       INVALID_FILE_SEPARATOR , 
				       VALID_FILE_SEPARATOR);
      }
    return mLocalDatabasePath;    
  }

  //========================================================================

  //========================================================================
  void Gimmick::CreateUserSettingsDirectory()
  {
	  
	 // std::string st("C:/Documents and Settings/cervenansky/.gimmick/");
	 // 	boost::algorithm::replace_all( st, 
		//		       INVALID_FILE_SEPARATOR , 
		//		       VALID_FILE_SEPARATOR);
		//const boost::filesystem::path mpath(st);
//C:\Documents and Settings\cervenansky\.gimmick");
	  //if ( !boost::filesystem::exists( path ) ) 		  return ;
	 //  boost::filesystem::directory_iterator end_itr; // default construction yields past-the-end
	 //  for ( boost::filesystem::directory_iterator itr( path );  itr != end_itr;  ++itr )
	 // {
		//// If is directory & recurse : do recurse
		//	if ( boost::filesystem::is_directory(itr->status()) )
		//	return;
	 //  }

	  //JCP settings dir 02/10/2009
	  const std::string settingsdirectory = GetUserSettingsDirectory();
	  	//boost::algorithm::replace_all( mUserSettingsDirectory, 
				//       INVALID_FILE_SEPARATOR , 
				//       VALID_FILE_SEPARATOR);
;//("E:\frederic");
		  //("C:\\Documents and Settings\\cervenansky\\.gimmick\\"); // settingsdirectory );
		bool isdir = false;
	   isdir = boost::filesystem::is_directory(settingsdirectory); // settingsdirectory );
    if (! isdir )
      {
	GimmickMessage(1,"Directory '"<<GetUserSettingsDirectory()<<"' "
 		       << "does not exist : creating it"<<std::endl);
	
	if ( ! boost::filesystem::create_directory( GetUserSettingsDirectory() ) )
	  {
	    GimmickError("ERROR CREATING '"<<GetUserSettingsDirectory()<<"'");
	  }
      }

	std::string setDir=GetUserSettingsDirectory();
	boost::algorithm::replace_all( setDir,
				       INVALID_FILE_SEPARATOR , 
				       VALID_FILE_SEPARATOR);
	setDir+="share/";
	boost::filesystem::create_directory( setDir );
	setDir+="creaImageIO/";
	boost::filesystem::create_directory( setDir );
	setDir+=mLocalDescpName;

	if(!boost::filesystem::is_regular(setDir))
	{
		char name[PATH_MAX];
		crea::System::GetAppPath(name,PATH_MAX);
		std::cout<<name<<std::endl;
		
		std::string path=name;
		path=path.substr(0,path.size()-1);
		path=path.substr(0,path.find_last_of("/"));
		//Creating directories

// The following stuff works on Linux, NOT CHECKED on Windows // JPR
		
#if defined(_WIN32)		
		path+="/bin/share/creaImageIO/";
#endif

#if defined (LINUX)
		path+="/../share/creaImageIO/";
#endif	
#if defined(__APPLE__)
		path+="/../../../../share/creaImageIO/";
#endif 


path+= mLocalDescpName;
		
		std::cout <<"From: " << path   <<std::endl;
		std::cout <<"To: "   << setDir <<std::endl;
		boost::algorithm::replace_all(	path,
						INVALID_FILE_SEPARATOR , 
						VALID_FILE_SEPARATOR);
		boost::filesystem::copy_file(path,setDir);
	}
	  
  }
  //========================================================================


  //========================================================================
  /// Sets message level
  void Gimmick::SetMessageLevel(int l)
  {
    SetGimmickMessageLevel(l);
  }
  //========================================================================

  //========================================================================
  /// Sets message level
  void Gimmick::SetDebugMessageLevel(int l)
  {
    SetGimmickDebugMessageLevel(l);
  }
  //========================================================================

  //========================================================================
  /// Returns the tree handler with the given name
  TreeHandler* Gimmick::GetTreeHandler(const std::string& name) const 
  {  
    TreeHandlerMapType::const_iterator i;
    i = GetTreeHandlerMap().find(name);
    if ( i == GetTreeHandlerMap().end() )
      {
	GimmickError("TreeHandler '"<<name<<"' does not exist");
      }
    return i->second;
  }

  //========================================================================
  /// Add the files to the tree handler
  void Gimmick::AddFiles(const std::string& d, 
			const std::vector<std::string>& filenames)
  {
    GimmickMessage(2,"Adding files to '"<<d<<"'"<<std::endl);
 
	mImageAdder.SetCurrentDatabase(d);
	mImageAdder.SetTreeHandler(GetTreeHandler(d));
	mImageAdder.SetSynchronizer(mSynchronizer);
	mImageAdder.AddFiles(filenames);	
  }
  //========================================================================

  //========================================================================
  /// Add a dir to the local database
  void Gimmick::AddDir(const std::string& d, const std::string& f, 
		       bool recurse)
  {
	GimmickMessage(2,"Adding dir '"<<f<<"' to '"<<d<<"' recurse:"
		  	 <<recurse<<std::endl);

	TreeHandler * handler=GetTreeHandler(d);
	mImageAdder.SetCurrentDatabase(d);
	mImageAdder.SetTreeHandler(handler);
	mImageAdder.SetSynchronizer(mSynchronizer);
	mImageAdder.AddDirectory(f,recurse);  
  }

  //========================================================================

  //========================================================================
  void Gimmick::RemoveFile(const std::string& d, 
			   tree::Node* node)
  {
	  mImageAdder.SetCurrentDatabase(d);
	  mImageAdder.SetSynchronizer(mSynchronizer);
	  mImageAdder.RemoveFile(node);
  }
  //========================================================================

  //void Gimmick::Anonymize(std::vector<std::string>i_files)  {  }
  //========================================================================
 
  void Gimmick::CopyFiles(const std::vector<std::string>& filenames, const std::string& d )
  {
	  TreeHandler * handler=GetTreeHandler(d);
	  mImageAdder.SetCurrentDatabase(d);
	  mImageAdder.SetTreeHandler(handler);
	  mImageAdder.SetSynchronizer(mSynchronizer);
	  mImageAdder.CopyFiles(filenames, mSettings->getValue(SETTINGS_COPY_PATH));
  }

  //========================================================================
 
  std::string Gimmick::Synchronize(const std::string& d, bool repair, bool checkAttributes)
  {
	  TreeHandler * handler=GetTreeHandler(d);
	  mImageAdder.SetCurrentDatabase(d);
	  mImageAdder.SetTreeHandler(handler);
	  mImageAdder.SetSynchronizer(mSynchronizer);
	  return mImageAdder.Synchronize(repair, checkAttributes);
  }

  //========================================================================
  /// 
  void Gimmick::Print(const std::string& d)
  {
    GetTreeHandler(d)->GetTree().Print();
  }
  //========================================================================

  void Gimmick::GetSetting(const std::string& name, std::string& value)
  {
    value = mSettings->getValue(name);
  }
  //========================================================================

  //========================================================================

  void Gimmick::GetAttributes(const std::string& d, 
	  		const std::string& filename, 
	  		const std::vector<std::string>& params, 
	  		std::vector<std::string>& results)
  {
	  TreeHandler * handler=GetTreeHandler(d);
	  mImageAdder.SetCurrentDatabase(d);
	  mImageAdder.SetTreeHandler(handler);
	  mImageAdder.SetSynchronizer(mSynchronizer);
	  mImageAdder.GetAttributes(params, filename, results);
  }
  //========================================================================

  //========================================================================
  // get attributes values from database  for a given file from database 
  //========================================================================
   void Gimmick::GetAttributes(const std::string filename, std::map<std::string, std::string> &i_res, OutputAttr i_attr)
  {
	   if (i_attr.inside.size() > 0)
	   {
		   std::map<std::string, std::string> map_attr;
		   TreeHandler * handler=GetTreeHandler(i_attr.db);
		   handler->getAllAttributes(filename, map_attr);
		   if(i_attr.inside.front() == "ALL") // we  take all values
		   {
			   std::map<std::string, std::string>::iterator it = map_attr.begin();
			   for(; it != map_attr.end(); it++)
				   i_res[it->first] = it->second;
		   }
		   else
		   {
			    std::vector<std::string>::iterator it = i_attr.inside.begin();
			    for(; it != i_attr.inside.end(); it++)
				   i_res[(*it)] = map_attr[(*it)];
		   }
	   }
  }


  //========================================================================

  void Gimmick::UpdateSetting(const std::string& name, const std::string& value)
  {
	  mSettings->updateSetting(name,value);
	  mSettings->writeSettingsFile();
  }
  //========================================================================

  void Gimmick::DeleteDrive(const std::string& drive)
  {
	for( TreeHandlerMapType::const_iterator it = mTreeHandlerMap.begin();
					       it!= mTreeHandlerMap.end(); 
					       ++it)
	   {
		   mImageAdder.SetTreeHandler(it->second);
		   mImageAdder.DeleteDriveFromMainDB(drive);
	   }
	mImageAdder.SetSynchronizer(mSynchronizer);
	mImageAdder.DeleteDriveFromOtherDB(drive);
  }

  //========================================================================
  void Gimmick::EditField(tree::Node* node, const std::string& d, const std::string& name, const std::string& key, const std::string& val)
  {
	TreeHandler * handler=GetTreeHandler(d);
	mImageAdder.SetCurrentDatabase(d);
	mImageAdder.SetTreeHandler(handler);
	mImageAdder.EditField(node,name,key,val);
  }
  //========================================================================

  ////////////////////////////////////////////////////////////////////////
  // add DB from Settings file						//
  // @param : -								//
  // return : -								//
  ////////////////////////////////////////////////////////////////////////
  void Gimmick::addDBSettings()
  {

	 std::string pathSettings = mSettings->getValue(SETTINGS_DBPATH);
	 
	 // split to find all paths
	 std::vector<std::string> paths;
	 std::string separator = ";";
	 std::string::size_type last_pos = pathSettings.find_first_not_of(separator);
	 //find first separator
	 std::string::size_type pos = pathSettings.find_first_of(separator, last_pos);
	 while(std::string::npos != pos || std::string::npos != last_pos)
	 {
		paths.push_back(pathSettings.substr(last_pos, pos - last_pos));
		last_pos = pathSettings.find_first_not_of(separator, pos);
		pos = pathSettings.find_first_of(separator, last_pos);
	 }

	 std::vector<std::string>::iterator it_path = paths.begin();
	 for(; it_path != paths.end(); ++it_path)
	 {
		 pos = it_path->find_last_of("\\");
		 last_pos = it_path->find_last_of(".");
		 std::string name = it_path->substr(pos +1, last_pos -pos-1 );
		 addDB(name, it_path->c_str());
	 }
  }	


///////////////////////////////////////////////////////////////////////////////
// Fill attributes structure with attributes present in database (inside vector
// and not present (outside)
///////////////////////////////////////////////////////////////////////////////
void Gimmick::fillVectInfos(std::vector<std::string> i_attr, OutputAttr &infos)
{
	//test if a tag is present in Database descriptor
	TreeHandler * handler=GetTreeHandler(infos.db);
	mImageAdder.SetTreeHandler(handler);
	std::vector<std::string>::const_iterator it = i_attr.begin();
	for (;it != i_attr.end(); it++)
	{
		if( mImageAdder.isAttributeExist((*it)) != "" ) // in DB
		{
			infos.inside.push_back((*it));
		}
		else
		{
				infos.outside.push_back((*it)); // Need to scan again the files
		}
	}
}

const std::string Gimmick::getSummary()
{
      const AddProgress& p = GetAddProgress();
    std::stringstream mess;
    mess << "Dirs \tscanned\t: " << p.GetNumberScannedDirs()  << "\n";
    mess << "Files\tscanned\t: " << p.GetNumberScannedFiles() << "\n";
    mess << "Files\thandled\t: " << p.GetNumberHandledFiles() << "\n\n";
    mess << "Files\tadded  \t: " << p.GetNumberAddedFiles()   << "\n\n";
    return mess.str();
}

}