/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#include <creaImageIOImageReader.h>
#include <creaImageIOTreeAttributeDescriptor.h>
#include <creaImageIOSystem.h>

#include <creaImageIOVtkImageReader.h>
#if defined (USE_GDCM)
	#include <creaImageIODicomImageReader.h>
#endif
#if defined(USE_GDCM2)
	#include <creaImageIODicomImageReader2.h>
#endif
#include <creaImageIOUltrasonixImageReader.h>
#include <vtkPNGReader.h>
#include <vtkTIFFReader.h>
#include <vtkJPEGReader.h>
#include <vtkBMPReader.h>
#include <vtkSLCReader.h>
#include <vtkMetaImageReader.h>
//#include <vtkGESignalReader.h>

#include "boost/filesystem/path.hpp"

namespace creaImageIO
{

  //=====================================================================
  ImageReader::ImageReader()
    :   
    mUnreadableImage(0),
    mLastFilename("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&")
  {
    //    std::cout << "#### ImageReader::ImageReader()"<<std::endl;
    if (mUnreadableImage!=0) return;


	Register( boost::shared_ptr<AbstractImageReader>(new VtkImageReader(vtkPNGReader::New() , "PNG", ".png")));
    Register(boost::shared_ptr<AbstractImageReader>(new VtkImageReader(vtkTIFFReader::New(), "TIFF", ".tiff")));
    Register(boost::shared_ptr<AbstractImageReader>(new VtkImageReader(vtkJPEGReader::New(), "JPEG", ".jpeg")));
    Register(boost::shared_ptr<AbstractImageReader>(new VtkImageReader(vtkBMPReader::New(), "BMP", ".bmp")));
    Register(boost::shared_ptr<AbstractImageReader>(new VtkImageReader(vtkSLCReader::New())));
    Register(boost::shared_ptr<AbstractImageReader>(new VtkImageReader(vtkMetaImageReader::New(),"MHD",".mhd")));
    //   Register(new VtkImageReader(vtkGESignalReader::New()));
    Register(boost::shared_ptr<AbstractImageReader>(new DicomImageReader));
    Register(boost::shared_ptr<AbstractImageReader>(new UltrasonixImageReader));

	UnRegister(".txt");
 
    mUnreadableImage = vtkImageData::New();
    int dim[3];
    dim[0] = dim[1] = 128; 
    dim[2] = 1; 
    mUnreadableImage->SetDimensions ( dim );
    mUnreadableImage->SetScalarTypeToUnsignedChar();
    mUnreadableImage->AllocateScalars();    
    for (int i=0;i<dim[0];i++) 
      for (int j=0;j<dim[1];j++) 
	mUnreadableImage->SetScalarComponentFromFloat(i,j,0,0,0);
    for (int i=0;i<dim[0];i++) 
      {
	mUnreadableImage->SetScalarComponentFromFloat(i,i,0,0,255);
	mUnreadableImage->SetScalarComponentFromFloat(dim[0]-1-i,i,0,0,255);
      }
  }
  //=====================================================================

  //=====================================================================
  ImageReader::~ImageReader()
  {

 //   for (i=mReader.begin(); i!=mReader.end(); i++)
 //     {
	//delete (*i);
 //     }
//    mReader.clear();
    if (mUnreadableImage!=0) 
      {
	mUnreadableImage->Delete();
	mUnreadableImage = 0;
      }
  }
  //=====================================================================

  //=====================================================================
  void ImageReader::Register(boost::shared_ptr<AbstractImageReader> r)
  {
    mReader.push_back(r);

  }

  void ImageReader::UnRegister(const std::string i_val)
  {
    mUnReader.push_back(i_val);

  }
  //=====================================================================

   //=====================================================================
  // Returns true iff the file is readable
  bool ImageReader::ShallNotRead( const std::string& filename ) 
  {
    bool ok = true;
	if(filename != "")
	{
		std::vector<std::string >::iterator i ;
		for (i=mUnReader.begin(); i!=mUnReader.end(); i++)
		{
			
			if ( (*i).c_str() == filename) 
			{
				ok = false;
				break;
			}
		}
	}
	return ok;
  }

  //=====================================================================
  // Returns true iff the file is readable
  bool ImageReader::CanRead( const std::string& filename ) 
  {
    bool ok = false;

	if( !ShallNotRead(filename))
	{
		return ok;
	}
	if(filename != "")
	{
		std::vector<boost::shared_ptr<AbstractImageReader> >::iterator i;
		for (i=mReader.begin(); i!=mReader.end(); i++)
		{
			ok = (*i)->CanRead(filename);
			if (ok) 
			{
				mLastFilename = filename;
				mLastReader = *i;
				break;
			}
		}
	}
	return ok;
  }
  //=====================================================================

  //=====================================================================
  // Reads the file (CanRead must be called before : no test here)
  vtkImageData*  ImageReader::ReadImage( const std::string& filename)
  {
	  if (mLastFilename!=filename)
      {
	if (!CanRead(filename))  
	  { 
	    vtkImageData* im = vtkImageData::New();
	    im->ShallowCopy(mUnreadableImage);
	    return im;
	  }
      }
    vtkImageData* i = mLastReader->ReadImage(mLastFilename);
    if (i==0) 
      {
	i = vtkImageData::New();
	i->ShallowCopy(mUnreadableImage);
      }
    return i;
  }
  //=====================================================================
  // Another function to read attributes for a file
  void ImageReader::getAttributes(const std::string filename,
		std::map <std::string , std::string> &infos, std::vector<std::string> i_attr)
  {
	   if (mLastFilename!=filename)
      {
	if (!CanRead(filename)) 
	  { 
	    return;
	  }
      }
    mLastReader->getAttributes(filename, infos, i_attr);
  }
  //=====================================================================
   void ImageReader::ReadAttributes(const std::string& filename, 
				    std::map<std::string,std::string>& attr)
   {
    if (mLastFilename!=filename)
      {
	if (!CanRead(filename)) 
	  { 
	    return;
	  }
      }
    mLastReader->ReadAttributes(mLastFilename,attr);
   }
  //=====================================================================


  //=====================================================================
  /// Pushes back all kwown extensions (without dot) in the vector given
  void ImageReader::PushBackExtensions(std::vector<std::string>& v)
  {
	  std::vector<boost::shared_ptr<AbstractImageReader> >::iterator i;
    for (i=mReader.begin(); i!=mReader.end(); i++)
      {
	(*i)->PushBackExtensions(v);
      }
  }
  //=====================================================================
  
} // namespace creaImageIO
