/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#include <creaImageIOSettings.h>
#include <boost/filesystem/fstream.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>

#include <iostream>
#include <fstream>

// Memory tracking allocation
#ifdef _DEBUG
#define new DEBUG_NEW
#endif
using namespace boost;
namespace po = boost::program_options;

namespace creaImageIO
{
    Settings::Settings(const std::string i_path)
    {
        //need to position path in user directory first.
        m_SettingsFileName = i_path + "\\.creaImageIO\\share\\creaImageIO\\app.config";
        //Test if Settings File exist
        if(!boost::filesystem::exists(m_SettingsFileName) )
        {
            createFile();
        }
        std::ifstream ifs(m_SettingsFileName.c_str());
		std::string line;
		std::string sets;
   	    if (ifs.is_open())
		{
			while (! ifs.eof() )
			{
				getline(ifs,line);
				sets += line;
			}
			ifs.close();
		}
      	std::vector<std::string> Keys;
		Keys.push_back(SETTINGS_SYNC_EVENT);
		Keys.push_back(SETTINGS_DBPATH);
		Keys.push_back(SETTINGS_SYNC_FREQ);
		Keys.push_back(SETTINGS_COPY_PATH);
		Keys.push_back(SETTINGS_REMOVE_PATIENT_DISPLAY);
		Keys.push_back(SETTINGS_OUTPUT_ASK);
		Keys.push_back(SETTINGS_OUTPUT_DIM);
		readSettings(Keys, sets);

    }

    Settings::~Settings()
    {
        
    }


   ////////////////////////////////////////////////////////////////////////////////////////////////
    // create the config file                                                                                 //
    //@param : -                                                                                               //
    // return  : -                                                                                                // 
    ///////////////////////////////////////////////////////////////////////////////////////////////
   void Settings::createFile()
   {
       m_SettingsMap[SETTINGS_SYNC_FREQ] = "12";
       m_SettingsMap[SETTINGS_SYNC_EVENT] = "end";
       m_SettingsMap[SETTINGS_DBPATH] = "";
       m_SettingsMap[SETTINGS_DICOM_LIBRARY] = "gdcm";
	   m_SettingsMap[SETTINGS_COPY_PATH] = m_SettingsFileName.substr(0,m_SettingsFileName.find_last_of('\\')+1)+"Copied files";
	   m_SettingsMap[SETTINGS_REMOVE_PATIENT_DISPLAY] = "0";
	   m_SettingsMap[SETTINGS_OUTPUT_ASK] ="true";
	   m_SettingsMap[SETTINGS_OUTPUT_DIM] = "1";
       writeSettingsFile();
   }

	////////////////////////////////////////////////////////////////////////////////////////////////
    // read Settings from config file                                                             //
    // @param i_keys : list of keys                                                               //
    // @param  i_file : text from config file                                                     //
	// return : -
    ///////////////////////////////////////////////////////////////////////////////////////////////
	void Settings::readSettings(std::vector<std::string> &i_Keys, const std::string &i_file)
	{
		std::vector<std::string>::iterator it_key = i_Keys.begin();
		for(;  it_key< i_Keys.end(); ++it_key)
		{
			size_t fpos = i_file.find(it_key->c_str());
			size_t lpos = i_file.rfind(it_key->c_str());
			if(fpos != std::string::npos && lpos != std::string::npos)
			{
				m_SettingsMap[it_key->c_str()] = i_file.substr(fpos + it_key->size(),lpos-fpos - it_key->size());
			}
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////
    // Update settings in config file                                                             //
    // @param key : Key to update                                                                 //
    // @param value: New value to set                                                             //
	// return : -
    ///////////////////////////////////////////////////////////////////////////////////////////////
	void Settings::updateSetting(const std::string& key, const std::string &val)
	{
		m_SettingsMap[key.c_str()] = val;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////
    // add a path to a DB			                                                              //
    // @param i_path : DB path to add                                                             //
    // return : -																				  //
    ///////////////////////////////////////////////////////////////////////////////////////////////
	void Settings::addDB(const std::string &i_path)
	{
		if(m_SettingsMap[SETTINGS_DBPATH].find(i_path) == std::string::npos)
		{
			m_SettingsMap[SETTINGS_DBPATH] += i_path + ";";
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////
    // remove a path to a DB			                                                          //
    // @param i_path : DB path to delete (don't exist anymore)                                    //
    // return : -
	///////////////////////////////////////////////////////////////////////////////////////////////
	
	void Settings::removeDB(const std::string &i_path)
	{
		boost::algorithm::replace_all(m_SettingsMap[SETTINGS_DBPATH],i_path + ";","");
	}

	///////////////////////////////////////////////////////////////////////////////////////////////
    // write Settings buffer from                                                                //
    // @param o_file : settings buffer                                                           //
    //																							 //
	// return : -																				 //
    ///////////////////////////////////////////////////////////////////////////////////////////////
	void Settings::writeSettings(std::ofstream &o_file)
	{
		std::map<std::string, std::string>::iterator it_map = m_SettingsMap.begin();
		std::stringstream st;
		for(; it_map != m_SettingsMap.end(); ++it_map)
		{
			o_file << it_map->first.c_str();
			o_file <<  it_map->second.c_str();
			o_file << it_map->first.c_str();
			o_file << std::endl;
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////
    // write Settings file                                                             //
    // @param : -                                                               //
    // return : -
    ///////////////////////////////////////////////////////////////////////////////////////////////
	void Settings::writeSettingsFile()
	{	
	    std::ofstream ofs(m_SettingsFileName.c_str());
		ofs.clear();
		writeSettings(ofs);
		ofs.close();
	}  
}

