/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#include <creaImageIOSynchron.h>
#include <creaImageIOSystem.h>
#include <boost/filesystem.hpp>
#include <boost/algorithm/string.hpp>


//namespace fs = boost::filesystem;

//=====================================================================
 

namespace creaImageIO
{

    //=====================================================================
    #define QUERYSYNCDB(QUER,RES)					\
    try									\
    {									\
	RES = mDB->execQuery(QUER.c_str());				\
    }									\
    catch (CppSQLite3Exception& e)					\
    {									\
      GimmickError("SQLite query '"<<QUER<<"' Error : "			\
		   << e.errorCode() << ":"				\
		   << e.errorMessage() );				\
    }									                                                                   
   //=====================================================================
    #define UPDATESYNCDB(UP)				        	\
    try									\
    {									\
      mDB->execDML(UP.c_str());						\
    }									\
    catch (CppSQLite3Exception& e)					\
    {									\
      GimmickError("SQLite update '"<<UP<<"' Error : "			\
		   << e.errorCode() << ":"				\
		   << e.errorMessage() );				\
    }				
   //=====================================================================

    Synchronizer::Synchronizer(const std::string& path)
    {
		pathDB = path + "maintenance_database.db3";
		mDB = new CppSQLite3DB;
		Initialize();
    }

   //=====================================================================
    Synchronizer::~Synchronizer()
    {
       delete mDB;
    }

	//=====================================================================
	void Synchronizer::Initialize()
	{    
		if (!boost::filesystem::exists(pathDB)) 
		{
			CreateDB();
		}
    
		// OPENING
		else
		{
		try
		{
		mDB->open(pathDB.c_str());
		}
		catch (CppSQLite3Exception& e)
		{
		GimmickError("Opening '"<<pathDB<<"' : "
		     << e.errorCode() << ":" 
		     << e.errorMessage());
		}
		}
		// get the ADD operations List 
		//UpdateAddList(pathDB);
	}

   //=====================================================================
    void Synchronizer::CreateDB()
    {
		mDB->open(pathDB.c_str());
        // CREATING TABLES
    	std::string command;
	command = "CREATE TABLE ";
	command += "ADD_OPS";
	command += "\n(\nADD_KEY INTEGER PRIMARY KEY";
	command += ",\nPATH text";
	command += ",\nRECURSIVE boolean";
	command += ",\nFILES_ADDED int";
	command += ",\nREFERENCEDDB text";
	command += "\n)";
	UPDATESYNCDB(command);

	command = "CREATE TABLE ";
	command += "IGNORED_FILES";
	command += "\n(\nID INTEGER PRIMARY KEY";
	command += ",\nADD_KEY integer";
	command += ",\nPATH text";
	command += ",\nREMOVE boolean";
	command += ",\nTIME datetext";
	command += "\n)";
	UPDATESYNCDB(command);
    }

	//=====================================================================
	void Synchronizer::CleanName(std::string& str) const
	{
		size_t pos;
		do
		{
			pos = str.find('\\');
			if ( pos != (size_t)-1 )  
			{
				str.replace(pos, 1, "/");
			}
		}
		while ( pos != (size_t)-1 );
	}

	//=====================================================================
    void Synchronizer::GetFileList(std::vector<AddList> & list, const std::string& refdb)
    {
       CleanList(refdb);
       list=mAddList;
    }

	//=====================================================================
   void Synchronizer::GetIgnoredFiles(const std::string& key, std::vector<std::string> &ignoreList)
   {
        ignoreList=GetIgnoreList(key);
   }

//=====================================================================
    void Synchronizer::UpdateAddList(const std::string& refdb)
    {
        std::string query = "SELECT * FROM ADD_OPS WHERE REFERENCEDDB = '"+refdb+"';";
        CppSQLite3Query res;
        QUERYSYNCDB(query, res);
	while (!res.eof())
        {
	    AddList temp = AddList(res);
            mAddList.push_back(temp);
	    res.nextRow();
        }
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////
    // remove an entry of the DB
    //@param i_table : table where to do the remove
    // @param i_key : the add_key reference (one entry to remove for ADD_OP table, many for IGNORED_FILES table
    //@result : -
    /////////////////////////////////////////////////////////////////////////////////////////////////
    void Synchronizer::RemoveEntry(const std::string i_table, const std::string i_key)
    {
        std::string query = "DELETE  FROM " + i_table + " WHERE ADD_KEY = '" + i_key +"'";
        UPDATESYNCDB(query);
    }

	/////////////////////////////////////////////////////////////////////////////////////////////////
    // remove several entries of the DB
    // @param i_table : table where to do the remove
    // @param i_attribute: attribute to match
    // @param i_operand : operand to use
    // @param i_val : the reference
    //@result : -
    /////////////////////////////////////////////////////////////////////////////////////////////////
    void Synchronizer::RemoveEntries(const std::string i_table, 
		const std::string i_attribute, 
		const std::string i_operand, 
		const std::string i_val)
    {
        std::stringstream query;
	query<<"DELETE  FROM "<<i_table<<" WHERE "<<i_attribute<<" "<<i_operand<<" '"<<i_val<<"'";
        UPDATESYNCDB(query.str());
	}

    /////////////////////////////////////////////////////////////////////////////////////////////////
    // clean DataBase if an operation has no child anymore
	// @param refdb: the database segement to clean
    // @result : -
    /////////////////////////////////////////////////////////////////////////////////////////////////
    void Synchronizer::CleanList(const std::string& refdb)
    {
	mAddList.clear();
	UpdateAddList(refdb);
        std::vector<AddList>::iterator it_add = mAddList.begin();
        for(;it_add <mAddList.end(); ++it_add)
        {
                if(it_add->nbFiles == "0")
                {
			RemoveEntry("ADD_OPS", it_add->key);
			RemoveEntry("IGNORED_FILES", it_add->key);

                }
        }
		mAddList.clear();
		UpdateAddList(refdb);
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////
    // Inserts a new add operation in the database
    // @param path: the path of the directory that was added
    // @param recursive: shows if the action was called recursively or not
    // @param nChildren: the number of files affected by the operation
    // @param refdb: the referenced database
    // @result : The operation has been added
    /////////////////////////////////////////////////////////////////////////////////////////////////
	void Synchronizer::InsertAddOp(const std::string& path, const std::string& recursive, const std::string& nChildren, const std::string& refdb)
	{
		std::string insert;
		std::string pat=path.c_str();
		CleanName(pat);
		insert="INSERT INTO ADD_OPS (PATH,RECURSIVE,FILES_ADDED,REFERENCEDDB) VALUES('";
		insert+=convert(pat)+"','";
		insert+=recursive+"',";
		insert+=nChildren+",'";
		insert+=refdb+"');";
		UPDATESYNCDB(insert);
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////
    // Inserts a new ignored file in the database
	// @param add_key: the key of the add_op to which it corresponds
    // @param path: the path of the directory that was added
	// @param remove: shows if the file was removed or not
	// @param time: the time in which the file was removed
    // @result : The file has been inserted
    /////////////////////////////////////////////////////////////////////////////////////////////////

	void Synchronizer::InsertIgnoreFile(const std::string& addKey, const std::string& path, const std::string& remove, const std::string& time, const std::string& refdb )
	{
		std::string pat=path.c_str();
		CleanName(pat);
		std::string id=GetAttribute("ID","IGNORED_FILES","PATH",pat,refdb);
		if(id.compare("")==0)
		{
			std::string insert;
			insert="INSERT INTO IGNORED_FILES (ADD_KEY,PATH,REMOVE,TIME) VALUES('";
			insert+=addKey+"','";
			insert+=convert(pat)+"','";
			insert+=remove+"',";
			insert+=time+");";
			UPDATESYNCDB(insert);
		}
		else
		{
			//Gets the add key
			std::string ak=GetAttribute("ADD_KEY","IGNORED_FILES","ID",id,refdb);
			//gets the parent database to check if the file has been added to the current database
			std::string parentDB=GetAttribute("*","ADD_OPS","ADD_KEY",ak,refdb);
			//If there is no such entry, add it
			if(parentDB.compare("")==0)
			{
				std::string insert;
				insert="INSERT INTO IGNORED_FILES (ADD_KEY,PATH,REMOVE,TIME) VALUES('";
				insert+=addKey+"','";
				insert+=convert(pat)+"','";
				insert+=remove+"',";
				insert+=time+");";
				UPDATESYNCDB(insert);
			}
			else
			{
				//Sets the new add key attribute for the file
				SetAttribute("ADD_KEY","IGNORED_FILES",addKey,"ID", id,refdb);
				//Sets the new remove attribute for the file
				SetAttribute("REMOVE","IGNORED_FILES",remove,"ID", id,refdb);
				//Sets the new time attribute for the file
				SetAttribute("TIME","IGNORED_FILES",time,"ID", id,refdb);
			}
		}
	}

    /////////////////////////////////////////////////////////////////////////////////////////////////
    // get the files name to ignore for a add operation synchronization
    // @param : the add key
    //@result : list (path) of ignore files
    /////////////////////////////////////////////////////////////////////////////////////////////////
    std::vector<std::string> Synchronizer::GetIgnoreList(const std::string &i_key)
    {
        mIgnoreList.clear();
        std::vector<std::string> i_names;
	std::string query = "SELECT * FROM IGNORED_FILES WHERE ADD_KEY = ";
	query+=i_key;
        CppSQLite3Query res;
        QUERYSYNCDB(query, res);
	while (!res.eof())
        {
		RemoveList temp = RemoveList(res);
		if(temp.remove.compare("0")==0)
		{
                   mIgnoreList.push_back(temp);
		}
		res.nextRow();
        }
        std::vector<RemoveList>::iterator it;

        for(it = mIgnoreList.begin();it != mIgnoreList.end(); ++it)
        {
            i_names.push_back((*it).path);
        }
	return i_names;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////
    // Gets the required attribute in the required table
    // @param attribute: the attribute to look for
    // @param table: the table to look in
    // @param searchParam: the search parameter
    // @param searchValue: the search value
    // @result : required attribute
    /////////////////////////////////////////////////////////////////////////////////////////////////
    std::string Synchronizer::GetAttribute(const std::string& attribute, 
					const std::string& table, 
					const std::string& searchParam,
					const std::string& searchValue, 
					const std::string& refdb)
    {
        std::stringstream query;
	std::string result;
	std::string sVal=convert(searchValue.c_str());
	CleanName(sVal);
	query<<"SELECT "<<attribute<<" FROM "<<table<<" WHERE "<<searchParam<<" = '"<<sVal;
	if(table.compare("ADD_OPS")==0)
	{
		query<<"' AND REFERENCEDDB = '"<<refdb<<"';";
	}
	else
	{
		query<<"';";
	}
        CppSQLite3Query res;
        QUERYSYNCDB(query.str(), res);
	while (!res.eof())
        {
		result=res.getStringField(0);
		res.nextRow();
        }
	return result;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////
    // Sets the attribute value in the required table and column
    // @param attribute: the attribute to look for
    // @param table: the table to look in
    // @param value: the value to set
    // @param searchParam: the search parameter
    // @param searchValue: the search value
    // @result : attribute value changed
    /////////////////////////////////////////////////////////////////////////////////////////////////
    void Synchronizer::SetAttribute(const std::string& attribute, 
				const std::string& table, 
				const std::string& value,
				const std::string& searchParam,
				const std::string& searchValue,
				const std::string& refdb)
    {
	std::string val=value.c_str();
	std::string sVal=convert(searchValue.c_str());
	CleanName(val);
	CleanName(sVal);
	std::string sql = "UPDATE ";
	sql+=table;
	sql+=" SET ";
	sql += attribute;
	sql += " = '";
	sql += val;
	sql += "' WHERE ";
	sql += searchParam;
	sql += " = '";
	sql += sVal;
	if(table.compare("ADD_OPS")==0)
	{
		sql += "' AND REFERENCEDDB = '";
		sql += refdb;
	}
	sql += "';";
	UPDATESYNCDB(sql);
    }



/////////////////////////////////////////////////////////////////////////////////////////////////
    // get the files name to ignore for a add operation synchronization
    // @param : the add key
    //@result : list (path) of ignore files
    /////////////////////////////////////////////////////////////////////////////////////////////////
	void Synchronizer::GetList(const std::string i_db)
    {
		mList.clear();
        std::vector<std::string> i_names;
		std::vector<std::string> keys;
		CppSQLite3Query res;
		std::string query ="SELECT ADD_KEY, REFERENCEDDB FROM  ADD_OPS";
		QUERYSYNCDB(query, res);
		keys.clear();
		while (!res.eof())
        {
			std::string key(res.getStringField(0));
			std::string db(res.getStringField(1));
			if (db == i_db)
			{
				keys.push_back(key);
			}
			res.nextRow();
		}
		query = "SELECT PATH, REMOVE FROM IGNORED_FILES WHERE";
		if(keys.size() > 0)
		{
			for (int i=0; i < (int)keys.size(); i++)
			{
				query += " ADD_KEY = " + keys[i];
				query += " AND";
			}
			query = query.substr(0,query.size() - 4);
		}
		else 
		{
			query += " ADD_KEY = -1";
		}
        QUERYSYNCDB(query, res);
		while (!res.eof())
        {
		std::string file(res.getStringField(0));
		std::string ignore(res.getStringField(1));
		mList[file] = ignore == "0"? true : false;
		res.nextRow();
        }
    }

	bool Synchronizer::isIndexed(const std::string filename)
	{
		bool valid = true;
		std::string name(filename);
		boost::algorithm::replace_all( name,"\\" , "/");
		std::map <std::string, bool>::iterator it_list = mList.begin();
		for(;it_list != mList.end(); it_list++)
		{
			if(it_list->first == name)
			{
				valid = false;
				break;
			}
		}
		return valid;
	}
	  const std::string Synchronizer::convert(const std::string &i_word)
	  {
		  std::string temp = i_word;
		  boost::algorithm::replace_all(temp,"'","''");

	  return temp.c_str();
  }
}


