/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#ifndef __creaImageIOTimestampDatabaseHandler_h_INCLUDED__
#define __creaImageIOTimestampDatabaseHandler_h_INCLUDED__
#include <vector>
#include <map>
#include <creaImageIOTree.h>
class CppSQLite3DB;

namespace creaImageIO
{
	using namespace std;
//=======================================================================
  /// Concrete TreeHandler which manages a Tree stored in a sqlite database
  class TimestampDatabaseHandler 
  {
  public:
    //====================================================================
    /// Ctor with database file name 
    TimestampDatabaseHandler(const std::string& filename);
    /// Dtor
    virtual ~TimestampDatabaseHandler();
    //====================================================================

    //====================================================================
    /// Returns the sqlite db file name 
    const std::string& GetFileName() const { return mFileName; }
    //====================================================================
 
    //====================================================================
    // INITIALIZATION / FINALIZATION
    //====================================================================

    //====================================================================
    /// Opens an existing 'source' 
     bool Open();
    /// Closes the 'source'
    bool Close();
    /// Creates a new 'source' 
    bool Create();
    /// Destroys the 'source'
    bool Destroy();
    //====================================================================

	//====================================================================
    // READ / WRITE
    //====================================================================
	//====================================================================
	///Returns the id of the path if it's indexed, blank otherwise
	std::string IsIndexed(const std::string& path, const std::string& refdb);
    ///Sets the current path's parent
	bool AddDirectory(const std::string& parent,
				   const std::string& path, 
				   const time_t lastModif, 
				   const time_t lastRead,
				   const std::string& refdb);
	///Adds a new file to the database without a parent
	void AddFile(const std::string& path, const time_t lastModif, const time_t lastRead, const std::string& refdb);
	///Adds a new file to the database with a parent
	void AddFile(const std::string& parentId,const std::string& path, const time_t lastModif, const time_t lastRead, const std::string& refdb);
	///Sets the attribute to the value passed as parameter where the searchParameter is searchValue
	void SetAttribute(const std::string& attName, 
					const std::string& attValue,
					const std::string& searchParam,
					const std::string& searchValue);
	///Removes the given node
	void RemoveNode(const std::string& searchAtt, const tree::Node* node, const std::string& refdb);
	///Removes the filename with the given pathname
	void RemoveFile(const std::string& searchAtt, const std::string& searchVal, const std::string& refdb);
	///Cleans the path name
	void CleanPath(std::string& str) const;
	///Checks the timestamp in the database and compares it with the given one. 
	//If there is a difference, it will return false, otherwise it will return true.
	bool CheckTimestamp(const std::string pathId, const time_t lastModif, const std::string& refdb);
	///Removes the entries that match the given parameters
	void RemoveEntries(const std::string i_table, 
		const std::string i_attribute, 
		const std::string i_operand, 
		const std::string i_val);

	//====================================================================


  protected:
    //======================================================================
    /// Open the database
    bool DBOpen();
    //======================================================================
    //======================================================================
    // Creation
    /// Creates a new database on disk and the tables
    bool DBCreate();
    //======================================================================
	//======================================================================
    // Removes a file from the database
	void DBRemove(const std::string& searchAtt, const std::string& searchVal, const std::string& refdb);
 
  private:
    /// The DB
    CppSQLite3DB* mDB;
   /// The physical location associated to the DicomDatabase (directory, db file...)
    std::string mFileName;
  
  };
  // EO class
  //=======================================================================


} // EO namespace creaImageIO

// EOF
#endif  

