/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#include <creaImageIOTreeAttributeDescriptor.h>


#if defined(_WIN32)
#pragma warning(disable: 4996)
#endif

#if defined(USE_GDCM)
#include <gdcmGlobal.h>
#include <gdcmDictSet.h>
#endif

#if defined(USE_GDCM2)
#include <gdcmGlobal.h>
#include <gdcmDicts.h>
#include <gdcmDict.h>
#endif

#include <boost/algorithm/string/replace.hpp>

namespace creaImageIO
{

  namespace tree
  {

    //========================================================================
    void AttributeDescriptor::CleanName(std::string& str) const
    {
      // quote must be doubled for SQL
      //    crea::Utils::Replace( str, "'", "''" );
      boost::algorithm::replace_all(str,"'","''");
      // Found strange strings which contained NULL char INSIDE string 
      int i,size=(int)str.size();
      for (i=0;i<size;++i) 
	{
	  if (str[i]==0) 
	    {
	      str = str.substr(0,i);
	      break;
	    }
	}
    }
    //========================================================================

    //=====================================================================
    // Ctor with key, name and flags
     AttributeDescriptor::AttributeDescriptor(const std::string& key,
					     const std::string& name,
					     unsigned int flags)
      : mKey(key), mName(name), mGroup(0), mElement(0), mFlags(flags)
    {

      CleanName(mName);
      GimmickDebugMessage(3,"AttributeDescriptor : '"<<key
			  <<"' ["<<flags<<"]"<<std::endl);
      GimmickDebugMessage(3,"='"<<mName<<"'"<<std::endl);
    }

    //=====================================================================

     //=====================================================================
   // Ctor with dicom group, elem and flags
    // The key is built as 'Dgroup_elem'
    // The user name is retreived from dicom dictionnary
    AttributeDescriptor::AttributeDescriptor(unsigned short group,
					     unsigned short element,
					     unsigned int flags)
      : mGroup(group), mElement(element), mFlags(flags)
    {
      
      //GDCM_NAME_SPACE::TagKey tag(group,element);
      char ctag[12];
      sprintf(ctag,"D%04x_%04x",group,element);
      mKey = ctag;

      GimmickDebugMessage(3,"AttributeDescriptor : '"<<mKey
			  <<"' ["<<flags<<"]"<<std::endl);

#if defined(USE_GDCM)
      // Retrieve the name from gdcm dict
      GDCM_NAME_SPACE::DictEntry* entry =
	GDCM_NAME_SPACE::Global::GetDicts()
	->GetDefaultPubDict()->GetEntry(mGroup,mElement);
	  
      if (entry)
	{
	  mName = entry->GetName();
	  CleanName(mName);
	  GimmickDebugMessage(3,"='"<<mName<<"'"<<std::endl);
	}
      else
	{
	  GimmickMessage(1,"!! WARNING : tag '"<<mKey
			 <<"' is not in DICOM dictionnary ! "
			 <<"Considering it as a user attribute"
			 << std::endl);
	  mName = "UNKNOWN";
	  mGroup = mElement = 0;
	}
#endif


	  
	  
#if defined(USE_GDCM2)
      // Retrieve the name from gdcm dict
	const gdcm::Global& g = gdcm::Global::GetInstance(); 
	 const gdcm::Dicts &dicts = g.GetDicts();
  const gdcm::Dict &dict = dicts.GetPublicDict();
	  gdcm::DictEntry dictentry =  dict.GetDictEntry(gdcm::Tag(mGroup, mElement));
	  
      mName = dictentry.GetName();
	  if(!mName.empty())
	  {
		  CleanName(mName);
		  GimmickDebugMessage(3,"='"<<mName<<"'"<<std::endl);
	  }
      else
	  {
		GimmickMessage(1,"!! WARNING : tag '"<<mKey
			 <<"' is not in DICOM dictionnary ! "
			 <<"Considering it as a user attribute"
			 << std::endl);
		mName = "UNKNOWN";
		mGroup = mElement = 0;
		}
#endif

    }
    //=====================================================================


    //=====================================================================
    /// Extracts group and element from a key of the form "Dgroup_elem" 
    void AttributeDescriptor::GetDicomGroupElementFromKey(const std::string& key,
							  unsigned short& group,
							  unsigned short& elem)
    {
      group = elem = 0;
      if ( (key.size()==10) &&
	   (key[0] == 'D') &&
	   (key[5] == '_') )
	  {
	 sscanf(key.c_str(),"D%04hx_%04hx ",&group,&elem);  
	  GimmickDebugMessage(3,"GetDicomGroupElementFromKey '"<<key<<"' : "			 <<group<<"|"<<elem<<std::endl);
	}
      else 
	{ 
	  GimmickMessage(5,"GetDicomGroupElementFromKey '"<<key<<"' : "
			 <<" not a DICOM key format"<<std::endl);
	}
      return;
    }

	//=====================================================================
	/// test if the type is a date
	bool AttributeDescriptor::isDateEntry() const
	{
		 
		bool btest = false;
		// Retrieve the name from gdcm dict
#if defined(USE_GDCM)
		GDCM_NAME_SPACE::DictEntry* entry =	GDCM_NAME_SPACE::Global::GetDicts()->GetDefaultPubDict()->GetEntry(GetGroup(),GetElement());
		if(	entry != 0)
		{
			if( entry->GetVR().str() == "DA" )
			{
				btest = true;
			}
		}
#endif
#if defined(USE_GDCM2)
	 const gdcm::Global& g = gdcm::Global::GetInstance(); 
	 const gdcm::Dicts &dicts = g.GetDicts();
  const gdcm::Dict &dict = dicts.GetPublicDict();
	  if(mGroup != 0 && mElement != 0)
	  {
		  gdcm::DictEntry dictentry =  dict.GetDictEntry(gdcm::Tag(GetGroup(), GetElement()));
		if( gdcm::VR::GetVRString(dictentry.GetVR()) == "DA")
		{
				btest = true;
		}
	  }
#endif
		return btest;
	}

	//=====================================================================
	/// test if the type is a time
	bool AttributeDescriptor::isTimeEntry() const
	{
		 
		bool btest = false;
#if defined(USE_GDCM)
		// Retrieve the name from gdcm dict
		GDCM_NAME_SPACE::DictEntry* entry =	GDCM_NAME_SPACE::Global::GetDicts()->GetDefaultPubDict()->GetEntry(GetGroup(),GetElement());
		if(	entry != 0)
		{
			if( entry->GetVR().str() == "TM" )
			{
				btest = true;
			}
		}
#endif

#if defined(USE_GDCM2)
	 const gdcm::Global& g = gdcm::Global::GetInstance(); // sum of all knowledge !
	 const gdcm::Dicts &dicts = g.GetDicts();
  const gdcm::Dict &dict = dicts.GetPublicDict(); // Part 6
	  if(mGroup != 0 && mElement != 0)
	  {
		gdcm::DictEntry dictentry =  dict.GetDictEntry(gdcm::Tag(mGroup, mElement));
		if(gdcm::VR::GetVRString(dictentry.GetVR()) == "TM")
		{
				btest = true;
		}
	  }
#endif

		return btest;
	}


    //=====================================================================
	/// Decodes the type of the attribute
     void AttributeDescriptor::DecodeType(unsigned int& typ) const
	  {
		 std::string type=""; 
#if defined(USE_GDCM)	
		  // Retrieve the name from gdcm dict
		GDCM_NAME_SPACE::DictEntry* entry =
		GDCM_NAME_SPACE::Global::GetDicts()
		->GetDefaultPubDict()->GetEntry(GetGroup(),GetElement());

		if (entry==0) 
		{
			typ = 2;
			return;
		}
		 type = entry->GetVR().str();
#endif
#if defined(USE_GDCM2)
	 const gdcm::Global& g = gdcm::Global::GetInstance(); // sum of all knowledge !
	 const gdcm::Dicts &dicts = g.GetDicts();
  const gdcm::Dict &dict = dicts.GetPublicDict(); // Part 6
	  gdcm::DictEntry dictentry =  dict.GetDictEntry(gdcm::Tag(mGroup, mElement));
	  type = gdcm::VR::GetVRString(dictentry.GetVR());
#endif

		GimmickDebugMessage(3,"VR Value is "<<type<<"!"<<std::endl);
		if(type=="AS" ||
		type=="DA" ||
		type=="FL" ||
		type=="FD" ||
		type=="IS" ||
		type=="SL" ||
		type=="SS" ||
		type=="UI" ||
		type=="US" ||
		type=="SH")
		{
			// Numerical 
			typ = 1;
		}
		else
		{
			// String
			typ = 2;
		}
		
	  }
	  //=====================================================================

  } // EO namespace tree

} // EO namespace creaImageIO
