/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#include <creaImageIOTreeDescriptor.h>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <creaImageIOSystem.h>
#include <boost/filesystem.hpp>


#include <fstream>


namespace creaImageIO
{

  namespace tree
  {

    //==================================================================
    /// The attribute is hidden (not visible to user)
    const unsigned int AttributeDescriptor::PRIVATE = 1;
    /// The attribute enters in unique identifier constitution 
    const unsigned int AttributeDescriptor::IDENTIFIER = 2;
    /// The attribute can be edited
    const unsigned int AttributeDescriptor::EDITABLE = 3;
    /// the attribute describes the node
    const unsigned int AttributeDescriptor::LABEL = 4;
    //==================================================================

    //==================================================================
    Descriptor::Descriptor()
    {
      CreateLevel0Descriptor();
    }
    //==================================================================

   //==================================================================
    Descriptor::~Descriptor()
    {
    }
    //==================================================================

    //==================================================================
    void Descriptor::CreateLevel0Descriptor()
    {
      Add(LevelDescriptor("Root"));
    }
    //==================================================================
    
    //==================================================================
    /// Creates the default descriptor
    void Descriptor::CreateDefault()
    {
      // clears the existing one
      Clear();
      
      // Creates the level 0 descriptor 
      CreateLevel0Descriptor();
      // Creates the attribute "Name"
      Add(AttributeDescriptor("Name","Name",
			      AttributeDescriptor::LABEL),0);
      
      // Patient level
      Add(LevelDescriptor("Patient"));
      Add(AttributeDescriptor("NumberOfChildren","#Series",0),1);   // Number of Series
      Add(AttributeDescriptor(0x0010,0x0010,   // Patient name
			      AttributeDescriptor::LABEL),1);
      Add(AttributeDescriptor(0x0010,0x0040),1); // Patient sex
      Add(AttributeDescriptor(0x0010,0x0030),1); // Patient birthday
      Add(AttributeDescriptor(0x0010,0x0020,   // Patient ID
			      AttributeDescriptor::IDENTIFIER),1);

      // Study-series level
      Add(LevelDescriptor("Series"));
      Add(AttributeDescriptor("NumberOfChildren","#Images",0),2);   // Number of images
      Add(AttributeDescriptor(0x0008,0x0060,    // Modality
			      AttributeDescriptor::LABEL),2);
      Add(AttributeDescriptor(0x0008,0x1030),2); // Study Description
      Add(AttributeDescriptor(0x0008,0x103E),2); // Description
      Add(AttributeDescriptor(0x0008,0x0080),2); // Institution Name
      Add(AttributeDescriptor(0x0008,0x0081),2); // Institution Adress
      Add(AttributeDescriptor(0x0008,0x1010),2); // Station Name
      Add(AttributeDescriptor(0x0008,0x1048),2); // Physician of Record
      Add(AttributeDescriptor(0x0008,0x1050),2); // Performing Physician's Name
      Add(AttributeDescriptor(0x0018,0x1030),2); // Protocol Name

      Add(AttributeDescriptor(0x0020,0x0010),2); // Study ID
      Add(AttributeDescriptor(0x0008,0x0020),2); // Study Date
      Add(AttributeDescriptor(0x0008,0x0030),2); // Study Time
      Add(AttributeDescriptor(0x0008,0x0050),2); // Study Accession Number
      Add(AttributeDescriptor(0x0008,0x0005),2); // Specific character set
      Add(AttributeDescriptor(0x0008,0x0021),2); // Series Date
      Add(AttributeDescriptor(0x0008,0x0031),2); // Series time

      Add(AttributeDescriptor(0x0020,0x000D   // Study Instance UID  
						    ),2);//AttributeDescriptor::IDENTIFIER),2);
      Add(AttributeDescriptor(0x0020,0x000E,   // Series Instance UID  
				AttributeDescriptor::IDENTIFIER),2);
      // |
      //			AttributeDescriptor::LABEL),2);
     

      // Image level
      Add(LevelDescriptor("Image"));

      Add(AttributeDescriptor(0x0020,0x0013),3); // Image Number

      Add(AttributeDescriptor(0x0028,0x0010),3); // Rows
      Add(AttributeDescriptor(0x0028,0x0011),3); // Columns
      Add(AttributeDescriptor(0x0028,0x0012),3); // Planes
      Add(AttributeDescriptor(0x0028,0x0002),3); // Sample per pixels
      Add(AttributeDescriptor(0x0028,0x0008),3); // Number of Frames 
      Add(AttributeDescriptor(0x0028,0x0004),3); // Photometric Interpretation
      Add(AttributeDescriptor(0x0028,0x0103),3); // Pixel Representation

      Add(AttributeDescriptor(0x0020,0x0032),3); // Image Position Patient
      Add(AttributeDescriptor(0x0020,0x0037),3); // Image Orientation Patient
      Add(AttributeDescriptor(0x0020,0x1041),3); // Slice Location
      Add(AttributeDescriptor(0x0028,0x0006),3); // Planar Configuration

      Add(AttributeDescriptor(0x0028,0x0030),3); // Pixel Spacing
      Add(AttributeDescriptor(0x0028,0x0100),3); // AlocatedBits
      Add(AttributeDescriptor(0x0028,0x0101),3); // StoredBits

      Add(AttributeDescriptor(0x0008,0x0008),3); // Image Type
      Add(AttributeDescriptor(0x0008,0x0023),3); // Content Date
      Add(AttributeDescriptor(0x0008,0x0033),3); // Content Time

      Add(AttributeDescriptor(0x0020,0x4000),3); // Image Comments

      Add(AttributeDescriptor(0x0004,0x1500,   // File Name
				AttributeDescriptor::LABEL),3);
      Add(AttributeDescriptor(0x0028,0x1052),3); // Rescale Intercept
      Add(AttributeDescriptor(0x0028,0x1053),3); // Rescale Slope

      Add(AttributeDescriptor(0x0050,0x0004),3); // Calibration Image

      Add(AttributeDescriptor(0x0020,0x0052   // Frame Reference UID
					   ),3);
      Add(AttributeDescriptor(0x0008,0x0016),3); // SOP Class UID
      Add(AttributeDescriptor("FullFileName",  // Full file name
				"Full file name",
				AttributeDescriptor::IDENTIFIER),3); 

    }

	//////////////////////////////////////////////////////////////
	// create a descriptor (name, attributes...) from a file)	//
	// @param : file path						//
	// return : -							//
	//////////////////////////////////////////////////////////////
	void Descriptor::createDescriptorfromFile(const std::string &i_name)
	{
		Clear();
		
		// read file and put in buffer
		std::ifstream i_file(i_name.c_str());
		std::stringstream buffer;
		buffer << i_file.rdbuf();
		std::string line;
		bool bname;
		int ilevel = -1;

		
		while(std::getline(buffer, line))
		{
			if(line =="<level>")
			{	//increment levels.
				ilevel++;
				bname = true;
			}
			else if(bname)
			{
				// For each level, a name to describe it
				Add(LevelDescriptor(line));
				bname = false;
			}
			else if(line.empty()) // to avoid end line
			{
				return;
			}
			else
			{ 
				// split line to find all tags
				std::vector<std::string> descriptors;
				std::string separator = " ";
				std::string::size_type last_pos = line.find_first_not_of(separator);
				//find first separator
				std::string::size_type pos = line.find_first_of(separator, last_pos);
				while(std::string::npos != pos || std::string::npos != last_pos)
				{
   					descriptors.push_back(line.substr(last_pos, pos - last_pos));
					last_pos = line.find_first_not_of(separator, pos);
					pos = line.find_first_of(separator, last_pos);
				}
				
				// By default, the last tag is at zero and not recorded but if take in count
				unsigned int flag = 0;
				if(descriptors.size() == 4)
				{
					std::stringstream val;
					val << std::dec << descriptors[3];
					val>> flag;
				}

				// if Dicom tag, use "group" and "element" descriptor
				if(descriptors[0] == "D")
				{	std::stringstream val, val2;
					unsigned short group;
					unsigned short element;
					val <<   std::dec << descriptors[1] ;
					val >> std::hex >> group;
					val2 << std::dec <<  descriptors[2];
					val2 >> std::hex >> element;
					Add(AttributeDescriptor( group,element,flag), ilevel);
				}

				else if(descriptors[0].find("#") != -1)
				{
					// commented line continue to next line
				}
				else
				{	boost::algorithm::replace_all(descriptors[2],"_"," ");
					Add(AttributeDescriptor( descriptors[1].c_str(),descriptors[2].c_str(),flag), ilevel);
				}
			}
		}
  	}

    //==================================================================

    //==================================================================
    /// Adds a LevelDescriptor at the end of the list
    void Descriptor::Add(const LevelDescriptor& d)
    {
      mLevelDescriptorList.push_back(d);
    }
    //==================================================================
    
    //==================================================================  
    /// Adds an AttributeDescriptor to level l
    void Descriptor::Add(const AttributeDescriptor& d, int l)
    {
      mLevelDescriptorList[l].Add(d);
      // TO DO : update DicomTagToName and NameToDicomTag map
    }
    //==================================================================

    //==================================================================
    /// Clears the Descriptor
    void Descriptor::Clear()
    {
      mLevelDescriptorList.clear();
    }

    //==================================================================

    //==================================================================
    /// Builds the key to value map of all the attributes of the tree
    void Descriptor::BuildAttributeMap( std::map<std::string,std::string>& map ) const
    {
      map.clear();
      LevelDescriptorListType::const_iterator l;
      for (l = GetLevelDescriptorList().begin();
	   l!= GetLevelDescriptorList().end();
	   ++l)
	{
	  LevelDescriptor::AttributeDescriptorListType::const_iterator a;
	  for (a = l->GetAttributeDescriptorList().begin();
	       a!= l->GetAttributeDescriptorList().end();
	       ++a)
	    {
	      map[a->GetKey()]="";
	    }
	}
    }

	
    //==================================================================
	// test if an attribute is present in DescriptionList
	// return level's name
	const std::string Descriptor::isExist(const std::string i_attr)
	{
		std::string name = "";
		LevelDescriptorListType::const_iterator l = GetLevelDescriptorList().begin();
		for (;	   l!= GetLevelDescriptorList().end(); ++l)
		{
			LevelDescriptor::AttributeDescriptorListType::const_iterator a = l->GetAttributeDescriptorList().begin();
	        for(;a!= l->GetAttributeDescriptorList().end(); ++a)
			{
				if (a->GetKey() == i_attr)
				{
					name = l->GetName();
					break;
				}
			}
		}
		return name.c_str();
	}
  }
}
