/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#ifndef __creaImageIOTreeHandler_h_INCLUDED__
#define __creaImageIOTreeHandler_h_INCLUDED__

#include <creaImageIOTree.h>

namespace creaImageIO
{


  //=======================================================================
  //class TreeHandlerStatistics;
  //=======================================================================
  /**
   * \ingroup Model
   */
  //=======================================================================

  /// Abstract class which 'handles' a Tree structure 
  class TreeHandler
  {
  public:

    //====================================================================
    //  typedef TreeHandlerStatistics Statistics;
    //====================================================================

    //====================================================================
    /// Ctor
    TreeHandler() {}
    /// Virtual dtor
    virtual ~TreeHandler() {}
    //====================================================================

    //====================================================================
    /// Returns the Tree handled 
    tree::Tree& GetTree() { return mTree; }
    /// Returns the Tree handled (const)
    const tree::Tree& GetTree() const { return mTree; }
    //====================================================================

    //====================================================================
    // QUERY METHODS
    /// Is the 'source' readable ?
    virtual bool IsReadable() { return false; }
    /// Is the 'source' writable ?
    virtual bool IsWritable() { return false; }
    //====================================================================


    //====================================================================
    // INITIALIZATION / FINALIZATION
    //====================================================================

    //====================================================================
    /// Opens an existing 'source' 
    // Default mode is read only 
    // If IsWritable and writable==true then opens in read/write mode
    virtual bool Open(bool writable = false) { return false; }
    /// Closes the 'source'
    virtual bool Close() { return false; }
    /// Creates a new 'source' 
    // Default mode is read only 
    // If IsWritable and writable==true then opens in read/write mode
    virtual bool Create(bool writable = false) { return false; }
    /// Destroys the 'source'
    virtual bool Destroy() { return false; }
	/// Begins a transaction
	virtual void BeginTransaction(){}
	///Commits results and ends transaction
	virtual void EndTransaction(){}
    //====================================================================


    //====================================================================
    // READ METHODS
    //====================================================================


    //====================================================================
    /// Returns the number of children of the Node *WITHOUT LOADING THEM*
    // REM : The Tree itself is a Node and asking for its number of 
    //       children returns the number of children of level 1.
    virtual unsigned int GetNumberOfChildren(tree::Node* n) { return 0; }
    //====================================================================

	//====================================================================
    /// Returns the attribute requested. Useful for synchronization.
	virtual void GetAttribute(std::string levelDescriptor,
									   std::string searchParam, 
									   std::string searchVal, 
									   std::string key, 
									   std::string& result){}
    //====================================================================

    //====================================================================
    /// Recursively loads the children of node 'parent' until maxlevel 
    // is reached.
    // If maxlevel <= 0 then loads all the sub-tree rooted at parent 
    // If parent == NULL or parent == tree then starts with the 'children' of 
    // the tree itself.
    // Returns the total number of children loaded.
    virtual int LoadChildren(tree::Node* parent, int maxlevel) 
    { return 0; }
    //====================================================================

    //====================================================================
    /// Unloads the Node and its descendants
    // WITHOUT altering the source, e.g. the database
    virtual void UnLoad(tree::Node* n) { return; }
    //====================================================================

	//====================================================================
    /// Returns the top level node id for the given search param and search value
    virtual void GetTopLevelNodeId(const std::string& searchParam, 
											const std::string& searchValue, 
											std::string& parent_id){ return; }
    ///====================================================================


    //====================================================================
    // WRITE METHODS : WORK ONLY IN WRITE MODE
    //====================================================================
    typedef tree::Node::AttributeMapType AttributeMapType;
    /// Adds a branch in the tree with the attributes provided
    // returns the Level in the tree where the branch was connected 
    // (-1 for error, 0 for top level, etc. ) 
    // Of course the branch is loaded on exit
    virtual int AddBranch( const AttributeMapType& ) { return -1; }
    /// Removes the node and its descendants 
    virtual bool Remove(tree::Node*)  { return false; }
    /// Sets an attribute of a Node
    virtual bool SetAttribute(tree::Node*, 
			      const std::string& key,
			      const std::string& value) { return false; }
	// Sets an attribute
    virtual void SetAttribute(const std::string& levelDescriptor, 
			      const std::string& key,
			      const std::string& value,
				  const std::string& searchParam, 
				  const std::string& searchVal){}
	//Deletes the tuple that matches the parameters given
	virtual void DeleteTuple(std::string levelDescriptor, std::string key, std::string value){}
	//Deletes the entries that match the parameters given
	virtual void RemoveEntries(const std::string i_table, 
		const std::string i_attribute, 
		const std::string i_operand, 
		const std::string i_val){}
 
    //====================================================================
	/// get all attributes for a file
	virtual void getAllAttributes(std::string i_filename, std::map<std::string, std::string> &i_results) =0;

  private:
    /// The handled tree
    tree::Tree mTree;

  };
  // EO class TreeHandler
  //=======================================================================
  /*
  //=======================================================================
  /// Memorizes statistics on operations done by a tree handler
  // (nodes created, removed, ...)
  class TreeHandlerStatistics
  {
  public:
    //====================================================================
    /// Ctor
    TreeHandler(TreeHandler* tree) : mTreeHandler(tree) { Reset(); }
    /// Dtor
    ~TreeHandler() {}
    /// Resets the stats
    void Reset();
    /// Prints the stats
    void Print();

    /// 
    void CreateNode(int level) { mNumberCreatedNode[level]++; }
    void DeleteNode(int level) { mNumberDeletedNode[level]++; }

  protected:
      TreeHandler* mTreeHandler;
    std::vector<int> mNumberCreatedNode;
    std::vector<int> mNumberDeletedNode;
    
    
    ///====================================================================
  };
  // EO class TreeHandlerStatistics
  //=======================================================================
  */

} // EO namespace creaImageIO

// EOF
#endif  

