/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#include <creaImageIOTreeNode.h>
#include <creaImageIOTree.h>
#include <creaImageIOSystem.h>
#include <algorithm>
#include <creaImageIOGimmick.h>

namespace creaImageIO
{
  namespace tree
  {

    //=============================================================
    /// Ctor with parent
    Node::Node(Node* parent)
      : mParent(parent),//mData(0),
	mChildrenLoaded(false)
    {
	mData.reset();
      if (parent) 
	{
	  GimmickDebugMessage(6,"Default Node constructor (level "<<GetLevel()<<")"
			 << std::endl);
	  // Insert into parent's children list
	  InitializeAttributeMap();
	 parent->GetChildrenList().push_back(this);
	}
      else
	{
	  GimmickDebugMessage(6,"Default Node constructor without parent"	
			 << std::endl);
	}
    }
    //=============================================================

    //=============================================================
    /// Ctor with parent and attributes map 
    Node::Node(Node* parent, const AttributeMapType& attr)
     : mParent(parent),//mData(0),
	mChildrenLoaded(false)
    {
	mData.reset();
      GimmickDebugMessage(6,"Node constructor (level "<<GetLevel()<<")"
		     << std::endl);

     if (parent) 
	{
	  // Insert into parent's children list
	  parent->GetChildrenList().push_back(this);
	  // Initialize attributes
	  LevelDescriptor::AttributeDescriptorListType::const_iterator a;
	  for (a = GetTree()->GetAttributeDescriptorList(GetLevel()).begin();
	       a!= GetTree()->GetAttributeDescriptorList(GetLevel()).end();
	       ++a)
	    {
	      std::string v;
	      AttributeMapType::const_iterator i = attr.find(a->GetKey());
	      if ( i != attr.end() )  
		{
		  v = i->second;
		}
	      GimmickDebugMessage(6,"Setting attribute '"<<a->GetName()<<"' = '"
			     <<v<<"'"<<std::endl);
	      UnsafeSetAttribute( a->GetKey(), v );
	    }
	}
      
    }
    //=============================================================


    //=============================================================
    Node::~Node()
    {
      GimmickDebugMessage(6,"Node destructor"
		     << std::endl);
      ChildrenListType::iterator i;
      for (i=GetChildrenList().begin(); i!=GetChildrenList().end(); i++)
	{
	  delete *i;
	}
	mData.reset();
    }
    //=============================================================


    //=============================================================
    /// Initializes the attribute map i.e. creates the entries
    void Node::InitializeAttributeMap()
    {
      // Initialize attributes
      LevelDescriptor::AttributeDescriptorListType::const_iterator a;
      for (a = GetTree()->GetAttributeDescriptorList(GetLevel()).begin();
	   a!= GetTree()->GetAttributeDescriptorList(GetLevel()).end();
	   ++a)
	{
	  UnsafeSetAttribute( a->GetKey(), "" );
	}
    }
    //=============================================================

    //=============================================================
    /// Returns the level descriptor of the node
    const LevelDescriptor& Node::GetLevelDescriptor() const
    { 
      return GetTree()->GetLevelDescriptor(GetLevel()); 
    }

	//=============================================================

    //=============================================================
    /// Returns the attribute descriptor of the passed parameter
	const AttributeDescriptor& Node::GetAttributeDescriptor(const std::string& k)const
	{
		LevelDescriptor::AttributeDescriptorListType::const_iterator a;
      for (a = GetTree()->GetAttributeDescriptorList(GetLevel()).begin();
	   a!= GetTree()->GetAttributeDescriptorList(GetLevel()).end();
	   ++a)
		{
		
			if(a->GetKey()==k)
			{
				return *a;
			}
		
		}
		return *a;
	}
    //=============================================================

    //=============================================================
    int Node::RemoveChildrenFromList(Node* node)
    {
      ChildrenListType::iterator i = find(GetChildrenList().begin(),
					  GetChildrenList().end(),
					  node);
      if (i != GetChildrenList().end())
	{
	  GetChildrenList().erase(i);
	}
	return (int)GetChildrenList().size();
    }
    //=============================================================

    //=============================================================
    const std::string& Node::GetAttribute(const std::string& k) const
    {
      //    std::cout << "this = "<<(void*)this<<std::endl;
      //    std::cout << "mFieldValueMap="<<(void*)(&mFieldValueMap)<<std::endl;
      AttributeMapType::const_iterator i = mAttributeMap.find(k);
      if (i == mAttributeMap.end())
	{
	  static std::string def("");
	  return def;
	  //	CREAIMAGEIO_ERROR("DicomNode::GetFieldValue : no field with key '"<<k<<"'");
	}
      return i->second;
  }
  //=============================================================

  //=============================================================
    void Node::SetAttribute(const std::string& k, 
				 const std::string& v)
  {
    AttributeMapType::iterator i = mAttributeMap.find(k);
    if (i==mAttributeMap.end())
      {
	std::cout<<"[Gimmick!] Node::SetAttribute : no attribute with key '"
		 <<k<<"'"<<std::endl;
	creaError( "[Gimmick!] Node::SetAttribute : no attribute with key '"
		   <<k<<"'");
      }
    i->second = v;
  }
    //=============================================================

    //=============================================================
    bool Node::Matches(  const AttributeMapType& m ) const
    {
      GimmickDebugMessage(2,"'"<<GetLabel()<<"' matching..."<<std::endl);
      const std::vector<std::string>& id 
	= GetLevelDescriptor().GetIdentifierList();
      std::vector<std::string>::const_iterator i;
      for (i = id.begin(); i != id.end(); ++i)
	{
 	  if (mAttributeMap.find(*i)->second != m.find(*i)->second ) 
	    {
	      GimmickDebugMessage(2,"IDENTIFIER '"<<*i<<"' values do not match"<<std::endl);
	      return false;
	    }
	  GimmickDebugMessage(2,"IDENTIFIER '"<<*i<<"' values match"<<std::endl);
	}
      return true;
    }
    //=============================================================
    
    //=============================================================
    void Node::Print() const
    {
      std::string mess;
      for (int i = 0; i<GetLevel(); ++i) mess += "  ";
      mess += "|_ " + GetLabel();
      GimmickMessage(1,mess<<std::endl);
      ChildrenListType::const_iterator j;
      for (j=GetChildrenList().begin(); j!=GetChildrenList().end(); j++)
	{
	  (*j)->Print();
	} 
    }
    //=============================================================

    //=============================================================
    std::string Node::GetLabel() const
    {
      std::string l;
      const std::vector<std::string>& label 
	= GetLevelDescriptor().GetLabelList();
      
      std::vector<std::string>::const_iterator i;
      for (i = label.begin(); i != label.end(); )
	{
	  GimmickDebugMessage(9,"LABEL '"<<*i<<"'"<<std::endl);
	  AttributeMapType::const_iterator j = mAttributeMap.find(*i);
	  if (j != mAttributeMap.end())
	    {
	      l += j->second;
	      ++i;
	      if (i != label.end()) l += "|";
	    }
	  else 
	    {
	      GimmickError("Node::GetLabel() : LABEL attribute '"
			   <<*i
			   <<"' is not in node attribute map (should be!)" );
	    }
	}
      if (l.size()==0) l="?";
      return l;
    }
    //=============================================================

  }

}
 
