/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/


#include "creaImageIOSystem.h"
#include "creaImageIOUltrasonixImageReader.h"
#include <creaVtk.h>
#include <boost/filesystem/path.hpp>

#if defined(_WIN32)
#pragma warning(disable: 4996)
#endif

namespace creaImageIO
{
#define HEADER_SIZE	19
#define TYPE_RF		16
#define TYPE_B8		4
#define TYPE_B32	8


  //=====================================================================
  UltrasonixImageReader::UltrasonixImageReader()
  {
    SetName("Ultrasonix");
  }
  //=====================================================================
  
  //=====================================================================
  UltrasonixImageReader::~UltrasonixImageReader()
  {
  }
  //=====================================================================

  //=====================================================================
  struct Ultrasonix_header
  {
    // frames, width, height, ultrasounds frequency, sampling rate
    int type, frame, width, height, frequency, samplingRate;
  };
  //=====================================================================


  //=====================================================================
  bool ReadHeader( FILE *Ultrasonix_file, Ultrasonix_header& h )
  {
    //int *header=(int*)malloc(sizeof(int)*HEADER_SIZE);
    int header[HEADER_SIZE];
    fread(header, sizeof(int), HEADER_SIZE, Ultrasonix_file);
    if (ferror(Ultrasonix_file))
    	return false;
    h.type         = header[1];
    h.frame        = header[2];
    h.height       = header[3];
    h.width        = header[4];
    h.frequency    = header[14];
    h.samplingRate = header[15];
    //free(header);  
    return true;
  }
  //=====================================================================

  //=====================================================================
  bool UltrasonixImageReader::CanRead(const std::string& filename)
  { 
    long size = -1;
    bool ok = false;
    FILE *Ultrasonix_file=fopen(filename.c_str(), "rb");
    if (Ultrasonix_file) 
    {
	Ultrasonix_header h;
	if (!ReadHeader(Ultrasonix_file, h) )
	{
		fclose(Ultrasonix_file);
        	std::cout << "cannot read Ultrasonix header for file [" << filename << "]" << std::endl;				
		return false;	
	}

	fseek(Ultrasonix_file,0,SEEK_END);		// go to end of file
	if (h.type == TYPE_RF)
		size = (ftell(Ultrasonix_file) - (HEADER_SIZE+h.frame) * sizeof(int)) / sizeof(short);
	else if (h.type == 1)//TYPE_B8)
		size = (ftell(Ultrasonix_file) - (HEADER_SIZE+h.frame+4) *  sizeof(int)) / sizeof(char);
	else if (h.type == TYPE_B32)
		size = (ftell(Ultrasonix_file) - HEADER_SIZE * sizeof(int)) / sizeof(int);

	// check if the data size corresponds to the dimensions of the images
	if (size == h.width * h.height * h.frame )
		ok = true;

	fclose(Ultrasonix_file);
    }
    return ok;
  }
  //=====================================================================
  void UltrasonixImageReader::getAttributes(const std::string filename,
		std::map <std::string , std::string> &infos, std::vector<std::string> i_attr)
  {
	  //TO DO
  }
  
  //=====================================================================
  vtkImageData* UltrasonixImageReader::ReadImage(const std::string& filename)
  {
    FILE *Ultrasonix_file=fopen(filename.c_str(),"rb");
    if (!Ultrasonix_file) 
    {
    	std::cout << "cannot open file [" << filename << "]" << std::endl;
    	return 0;
    }
    Ultrasonix_header h;
    if (!ReadHeader(Ultrasonix_file,h)) 
    {
        std::cout << "cannot read Ultrasonix header for file [" << filename << "]" << std::endl;
	fclose(Ultrasonix_file);  
	return 0;
    }

    long frame_size = h.height   * h.width;      
    long im_size    = frame_size * h.frame;

    short *dataRF,  *ptrRF;
    char  *dataB8,  *ptrB8;
    int   *dataB32, *ptrB32;
    vtkImageData* im;
    int temp;

    switch (h.type)
    {
	case TYPE_RF:
		dataRF = (short*)malloc(sizeof(short)*im_size);
		ptrRF  = dataRF;

		for (int k=0; k<h.frame; k++)
		{
			int frame_number;
			fread(&frame_number, sizeof(int), 1, Ultrasonix_file);
			fread(ptrRF,sizeof(short), frame_size, Ultrasonix_file);
			ptrRF += frame_size;
		}
		fclose(Ultrasonix_file);  

		im = crea::NewVtkImageDataFromRaw( dataRF, h.width, h.height, h.frame);
	break;

	case TYPE_B8:
		dataB8 = (char*)malloc(sizeof(char)*im_size);
		ptrB8  = dataB8;
		for (int k=0; k<h.frame; k++)
		{
			fread(ptrB8,sizeof(char), frame_size, Ultrasonix_file);
			ptrB8 += frame_size;
		}
		// in mode b frames width and height are inverted
		temp     = h.width;
		h.width  = h.height;
		h.height = temp;

		fclose(Ultrasonix_file);  

	im = crea::NewVtkImageDataFromRaw( dataB8, h.width, h.height, h.frame);
	break;

	case TYPE_B32:
		dataB32 = (int*)malloc(sizeof(int)*im_size);
		ptrB32  = dataB32;
		for (int k=0; k<h.frame; k++)
		{
			fread(ptrB32, sizeof(int), frame_size, Ultrasonix_file);
			ptrB32 += frame_size;
		}
		// in B mode frames width and height are inverted
		temp     = h.width;
		h.width  = h.height;
		h.height = temp;

		fclose(Ultrasonix_file);  

		im = crea::NewVtkImageDataFromRaw( dataB32, h.width, h.height, h.frame);
	break;
    }

    return im;
}
  //=====================================================================
  

  //=====================================================================
  void UltrasonixImageReader::PushBackExtensions(std::vector<std::string>& v)
  {
    v.push_back("Ultrasonix");
  }
  //=====================================================================
 


  //=====================================================================
 void UltrasonixImageReader::ReadAttributes(const std::string& filename, 
				      std::map<std::string,std::string>& attr)
 {
    GimmickMessage(2,"Reading attributes from '" << filename << std::endl);

    FILE *Ultrasonix_file = fopen(filename.c_str(), "rb");
    if (!Ultrasonix_file)
    {
    	std::cout << "cannot open RF file [" << filename << "]" << std::endl;
    	return;
    }

    Ultrasonix_header h;
    if (!ReadHeader(Ultrasonix_file, h)) 
    {
        fclose(Ultrasonix_file);
        std::cout << "cannot read Ultrasonix Attributes for RF file [" << filename << "]" << std::endl;  
	return;
    }

    fclose(Ultrasonix_file);  
  
    // Columns
    char cols[128];
    sprintf(cols,"%i", h.width);
    // Rows
    char rows[128];
    sprintf(rows,"%i", h.height);
    // Planes 
    char planes[128];
    sprintf(planes,"%i", h.frame);
    // Sampling frequency
    char samplingFrequency[128];
    sprintf(samplingFrequency,"%i", h.samplingRate);
    // Transducer frequency
    char transducerFrequency[128];
    sprintf(transducerFrequency,"%i", h.frequency);
   
    // 
    std::map<std::string,std::string>::iterator i;
    if ( (i = attr.find("FullFileName")) != attr.end())
    {
	   i->second = filename;
    }
    if ( (i = attr.find("D0004_1500")) != attr.end())
    {
	   boost::filesystem::path full_path(filename);
	   std::string f = full_path.leaf().string();
	   i->second = f;
    }
    if ( (i = attr.find("D0028_0010")) != attr.end())
    {
	   i->second = rows;
    }
    if ( (i = attr.find("D0028_0011")) != attr.end())
    {
	   i->second = cols;
    }
    if ( (i = attr.find("D0028_0012")) != attr.end())
    {
	   i->second = planes;
    }
    if ( (i = attr.find("D003a_001a")) != attr.end())
    {
       i->second = samplingFrequency;
    }
    if ( (i = attr.find("D0018_6030")) != attr.end())
    {
	   i->second = transducerFrequency;
    }
    
    GimmickMessage(2,"Attributes map:"<<std::endl<<attr<<std::endl);
    return;
}
  //=====================================================================

} // namespace creaImageIO
