/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/



#include <creaImageIOWxPACSConnectionPanel.h>
#include <creaImageIOWxTreeView.h>
#include <creaImageIOWxCustomizeConfigPanel.h>
#include <creaImageIOWxListenerPanel.h>
#include <creaImageIOWxEditFieldsPanel.h>
#include <creaImageIOWxAttributeSelectionPanel.h>
#include <creaImageIOWxDescriptorPanel.h>
#include <creaImageIOWxDumpPanel.h>
#include <creaImageIOWxExportDlg.h>
#include <creaImageIOWxOutputDlg.h>
#include <creaImageIOOutputModel.h>

#if defined(_WIN32)
#pragma warning(disable: 4996)
#endif

using namespace crea;
// Icons
#include "icons/accept.xpm"
#include "icons/add.xpm"
#include "icons/folder-down.xpm"
#include "icons/page-down.xpm"
#include "icons/remove.xpm"
#include "icons/database-add.xpm"
#include "icons/create-database.xpm"
#include "icons/help.xpm"
#include "icons/synchronize.xpm"
#include "icons/settings.xpm"
#include "icons/tools.xpm"
//#include "icons/import.xpm"

#include <wx/imaglist.h>
#include <wx/popupwin.h>
#include<boost/filesystem/operations.hpp>
#if defined(BUILD_BRUKER)
	#include "bruker2dicom.h"
#endif


#include <creaImageIOGimmick.h>
#ifdef _DEBUG
#define new DEBUG_NEW
#endif

namespace creaImageIO
{
   
  //======================================================================
  // The ids of the different tools
  enum
    {
        TOOL_ADDFILES_ID    = 1,
        TOOL_ADDDIR_ID      = 2,
        TOOL_ADDDATABASE_ID = 3,
        TOOL_REMOVE_ID      = 4,
        TOOL_SYNCHRONIZE_ID = 5,
        TOOL_HELP_ID        = 6,
        TOOL_SETTINGS_ID    = 7,
        TOOL_TOOLS_ID       = 8,
        TOOL_CREATEDB_ID    = 9,
        TOOL_PACS_ID        = 10
    };
  //======================================================================

  //================================================================
  // 
  const int icon_number = 11;
  // Icon ids
  typedef enum
    {
      Icon_create_database,
      Icon_accept,
      Icon_add,
      Icon_folder_down,
      Icon_page_down,
      Icon_database_add,
      Icon_remove,
      Icon_synchronize,
      Icon_help,
      Icon_settings,
      Icon_tools
    }
    icon_id;
  //================================================================

  //================================================================
  /*
  const icon_id Icon[5] = { Icon_Database,  
			    Icon_Patient,
			    Icon_Study,
			    Icon_Series,
			    Icon_Image };
  */
  //================================================================


  //======================================================================
  // CTor
	WxGimmickView::WxGimmickView(boost::shared_ptr<Gimmick> gimmick,
			       wxWindow *parent, 
			       const wxWindowID id,
			       const wxPoint& pos, 
			       const wxSize& size,
			       int min_dim,
                               int max_dim,
			       int number_of_threads)
    : wxPanel(parent,id,pos,size),
      GimmickView(gimmick, number_of_threads),
      mProgressDialog(0),
      mConstructed(false)
  {
    GimmickDebugMessage(1,"WxGimmickView::WxGimmickView"
			<<std::endl);
    // Sets the current directory to the home dir
    mCurrentDirectory =  std2wx(gimmick->GetHomeDirectory());


     // Connect the AddProgress callback
    gimmick->ConnectAddProgressObserver 
     ( boost::bind( &WxGimmickView::OnAddProgress , this, _1 ) );

    // Create the list of icons (mIcon)
    CreateIconList();

    // Global sizer
    msizer = new wxBoxSizer(wxVERTICAL);

    // Create the tool bar
    CreateToolBar(); 
    msizer->Add( mToolBar, 0, wxGROW, 0);

    // Split part below toolbar into notebook for views and panel
    // for preview, messages...
    mSplitter = new wxSplitterWindow( this , -1);

    // Notebook
    mNotebook = new wxNotebook(mSplitter,
			       -1, wxDefaultPosition, wxDefaultSize, 0);

    //Gimmick
    mGimmick=gimmick;

    mSelectionMaxDimension = max_dim;
    mSelectionMinDimension = min_dim;
    
    // Create the views
    CreateTreeViews();

    // Bottom panel 
    mBottomPanel = new wxPanel(mSplitter,-1);
    
    mbottom_sizer = new wxBoxSizer(wxVERTICAL); //HORIZONTAL);
    
    
    // Previewer
    mViewer = new WxViewer(mBottomPanel, wxID_ANY, wxT("Gimmick! Viewer"),wxDefaultPosition, wxDefaultSize );
	//pointers.push_back(new ImagePointerHolder(GetDefaultImage())
	pointers.push_back(boost::shared_ptr<creaImageIO::ImagePointerHolder>(new ImagePointerHolder(GetDefaultImage())));

	mViewer->SetImageVector(pointers);
	mViewer->StartPlayer();


    mbottom_sizer->Add(mViewer,1,wxGROW,1);
    //    mViewer->Show();

    mText = new wxStaticText(mBottomPanel, wxID_ANY, wxT("Welcome to Gimmick!"));
    mbottom_sizer->Add(mText,0,wxGROW,0);

	  
	  
    mBottomPanel->SetSizer(mbottom_sizer);

    // Splitting
    /// \TODO fix warning: unused variable hsize
    int hsize = size.GetHeight();

    int top_minsize = 450;
    int bottom_minsize = 50;

    mSplitter->SetMinimumPaneSize( bottom_minsize );
    mSplitter->SplitHorizontally( mNotebook, mBottomPanel, 
				  top_minsize);

    msizer->Add( mSplitter, 1, wxGROW, 0);

    mProgressDialog=0;
    SetSizer( msizer );     
    SetAutoLayout(true);
    Layout();
    //mListener=new Listener();
    //mListener->ConnectObserver(boost::bind( &WxGimmickView::OnDriveMount, this, _1 ) );
    //mListener->Create();
   // mListener->Run();
   // mListener->Pause();

    mConstructed = true;
  }
  //======================================================================

  //======================================================================
  /// Destructor
  WxGimmickView::~WxGimmickView()
  {
	// stop the viewer before application exit.
    mViewer->StopPlayer();
    GimmickDebugMessage(1,"WxGimmickView::~WxGimmickView"
			<<std::endl);
	delete mIcon;
	delete mViewer;
    //if(mListener->IsAlive())   {        mListener->Delete();    }
  }
  //======================================================================
  
  //======================================================================
  /// Creates the tool bar
  void WxGimmickView::CreateToolBar()
  {
    long style = wxTB_HORIZONTAL | wxNO_BORDER | wxTB_TEXT;
    mToolBar = new wxToolBar(this,-1,wxDefaultPosition,wxDefaultSize,
			     style);

    mToolAddFile = mToolBar->AddTool( TOOL_ADDFILES_ID, 
				      _T("Add file(s)"),
				      mIcon->GetBitmap(Icon_page_down),
				      _T("Add one or more file to database")
				      );
    mToolAddDir = mToolBar->AddTool( TOOL_ADDDIR_ID, 
				      _T("Add folder"),
				      mIcon->GetBitmap(Icon_folder_down),
				      _T("Add the content of a folder to database")
				      );
    mToolAddDatabase = mToolBar->AddTool( TOOL_ADDDATABASE_ID, 
				      _T("Open database"),
				      mIcon->GetBitmap(Icon_database_add),
				      _T("Open a local or distant database")
				      );
    mToolRemove = mToolBar->AddTool( TOOL_REMOVE_ID, 
				      _T("Remove"),
				      mIcon->GetBitmap(Icon_remove),
				      _T("Remove selected items")
				      );
    mToolSynchronize = mToolBar->AddTool( TOOL_SYNCHRONIZE_ID, 
				      _T("Synchronize"),
				      mIcon->GetBitmap(Icon_synchronize),
				      _T("Synchronizes the database with disk")
				      );
    mToolHelp = mToolBar->AddTool( TOOL_HELP_ID, 
				      _T("Help"),
				      mIcon->GetBitmap(Icon_help),
				      _T("Open help window")
				      );
    mToolSettings = mToolBar->AddTool( TOOL_SETTINGS_ID, 
				      _T("System settings"),
				      mIcon->GetBitmap(Icon_settings),
				      _T("Allows the modification of various system settings")
				      );
    mToolTools = mToolBar->AddTool( TOOL_TOOLS_ID, 
				      _T("Tools"),
				      mIcon->GetBitmap(Icon_tools),
				      _T("Applies tools to images")
				      );
    mToolAddFile = mToolBar->AddTool( TOOL_CREATEDB_ID, 
				      _T("Create database"),
				      mIcon->GetBitmap(Icon_create_database),
				      _T("Create DB from an Attributes Descriptor file")
				      );
#if defined(BUILD_PACS)
	mToolAddFile = mToolBar->AddTool( TOOL_PACS_ID, 
				      _T("PACS Connection,"),
				      mIcon->GetBitmap(Icon_create_database),
				      _T("Echo, Find and Get to a PACS")
				      );
#endif
    //const wxBitmap& bitmap1, const wxString& shortHelpString = "", wxItemKind kind = wxITEM_NORMAL)

    mToolBar->Realize();
  }
  //======================================================================

 
  //======================================================================
  /// Create the tree view for TreeHandler provided
  void WxGimmickView::CreateTreeView( TreeHandler* h)
  {
    std::string name(h->GetTree().GetAttribute("Name"));
    GimmickMessage(2,"Creating the tree view for '"<<
		   name<<"'"<<std::endl);
    // Create the WxTreeView
    WxTreeView* view = new WxTreeView(h, this, mNotebook, -1);

    // TO DO : TEST THAT A VIEW WITH SAME NAME IS NOT
    // ALREADY IN THE MAP
    GetTreeViewMap()[name] = view;

    // Add Notebook page
    mNotebook->AddPage( view, crea::std2wx(name) );
	
  }

  //======================================================================
  void WxGimmickView::GetSelectedImages(std::vector<vtkImageData*>& s, int dim)
  {
	std::vector<std::string> files;
	GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetSelectedAsString(files);
	//ReadImagesNotThreaded(s, files, dim);


		std::string db_name = crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()));
	//		ReadImagesNotThreadedInfosInVector(db_name,output, files,dim,1);
  }



  //======================================================================
  void WxGimmickView::GetSelectedImagesInVector(std::vector<vtkImageData*>& s, int dim)
  {
	std::vector<std::string> files;
	std::string db_name = crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()));
	GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetSelectedAsString(files);

	ReadImagesNotThreadedInVector(s, files, dim);
  }


  /// By default if out_infos is empty, we dont' provide informations, we return only vtkImageData 
  /// if out_infos has only one entry "all" we provide all database informations
  void WxGimmickView::getSelectedFiles(std::vector<OutStrGimmick> &outG, std::vector< std::string> i_attr, 
                                       bool mult, const std::string out_model)
  {
	// First we select the files
   std::vector<std::string> files;
	std::string db_name = crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()));
	GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetSelectedAsString(files);

	std::string asking;
	std::string dim;
	bool bsingle = isSingle(files.front());
	int i_dim_out;
	mGimmick->GetSetting(SETTINGS_OUTPUT_ASK, asking);
	mGimmick->GetSetting(SETTINGS_OUTPUT_DIM, dim);


	if (asking == "true")
	{
		// display the output dialog box
		// get dim
		int idim;
		sscanf(dim.c_str(),"%d",&idim);
		WxOutputDlg *dlg = new WxOutputDlg(this,files, idim -1, bsingle);
		if (dlg->ShowModal() == wxID_OK)
		{
			dim = dlg->getDim();
			mGimmick->UpdateSetting(SETTINGS_OUTPUT_ASK, dlg->getAsking());
			mGimmick->UpdateSetting(SETTINGS_OUTPUT_DIM, dim);
		} else {
			return;
		} 
	} else {
	}
	sscanf(dim.c_str(),"%d",&i_dim_out);

	// Next we create the structure for dicom output infos
	OutputAttr Oattr;
//EED 01/09/2014
	   if (i_dim_out==2)
		{
			Oattr.mult=true;
		} // i_dim_out


	Oattr.db = crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()));
	if(i_attr.empty())
	{
		// We don't send informations!
	} else if( i_attr.size() == 1 && i_attr.front() == "ALL")
	{
		// we send all database
		Oattr.inside.push_back("ALL");
	} else if( (i_attr.size() == 1 && i_attr.front() != "ALL") 
		|| (i_attr.size() >1) )
	{
		mGimmick->fillVectInfos(i_attr, Oattr);
	} else {
		// nothing
	}

	// Next we create model and sort files
	std::vector<std::string> sort_files(files);
	double zspacing = 1;
	if( !out_model.empty() )
	{
	/*	OutputModel mod(out_model);
		mGimmick->isAttributeExist(db_name, mod);
		mod.sort(files, sort_files);
	*/}

	if ( sort_files.size() > 1 && ( (bsingle && i_dim_out == 1) || (!bsingle && i_dim_out ==3) ) )
	{
		OutputModel *mod = new OutputModel();
		zspacing = mod->orderFilesWithZspacing(sort_files);
		delete mod;
	}
	// Now we read and create the waiting output (vtkImageData * or OutGimmickData *)
	readImages(outG, sort_files, Oattr, i_dim_out, zspacing);
  }




  //======================================================================

  //======================================================================


  //======================================================================
  void WxGimmickView::GetSelectedFiles(std::vector<std::string>& s)
  {
	GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetSelectedAsString(s);
  }
  //======================================================================

  //======================================================================
  void WxGimmickView::GetImages(int dim, 
				const std::vector<std::string>& files, 
				std::vector<vtkImageData*>& s)
  {
	ReadImagesNotThreaded(s,files,dim);
  }
  //======================================================================


  //=================================================
  void WxGimmickView::CreateIconList()
  {
    // Size of the icons;
    int size = 16;

    wxIcon icons[20];
    // should correspond to Icon_xxx enum
    icons[Icon_accept]          = wxIcon(accept_xpm);
    icons[Icon_add]             = wxIcon(add_xpm);
    icons[Icon_folder_down]     = wxIcon(folder_down_xpm);
    icons[Icon_page_down]       = wxIcon(page_down_xpm);
    icons[Icon_remove]          = wxIcon(remove_xpm);
    icons[Icon_database_add]    = wxIcon(database_add_xpm);
    icons[Icon_help]            = wxIcon(help_xpm);
    icons[Icon_synchronize]     = wxIcon(synchronize_xpm);
    icons[Icon_create_database] = wxIcon(create_database_xpm);
    icons[Icon_settings]        = wxIcon(settings_xpm);
    icons[Icon_tools]           = wxIcon(tools_xpm);

    //   unsigned int NbIcons = 8;
    // Make an image list containing small icons
    mIcon = new wxImageList(size,size,true);
    
    // Make all icons the same size = size of the first one
    int sizeOrig = icons[0].GetWidth();
    for ( size_t i = 0; i < icon_number; i++ )
      {
	if ( size == sizeOrig )
	  {
	    mIcon->Add(icons[i]);
	  }
        else
	  {
            mIcon->Add(wxBitmap(wxBitmap(icons[i]).ConvertToImage().Rescale(size, size)));
	  }
      }
  }
  //=================================================


  //=================================================
  void WxGimmickView::OnAddFiles(wxCommandEvent& event)
  {
    mViewer->StopPlayer();
    long style = wxOPEN | wxFILE_MUST_EXIST | wxFD_MULTIPLE;
    std::string wc("*");
    wxFileDialog* FD = new wxFileDialog( 0, 
					 _T("Select file"),
					 _T(""),
					 _T(""),
					 crea::std2wx(wc),
					 style,
					 wxDefaultPosition);
    
    if (FD->ShowModal()==wxID_OK)
      {
	wxBusyCursor busy;

	wxArrayString files;
	FD->GetPaths(files);
	unsigned int i;
	std::vector<std::string> filenames;
	for (i=0;i<files.GetCount();++i)
	{
	  filenames.push_back(wx2std(files[i]));
	  GimmickMessage(2,"Adding File "<<files[i]<<"."<<std::endl);
	}

	mProgressDialog = 
	  new wxProgressDialog(_T("Adding file(s)"),
			       _T(""),
			       1000,
//			       this,
			       NULL,
			       wxPD_ELAPSED_TIME |
			       // wxPD_ESTIMATED_TIME |
			       // wxPD_REMAINING_TIME |
			       wxPD_CAN_ABORT );

	// TO DO : select the current tree handler
	mGimmick->AddFiles(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),filenames);

	mProgressDialog->Pulse(_T("Updating view..."));

	UpdateTreeViewLevel(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),1);
	killProgress();
	DisplayAddSummary();	

      }
	mViewer->StartPlayer();	
  }
  //=================================================

  //=================================================
  void WxGimmickView::OnAddDir(wxCommandEvent& event)
  {
    mViewer->StopPlayer();
	std::string name = crea::wx2std(mNotebook->GetCurrentPage()->GetName());
    long style = wxDD_DEFAULT_STYLE | wxDD_DIR_MUST_EXIST;
    wxDirDialog* FD = 
         new wxDirDialog( 0, 
		       _T("Select directory"),
		       mCurrentDirectory,
		       style);
    
    if (FD->ShowModal()==wxID_OK)
    {
			time(&mstart);
	std::string dirname = wx2std (FD->GetPath());
	bool recurse =  isNeedRecursive(dirname);
	if (recurse)
	{
             recurse = wxMessageBox(_T("Recurse into sub-directories ?"), _T("Scan directory"), wxYES_NO,this ) == wxYES ? true : false;
	}

		wxBusyCursor busy;
		wxString title(_T("Adding directory"));
		if (recurse) 
		title = _T("Adding directory (recursive)");
		mProgressDialog = 
		new wxProgressDialog(_T("Adding directory"),
					_T(""),
					NumberFilesToAdd(dirname,recurse),

//EED					this,
					NULL,

					wxPD_ELAPSED_TIME | 
					wxPD_SMOOTH |
					// wxPD_ESTIMATED_TIME |
					// wxPD_REMAINING_TIME |
					wxPD_CAN_ABORT );

		mCurrentDirectory = FD->GetPath();  
		mGimmick->AddDir(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),dirname,recurse);
		mProgressDialog->Pulse(_T("Updating view..."));

		UpdateTreeViewLevel(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),1);
		killProgress();
		DisplayAddSummary();
	  }
    mViewer->StartPlayer();
	delete FD;
  }


  //=================================================
  // Determines number of files potentially to add to database
   int WxGimmickView::NumberFilesToAdd(const std::string &dirpath, bool recursive)
  {
	   int nb = 0;
	   if ( !boost::filesystem::exists( dirpath ) ) return nb;
	   boost::filesystem::directory_iterator end_itr; // default construction yields past-the-end
	   for ( boost::filesystem::directory_iterator itr( dirpath );  itr != end_itr;  ++itr )
	  {
		// If is directory & recurse : do recurse
		if ( boost::filesystem::is_directory(itr->status()) )
		{
			if (recursive) 
			{
				nb += NumberFilesToAdd(itr->path().string(), recursive);
			}
		}
		else
		{
			nb++;
		}
	  }
	return nb;
  }

   //=================================================
   // Test a directory to know if contains sub-directory to analyze
  bool WxGimmickView::isNeedRecursive(std::string i_name)
  {
      boost::filesystem::directory_iterator iter(i_name), end_iter;
      //bool bfindir = false;
		  for(; iter != end_iter; ++iter)
		  {
			  if(boost::filesystem::is_directory(*iter))
			  {
				  return true;
			  }
		  }
		  return false;
  }
  //=================================================

  //=================================================
  void WxGimmickView::OnSelectionChange(const std::vector<tree::Node*>& sel, bool isSelection, int selection, bool needProcess)
  {      
    GimmickDebugMessage(5,
			"WxGimmickView::OnSelectionChange"
			<<std::endl);
    wxBusyCursor busy;
	bool valid=true;
	
	if(sel.size()==0)
	{
		valid= ValidateSelected(NULL,
				mSelectionMinDimension,
				mSelectionMaxDimension );
	}
	else if(needProcess)
	{
		ResetExtent();
		std::vector<tree::Node*>::const_iterator i;
		for(i=sel.begin();i!=sel.end()&&valid;++i)
		{
			valid= ValidateSelected((*i),
				mSelectionMinDimension,
				mSelectionMaxDimension );
		}
	}
	else if(isSelection)
	{
		valid= ValidateSelected(sel.front(),
				mSelectionMinDimension,
				mSelectionMaxDimension );
	}
	else
	{
		ResetExtent();
		std::vector<tree::Node*>::const_iterator i;
		for(i=sel.begin();i!=sel.end()&&valid;++i)
		{
			valid= ValidateSelected((*i),
				mSelectionMinDimension,
				mSelectionMaxDimension );
		}
	}
	mText->SetLabel(crea::std2wx(GetMessage()));
    /*if(valid)
      {
	ReadImageThreaded(sel);
      }
    else
      {
		  ClearSelection();
      }*/
	ReadImageThreaded(sel);
   }

  //==================================================

  //==================================================
  ///Reads Images (Threaded)
  void WxGimmickView::ReadImageThreaded(const std::vector<tree::Node*>& sel)
  {	
   GimmickDebugMessage(5,
		       "ReadImageThreaded"
		       <<std::endl);
   int maxprio = GetMaximalPriority();
   int prio = maxprio + 2000;

   if(sel.size()>0)
   {
   //First load the selected images
   mCurImageItemToShow = sel.front();
   pointers.clear();
   int index = 0;
   std::vector<tree::Node*>::const_iterator selected;
   for(selected=sel.begin();selected!=sel.end();++selected)
     {
       GimmickDebugMessage(5,
			   "Requesting image from selected "
			   <<(*selected)->GetAttribute("FullFileName")
			   <<std::endl);
	   //ImagePointerHolder* ph=new ImagePointerHolder(GetDefaultImage());
	   boost::shared_ptr<ImagePointerHolder> ph(new ImagePointerHolder(GetDefaultImage()));
	   pointers.push_back(ph);
       RequestReading(*selected,prio,index,ph);
       //       AddEntryToMap(*selected);
       prio--;
       index++;
     }
	mViewer->SetImageVector(pointers);
	//Going up
	prio = maxprio + 20;
	std::vector<tree::Node*> up;
	GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetNodes(up,true);
	std::vector<tree::Node*>::iterator iterUp;
	for(iterUp=up.begin();iterUp!=up.end();++iterUp)
	{
		GimmickDebugMessage(5,
				"Requesting image from neighbors up "
				<<(*iterUp)->GetAttribute("FullFileName")
				<<std::endl);
//		ImagePointerHolder* ph=new ImagePointerHolder(GetDefaultImage());
		boost::shared_ptr<ImagePointerHolder> ph(new ImagePointerHolder(GetDefaultImage()));
		RequestReading(*iterUp,prio,-1,ph);
		//		AddEntryToMap(*iterUp);
		prio--;
		if (prio == maxprio) break;
	}

	//Going down
	prio = maxprio + 19;
	std::vector<tree::Node*> down;
	GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetNodes(down,false);
	std::vector<tree::Node*>::iterator iterDown;
	for(iterDown=down.begin();iterDown!=down.end();++iterDown)
	{
		GimmickDebugMessage(5,
				"Requesting image from neighbors down "
				<<(*iterDown)->GetAttribute("FullFileName")
				<<std::endl);
		//ImagePointerHolder* ph=new ImagePointerHolder(GetDefaultImage());
		boost::shared_ptr<ImagePointerHolder> ph(new ImagePointerHolder(GetDefaultImage()));
		RequestReading(*iterDown,prio,-1,ph);
		//		AddEntryToMap(*iterDown);
		prio--;
		if (prio == maxprio) break;
	}
   }
   else
   {
	   pointers.clear();
	   //ImagePointerHolder* ph=new ImagePointerHolder(GetDefaultImage());
	   boost::shared_ptr<ImagePointerHolder> ph(new ImagePointerHolder(GetDefaultImage()));
	   pointers.push_back(ph);
	   mViewer->SetImageVector(pointers);
   }
  }

  //==================================================

  //==================================================

#if defined(WIN32)
  //==================================================
   void  WxGimmickView::OnInternalIdle()
   {
     if (!mConstructed) return;
     static bool first_time = true;
     if (false)
     {
       first_time = false;
     }
   //   GimmickMessage(1,"WxGimmickView : Refresh viewer"<<std::endl);
	//  mViewer->StartPlayer();
     if(mViewer)
     {
        mViewer->RefreshIfNecessary();
     }
  }
#else
  void WxGimmickView::UpdateWindowUI(long flags)
  {
	  if(mViewer)
     {
        mViewer->RefreshIfNecessary();
     }
  }
#endif
   //==================================================

  //==================================================
   void  WxGimmickView::ClearSelection()
   {
	pointers.clear();
	pointers.push_back(boost::shared_ptr<creaImageIO::ImagePointerHolder>(new ImagePointerHolder(GetDefaultImage())));
	//pointers.push_back(new ImagePointerHolder(GetDefaultImage()));
	mViewer->SetImageVector(pointers);
	mViewer->RefreshIfNecessary();
	ResetExtent();
  }

  //=================================================
 
  //=================================================
  void WxGimmickView::OnRemove(wxCommandEvent& event)
  {
  	//TODO Select current tree handler       
     wxBusyCursor busy;
	 std::string remove;
	 mGimmick->GetSetting(SETTINGS_REMOVE_PATIENT_DISPLAY,remove);
	 GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->RemoveSelected(remove);
	 mGimmick->UpdateSetting(SETTINGS_REMOVE_PATIENT_DISPLAY,remove);
     ClearSelection();
  }
  //=================================================


  //=================================================
  void WxGimmickView::AddIgnoreFile(tree::Node* toRemove)
  {
     mGimmick->RemoveFile(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),toRemove);
   //  GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->UpdateLevel(1);
  }

  //=================================================
  void WxGimmickView::CopyFiles(const std::vector<std::string>& filenames)
  {
     mGimmick->CopyFiles(filenames, crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())));
     wxMessageBox(std2wx("The selected files have been copied"),_T("Copy files"),wxOK,this);
  }

   //=================================================
  void WxGimmickView::AddDir(std::string dirName)
  {
     mProgressDialog = new wxProgressDialog(_T("Adding directory"),_T(""),1000,

//EED						this,
						NULL,
	
						wxPD_ELAPSED_TIME |wxPD_CAN_ABORT );
     mCurrentDirectory = crea::std2wx(dirName);
     mGimmick->AddDir(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),dirName,true);
     mProgressDialog->Pulse(_T("Updating view..."));

     UpdateTreeViewLevel(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),1);
     killProgress();
     DisplayAddSummary();
  }

   //=================================================
  void WxGimmickView::OnSynchronize(wxCommandEvent& event)
  {       
    wxBusyCursor busy;
    const wxString choices[] = { _T("Check database for files deletion and addition and give a report."), 
				_T("Check database for files deletion, addition and attributes change. Then give a report."), 
				_T("Repair database (remove deleted files and add new files)."), 
				_T("Repair database (remove deleted files, add new files and reset changed attributes).") } ;

    wxSingleChoiceDialog dialog(this,
				_T("Select one of the following synchronization actions:\n")
                                _T("Please note that, due to the heavy amount of operations required, this action might take a while."),
                                _T("Synchronization Settings"),
                                WXSIZEOF(choices), choices);

    //dialog.SetSelection(0);

    if (dialog.ShowModal() == wxID_OK)
    {
			wxBusyCursor busy;
        int sel=dialog.GetSelection();
	bool repair=false;
	bool checkAttributes=false;
	if(sel==2 || sel==3){repair=true;}
	if(sel==1 || sel==3){checkAttributes=true;}
	std::string mess=mGimmick->Synchronize(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),repair, checkAttributes);
	wxMessageBox(std2wx(mess),_T("Synchronization result"),wxOK,this);
	if(sel==2 || sel==3){
		GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->UpdateLevel(1);
	}
		
    }
  }
  //=================================================

  //=================================================
  void WxGimmickView::OnSettings(wxCommandEvent& event)
  {
    wxDialog* dial= new wxDialog (this,-1,_T("System Settings"),wxDefaultPosition, wxSize(450,220));
    wxBoxSizer	*siz = new wxBoxSizer(wxVERTICAL);
    // Notebook
    wxNotebook* nb= new wxNotebook(dial, -1, wxDefaultPosition, wxDefaultSize, 0);
    
    siz->Add( nb,1,wxGROW  ,0);  
    CreateSettingsDialog(nb,dial);
    dial->SetSizer(siz);
    dial->ShowModal();
  }

  //=================================================
  void WxGimmickView::OnImportExport(wxCommandEvent &Event)
  {
	wxBusyCursor busy;
	// Test if one image is selected => export
	// if not =>import
    if (GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))])
	{
		ExportImages();
	}
	else
	{
		ImportImages();
	}
  }

  void WxGimmickView::ExportImages()
  {
		//Archive selection: name, emplacement
		//same process than copy local but to a zip
		// if settings are yes "always ask for descriptor addition", ask
		// if settings are yes, adding descriptor
  }

  void WxGimmickView::ImportImages()
  {
		//Find the *.zip
		//dezip
		// Contain a descriptor.text
		// create a new database, and add to database
		// if not, add to current database
		// 
  }

  //=================================================
  //AndresDonadio
  void WxGimmickView::OnTools(wxCommandEvent& event)
  {
	mViewer->StopPlayer();
	
	wxDialog* dial = new wxDialog (this,-1,_T("Tools"),wxDefaultPosition, wxSize(550,350));

	wxSizer* buttonsSizer = dial->CreateSeparatedButtonSizer(wxOK|wxCANCEL);
	wxNotebook* nb= new wxNotebook(dial, -1, wxDefaultPosition, wxDefaultSize, 0);
	wxBoxSizer *dialSizer = new wxBoxSizer(wxVERTICAL);	
	dialSizer->Add(nb,1,wxGROW,0);
	dialSizer->Add(buttonsSizer,0,wxGROW);

#if defined(BUILD_BRUKER)
	//First page: Bruker Image Reader
	WxGimmickTools * gimmickTools = new WxGimmickTools(nb, mCurrentDirectory);
	nb->AddPage( gimmickTools, _T("Bruker Image Reader") );
#endif

	dial->SetSizer(dialSizer, true);
	dial->Layout();
	dial->ShowModal();

	if (dial->GetReturnCode() == wxID_OK)
	{
#if defined(BUILD_BRUKER)
		if (nb->GetSelection()==0)//Selection: Bruker Image Reader  
		{
			std::string inputDir  = crea::wx2std(gimmickTools->getInputDir());
			std::string outputDir = crea::wx2std(gimmickTools->getOutputDir());

			bool addToDB = gimmickTools->getAddToDBCheckBoxValue();

			if (inputDir.compare("")!=0 && outputDir.compare("")!=0)
			{
				if ( wxMessageBox(_T("Depending on the amount of Data the process can take several minutes. Do you want to continue?"),
						  _T("Please confirm"), wxICON_QUESTION | wxYES_NO) == wxYES )
				{
					Bruker2Dicom b2d;    
					b2d.SetInputDirectory(inputDir);
					b2d.SetOutputDirectory(outputDir);
					b2d.SetConvertModeToDicom();
					b2d.verbose=false;
					b2d.Execute();
					if (addToDB)
					{
	std::cout <<"after addToDB" << std::endl;
						mProgressDialog = new wxProgressDialog(_T("Adding directory"),_T(""),1000,

//EED											this,
											NULL,

											wxPD_ELAPSED_TIME |wxPD_CAN_ABORT );
	std::cout <<"after new wxProgressDialog" << std::endl;						
						mCurrentDirectory = gimmickTools->getOutputDir();
	std::cout <<"after gimmickTools->getOutputDir[" <<mCurrentDirectory << std::endl;						
						mGimmick->AddDir(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),outputDir,true);
	std::cout <<"after mGimmick->AddDir" << std::endl;					
						mProgressDialog->Pulse(_T("Updating view..."));
	std::cout <<"after mProgressDialog->Pulse" << std::endl;
						UpdateTreeViewLevel(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),1);
	std::cout <<"after UpdateTreeViewLevel" << std::endl;						
						killProgress();
	std::cout <<"after delete mProgressDialog" << std::endl;							
						DisplayAddSummary();
	std::cout <<"after dDisplayAddSummary" << std::endl;						
					}	
				}
			}
			else
			{
				wxMessageBox(_T("One or both of the directory fields are empty"),_T("Empty Fields"),wxOK,this);
			}
	}
	delete gimmickTools;
#endif
	}
	mViewer->StartPlayer();
  }

  //=================================================

  void WxGimmickView::CreateSettingsDialog(wxNotebook* nb, wxDialog* dial)
  {
	  //First page: Customization of configurations
	  //Copy Path string
	  std::string cp;
	  mGimmick->GetSetting(SETTINGS_COPY_PATH,cp);
	  //Database Path String
	  std::string dp;
	  mGimmick->GetSetting(SETTINGS_DBPATH,dp);
	  //Syncronization Event String
	  std::string se;
	  mGimmick->GetSetting(SETTINGS_SYNC_EVENT,se);
	  //Syncronization Frequency String
	  std::string sf;
	  mGimmick->GetSetting(SETTINGS_SYNC_FREQ,sf);

	  WxCustomizeConfigPanel * customConfig=new WxCustomizeConfigPanel(nb,dial,this,cp,dp,se,sf);

	  nb->AddPage( customConfig, crea::std2wx("Customize Configuration") );

	  //Second page: Creation of Databases
	  /*wxPanel* databaseCreation=new wxPanel(nb);
	  nb->AddPage( databaseCreation, crea::std2wx("Create Database") );*/

	  //Second page (temporary): Connection to PACS
	  WxPACSConnectionPanel* pacs=new WxPACSConnectionPanel(nb,dial, this);
	  nb->AddPage( pacs, crea::std2wx("Connect to PACS") );

	  //Third page: CD/DVD Watch
	  WxListenerPanel* cdWatch=new WxListenerPanel(nb,dial, this,true);//, mListener->IsPaused());
	  nb->AddPage( cdWatch, crea::std2wx("CD/DVD") );

	  //Fourth page: Selection of attributes to show
	  std::vector<std::string> shown;
	  std::vector<std::string> nShown;
	  GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetAttributes(shown,nShown,1);
	  int nLev=GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetNumberOfLevels();
	  WxAttributeSelectionPanel* attSelection=new WxAttributeSelectionPanel(nb,dial,this,shown,nShown,nLev);
	  nb->AddPage( attSelection, crea::std2wx("Selection of Attributes") );
  }

  //===================================================================
  void WxGimmickView::GetVisibleAttributes(std::vector<std::string>& shown, 
	  std::vector<std::string>& nShown, int level)
  {
	  GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->GetAttributes(shown,nShown,level);
  }

  //===================================================================
  void WxGimmickView::OnAttributesChanged(const std::vector<std::string>& nShown, int level)
  {
	  GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->SetNonVisibleAttributes(nShown,level);
	  std::vector<std::string> n=nShown;
	  GetTreeViewMap()[crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection()))]->CreateCtrl(n,level);
  }
  //===================================================================
  void WxGimmickView::OnSaveSettingsCallback(const std::string& copyPath,
	  const std::string& dbPath,
	  const std::string& syncEvent,
	  const std::string& syncFreq)
  {
	  mGimmick->UpdateSetting(SETTINGS_COPY_PATH,copyPath);
	  mGimmick->UpdateSetting(SETTINGS_DBPATH,dbPath);
	  mGimmick->UpdateSetting(SETTINGS_SYNC_EVENT,syncEvent);
	  mGimmick->UpdateSetting(SETTINGS_SYNC_FREQ,syncFreq);
  }

  //===================================================================
  void WxGimmickView::OnListenerCallback(const std::string& drive, bool addFiles, bool removeFiles)
  {
	 mListener->SetMonitoredDrive(drive);
	 mListener->SetAddFilesState(addFiles);
	 mListener->SetRemoveFilesState(removeFiles);
  }

  //========================================================================

  void WxGimmickView::OnDriveMount(bool mount)
  {
	  GimmickMessage(1, "Gimmick::OnDriveMount"<<std::endl);
	  std::string drive;
	  mListener->GetMonitoredDrive(drive);
	  
	  if(mount)
	  {
		mViewer->StopPlayer();
		wxBusyCursor busy;
		wxString title(_T("Adding drive"));
		mProgressDialog = 
		new wxProgressDialog(_T("Adding drive"),
					_T(""),
					1000,

//EED					this,
					NULL,

					wxPD_ELAPSED_TIME |
					//			       wxPD_ESTIMATED_TIME | 
					//			       wxPD_REMAINING_TIME |
					wxPD_CAN_ABORT );
		mCurrentDirectory = crea::std2wx(drive);
		mGimmick->AddDir(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),drive,true);
		mProgressDialog->Pulse(_T("Updating view..."));
		
		UpdateTreeViewLevel(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),1);
		killProgress();
		DisplayAddSummary();
		mViewer->StartPlayer();  
	  }
	  else
	  {  
		  mGimmick->DeleteDrive(drive);
		  UpdateTreeViewLevel(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),1);
	  } 
  }

   //========================================================================

  void WxGimmickView::StartListeningThread()
  {
	  mListener->Resume();
  }

   //========================================================================

  void WxGimmickView::StopListeningThread()
  {
	  mListener->Pause();
  }

  //========================================================================
  void WxGimmickView::CreateEditFieldsDialog(tree::Node* node, std::vector<std::string> names, std::vector<std::string> keys)
  {
    wxDialog* dial= new wxDialog (this,-1,crea::std2wx("Edit Fields for node "+node->GetLabel()),wxDefaultPosition, wxSize(350,155));
    wxBoxSizer	*siz = new wxBoxSizer(wxVERTICAL);
    WxEditFieldsPanel* ef = new WxEditFieldsPanel(dial, dial, this, node, names, keys);

    siz->Add( ef,1,wxGROW  ,0); 
    dial->SetSizer(siz);
    dial->ShowModal();  
  }

	

 //========================================================================
  void WxGimmickView::DumpTags(std::string i_filename)
  {
    WxDumpPanel* pan= new WxDumpPanel (this,i_filename);
    pan->ShowModal();  
  }

  //========================================================================
  void WxGimmickView::ExportToStorage(const std::vector<std::string> i_filenames)
  {
	std::vector<std::string> storages;
	Gimmick::TreeHandlerMapType::iterator it = mGimmick->GetTreeHandlerMap().begin();
	for(;it != mGimmick->GetTreeHandlerMap().end(); it++)
	{
		storages.push_back(it->first);
	}

    WxExportDlg* exp= new WxExportDlg(this,storages);
	if ( exp->ShowModal() ==ID_EXPORT_OK)
	{
		std::string storage = exp->GetStorage();
		mProgressDialog = 
	    new wxProgressDialog(_T("Adding file(s)"),
			       _T(""),
			       1000,

//EED				this,
				NULL,

			       wxPD_ELAPSED_TIME |
			       // wxPD_ESTIMATED_TIME |
			       // wxPD_REMAINING_TIME |
			       wxPD_CAN_ABORT );
		mGimmick->AddFiles(storage,i_filenames);
		mProgressDialog->Pulse(_T("Updating view..."));
		UpdateTreeViewLevel(storage,1);
		killProgress();
		DisplayAddSummary();	
	}
  }



  //========================================================================
  void WxGimmickView::OnFieldsEdited(tree::Node* node, const std::string& name, const std::string& key, const std::string& val)
  {
     mGimmick->EditField(node, crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())), name, key, val);
     UpdateTreeViewLevel(crea::wx2std(mNotebook->GetPageText(mNotebook->GetSelection())),1);
  }

  //=================================================
  /// AddProgress Gimmick callback
  void WxGimmickView::OnAddProgress( Gimmick::AddProgress& p)
  {
    char mess[200];
    sprintf(mess,"%i dirs : %i files :\n            %i handled - %i added",
	   p.GetNumberScannedDirs(),
	   p.GetNumberScannedFiles(),
	   p.GetNumberHandledFiles(),
	   p.GetNumberAddedFiles());
    //    std::cout << "OnAddProgress "<<mess<<std::endl;
    wxString s(wxString::From8BitData(mess));
    //  std::cout << "Pulse"<<std::endl;

	if(mProgressDialog != 0)
	{
		if (!mProgressDialog->Pulse(s)) 
		  {
		p.SetStop();
		  }
	}
    //  std::cout << "OnAddProgress ok"<<std::endl;
  }
  //=================================================

  //=================================================
  void WxGimmickView::DisplayAddSummary()
  {
      std::stringstream mess(mGimmick->getSummary());
    time_t end;
	time(&end);
	mess << "time to scan :" << difftime(end,mstart) << "sec";
    wxMessageBox(std2wx(mess.str()),_T("Addition result"),wxOK,this);
  }

  ////////////////////////////////////////////////
  // Add a DB to application			//
  // @param event : WxEvent			//
  // @return : -				//
  ////////////////////////////////////////////////
  void WxGimmickView::OnAddDB(wxCommandEvent& event)
  {
	  //Select DB
	  long style = wxDD_DEFAULT_STYLE | wxDD_DIR_MUST_EXIST;
	  std::string wc("*.sqlite3*");
	  wxFileDialog* FD = new wxFileDialog( 0, 
					 _T("Select file"),
					 _T(""),
					 _T(""),
					 crea::std2wx(wc),
					 style,
					 wxDefaultPosition);
    
    if (FD->ShowModal()==wxID_OK)
	{
		wxBusyCursor busy;
		wxArrayString files;
		FD->GetPaths(files);
		std::stringstream st;
		for(int i = 0; i< files.size(); i++)
		{
			//get name of DB (file name)
		  size_t pos = files[i].find_last_of(_T("\\"));
		  std::string name = crea::wx2std(files[i].substr(pos+1));
		  pos = name.find_last_of(".");
		  name = name.substr(0,pos);
		  //create TreeHandler
		  mGimmick->addDB(name, crea::wx2std(files[i]));
		  //create TreeView
		  CreateSingleTreeView(name);
		}
	}
  }
    ////////////////////////////////////////////////////
  // Create a DB from an Attributes Descriptor files  //
  // @param event : WxEvent					//
  // @return : -						//
  //////////////////////////////////////////////////
  void WxGimmickView::OnCreateDB(wxCommandEvent& event)
  {
	//  PACSConnection("");
    	WxDescriptorPanel * DescriptorPan = new WxDescriptorPanel(this, mGimmick->GetHomeDirectory());
	DescriptorPan->Layout();
	if ( DescriptorPan->ShowModal() == ID_DSCP_APPLY)
	{
		wxBusyCursor busy;
		std::string file(DescriptorPan->GetDescriptor());
		if (!file.empty())
		{
			size_t pos = file.find_last_of("\\");
			std::string name = file.substr(pos+1);
			std::string directory = file.substr(0,pos);
			pos = name.find_last_of(".");
			name = name.substr(0,pos);
			//get directory to store DB
			directory +=  "\\" + name + ".sqlite3";
			//create createDB
			mGimmick->createDB(name, file,directory);
			//create TreeHandler
			mGimmick->addDB(name, directory);
			//create TreeView
			CreateSingleTreeView(name);
		}
	}
  }

  void WxGimmickView::killProgress()
  {
	  	mProgressDialog->Resume();
		mProgressDialog->Destroy();
		mProgressDialog = 0;
  }

   //=================================================

   //=================================================
  BEGIN_EVENT_TABLE(WxGimmickView, wxPanel)
    EVT_TOOL(TOOL_CREATEDB_ID, WxGimmickView::OnCreateDB)
    EVT_TOOL(TOOL_ADDFILES_ID, WxGimmickView::OnAddFiles)
    EVT_TOOL(TOOL_ADDDIR_ID, WxGimmickView::OnAddDir)
    EVT_TOOL(TOOL_ADDDATABASE_ID, WxGimmickView::OnAddDB)
    EVT_TOOL(TOOL_REMOVE_ID, WxGimmickView::OnRemove)
    EVT_TOOL(TOOL_SYNCHRONIZE_ID, WxGimmickView::OnSynchronize)
    EVT_TOOL(TOOL_SETTINGS_ID, WxGimmickView::OnSettings)
    EVT_TOOL(TOOL_TOOLS_ID, WxGimmickView::OnTools)
  END_EVENT_TABLE()
  //=================================================

} // EO namespace creaImageIO

