/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#include <creaImageIOWxTreeView.h>
#include <creaImageIOGimmickView.h>

#include <wx/splitter.h>
#include <wx/gdicmn.h>
#include <boost/date_time/gregorian/gregorian.hpp>
#include <creaImageIOGimmick.h>
#ifdef _DEBUG
#define new DEBUG_NEW
#endif
//=====================================================================
namespace creaImageIO
{

  //=====================================================================
}
//=====================================================================

//=====================================================================
///Comparing function for ordering algorithm. Takes parameters as strings.
int wxCALLBACK CompareFunctionStrings(long item1, long item2, long sortData)
{	
  creaImageIO::ItemData* data1 = (creaImageIO::ItemData*)item1;
  creaImageIO::ItemData* data2 = (creaImageIO::ItemData*)item2;

  const std::string& s1(*(data1->attr));
  const std::string& s2(*(data2->attr));
  if(sortData==1)
    {
      // inverse the order
      if (s1 < s2)
	return 1;
      if (s1 > s2)
	return -1;
      
      return 0;
    }
  else
    {
      if (s1 < s2)
	return -1;
      if (s1 > s2)
	return 1;
      
      return 0;
      
    }
}
//=====================================================================

//=====================================================================
///Comparing function for ordering algorithm. Takes parameters as ints.
int wxCALLBACK CompareFunctionInts(long item1, long item2, long sortData)
{	
  creaImageIO::ItemData* data1 = (creaImageIO::ItemData*)item1;
  creaImageIO::ItemData* data2 = (creaImageIO::ItemData*)item2;

  const std::string& s1(*(data1->attr));
  const std::string& s2(*(data2->attr));

  int val1=atoi(s1.c_str());
  int val2=atoi(s2.c_str());

  if(sortData==1)
    {
      // inverse the order
      if (val1 < val2)
	return 1;
      if (val1 > val2)
	return -1;
      
      return 0;
    }
  else
    {
      if (val1 < val2)
	return -1;
      if (val1 > val2)
	return 1;

      return 0;
      
    }
  
}

//=====================================================================


//=====================================================================
namespace creaImageIO
{
  //=====================================================================
  // CTor
  WxTreeView::WxTreeView(TreeHandler* handler,
			 GimmickView* gimmick,
			 wxWindow* parent,
			 const wxWindowID id)
    : wxPanel(parent,id),
      TreeView(handler, gimmick)
  {
    GimmickDebugMessage(1,"WxTreeView::WxTreeView"
			<<std::endl);

    
    // Split part below toolbar into notebook for views and panel
    // for preview, messages...
    // TO DO : Splitter
    //    mSplitter = new wxSplitterWindow( this , -1);

    // Global sizer
    msizer = new wxBoxSizer(wxHORIZONTAL);
    
    int ctrl_style = wxLC_REPORT | wxLC_VRULES;
    int col_style = wxLIST_FORMAT_LEFT;

    // Creating the ListCtrl for the levels > 0 (not for Root level)
    for (int i = 0;
	 i < handler->GetTree().GetNumberOfLevels() -1;
	 ++i)
      {
	GimmickDebugMessage(5,"Creating view for level "<<i
			    <<std::endl);
	LevelType level;
	level.SelectedUpToDate = true;
	level.SortColumn = 0;

	// If the first level : parent = this
	wxWindow* sparent = this;
	// else parent = last splitter
	if (i>0) 
		sparent = mLevelList[i-1].wxSplitter;

	level.wxSplitter = new wxSplitterWindow( sparent , -1);
	if(i!=0)
	{
	level.wxSplitter->Show(false);
	}
	//	    level.wxSplitter->SetMinimumPaneSize(100);
	
	wxListCtrl* ctrl = new wxListCtrl(level.wxSplitter,
					  i,
					  wxDefaultPosition, 
					  wxDefaultSize,
					  ctrl_style);
	level.wxCtrl = ctrl;
	level.wxSplitter->Initialize(ctrl);
   
	// Create the columns : one for each attribute of the level
	int col = 0;
	std::string title;

  	tree::LevelDescriptor::AttributeDescriptorListType::const_iterator a;
	for (a  = handler->GetTree().GetAttributeDescriptorList(i+1).begin();
	     a != handler->GetTree().GetAttributeDescriptorList(i+1).end();
	     ++a)

{
	
	    GimmickDebugMessage(5,"Creating column "<<col<<" : "
				<<a->GetName()
				<<std::endl);
	    
	    if(a->GetFlags()!=creaImageIO::tree::AttributeDescriptor::PRIVATE)
	      {
		
		if(a->GetName()=="UNKNOWN")
		  {
		    title = "#";
		    title += handler->GetTree().GetLevelDescriptor(i+1).GetName();
		    if (title[title.size()-1]!='s')
		      title += "s";
		    
		  }
		else
		  {
		    title=a->GetName();
		  }
		  std::string temp = a->GetKey();
		  if (temp.compare("ID") != 0)
		  {
		
		ctrl->InsertColumn(col, 
				   crea::std2wx(title),
				   col_style);
		col++;
		  }
		level.key.push_back(a->GetKey());
	      }
		
	  }
	  
	mLevelList.push_back(level);
      }
    
#if wxUSE_MENUS

	 // Column Menu
    menu =new wxMenu;
	wxMenuItem* m1=menu->Append(wxID_ANY, _T("&Sort ascending"));
	wxMenuItem* m2=menu->Append(wxID_ANY, _T("&Sort descending"));
	wxMenuItem* m3=menu->Append(wxID_ANY, _T("&Filter"));
	mAscendingID=m1->GetId();
	mDescendingID=m2->GetId();
	mFilterID=m3->GetId();
	Connect( mAscendingID, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnPopupSort) );
	Connect( mDescendingID, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnPopupSort) );
	Connect( mFilterID, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnPopupFilter) );


	////SubMenuItem EXPORT
	subExportMenu = new wxMenu;
	wxMenuItem *subExp1 = subExportMenu->Append(wxID_ANY, _T("&Export to Storage"));
	Connect( subExp1->GetId(), wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnExportToStorage) );

	//ItemMenu
	menuItem =new wxMenu;


	wxMenuItem* m2Item=menuItem->Append(wxID_ANY, _T("&Local Copy"));
	wxMenuItem* m3Item=menuItem->Append(wxID_ANY, _T("&Edit Fields"));
	wxMenuItem* m4Item=menuItem->Append(wxID_ANY, _T("&Display Dicom Tags"));
	menuItem->AppendSubMenu(subExportMenu, wxT("&Export"));


	wxMenuItem* m1Item=menuItem->Append(wxID_ANY, _T("&Anonymize"));
	mAnonymizingID=m1Item->GetId();
	Connect( mAnonymizingID, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnAnonymizer) );

	mLocalCopyID=m2Item->GetId();
	mEditFieldID=m3Item->GetId();
	mDumpID=m4Item->GetId();
	
	
	Connect( mLocalCopyID, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnLocalCopy) );
	Connect( mEditFieldID, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnEditField) );
	Connect( mDumpID, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler(WxTreeView::OnDumpTags) );
	

	
#endif // wxUSE_MENUS
	/// Initialize the first level splitter
	  
	msizer->Add( mLevelList[0].wxSplitter ,1, wxGROW  ,0);
	//	mColumnSelected=1;
	mLastSelected=0;
	mLastLevel=0;
	//	mDirection=true;

	mIgnoreSelectedChanged = false;

	//CreateColorPalette();
    UpdateLevel(1);

    SetSizer( msizer );     
    SetAutoLayout(true);
    Layout();

  }
  //=====================================================================

  //=====================================================================
  /// Destructor
  WxTreeView::~WxTreeView()
  {
    GimmickDebugMessage(1,"WxTreeView::~WxTreeView"
			<<std::endl);
	delete menu;
	delete menuItem;

  }
  //=====================================================================
  
  
  
  //=====================================================================
  const std::vector<tree::Node*>& WxTreeView::GetSelected(int level)
  {
	  std::vector<tree::Node*>& sel = mLevelList[0].Selected;
    //  if (GetSelectedUpToDate(level)) 
    int l = level - 1;
    // the selection of upper level
	if(mLevelList.size() == level -1)
		 sel = mLevelList.back().Selected;
	else
		  sel=  mLevelList[l].Selected;
	if (sel.size() > 0)
	{
		sel.clear();
	}
	  if (level == 1) 
      {
	sel.push_back(GetTreeHandler()->GetTree().GetTree());
      }
	  else if (level < mLevelList.size()+2 ) 
    {
		long item = -1;
		for ( ;; )
		{
			item = GetCtrl(l-1)->GetNextItem(item,
										 wxLIST_NEXT_ALL,
										 wxLIST_STATE_SELECTED);
			if ( item == -1 )
				break;
			long adr = (long)GetCtrl(l-1)->GetItemData(item);
			tree::Node* n = ((ItemData*)adr)->node;
			/* FCY 18-04-2011: don't understand the real purpose of these lines,
			 if uncomment add last frame in first place 
				if(mLastSelected==item)
			{
				std::vector<tree::Node*>::iterator it;
				it = sel.begin();
				it = sel.insert ( it , n );
			}
			else
			{*/
				
				sel.push_back(n);
			//}			
			
		}
		/*int n = GetCtrl(l-1)->GetItemCount();
		for (int i = 0; i<n; i++)
		{
			std::cout<<GetCtrl(l-1)->GetItemState(i,wxLIST_STATE_SELECTED)<<std::endl;
			if ( GetCtrl(l-1)->GetItemState(i,wxLIST_STATE_SELECTED))
			{
				long adr = GetCtrl(l-1)->GetItemData(i);
				tree::Node* n = ((ItemData*)adr)->node;
				if(mLastSelected==i)
				{
					std::vector<tree::Node*>::iterator it;
					it = sel.begin();
					it = sel.insert ( it , n );
				}
				else
				{
					
					sel.push_back(n);
				}
			}
	      }*/
	  }
	else
	{
		// NOTHING
	}   
	 
    //    return mLevelList[level-1].Selected;
    return sel;
  }

  //=====================================================================
  
  //=====================================================================
  ///Removes selected nodes on last selected level
   // NOT SPECIFIC 
  void WxTreeView::RemoveSelected(std::string &i_save)
  {
	 bool erase=false;
	 
	 unsigned int tempLevel = mLastLevel;
    mLastLevel+=1;
    const std::vector<tree::Node*>& sel=GetSelected(mLastLevel+1);
	// if no selection, no remove action.
    if(sel.size() != 0)
	{
	
	    std::stringstream out;
	    std::string levelName=GetTreeHandler()->GetTree().GetLevelDescriptor(mLastLevel).GetName();
	    out<<"Delete ";
	    out<<sel.size();
	    if(sel.size()>1&&levelName.at(levelName.size()-1)!='s')
	      {
		out<<" "<<levelName;
		out<<"s?";
	      }
	    else
	      {
		out<<" "<<GetTreeHandler()->GetTree().GetLevelDescriptor(mLastLevel).GetName()<<"?";
	      }
	    if (wxMessageBox(crea::std2wx(out.str()),
			     _T("Remove Files"),
			     wxYES_NO,this ) == wxYES)
	      {
		erase = true;
	      }
	    if(erase)
		  {
            GetGimmickView()->modifyValidationSignal(false);
		    bool needRefresh=false;
		    std::vector<tree::Node*>::const_iterator i;
		    for (i=sel.begin(); i!=sel.end(); ++i)
		      {
			GimmickMessage(1,
				       "deleting '"
				       <<(*i)->GetLabel()
				       <<"'"<<mLastLevel
				       <<std::endl);
			if((*i)->GetParent()->GetNumberOfChildren()<2)
			  {
			    needRefresh=true;
			  }
			  //tree::Node* n = new (tree::Node*)(*i);
			  GetTreeHandler()->LoadChildren((*i),4);
			  GetGimmickView()->AddIgnoreFile(*i);
			  GetTreeHandler()->Remove(*i);
		      }
		    
		    if(needRefresh && mLastLevel>1)
		      {
			UpdateLevel(mLastLevel-2);
		      }
		    else if(mLastLevel>1)
		      {
			UpdateLevel(mLastLevel-1);
		      }
		    else
		      {
			UpdateLevel(mLastLevel);
		      }
		  }
	}
	else
	{
		// no need to incremente level
		mLastLevel = tempLevel;
	}
    
	if (erase && mLastLevel == 1 && i_save == "0")
	{
	
		RemoveAlertDlg *dial = new RemoveAlertDlg(this,  crea::std2wx("Remove files"), wxSize(370,100));
		//dial->ShowModal();
		if (dial->ShowModal() == wxID_OK)
		{
			i_save = dial->isChecked() == false? "0" : "1";
		}
		
	}
  }
  
  
  //=====================================================================
  /// Updates a level of the view (adds or removes children, etc.)
  void WxTreeView::UpdateLevel( int level )
  {
    GimmickDebugMessage(1,
			GetTreeHandler()->GetTree().GetLabel()
			<<"WxTreeView::UpdateLevel(level "
			<<level
			<<")"
			<<std::endl);
    
    wxBusyCursor busy;
    RecursiveUpdateLevel(level);
    int i;
    for (i=0; i<level-1; i++)
      {
	if (!GetSplitter(i)->IsSplit()) 
	  GetSplitter(i)->SplitVertically(  GetCtrl(i), GetSplitter(i+1),
					    100 );
      }
    if (GetSplitter(i)->IsSplit()) GetSplitter(i)->Unsplit();    
    
  }
  //=====================================================================
  
  //=====================================================================
  /// Recursive method called upon by UpdateLevel to refresh all windows
  void WxTreeView::RecursiveUpdateLevel( int level )
  {
    GimmickDebugMessage(1,
			GetTreeHandler()->GetTree().GetLabel()
			<<"WxTreeView::RecursiveUpdateLevel(level "
			<<level
			<<")"<<std::endl);
    
    
    const std::vector<tree::Node*>& sel(GetSelected(level));
    
    int l = level - 1;
    
    // to speed up inserting we hide the control temporarily
    GetCtrl(l)->Hide();
    GetCtrl(l)->DeleteAllItems();
    
    std::vector<tree::Node*>::const_iterator i;
    
    for (i=sel.begin(); i!=sel.end(); ++i)
      {
	GimmickDebugMessage(1,
			    "adding children of '"
			    <<(*i)->GetLabel()
			    <<"'"
			    <<std::endl);
	int _id=0;
	
	//Adds items and sets their attributes 
	
	GetTreeHandler()->LoadChildren(*i,1);
	tree::Node::ChildrenListType::reverse_iterator j;
	for (j = (*i)->GetChildrenList().rbegin(); 
	     j!= (*i)->GetChildrenList().rend(); 
	     ++j)
	  {
	    GimmickDebugMessage(1,
				"adding children "
				<<(*j)->GetLabel()
				<<"'"
				<<std::endl);
	    
	    wxListItem item;
	    item.SetMask(wxLIST_MASK_STATE | 
			 wxLIST_MASK_TEXT |
			 //			 wxLIST_MASK_IMAGE |
			 wxLIST_MASK_DATA |
			 //			 wxLIST_MASK_WIDTH |
			 wxLIST_MASK_FORMAT
			 );
	    
		ItemData* data = new ItemData();
	    data->node = *j;
	    data->id = _id;
		
	    item.SetId(_id);
	    item.SetData(data);

	    _id++;
	    GetCtrl(l)->InsertItem(item);
	    
	    //Setting attributes
	    for (int k=0; k<GetCtrl(l)->GetColumnCount(); ++k)				
	      {
		std::string val;
		//  Temporary correction : it works but no explanation about the problem FCY
		
		if(k==0 && level <3)
		{
		  val = (*j)->GetAttribute("NumberOfChildren");
		}
		else
		  val = (*j)->GetAttribute(mLevelList[l].key[k]);
		if(((*j)->GetAttributeDescriptor(mLevelList[l].key[k])).isDateEntry()) // Date
		  {
		    //					  std::cout << "["<<val<< "]" << std::endl;
		    std::string valtmp(val);
		    try
		      {
			boost::gregorian::date d1(boost::gregorian::from_undelimited_string(val));				   
			val = to_iso_extended_string(d1);
		      }
		    catch (...)
		      {
			val =  valtmp;
		      }
		    //					  std::cout << "["<<val<< "]" << std::endl;	
		  }
		else if(((*j)->GetAttributeDescriptor(mLevelList[l].key[k])).isTimeEntry()) // Time
		  {
		    if ((val.size()>6) && 
			(val != "" || val != " "))
		      val = val.substr(0,2) + " : " 
			+ val.substr(2,2) + " : " 
			+ val.substr(4,2);
		  }
		else
		  {
		    if (val.size()==0) val = "?";
		  }
		if (val.size()==0) val = "X";
		item.SetText( crea::std2wx(val));
		item.SetColumn(k);
		  
		  GetCtrl(l)->SetItem(item);
		  }
		item.Clear();
	    
	  }
      }
    
    SortLevel(l);
    GetCtrl(l)->Show();
  }
  //=====================================================================
  
  
  //================================================================
  void WxTreeView::OnItemDeSelected(wxListEvent& event)
  { 
    GimmickDebugMessage(1,
			GetTreeHandler()->GetTree().GetLabel()
			<<" WxTreeView::OnItemDeselected"<<std::endl);
    // retrieve the level
    wxObject* obj = event.GetEventObject();   
    unsigned int level = 0;
    for (level = 0; level<mLevelList.size(); ++level)
      {
	if ( GetCtrl(level) == obj ) break;
      } 
    SetSelectedUpToDate(level,false);
    // to allow a first selection in images TreeView
    if (level==mLevelList.size()-1) 
      OnItemSelected(event);
  }
  //================================================================
  
  //================================================================
  void WxTreeView::OnItemSelected(wxListEvent& event)
  { 
	  
    GimmickDebugMessage(1,
			GetTreeHandler()->GetTree().GetLabel()
			<<" WxTreeView::OnItemSelected"<<std::endl);

    if (mIgnoreSelectedChanged) 
      {
	GimmickDebugMessage(1,
			    " mIgnoreSelectedChanged true: returning"
			    <<std::endl);
	return;
      }
    

    
    wxListItem info;
    info.m_itemId = event.m_itemIndex;
    mLastSelected = event.m_itemIndex;
    // retrieve the level
    wxObject* obj = event.GetEventObject();   
    unsigned int level = 0;
    for (level = 0; level<mLevelList.size(); ++level)
      {
	if ( GetCtrl(level) == obj ) break;
      }
	mLastLevel=level;
    GimmickDebugMessage(1,
			" Level "<<level+1
			<<std::endl);
    
    // Update the children level (if selection not at last level)
    if (level<mLevelList.size()-1) 
      {
		
	UpdateLevel( level + 2 ); 
	// Reset the viewer setting the default image
	GetGimmickView()->ClearSelection();
      }
    // Select all images if the selection is at series level
    if (level==mLevelList.size()-2) 
		SelectAll(level+1);
    // Validate selected images if the selection is at image level
    if (level==(mLevelList.size()-1)) //&&mProcess) 
      {
	if(event.GetEventType()==wxEVT_COMMAND_LIST_ITEM_SELECTED)
	  {
		  ValidateSelectedImages (true);
	  }
	else
	  {
		  ValidateSelectedImages (false);
	  }
      }
    
  }
  //================================================================

  //================================================================
  void WxTreeView::SelectAll(int level)
  {
    long item = -1;
    //    int level=mLevelList.size()-1;
    for ( ;; )
      {
	item = GetCtrl(level)->GetNextItem(item,
					   wxLIST_NEXT_ALL);
        if ( item == -1 )
	  break;
	
	if(item==(GetCtrl(level)->GetItemCount()-1))
	  {
	    mIgnoreSelectedChanged = false;//mProcess=true;
	  }
	else
	  {
	    mIgnoreSelectedChanged = true;//	mProcess=false;
	  }
	GetCtrl(level)->SetItemState(item,wxLIST_STATE_SELECTED, wxLIST_MASK_STATE 
 				     | wxLIST_MASK_TEXT |wxLIST_MASK_IMAGE | wxLIST_MASK_DATA | wxLIST_MASK_WIDTH | wxLIST_MASK_FORMAT);
      }
  }

  //================================================================
  //================================================================

  void WxTreeView::OnColClick(wxListEvent& event)
  { 
    mColumnSelected = event.m_col;
    wxPoint clientpt;
    clientpt.x = wxGetMousePosition().x - this->GetScreenPosition().x;
    clientpt.y = wxGetMousePosition().y - this->GetScreenPosition().y;
    senderCtrl = event.GetEventObject(); 
    unsigned int level = 0;
    for (level = 0; level<mLevelList.size(); ++level)
      {
	if ( GetCtrl(level) == senderCtrl ) break;
      }
    PopupMenu(menu, clientpt);
    
  }

   //================================================================
  //================================================================

  void WxTreeView::OnItemMenu(wxListEvent &event)
  {
	 wxPoint clientpt;
    clientpt.x = wxGetMousePosition().x - this->GetScreenPosition().x;
    clientpt.y = wxGetMousePosition().y - this->GetScreenPosition().y;
    senderCtrl = event.GetEventObject();
    unsigned int level = 0;
    for (level = 0; level<mLevelList.size(); ++level)
      {
		if ( GetCtrl(level) == senderCtrl ) break;
      }
	  long* ptr=0;
	  int flag;
	  mLastRightLevel=level;
	  mLastRightSelected=GetCtrl(level)->HitTest(wxPoint(0,clientpt.y-8),flag,ptr);
    PopupMenu(menuItem, clientpt);
    
  }
  
  //================================================================
  //================================================================

  void WxTreeView::OnPopupFilter(wxCommandEvent& event)
  {
    wxBusyCursor busy;
    GimmickDebugMessage(7,
			"WxTreeView::OnEndLabelEdit" 
			<<std::endl);
    unsigned int level = 0;
    for (level = 0; level<mLevelList.size(); ++level)
      {
	if ( GetCtrl(level) == senderCtrl ) break;
      }
    std::string filter = crea::wx2std(wxGetTextFromUser(_T("Enter the filter to apply"), _T("Filter On Column")));
    
    std::string att;
    
    long it = -1;
    UpdateLevel(level+1);
    
    std::vector<long> items;
    bool in=false;
    int del=0;
    for ( ;; )
      {
	it = GetCtrl(level)->GetNextItem(it,
					 wxLIST_NEXT_ALL);
	if ( it == -1 )
	  break;
	
	long adr = (long)GetCtrl(level)->GetItemData(it);
	tree::Node* nod = ((ItemData*)adr)->node;
	att=(*nod).GetAttribute(mLevelList[level].key[mColumnSelected]);
	
	
	if(att.find(filter)>900)
	  {
	    
	    if(!in)
	      {
		in=true;
	      }
	    else
	      {
		del+=1;
	      }
	    
	    items.push_back(it-del);
	  }
	
      }
    std::vector<long>::iterator iter;
    for(iter=items.begin();iter!=items.end();++iter)
      {
	GetCtrl(level)->DeleteItem(*iter);
      }
    GetGimmickView()->ClearSelection();
  }
  //================================================================
  
  //================================================================
  void WxTreeView::OnPopupSort(wxCommandEvent& event)
  {
    wxBusyCursor busy;
    unsigned int level = 0;
    for (level = 0; level<mLevelList.size(); ++level)
      {
	if ( GetCtrl(level) == senderCtrl ) break;
      }
    mLevelList[level].SortColumn = mColumnSelected;

    if(event.GetId()==mAscendingID)
      {
	mLevelList[level].SortAscending = true;
      }
    else if(event.GetId()==mDescendingID)
      {
	mLevelList[level].SortAscending = false;
      }
	  
    SortLevel(level);
  }
  //================================================================

  void WxTreeView::OnAnonymizer(wxCommandEvent &event)
  {
	   wxBusyCursor busy;
	   std::vector<std::string> filesname;
	   std::vector<tree::Node*> nodes;
	   nodes.push_back(((ItemData*)GetCtrl(mLastRightLevel)->GetItemData(mLastRightSelected))->node);
  	   if(nodes.size() != 0)
	   {
		   GetFilenamesAsString(nodes,filesname);
		   GetGimmickView()->Anonymize(filesname,0);
	   }
	
  }

  //================================================================
  void WxTreeView::OnLocalCopy(wxCommandEvent& event)
  {
    wxBusyCursor busy;
    
	unsigned int tempLevel = mLastLevel;
    mLastLevel+=1;
    const std::vector<tree::Node*>& sel=GetSelected(mLastLevel+1);
	
    if(sel.size() != 0)
	{
	    bool copy=false;
	    std::stringstream out;
	    std::string levelName=GetTreeHandler()->GetTree().GetLevelDescriptor(mLastLevel).GetName();
	    out<<"Copy ";
	    out<<sel.size();
	    if(sel.size()>1&&levelName.at(levelName.size()-1)!='s')
	      {
		out<<" "<<levelName;
		out<<"s to .creaImageIO?";
	      }
	    else
	      {
		out<<" "<<GetTreeHandler()->GetTree().GetLevelDescriptor(mLastLevel).GetName()<<" to .creaImageIO?";
	      }
	    if (wxMessageBox(crea::std2wx(out.str()),
			     _T("Remove Files"),
			     wxYES_NO,this ) == wxYES)
	      {
		copy = true;
	      }
	    if(copy)
		  {
			std::vector<std::string> s;
			GetFilenamesAsString(sel,s);
            GetGimmickView()->CopyFiles(s);
		  }
	}
	else
	{
		mLastLevel = tempLevel;
	}
    
    
  }
  //================================================================

   //================================================================
  void WxTreeView::OnEditField(wxCommandEvent& event)
  {
	if(mLastRightSelected!=-1)
	{
    tree::Node* node=((ItemData*)GetCtrl(mLastRightLevel)->GetItemData(mLastRightSelected))->node;
  	tree::LevelDescriptor::AttributeDescriptorListType::const_iterator a;
	std::vector<std::string> names;
	std::vector<std::string> keys;
	for (a  = GetTreeHandler()->GetTree().GetAttributeDescriptorList(mLastRightLevel+1).begin();
	     a != GetTreeHandler()->GetTree().GetAttributeDescriptorList(mLastRightLevel+1).end();
	     ++a)
	{
		if(a->GetFlags()==creaImageIO::tree::AttributeDescriptor::EDITABLE)
	    {
			names.push_back(a->GetName());
			keys.push_back(a->GetKey());
		}
	}
	GetGimmickView()->CreateEditFieldsDialog(node,names,keys);
	}
  }

  //================================================================

  //================================================================

  void WxTreeView::OnExportToStorage(wxCommandEvent &event)
  {
	std::vector<std::string> filesname;
	std::vector<tree::Node*> nodes;
	nodes.push_back(((ItemData*)GetCtrl(mLastRightLevel)->GetItemData(mLastRightSelected))->node);
  	GetFilenamesAsString(nodes,filesname);
	GetGimmickView()->ExportToStorage(filesname);
  }

  //================================================================

  //================================================================

  void WxTreeView::OnDumpTags(wxCommandEvent &event)
  {
	  if(mLastRightSelected!=-1)
	{
		tree::Node* node=((ItemData*)GetCtrl(mLastRightLevel)->GetItemData(mLastRightSelected))->node;
  		tree::LevelDescriptor::AttributeDescriptorListType::const_iterator a;
		std::vector<std::string> names;
		std::vector<std::string> keys;
		for (a  = GetTreeHandler()->GetTree().GetAttributeDescriptorList(mLastRightLevel+1).begin();
			 a != GetTreeHandler()->GetTree().GetAttributeDescriptorList(mLastRightLevel+1).end();
			 ++a)
			{
				if(a->GetKey()=="FullFileName")
				{
					GetGimmickView()->DumpTags(node->GetAttribute("FullFileName"));
					return;
				}
			}
	  }
  }
  

  //================================================================

  //================================================================
  void WxTreeView::SortLevel(int level)
  { 	  
    GimmickDebugMessage(1,
			"WxTreeView::SortLevel(" 
			<<level<<")"
			<<std::endl);  
    //Obtain the column name and the level that needs to be organized
    
    //    int l = level - 1;
    //Sets the data for the items to be sorted
    //    std::string att;
    unsigned int ty=0;
    int nbselected = 0;
    int n = GetCtrl(level)->GetItemCount();
    for (int i = 0; i < n; i++)
      {
	
	//Gets current item data
	ItemData* data = (ItemData*)GetCtrl(level)->GetItemData(i);
	
	//Extracts the node and the type of attribute	
	tree::Node* nod = data->node;
	if(i==0)
	  {
	    (*nod).GetAttributeDescriptor
	      (mLevelList[level].key[mLevelList[level].SortColumn])
	      .DecodeType( ty );
	  }
	//Obtains the organizing attribute
	data->attr = & (*nod).GetAttribute
	  (mLevelList[level].key[mLevelList[level].SortColumn]);
	//Selected ?
	data->selected = false;
	if (GetCtrl(level)->GetItemState(i,wxLIST_STATE_SELECTED)>0)
	  {
	    data->selected = true;
	    nbselected++;
	  }

      }	
    GimmickDebugMessage(1,
			"WxTreeView::OnSort : " 
			<<nbselected<<" selected before sorting"
			<<std::endl);  

    mIgnoreSelectedChanged = true; 
    // 
    if (mLevelList[level].SortAscending)
      {
	
	if(ty==1)
	  {
	    GetCtrl(level)->SortItems(CompareFunctionInts, 0);
	  }
	else
	  {
	    GetCtrl(level)->SortItems(CompareFunctionStrings, 0);
	  }
	
      }
    else
      {
	if(ty==1)
	  {
	    GetCtrl(level)->SortItems(CompareFunctionInts, 1);
	  }
	else
	  {
	    GetCtrl(level)->SortItems(CompareFunctionStrings, 1);
	  }
      }
 

    // Reselects the unselected 
    n = GetCtrl(level)->GetItemCount();
    int after = 0;
    for (int i = 0; i < n; i++)
      {
	
	//Gets current item data
	ItemData* data = (ItemData*)GetCtrl(level)->GetItemData(i);
  
	//  long item = -1;
	//    for ( ;; )
	//      {
	//	item = GetCtrl(level)->GetNextItem(item,wxLIST_NEXT_ALL);
	//	if ( item == -1 ) break;
	//Gets current item data
	//	ItemData* data = (ItemData*)GetCtrl(level)->GetItemData(item);
	// was selected ?
	
	if (data->selected)
	  {
	    nbselected--;
	    if (nbselected==0)
	      {
		// if it is the last one we must process the selection
		mIgnoreSelectedChanged = false;
	      }
	    GetCtrl(level)->SetItemState(i,
					 wxLIST_STATE_SELECTED, 
					 wxLIST_MASK_STATE 
					 | wxLIST_MASK_TEXT 
					 | wxLIST_MASK_IMAGE 
					 | wxLIST_MASK_DATA 
					 | wxLIST_MASK_WIDTH 
					 | wxLIST_MASK_FORMAT);   
	  }
	if (GetCtrl(level)->GetItemState(i,wxLIST_STATE_SELECTED)>0)
	  {
	    after++;
	  }

	
      }
    mIgnoreSelectedChanged = false; 
     GimmickDebugMessage(1,
			"WxTreeView::SortLevel : " 
			<<after<<" selected after sorting"
			<<std::endl);  
  
  }
  //================================================================

  
  //================================================================
  void WxTreeView::ValidateSelectedImages(bool isSelection)
  {
    GimmickDebugMessage(7,
			"WxTreeView::ValidateSelectedImages" 
			<<std::endl);
    const std::vector<tree::Node*>& sel(GetSelected((int)mLevelList.size()+1));
    GetGimmickView()->OnSelectionChange(sel,
					isSelection,(mLastSelected-1),
					!mIgnoreSelectedChanged);
 
  }
  //================================================================


  //================================================================
  void WxTreeView::GetNodes(std::vector<tree::Node*>& nodes, bool direction)
  {
	long item = mLastSelected;
	int level=(int)mLevelList.size()-1;
	//Gets current item data
	long adr = (long)GetCtrl(level)->GetItemData(item);
	//Extracts the node
	tree::Node* nod = ((ItemData*)adr)->node;
    for ( ;; )
    {
		if(direction)
		{
			item = GetCtrl(level)->GetNextItem(item,
                                     wxLIST_NEXT_ABOVE);
		}
		else
		{
			item = GetCtrl(level)->GetNextItem(item,
                                     wxLIST_NEXT_BELOW);
		}
        if ( item == -1 || item==0  )
		{
            break;
		}
		if(GetCtrl(level)->GetItemState(item, wxLIST_STATE_SELECTED)==0 )
		{

			adr = (long)GetCtrl(level)->GetItemData(item);
			nod = ((ItemData*)adr)->node;
			nodes.push_back(nod);
		}
    }

  }
  //================================================================
   //=================================================
  void WxTreeView::OnKeyDown(wxListEvent &event)
  {
	  if(event.GetKeyCode() == WXK_DELETE)
	  {
		   wxBusyCursor busy;
		   std::string temp = "0";
		   RemoveSelected(temp);
		   GetGimmickView()->ClearSelection();
	  }
		  
  }
  //================================================================

  //================================================================
  // Should be in another place : not specific !
  void WxTreeView::GetSelectedAsString(std::vector<std::string>&s)
  {
    int level= (int)mLevelList.size();
    const std::vector<tree::Node*>& sel=GetSelected(level+1);
    std::vector<tree::Node*>::const_iterator i;
    
    for (i=sel.begin(); i!=sel.end(); ++i)
      {
	std::string filename=(*i)->GetAttribute("FullFileName");
	s.push_back(filename);
      }
  }



  //================================================================
  void WxTreeView::GetFilenamesAsString(const std::vector<tree::Node*>& nodes, std::vector<std::string>&s)
  {
    std::vector<tree::Node*>::const_iterator i;
    
    for (i=nodes.begin(); i!=nodes.end(); ++i)
      {
		  if((*i)->GetLevel()<mLevelList.size())
		  {
			 GetTreeHandler()->LoadChildren(*i,0);
			 GetFilenamesAsString((*i)->GetChildrenList(),s);
		  }
		  else
		  {
			std::string filename=(*i)->GetAttribute("FullFileName");
			s.push_back(filename);
		  }
      }
  }

   //================================================================

   //================================================================
  void WxTreeView::GetAttributes(std::vector<std::string>& areShown, std::vector<std::string>& notShown, int level)
  {
	  areShown.clear();
	  notShown.clear();
  	tree::LevelDescriptor::AttributeDescriptorListType::const_iterator a;
	for (a  = GetTreeHandler()->GetTree().GetAttributeDescriptorList(level).begin();
	     a != GetTreeHandler()->GetTree().GetAttributeDescriptorList(level).end();
	     ++a)
	{
		if(a->GetFlags()==creaImageIO::tree::AttributeDescriptor::EDITABLE && IsAttributeVisible(a->GetName(),level))
	    {
			areShown.push_back(a->GetName());
		}
	}
	notShown=mLevelList[level-1].notShownAtts;
  }

  //================================================================
  void WxTreeView::SetNonVisibleAttributes(const std::vector<std::string>& notShown, int nlevel)
  {
	mLevelList[nlevel].notShownAtts=notShown;
  }

  //================================================================
   void WxTreeView::CreateCtrl(std::vector<std::string>& notShown, int nlevel)
  {
	int ctrl_style = wxLC_REPORT | wxLC_VRULES;
    int col_style = wxLIST_FORMAT_LEFT;
	LevelType level;
	mLevelList[nlevel].SelectedUpToDate = true;
	mLevelList[nlevel].SortColumn = 0;
	mLevelList[nlevel].key.clear();
	
	mLevelList[nlevel].wxCtrl = new wxListCtrl(mLevelList[nlevel].wxSplitter,
					  nlevel,
					  wxDefaultPosition, 
					  wxDefaultSize,
					  ctrl_style);
	wxWindow* oldWin=mLevelList[nlevel].wxSplitter->GetWindow1();
	mLevelList[nlevel].wxSplitter->ReplaceWindow(oldWin,mLevelList[nlevel].wxCtrl);
	mLevelList[nlevel].wxSplitter->Initialize(mLevelList[nlevel].wxCtrl);
   
	// Create the columns : one for each attribute of the level
	int col = 0;
	std::string title;

  	tree::LevelDescriptor::AttributeDescriptorListType::const_iterator a;
	for (a  = GetTreeHandler()->GetTree().GetAttributeDescriptorList(nlevel+1).begin();
	     a != GetTreeHandler()->GetTree().GetAttributeDescriptorList(nlevel+1).end();
	     ++a)

	{   
	    if(a->GetFlags()!=creaImageIO::tree::AttributeDescriptor::PRIVATE && IsAttributeVisible(a->GetName(),nlevel+1))
	      {
		  title=a->GetName();
		  std::string temp = a->GetKey();
		  if (temp.compare("ID") != 0)
		  {
			mLevelList[nlevel].wxCtrl->InsertColumn(col, 
					crea::std2wx(title),
					col_style);
			col++;
		  }
		mLevelList[nlevel].key.push_back(a->GetKey());
	      }
		
	  }
	oldWin->Destroy();
	UpdateLevel(1);
	}

   //================================================================
  bool WxTreeView::IsAttributeVisible(const std::string& val, int level)
  {
	  std::vector<std::string> ns=mLevelList[level-1].notShownAtts;
	  std::vector<std::string>::iterator it;
	  bool found=false;
	  for(it=ns.begin();it!=ns.end()&&!found;++it)
	  {
		  if(val.compare(*it)==0)
		  {
			  found=true;
		  }
	  }
	  return !found;
  }
 //================================================================
  //================================================================

	RemoveAlertDlg::RemoveAlertDlg(wxWindow *parent, 
                                       wxString title,    
                                       const wxSize& size)
 :   wxDialog( parent, 
		  wxID_ANY, 
		  title,
		  wxDefaultPosition,
		  size,
		  wxDEFAULT_DIALOG_STYLE)
	{
	    wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);

		//std::string out("To reload deleted patient, you should synchronize your database before.");  // JPR
		//wxTextCtrl *text = new wxTextCtrl(this, wxID_ANY,crea::std2wx(out),wxDefaultPosition, wxSize(500,20));
		wxTextCtrl *text = new wxTextCtrl(this, wxID_ANY,
					_T("To reload deleted patient, you should synchronize your database before."),
					wxDefaultPosition, wxSize(650,20));  // 650 vs 500 ? // JPRx
		mcheck = new wxCheckBox(this, 5478, _T("Do not display this warning again!"));
		Connect( mcheck->GetId(), wxEVT_COMMAND_CHECKBOX_CLICKED , (wxObjectEventFunction) &RemoveAlertDlg::onCheck ); 
		wxSizer* buttonsSizer = this->CreateSeparatedButtonSizer(wxOK|wxCANCEL);
		
		topsizer->Add(text);
		topsizer->Add(mcheck,0,wxGROW);
		topsizer->Add(buttonsSizer,0,wxGROW);
		SetSizer(topsizer, true);
		mSave = false;
		Layout();
	}
	RemoveAlertDlg::~RemoveAlertDlg(){};
	bool RemoveAlertDlg::isChecked()
	{
		return mSave;
	}
	void RemoveAlertDlg::onCheck(wxCommandEvent &Event)
	{
		mSave = mcheck->IsChecked();
	}
	

  //================================================================
  //================================================================
  BEGIN_EVENT_TABLE(WxTreeView, wxPanel)   
  /*
    EVT_SIZE(MyFrame::OnSize)

    EVT_MENU(LIST_QUIT, MyFrame::OnQuit)
    EVT_MENU(LIST_ABOUT, MyFrame::OnAbout)
    EVT_MENU(LIST_LIST_VIEW, MyFrame::OnListView)
    EVT_MENU(LIST_REPORT_VIEW, MyFrame::OnReportView)
    EVT_MENU(LIST_ICON_VIEW, MyFrame::OnIconView)
    EVT_MENU(LIST_ICON_TEXT_VIEW, MyFrame::OnIconTextView)
    EVT_MENU(LIST_SMALL_ICON_VIEW, MyFrame::OnSmallIconView)
    EVT_MENU(LIST_SMALL_ICON_TEXT_VIEW, MyFrame::OnSmallIconTextView)
    EVT_MENU(LIST_VIRTUAL_VIEW, MyFrame::OnVirtualView)
    EVT_MENU(LIST_SMALL_VIRTUAL_VIEW, MyFrame::OnSmallVirtualView)

    EVT_MENU(LIST_FOCUS_LAST, MyFrame::OnFocusLast)
    EVT_MENU(LIST_TOGGLE_FIRST, MyFrame::OnToggleFirstSel)
    EVT_MENU(LIST_DESELECT_ALL, MyFrame::OnDeselectAll)
    EVT_MENU(LIST_SELECT_ALL, MyFrame::OnSelectAll)
    EVT_MENU(LIST_DELETE, MyFrame::OnDelete)
    EVT_MENU(LIST_ADD, MyFrame::OnAdd)
    EVT_MENU(LIST_EDIT, MyFrame::OnEdit)
    EVT_MENU(LIST_DELETE_ALL, MyFrame::OnDeleteAll)
    EVT_MENU(LIST_SORT, MyFrame::OnSort)
    EVT_MENU(LIST_SET_FG_COL, MyFrame::OnSetFgColour)
    EVT_MENU(LIST_SET_BG_COL, MyFrame::OnSetBgColour)
    EVT_MENU(LIST_TOGGLE_MULTI_SEL, MyFrame::OnToggleMultiSel)
    EVT_MENU(LIST_SHOW_COL_INFO, MyFrame::OnShowColInfo)
    EVT_MENU(LIST_SHOW_SEL_INFO, MyFrame::OnShowSelInfo)
    EVT_MENU(LIST_FREEZE, MyFrame::OnFreeze)
    EVT_MENU(LIST_THAW, MyFrame::OnThaw)
    EVT_MENU(LIST_TOGGLE_LINES, MyFrame::OnToggleLines)
    EVT_MENU(LIST_MAC_USE_GENERIC, MyFrame::OnToggleMacUseGeneric)

    EVT_UPDATE_UI(LIST_SHOW_COL_INFO, MyFrame::OnUpdateShowColInfo)
    EVT_UPDATE_UI(LIST_TOGGLE_MULTI_SEL, MyFrame::OnUpdateToggleMultiSel)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(MyListCtrl, wxListCtrl)
    EVT_LIST_BEGIN_DRAG(LIST_CTRL, MyListCtrl::OnBeginDrag)
    EVT_LIST_BEGIN_RDRAG(LIST_CTRL, MyListCtrl::OnBeginRDrag)
	
    EVT_LIST_BEGIN_LABEL_EDIT(-1, WxTreeView::OnBeginLabelEdit)
    EVT_LIST_END_LABEL_EDIT(-1, WxTreeView::OnEndLabelEdit)
	
    EVT_LIST_DELETE_ITEM(LIST_CTRL, MyListCtrl::OnDeleteItem)
    EVT_LIST_DELETE_ALL_ITEMS(LIST_CTRL, MyListCtrl::OnDeleteAllItems)
#if WXWIN_COMPATIBILITY_2_4
    EVT_LIST_GET_INFO(LIST_CTRL, MyListCtrl::OnGetInfo)
    EVT_LIST_SET_INFO(LIST_CTRL, MyListCtrl::OnSetInfo)
#endif
  */
    EVT_LIST_KEY_DOWN(-1, WxTreeView::OnKeyDown)
    EVT_LIST_ITEM_SELECTED(-1, WxTreeView::OnItemSelected)
	EVT_LIST_ITEM_RIGHT_CLICK(-1, WxTreeView::OnItemMenu)
    EVT_LIST_ITEM_DESELECTED(-1, WxTreeView::OnItemDeSelected)
	/*
    EVT_LIST_KEY_DOWN(LIST_CTRL, MyListCtrl::OnListKeyDown)
    EVT_LIST_ITEM_ACTIVATED(LIST_CTRL, MyListCtrl::OnActivated)
    EVT_LIST_ITEM_FOCUSED(LIST_CTRL, MyListCtrl::OnFocused)
*/
    EVT_LIST_COL_RIGHT_CLICK(-1, WxTreeView::OnColClick)
	
    EVT_LIST_COL_CLICK(-1, WxTreeView::OnColClick)

	//EVT_LEFT_DOWN(WxTreeView::OnMouseClick)
	/*
    EVT_LIST_COL_BEGIN_DRAG(LIST_CTRL, MyListCtrl::OnColBeginDrag)
    EVT_LIST_COL_DRAGGING(LIST_CTRL, MyListCtrl::OnColDragging)
    EVT_LIST_COL_END_DRAG(LIST_CTRL, MyListCtrl::OnColEndDrag)

    EVT_LIST_CACHE_HINT(LIST_CTRL, MyListCtrl::OnCacheHint)

#if USE_CONTEXT_MENU
    EVT_CONTEXT_MENU(MyListCtrl::OnContextMenu)
#endif
    EVT_CHAR(MyListCtrl::OnChar)

    EVT_RIGHT_DOWN(MyListCtrl::OnRightClick)
  */
END_EVENT_TABLE()
  
} // EO namespace creaImageIO

