/*
# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image 
#                        pour la Sant)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and 
#  abiding by the rules of distribution of free software. You can  use, 
#  modify and/ or redistribute the software under the terms of the CeCILL-B 
#  license as circulated by CEA, CNRS and INRIA at the following URL 
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html 
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability. 
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------
*/

#include <creaImageIOWxViewer.h>

#include <fstream>
#include <vtkCamera.h>
#include <vtkRenderer.h>
#include <vtkImageData.h>
#include <creaMessageManager.h>
#include <stdio.h>
#include <time.h>


using namespace crea;

namespace creaImageIO
{
  
  //=====================================================================

  //=====================================================================
  class WxViewerPlayer: public wxThread
  {
  public:
    WxViewerPlayer(WxViewer* v) :
      mWxViewer(v)
    {}
    
    void* Entry();
    void  OnExit();
	
  private:

    WxViewer* mWxViewer;
  };
  
  //=====================================================================



  

  //=====================================================================
  // CTor
  WxViewer::WxViewer(wxWindow *parent, 
		     wxWindowID id,
		     wxString title,
		     const wxPoint& pos,
		     const wxSize& size)
    :   wxPanel( parent, 
		 id, 
		 pos,
		 size)
  {
    wxMutexLocker lock(mMutex);
    GimmickDebugMessage(6,"WxViewer::WxViewer"
			<<std::endl);

    mNeedRefresh = false;
    mLastImageShown = NULL;
	
	// previewer	
    mInteractor = new crea::creawxVTKRenderWindowInteractor(this,-1);
    mInteractor->UseCaptureMouseOn();	
 
    mViewer	= vtkImageViewer2::New();
    mViewer->SetupInteractor ( mInteractor );
    
    mCurrent = 0;
	mPlayer = 0;

	// Grid to place checkbox and slider 
	mflexSizer = new wxFlexGridSizer(1,2,1,1);
	//Slider
	mslide = new wxSlider(this,-1,0,0,1, wxDefaultPosition, wxSize(400,40), wxSL_HORIZONTAL | wxSL_LABELS);
	Connect( mslide->GetId(), wxEVT_COMMAND_SLIDER_UPDATED , (wxObjectEventFunction) &WxViewer::OnSlide ); 
	//CheckBox
	mcheck = new wxCheckBox(this,5123,crea::std2wx("Cine Loop"));
	Connect( mcheck->GetId(), wxEVT_COMMAND_CHECKBOX_CLICKED , (wxObjectEventFunction) &WxViewer::OnCineLoop ); 
	mcheck->SetValue(false);
	mflexSizer->Add(mcheck,0, wxFIXED_MINSIZE);
	mflexSizer-> Add( mslide,1,wxALIGN_CENTER | wxFIXED_MINSIZE );

	// Sizer for Previewer and GridSizer
	mtopSizer = new wxBoxSizer(wxVERTICAL);
	mtopSizer->Add(mflexSizer,0);
    mtopSizer-> Add( mInteractor ,1,wxGROW,0);
	SetSizer(mtopSizer,true);
	
	Update();  
    Layout(); 
  }
  //=====================================================================

  //=====================================================================
  /// Destructor
  WxViewer::~WxViewer()
  {
    wxMutexLocker lock(mMutex);
    GimmickDebugMessage(6,"WxViewer::~WxViewer"
			<<std::endl);
    // TO DO : desallocate cleanly
	if(mPlayer)
	{
		mPlayer->Pause();
		mPlayer->Delete();
		mPlayer = 0;
	}
	delete mInteractor;
	//delete mslide;
	//delete mflexSizer;
  }
  //=====================================================================

  //================================================================
   void WxViewer::SetImageVector(std::vector<boost::shared_ptr<ImagePointerHolder> >& pointers)
  {
	wxMutexLocker lock(mMutex);
	GimmickDebugMessage(6,"WxViewer::SetImageVector"<<std::endl);
	imagePointers=pointers;
	
	mslide->SetMax((int)pointers.size());
	mslide->SetMin(1);
	// Refresh don't work, TO MODIFY
	mslide->Refresh();
	mslide->ClearTicks();
	mslide->Hide();
	mslide->Show();
	StartPlayer();
  }

  //================================================================

  void WxViewer::ShowNextImage()
  {
	 
	
	  mMutex.Unlock();
	wxMutexLocker lock(mMutex);
	    
    GimmickMessage(2,"WxViewer::ShowNextImage() "
		   <<mCurrent+1<<"/"
		   <<imagePointers.size()<<std::endl);
    
    if(imagePointers.size()>0)
	{
		if (mCurrent<imagePointers.size()) 
		{
			boost::shared_ptr<ImagePointerHolder> iph = imagePointers[mCurrent];
			//ImagePointerHolder* iph= imagePointers[mCurrent];
			vtkImageData* currIm=iph->Get();
			ShowImage(currIm);
			if ( currIm != mLastImageShown ) 
			{
				mNeedRefresh = true;
				mLastImageShown = currIm;
			}
		mCurrent++;
	    }
		else
		{
			mCurrent = 0;
			//ImagePointerHolder* iph=imagePointers[mCurrent];
			boost::shared_ptr<ImagePointerHolder> iph = imagePointers[mCurrent];
			vtkImageData* currIm=iph->Get();
			ShowImage(currIm);
			if ( currIm != mLastImageShown ) 
			{
				mNeedRefresh = true;
				mLastImageShown = currIm;
			}
			mCurrent++;
		}
	}
  }
  //================================================================

  //=====================================================================
  void WxViewer::ShowImage(vtkImageData* im)
  {
    GimmickDebugMessage(6,"WxViewer::ShowImage"
			<<std::endl);
    if (im==0) return;

    mViewer->SetInput(im);

    mViewer->SetSlice( 0 );

    int x1,x2,y1,y2,z1,z2;
    double spx,spy,spz;
    im->Update();

//std::cout << "in WxViewer::ShowImage PrintSelf() =";
//im->PrintSelf(std::cout, vtkIndent(2));

    im->GetSpacing(spx,spy,spz);
    //im->GetExtent (x1,x2,y1,y2,z1,z2);  // JPR
    im->GetWholeExtent (x1,x2,y1,y2,z1,z2); 
/*       
std::cout << "in WxViewer::ShowImage GetWholeExtent ext =";
	std::cout << "   [x1]=" << x1;
	std::cout << "   [x2]=" << x2;
	std::cout << "   [y1]=" << y1;
	std::cout << "   [y2]=" << y2;
	std::cout << "   [z1]=" << z1;
	std::cout << "   [z2]=" << z2;
std::cout << std::endl; 
*/   
    if ((x1!=mx1) ||
	(x2!=mx2) ||
	(y1!=my1) ||
	(y2!=my2) ||
	(z1!=mz1) ||
	(z2!=mz2) ||
	(spx!=mspx) ||
	(spy!=mspy) ||
	(spz!=mspz) 
	)
      {
	mx1 = x1;
	mx2 = x2;
	my1 = y1;
	my2 = y2;
	mz1 = z1;
	mz2 = z2;
	mspx = spx;
	mspy = spy;
	mspz = spz;

	double *range = im->GetScalarRange();
	mViewer->SetColorWindow(range[1] - range[0]);
	mViewer->SetColorLevel(0.5 * (range[1] + range[0]));

	mViewer->GetRenderer()->ResetCamera();
		  
	//EED 21 mars 2012  FLIP problem  ..PLOP..
	vtkCamera *camera =mViewer->GetRenderer()->GetActiveCamera();		  
	camera->SetViewUp		(	0		,	-1		,	0	);
	camera->SetPosition		((x1+x2)/2	, (y1+y2)/2	, -10000);
	camera->SetFocalPoint	((x1+x2)/2	, (y1+y2)/2	,	0	);
	camera->SetParallelScale( (x2-x1)/3.0 );
		  
		  
	double bounds[6];

	mViewer->GetRenderer()->ComputeVisiblePropBounds(bounds);

	mViewer->GetRenderer()->ResetCameraClippingRange(bounds);
	mViewer->GetRenderer()->SetBackground(0.1,0.1,0.2);  
      }
  } 
  //================================================================
  
  //================================================================
  bool WxViewer::RefreshIfNecessary()
  {
    if (mNeedRefresh)
      {
	GimmickDebugMessage(10,"WxViewer : Refreshing"<<std::endl);

	mInteractor->Render();
	mNeedRefresh = false;
	return true;
      }
    return false;
  }
  //================================================================
  
  //==================================================
  void WxViewer::StopPlayer()
  {
	  wxMutexLocker lock(mMutex);
	  if (mPlayer==0 ) return;
	  mPlayer->Delete();  
	  mPlayer=0;
	  mMutex.Unlock();
  }
  //================================================================
  
  //==================================================
  void WxViewer::StartPlayer()
	{
		if(mcheck->IsChecked())
		{
			//	wxMutexLocker lock(mMutex);
			if (mPlayer != 0) return;
			mPlayer = new WxViewerPlayer(this);
			mPlayer->Create();
			mPlayer->Run();  
		}
		else
		{
			ShowNextImage();
		}
	}

  //================================================================
  
  //==================================================

    void WxViewer::OnCineLoop(wxCommandEvent &Event)
	{
		if(!mcheck->IsChecked())
		{
			mPlayer->Pause();
			mPlayer->Delete();
			mPlayer = 0;
		}
		StartPlayer();
	}
	
 //================================================================
  
  //==================================================
	
	void WxViewer::OnSlide(wxCommandEvent &Event)
	 {
		 mCurrent = mslide->GetValue() -1;
		 StartPlayer();
	 }
	 //================================================================
  
  //==================================================

	 void WxViewer::SetValue()
	 {
		 mslide->SetValue(mCurrent);
	 }

  //  BEGIN_EVENT_TABLE(WxGimmickFrame, wxDialog)
  //    END_EVENT_TABLE()
  //================================================================

//========================================================================
//========================================================================
//========================================================================
//========================================================================
//========================================================================
//========================================================================
//========================================================================
//========================================================================

  void*  WxViewerPlayer::Entry()
  {
	  
    GimmickDebugMessage(6,"WxViewerPlayer::Entry()"<<std::endl);
	
	while(!TestDestroy())
	    { 
			
			mWxViewer->ShowNextImage();
			mWxViewer->SetValue();
			::wxWakeUpIdle();
			clock_t endwait;
			endwait = clock () + 0.2 * CLOCKS_PER_SEC ;
			while (clock() < endwait ) {}
			
      }
    return 0;
  }

  //=====================================================================

  //=====================================================================
  void WxViewerPlayer::OnExit()
  {
    GimmickDebugMessage(6,"WxViewerPlayer::OnExit() "<<std::endl);
  }

 
} // EO namespace creaImageIO

