/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#include "imageplanewidget.h"

#include "vtkSmartPointer.h"
#include "vector"
#include "vtkImageActor.h"
#include "vtkImageAppendComponents.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkTextProperty.h"

using namespace std;

ImagePlaneWidget::ImagePlaneWidget()
{
    _img = 0;
    _xwidget = 0;
    _ywidget = 0;
    _zwidget = 0;
    m_Interactor = 0;
    _imageactorx = 0;
    _imageactory = 0;
    _imageactorz = 0;
}

void ImagePlaneWidget::initialize(vtkRenderWindowInteractor* interactor){
    m_Interactor = interactor;
    // Plane widget
    _xwidget = vtkImagePlaneWidget::New( );
    _ywidget = vtkImagePlaneWidget::New( );
    _zwidget = vtkImagePlaneWidget::New( );


    _xwidget->SetInteractor( interactor );
    _ywidget->SetInteractor( interactor );
    _zwidget->SetInteractor( interactor );

}
bool ImagePlaneWidget::showImageActors(bool show){
    if(show && m_Interactor){
        if(!_imageactorx){
            initializeImageActors();
        }
        m_Interactor->GetRenderWindow ()->GetRenderers ()->GetFirstRenderer ()->AddViewProp(_imageactorx);
        m_Interactor->GetRenderWindow ()->GetRenderers ()->GetFirstRenderer ()->AddViewProp(_imageactory);
        m_Interactor->GetRenderWindow ()->GetRenderers ()->GetFirstRenderer ()->AddViewProp(_imageactorz);
        return true;
    }else if(_imageactorx){
        m_Interactor->GetRenderWindow ()->GetRenderers ()->GetFirstRenderer ()->RemoveViewProp(_imageactorx);
        m_Interactor->GetRenderWindow ()->GetRenderers ()->GetFirstRenderer ()->RemoveViewProp(_imageactory);
        m_Interactor->GetRenderWindow ()->GetRenderers ()->GetFirstRenderer ()->RemoveViewProp(_imageactorz);
        return true;
    }

    return false;
}

void ImagePlaneWidget::initializeImageActors(){

    //cout<<w_ext[0]<<" "<<w_ext[1]<<" "<<w_ext[2]<<" "<<w_ext[3]<<" "<<w_ext[4]<<" "<<w_ext[5]<<" "<<endl;
    _imageactorx = vtkImageActor::New();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    int *w_ext = _img->GetWholeExtent();
    _imageactorx->SetInput(_img);
#else
    int *w_ext = _img->GetExtent();
    _imageactorx->SetInputData(_img);
#endif

    xslice =  (w_ext[4] + w_ext[5])/2;
    _imageactorx->SetDisplayExtent(w_ext[0], w_ext[1], w_ext[2], w_ext[3], xslice, xslice);


    _imageactory = vtkImageActor::New();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    _imageactory->SetInput(_img);
#else
    _imageactory->SetInputData(_img);
#endif

    yslice =  (w_ext[2] + w_ext[3])/2;
    _imageactory->SetDisplayExtent(w_ext[0], w_ext[1], yslice, yslice, w_ext[4], w_ext[5]);

    _imageactorz = vtkImageActor::New();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    _imageactorz->SetInput(_img);
#else
    _imageactorz->SetInputData(_img);
#endif

    zslice =  (w_ext[0] + w_ext[1])/2;
    _imageactorz->SetDisplayExtent(zslice, zslice, w_ext[2], w_ext[3], w_ext[4], w_ext[5]);


}

void ImagePlaneWidget::setImage(vtkImageData* img){
    _img = img;
    this->invariant();

    int *dims = _img->GetDimensions( );
    unsigned int sliceX = dims[0]/2;
    unsigned int sliceY = dims[1]/2;
    unsigned int sliceZ = dims[2]/2;

    // Set input X,Y and Z plane
    _xwidget->Off( );

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    _xwidget->SetInput( _img );
#else
    _xwidget->SetInputData( _img );
#endif

    _xwidget->SetPlaneOrientationToXAxes( );
    _xwidget->SetSliceIndex( sliceX );
    _xwidget->TextureInterpolateOff();
    _xwidget->SetResliceInterpolateToLinear();
    _xwidget->On( );
    _xwidget->DisplayTextOn();
    _xwidget->GetTextProperty()->SetColor(0.3, 0.3, 0.3);

    _ywidget->Off( );

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    _ywidget->SetInput( _img );
#else
    _ywidget->SetInputData( _img );
#endif

    _ywidget->SetPlaneOrientationToYAxes( );
    _ywidget->SetSliceIndex( sliceY );
    _ywidget->TextureInterpolateOff();
    _ywidget->SetResliceInterpolateToLinear();
    _ywidget->DisplayTextOn();
    _ywidget->GetTextProperty()->SetColor(0.3, 0.3, 0.3);
    _ywidget->On( );

    _zwidget->Off( );


//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    _zwidget->SetInput( _img );
#else
    _zwidget->SetInputData( _img );
#endif


    _zwidget->SetPlaneOrientationToZAxes( );
    _zwidget->SetSliceIndex( sliceZ );
    _zwidget->TextureInterpolateOff();
    _zwidget->SetResliceInterpolateToLinear();
    _zwidget->DisplayTextOn();
    _zwidget->On( );
    _zwidget->GetTextProperty()->SetColor(0.3, 0.3, 0.3);
}

void ImagePlaneWidget::SetLookupTable(vtkLookupTable *lookuptable){
    this->invariant();
    _xwidget->SetLookupTable(lookuptable);
    _ywidget->SetLookupTable(lookuptable);
    _zwidget->SetLookupTable(lookuptable);
}

void ImagePlaneWidget::SetColorTable(vtkColorTransferFunction *colortransfer){
    SetLookupTable((vtkLookupTable*)colortransfer);
}

void ImagePlaneWidget::showPlanes(bool show){
    this->invariant();
    if(show){
        _xwidget->On();
        _ywidget->On();
        _zwidget->On();
    }else{
        _xwidget->Off();
        _ywidget->Off();
        _zwidget->Off();
    }
}


void ImagePlaneWidget::invariant()
{
    if(_img == NULL){
        throw "Image data is not initialized";
    }
    if(!_xwidget ||!_ywidget || !_zwidget){
        throw "The widgets are not initialized";
    }
}


void ImagePlaneWidget::setSliceXImageActor(int value)
{
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    int *w_ext = _img->GetWholeExtent();
#else
    int *w_ext = _img->GetExtent();
#endif

    xslice = value;
    _imageactorx->SetDisplayExtent(w_ext[0], w_ext[1], w_ext[2], w_ext[3], xslice, xslice);
    m_Interactor->Render();
}

void ImagePlaneWidget::setSliceYImageActor(int value)
{
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    int *w_ext = _img->GetWholeExtent();
#else
    int *w_ext = _img->GetExtent();
#endif
    yslice = value;
    _imageactory->SetDisplayExtent(w_ext[0], w_ext[1], yslice, yslice, w_ext[4], w_ext[5]);
    m_Interactor->Render();
}


void ImagePlaneWidget::setSliceZImageActor(int value)
{

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
    int *w_ext = _img->GetWholeExtent();
#else
    int *w_ext = _img->GetExtent();
#endif

    zslice = value;
    _imageactorz->SetDisplayExtent(zslice, zslice, w_ext[2], w_ext[3], w_ext[4], w_ext[5]);
    m_Interactor->Render();
}
