/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

#include "volumerenderermanager.h"

/*=========================================================================

  Program:   wxMaracas
  Module:    $RCSfile: volumerenderermanager.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/15 14:16:37 $
  Version:   $Revision: 1.4 $

  Copyright: (c) 2002, 2003
  License:

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include <vtkMetaImageReader.h>
#include <vtkImageCast.h>

#include "vtkImageLuminance.h"
#include "vtkImageAppendComponents.h"
#include "vtkSmartPointer.h"

#include <map>

using namespace std;

/**
**	Start of the manager class
**/
VolumeRendererManager::VolumeRendererManager()
{
    _renderer 	= NULL;
    _idCount	= 0;
}

VolumeRendererManager::~VolumeRendererManager()
{
    _renderer 	= 0;
    _idCount	= 0;
    image 		= 0;
    for(unsigned i = 0; i < prop3Dvect.size();i++){
        prop3Dvect[i]->Delete();
    }
    prop3Dvect.clear();
}

/**
**	Sets the renderer to manage the prop3D from the view
**/
void VolumeRendererManager::setRenderer(vtkRenderer*  renderer)
{
        _renderer = renderer;
}

/**
** Gets the renderer which manage the prop3D from the view
**/
vtkRenderer* VolumeRendererManager::getRenderer(){
        return _renderer;
}

/**
** Updates Volume
**/
void VolumeRendererManager::Update(int ppid)
{
        VolumeRendererManagerData* data = this->getViewData(ppid);
        data->Update();
        _renderer->Render();
}

/**
* @pre   The image can have one or multiple components per voxel, and volume rendering is performed seprately over the
         three of them. If the image has multiple components and the separate components flag is set to false, then
         the vtkImageAppendComponents is used to create a single image.
* @post  The volume rendering is performed over the image vol
* @param vtkImageData* the image volume
* @param bool separatecomponents, if the image has multiple components, then a mapper is used for each of the channels
              if this flag is set to false and the volume has multiple components, vtkImageAppendComponents is used to create
              a single representation of the image.
*/
int VolumeRendererManager::addVolume(vtkImageData* img, vtkRenderWindowInteractor* interactor, bool independentcomponents)
{
    if(img->GetNumberOfScalarComponents() > 1 && !independentcomponents)
	{
        image = img;

        vtkSmartPointer< vtkImageLuminance > luminance = vtkSmartPointer< vtkImageLuminance >::New();
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
        luminance->SetInput(img);
#else
        luminance->SetInputData(img);
#endif

        luminance->Update();

        vtkSmartPointer< vtkImageAppendComponents > append = vtkSmartPointer< vtkImageAppendComponents >::New();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
        append->SetInput(0, img);
        append->SetInput(1, luminance->GetOutput());
#else
        append->SetInputData(0, img);
        append->SetInputData(1, luminance->GetOutput());
#endif

        append->Update();



        VolumeRendererManagerData* data = new VolumeRendererManagerData(append->GetOutput(), true);
        data->SetIndependentComponents(independentcomponents);


        prop3Dvect.push_back(data);

        data->setId(_idCount);
        _idCount++;

        return _idCount-1;


        /*image = img;

        vector< vtkImageData* > vectimg;
        GetImages(img, vectimg);
        vtkBoxWidget* boxw = 0;
        for(unsigned i = 0; i < vectimg.size(); i++){
            VolumeRendererManagerData* data = new VolumeRendererManagerData(vectimg[i], "");

            vtkColorTransferFunction* colorf = data->GetColorFunction();
            colorf->RemoveAllPoints();
            double r = 0, g = 0, b = 0;
            for(unsigned j = 0; j < 255; j++){

                if(i == 0){
                    r = j/255.0;
                    g = 0;
                    b = 0;
                }else if(i == 1){
                    r = 0;
                    g = j/255.0;
                    b = 0;
                }else if(i == 2){
                    r = 0;
                    g = 0;
                    b = j/255.0;
                }

                colorf->AddRGBPoint(j, r, g, b);
            }

            prop3Dvect.push_back(data);

            data->setId(_idCount);
            _idCount++;

            if(!boxw){
                EnableBoundingBox(interactor, data->getId());
                DisableBoundingBox(data->getId());
                boxw = data->GetBoxWidget();
            }else{
                data->SetBoxWidget(boxw);
            }
        }

        boxw->RemoveAllObservers();

        vtkBoxWidgetCallback *callback = vtkBoxWidgetCallback::New();

        for(unsigned i = 0; i < prop3Dvect.size(); i++){
            VolumeRendererManagerData* data = prop3Dvect[i];
            callback->AddMapper(data->GetVolumeMapper());
        }

        boxw->AddObserver(vtkCommand::InteractionEvent, callback);
        callback->Delete();

        return _idCount-1;*/

        /*vtkImageData* imgshort = 0;
        imgshort = vtkImageData::New();
        imgshort->SetNumberOfScalarComponents(1);
        imgshort->SetExtent(img->GetExtent());
        imgshort->SetSpacing(img->GetSpacing());
        imgshort->SetOrigin(img->GetOrigin());
        imgshort->SetScalarTypeToUnsignedShort();
        imgshort->AllocateScalars();
        GetImageDouble(img, imgshort);

        VolumeRendererManagerData* data = new VolumeRendererManagerData(imgshort, "");

        vtkColorTransferFunction* colorf = data->GetColorFunction();
        colorf->RemoveAllPoints();

        map< unsigned short, vector< double > > colormap;

        int *extent = img->GetExtent();

        for(unsigned i = extent[0]; i < extent[1]; i++){
            for(unsigned j = extent[2]; j < extent[3]; j++){
                for(unsigned k = extent[4]; k < extent[5]; k++){

                    unsigned char *imgpoint = ((unsigned char*)img->GetScalarPointer(i, j, k));
                    double temp = (double)(0.299*imgpoint[0] + 0.587*imgpoint[1] + 0.114*imgpoint[2]);
                    unsigned short val = temp*255.0;

                    vector< double > rgb;
                    rgb.push_back(0.299*imgpoint[0]);
                    rgb.push_back(0.587*imgpoint[1]);
                    rgb.push_back(0.114*imgpoint[2]);

                    colormap[val] = rgb;
                }
            }
        }



        map< unsigned short, vector< double > >::iterator it;
        for(it = colormap.begin(); it != colormap.end(); ++it){

            colorf->AddRGBPoint((*it).first, (*it).second[0] / 255.0, (*it).second[1] / 255.0, (*it).second[2] / 255.0);
        }

        prop3Dvect.push_back(data);

        data->setId(_idCount);
        EnableBoundingBox(interactor, data->getId());
        DisableBoundingBox(data->getId());
        _idCount++;*/


    }else{
        image = img;

        VolumeRendererManagerData* data = new VolumeRendererManagerData(img, "");
        prop3Dvect.push_back(data);


        data->setId(_idCount);
        _idCount++;

        EnableBoundingBox(interactor, data->getId());
        DisableBoundingBox(data->getId());



        return data->getId();

    }
}

/**
  *  @pre the image is not null and has more than one scalar component
  *  @post Each component in the image is put in a single image
  *  @param vtkImageData* img, multiple component image i.e. an image of vectors like an rgb
  *  @return vtkImageData* double type image
  */
void VolumeRendererManager::GetImageDouble(vtkImageData* img, vtkImageData* imgushort){


    int *extent = img->GetExtent();

    for(unsigned i = extent[0]; i < extent[1]; i++){
        for(unsigned j = extent[2]; j < extent[3]; j++){
            for(unsigned k = extent[4]; k < extent[5]; k++){
                if(img->GetScalarType() == VTK_UNSIGNED_CHAR){
                    unsigned char *imgpoint = ((unsigned char*)img->GetScalarPointer(i, j, k));

                    unsigned short *vectimgpoint = (unsigned short*)imgushort->GetScalarPointer(i, j, k);
                    double temp = (double)(0.299*imgpoint[0] + 0.587*imgpoint[1] + 0.114*imgpoint[2]);
                    *vectimgpoint = temp*255.0;

                }
            }
        }
    }
}

/**
  *  @pre the image is not null and has more than one scalar component
  *  @post Each component in the image is separated to form a different image
  *  @param vtkImageData* img, multiple component image i.e. an image of vectors like an rgb
  *  @return vector<vtkImageData* > a vector of images, one for each component
  */
void VolumeRendererManager::GetImages(vtkImageData* img, vector<vtkImageData* >& vectimg){

    for(unsigned i = 0; i < img->GetNumberOfScalarComponents(); i++){
        vectimg.push_back(vtkImageData::New());
        vectimg[i]->SetExtent(img->GetExtent());
        vectimg[i]->SetSpacing(img->GetSpacing());
        vectimg[i]->SetOrigin(img->GetOrigin());

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
        vectimg[i]->SetNumberOfScalarComponents(1);
        vectimg[i]->SetScalarType(img->GetScalarType());
        vectimg[i]->AllocateScalars();
#else
        vectimg[i]->AllocateScalars(img->GetScalarType(),1);
#endif

    }

    int *extent = img->GetExtent();

    for(unsigned i = extent[0]; i < extent[1]; i++){
        for(unsigned j = extent[2]; j < extent[3]; j++){
            for(unsigned k = extent[4]; k < extent[5]; k++){
                if(img->GetScalarType() == VTK_UNSIGNED_CHAR){
                    unsigned char *imgpoint = ((unsigned char*)img->GetScalarPointer(i, j, k));

                    for(unsigned l = 0; l < vectimg.size(); l++){
                        unsigned char *vectimgpoint = (unsigned char*)vectimg[l]->GetScalarPointer(i, j, k);
                        *vectimgpoint = imgpoint[l];
                    }
                }else if(img->GetScalarType() == VTK_CHAR){
                    char *imgpoint = ((char*)img->GetScalarPointer(i, j, k));

                   for(unsigned l = 0; l < vectimg.size(); l++){
                        char *vectimgpoint = ( char*)vectimg[l]->GetScalarPointer(i, j, k);
                       *vectimgpoint = imgpoint[l];
                   }
               }else if(img->GetScalarType() == VTK_UNSIGNED_SHORT){
                    unsigned short *imgpoint = ((unsigned short*)img->GetScalarPointer(i, j, k));

                    for(unsigned l = 0; l < vectimg.size(); l++){
                        unsigned short *vectimgpoint = (unsigned short*)vectimg[l]->GetScalarPointer(i, j, k);
                        *vectimgpoint = imgpoint[l];
                    }
                }else if(img->GetScalarType() == VTK_SHORT){
                     short *imgpoint = ((short*)img->GetScalarPointer(i, j, k));

                    for(unsigned l = 0; l < vectimg.size(); l++){
                         short *vectimgpoint = ( short*)vectimg[l]->GetScalarPointer(i, j, k);
                        *vectimgpoint = imgpoint[l];
                    }
                }
            }
        }
    }
}

/**
**	Adds a prop3D to the manager and returns the identifier
**/

// EED 2022-08-04	 C++17 throw
//int VolumeRendererManager::addVolume(int idTP, vtkImageData* vol, std::string dataname) throw(char*)
int VolumeRendererManager::addVolume(int idTP, vtkImageData* vol, std::string dataname) 
{
	try {
        checkInvariant();
        image = vol;
        if(vol != NULL){
                VolumeRendererManagerData* data = new VolumeRendererManagerData(vol, dataname);
                prop3Dvect.push_back(data);
                _renderer->AddActor(data->getProp3D());
                if(idTP == -1)
                {
                        data->setId(_idCount);
                        _idCount++;
                }
                else
                {
                        data->setId(idTP);
                }
                printf("VolumeRendererManager::addVolume->idVolumeRenderer: %i\n", data->getId());
                return data->getId();
        }else{
                throw "Check mhd imagefile file or input";
        }
        return -1;        
	} catch (...) {
	  throw ;
	}           
}
/**
**	adds or removes an actor depending of the bool value
**/

// EED 2022-08-04	 C++17 throw
//void VolumeRendererManager::addRemoveActor(int propid, bool addremove)  throw(char*)
void VolumeRendererManager::addRemoveActor(int propid, bool addremove)  
{
	try {
        checkInvariant();
        VolumeRendererManagerData* data = this->getViewData(propid);
        if(data->getProp3D()!=NULL){
                if(addremove){
                        _renderer->AddViewProp(data->getProp3D());
                }else{
                        _renderer->RemoveViewProp(data->getProp3D());
                }
                _renderer->Render();
        }
	} catch (...) {
	  throw ;
	}           
}
/**
**	Changes the opacity in a prop3D
**/
// EED 2022-08-04	 C++17 throw
//void VolumeRendererManager::setVolumeOpacity(int propid, std::vector<double> greylevel,std::vector<double> value)  throw(char*)
void VolumeRendererManager::setVolumeOpacity(int propid, std::vector<double> greylevel,std::vector<double> value) 
{
	try {
		checkInvariant();	
		VolumeRendererManagerData* volrenman = this->getViewData(propid);
		if (volrenman!=NULL) 
		{
		    volrenman->setVolumeOpacity(greylevel, value);
		} else {
			printf("EED  VolumeRendererManager::setVolumeOpacity  Warning volrenman NULL\n");
		}
		_renderer->Render();
	} catch (...) {
	  throw ;
	}           
}

/**
**	Set Volume Color
**/

// EED 2022-08-04	 C++17 throw
//void VolumeRendererManager::setVolumeColor(int volid, std::vector<double> greylevel,
//                                                                std::vector<double> red,
//                                                                std::vector<double> green,
//                                                                std::vector<double> blue)throw(char*)
void VolumeRendererManager::setVolumeColor(int volid, std::vector<double> greylevel,
                                                                std::vector<double> red,
                                                                std::vector<double> green,
                                                                std::vector<double> blue)
{                                                                
	try {
		checkInvariant();
		VolumeRendererManagerData* volrenman = this->getViewData(volid);
		if (volrenman!=NULL) 
		{
			this->getViewData(volid)->setVolumeColor(greylevel, red, green, blue);
		} else {
			printf("EED  VolumeRendererManager::setVolumeColor  Warning volrenman NULL\n");
		}
		_renderer->Render();
	} catch (...) {
	  throw ;
	}           
}

vtkImageData* VolumeRendererManager::getImageData(std::string filename){
        if(filename.compare("")!= 0){

                vtkMetaImageReader* reader =  vtkMetaImageReader::New();
                reader->SetFileName(filename.c_str());
                reader->Update();
                vtkImageData* img = reader->GetOutput();

                vtkImageCast* cast = vtkImageCast::New();
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
                cast->SetInput(img);
#else
                cast->SetInputData(img);
#endif

                cast->SetOutputScalarTypeToUnsignedShort();
                cast->Update();
                //reader->Delete();
                //img->Delete();
                return cast->GetOutput();
                //return img;
        }
        return NULL;
}

vtkImageData* VolumeRendererManager::getImageData(){
        return image;
}

// EED 2022-08-04	 C++17 throw
//void VolumeRendererManager::checkInvariant()  throw(char*)
void VolumeRendererManager::checkInvariant()  
{
	try{
		if(this->_renderer==NULL)
		{
		            throw "Renderer not set";
		}
	} catch (...) {
	  throw ;
	}           
}

// EED 2022-08-04	 C++17 throw
//VolumeRendererManagerData* VolumeRendererManager::getViewData(int id) throw(char*)
VolumeRendererManagerData* VolumeRendererManager::getViewData(int id) 
{
	try{
		int i;
		for(i = 0; i < (int)(prop3Dvect.size());i++){
		        if(prop3Dvect[i]->getId() == id){
		                return prop3Dvect[i];
		        }
		}
		throw "id not found in the data";
		return NULL;
	} catch (...) {
	  throw ;
	}           
}

// EED 2022-08-04	 C++17 throw
//void VolumeRendererManager::deleteActor(int propid) throw (char *)
void VolumeRendererManager::deleteActor(int propid) 
{
	try{
        checkInvariant();
        this->addRemoveActor(propid, false);
        int i,n;
        bool exit = false;
        for(i = 0; i < (int)(prop3Dvect.size())&&!exit;i++)
        {
                if(prop3Dvect[i]->getId() == propid){
                        n=i;
                        exit = true;
                }
        }
        if(exit){
                VolumeRendererManagerData* data = prop3Dvect[n];
                int j;
                for(j = i; j < (int)(prop3Dvect.size())-1;j++)
                {
                        prop3Dvect[j] = prop3Dvect[j+1];
                }
                delete data;
                prop3Dvect.pop_back();
        }else{
                throw "id not found in the data";
        }
	} catch (...) {
	  throw ;
	}           

}

vtkPiecewiseFunction* VolumeRendererManager::GetTransferFunction(int volumeid)
{
        return getViewData(volumeid)->GetTransferFunction();
}

vtkColorTransferFunction* VolumeRendererManager::GetColorFunction(int volumeid)
{
        return getViewData(volumeid)->GetColorFunction();
}

// EED 2022-08-04	 C++17 throw
//void VolumeRendererManager::changeCompositeMIPFunction(int id, int function) throw (char *)
void VolumeRendererManager::changeCompositeMIPFunction(int id, int function) 
{
	try{
		if(id == -1)
		{
		    for(unsigned i = 0; i < prop3Dvect.size(); i++)
		        prop3Dvect[i]->changeCompositeMIPFunction(function);
		}else{
		    getViewData(id)->changeCompositeMIPFunction(function);
		}
	} catch (...) {
	  throw ;
	}           		
}

/**
  Changes the interpolation of the volume rendering.
  type == 0 for linear interpolation
  type == 1 for nearest interpolation
  */
void VolumeRendererManager::changeInterpolationType(int type, int propid)
{
    if(propid == -1)
    {
        for(unsigned i = 0; i < prop3Dvect.size(); i++)
        {
            prop3Dvect[i]->changeInterpolationType(type);
        } // for i
    } else {
        getViewData(propid)->changeInterpolationType(type);
    }
}

/**
  * Set the lookuptable to the volumes in memory
  * if the id is set then it only changes the lookup table for a specific volume
 */
void VolumeRendererManager::SetLookupTable(vtkLookupTable* lookup, int id){
    if(id == -1){
        for(unsigned i = 0; i < prop3Dvect.size(); i++)
            prop3Dvect[i]->SetLookupTable(lookup);
    }else{
        getViewData(id)->SetLookupTable(lookup);
    }

}

/**
  * @returns all the props3D in this manager
*/
vector< vtkProp3D* > VolumeRendererManager::getProps3D(){

    vector< vtkProp3D* >  propvects;
    for(unsigned i = 0; i < prop3Dvect.size(); i++){
        propvects.push_back(prop3Dvect[i]->getProp3D());
    }
    return propvects;
}

/**
  *  @param std::vector<double> greylevel, the corresponding greylevel in the image
  *  @param std::vector<double> value, the corresponding value for the opacity
  *  @param int propid, the correspoding id, by default it applies the changes to the first volume in the array
  */
void VolumeRendererManager::setVolumeOpacity(std::vector<double> greylevel, std::vector<double> value, int propid)
{
    if(propid == -1)
    {
        for(unsigned i = 0; i < prop3Dvect.size(); i++)
        {
            prop3Dvect[i]->setVolumeOpacity(greylevel, value);
        } // for i
    }else{
        getViewData(propid)->setVolumeOpacity(greylevel, value);
    }
}

void VolumeRendererManager::EnableBoundingBox(vtkRenderWindowInteractor* interactor, int propid)
{
    if(propid == -1)
    {
        for(unsigned i = 0; i < prop3Dvect.size(); i++)
   		{
            prop3Dvect[i]->EnableBoundingBox(interactor);
        } // for i
    }else{
        getViewData(propid)->EnableBoundingBox(interactor);
    }
}

void VolumeRendererManager::DisableBoundingBox(int propid)
{
    if(propid == -1)
    {
        for(unsigned i = 0; i < prop3Dvect.size(); i++)
        {
            prop3Dvect[i]->DisableBoundingBox();
        } // for i
    }else{
        getViewData(propid)->DisableBoundingBox();
    }
}
