/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkInteractorStyleCutter.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/15 14:16:20 $
  Version:   $Revision: 1.2 $

  Copyright (c) 1993-2002 Ken Martin, Will Schroeder, Bill Lorensen 
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkInteractorStyleCutter.h"

#include <vtkPoints.h>
#include <vtkActor2D.h>
#include <vtkObjectFactory.h>
#include <vtkRenderer.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkCellArray.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkProperty2D.h>
#include <vtkCamera.h>

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
vtkCxxRevisionMacro(vtkInteractorStyleCutter, "$Revision: 1.2 $");
#else
// ..
#endif
vtkStandardNewMacro(vtkInteractorStyleCutter);

//----------------------------------------------------------------------------
vtkInteractorStyleCutter::vtkInteractorStyleCutter()
{
  this->CurrentPosition[0] = this->CurrentPosition[1] = 0;
  this->Direction[0] = this->Direction[1] = this->Direction[2] = 0.;
  this->Moving = 0;

  this->Points = vtkPoints::New();
  this->Lines = vtkCellArray::New();
  this->LoopPoints = vtkPoints::New();

  vtkPolyData *pd = vtkPolyData::New();
  pd->SetPoints( Points );
  pd->SetLines( Lines );
   
  vtkPolyDataMapper2D *bboxMapper = vtkPolyDataMapper2D::New();
//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
  bboxMapper->SetInput( pd );
#else
  bboxMapper->SetInputData( pd );
#endif
   
  this->BboxActor = vtkActor2D::New();
  this->BboxActor->SetMapper( bboxMapper );
  this->BboxActor->GetProperty()->SetColor(1, 0, 0);
  this->BboxActor->VisibilityOff();

  finished = false;

  //thanks
  pd->Delete();
  bboxMapper->Delete();
}

//----------------------------------------------------------------------------
vtkInteractorStyleCutter::~vtkInteractorStyleCutter()
{
  this->Points->Delete();
  this->BboxActor->Delete();
  this->Lines->Delete();
  this->LoopPoints->Delete();
}

//----------------------------------------------------------------------------
void vtkInteractorStyleCutter::OnMouseMove()
{
  if (!this->Interactor || !this->Moving)
    {
    return;
    }
  
  this->CurrentPosition[0] = this->Interactor->GetEventPosition()[0];
  this->CurrentPosition[1] = this->Interactor->GetEventPosition()[1];  
  
  //mouse move event
  this->Points->SetPoint(this->PointID, this->CurrentPosition[0], 
    this->CurrentPosition[1], 0);

  this->Interactor->Render();
}

//----------------------------------------------------------------------------
void vtkInteractorStyleCutter::OnLeftButtonDown()
{
  if (!this->Interactor)
    {
    return;
    }

	finished = false;
  
  this->CurrentPosition[0] = this->Interactor->GetEventPosition()[0];
  this->CurrentPosition[1] = this->Interactor->GetEventPosition()[1];  

  if(!this->Moving)
    {
    this->Initialize();

    //Call this before accessing CurrentRenderer
    this->FindPokedRenderer(this->CurrentPosition[0], this->CurrentPosition[1]);
	this->CurrentRenderer->AddViewProp(BboxActor);
    }

  this->Moving = 1;

  this->Points->SetPoint(this->PointID, this->CurrentPosition[0],
    this->CurrentPosition[1], 0);
  this->PointID = this->Points->InsertNextPoint( this->CurrentPosition[0], 
    this->CurrentPosition[1], 0);

  this->Lines->InsertCellPoint( this->PointID );
  this->Lines->UpdateCellCount( this->PointID + 1 );
  this->BboxActor->VisibilityOn();

  this->Interactor->Render();
}

//----------------------------------------------------------------------------
bool vtkInteractorStyleCutter::Finished()
{
    return finished;
}

//----------------------------------------------------------------------------
void vtkInteractorStyleCutter::OnRightButtonDown()
{
  if (!this->Interactor || !this->Moving)
    {
    return;
    }
  
  double xyz[3];
  this->Points->GetPoint( 0, xyz );
  this->Points->SetPoint(this->PointID, xyz);

  //Save current state
  this->EndLoop();

  this->Interactor->Render();
  this->Moving = 0;
  finished = true;
}

//----------------------------------------------------------------------------
void vtkInteractorStyleCutter::Initialize()
{
  this->Points->Reset();
  this->Lines->Reset();

  this->PointID = this->Points->InsertNextPoint( 0, 0, 0);
  this->Lines->InsertNextCell( 1 );
  this->Lines->InsertCellPoint( 0 );
}
//----------------------------------------------------------------------------
void vtkInteractorStyleCutter::EndLoop()
{
  double pi[3],fpi[4];
  int numPts = this->Points->GetNumberOfPoints()-1;
  this->LoopPoints->SetNumberOfPoints( numPts );
  vtkCamera *camera = this->CurrentRenderer->GetActiveCamera();
  //int state = camera->GetParallelProjection ();
  camera->ParallelProjectionOn();

  for (int i=0; i < numPts; i++)
  {
    this->Points->GetPoint(i, pi);
    this->CurrentRenderer->SetDisplayPoint(pi[0], pi[1], 0);
    this->CurrentRenderer->DisplayToWorld();

    this->CurrentRenderer->GetWorldPoint( fpi );
    if ( fpi[3] )
    {
      fpi[0] /= fpi[3];
      fpi[1] /= fpi[3];
      fpi[2] /= fpi[3];
    }
    this->LoopPoints->SetPoint( i, fpi[0], fpi[1], fpi[2] );
  }

  //Set direction of extrusion, should save this state before camera moves
  camera->GetDirectionOfProjection( this->Direction );
  //camera->SetParallelProjection( state );
}
//----------------------------------------------------------------------------
//Just a quick hack:
void vtkInteractorStyleCutter::VisibilityOff()
{
  this->BboxActor->VisibilityOff();
}
//----------------------------------------------------------------------------
void vtkInteractorStyleCutter::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}


