/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

/*=========================================================================

  Program:   wxMaracas
  Module:    $RCSfile: CutModelData.cxx,v $
  Language:  C++
  Date:      $Date: 2012/11/15 14:15:48 $
  Version:   $Revision: 1.10 $

  Copyright: (c) 2002, 2003
  License:

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "CutModelData.h"
#include "vtkRenderWindow.h"
/**
**	Start of the manager class
**/
CutModelData::CutModelData(int id, vtkRenderWindowInteractor* interactor, vtkCommand* observer, vtkImageData* img){
	
	initializeData(id, interactor, observer, img);
}

CutModelData::CutModelData(){

}
void CutModelData::initializeData(int id, vtkRenderWindowInteractor* interactor, vtkCommand* observer, vtkImageData* img){
	_id = id;
	_currentshape=0;
	createBoxWidget(interactor, observer);
	setTransform(img);
	createActor();
	createShapes();
	ChangeShape(0);
        interactor->GetRenderWindow ()->Render();
	checkInvariant();
}

CutModelData::~CutModelData(){
	checkInvariant();
	_boxWidgetVolume->RemoveAllObservers();
	_boxWidgetVolume->Off();
	_boxWidgetVolume->SetInteractor(NULL);
	_boxWidgetVolume->Delete();	
	_Mapper->Delete();	
	_Actor->Delete();	
	delete _cubefigure;
	delete _cylinderfigure;
	delete _spherefigure;
	currentmatrix->Delete();
	modeltransform->Delete();
	inversModel->Delete();
	_id = -1;
}

void CutModelData::RefreshViewBox(){
		
}
void CutModelData::createBoxWidget(vtkRenderWindowInteractor* interactor, vtkCommand* observer){
	_boxWidgetVolume = vtkBoxWidget::New();
	_boxWidgetVolume->SetInteractor( interactor );
	//_boxWidgetVolume->SetPlaceFactor(1);
	//_boxWidgetVolume->SetInput( img );	
	//_boxWidgetVolume->PlaceWidget();

	_boxWidgetVolume->AddObserver( vtkCommand::InteractionEvent		 , observer );
	_boxWidgetVolume->AddObserver( vtkCommand::StartInteractionEvent		 , observer );
	_boxWidgetVolume->AddObserver( vtkCommand::RightButtonReleaseEvent		 , observer );
	
	_boxWidgetVolume->HandlesOn ();
	_boxWidgetVolume->On();
	//_boxWidgetVolume->GetHandleProperty()->SetOpacity(0.5);	
	//_boxWidgetVolume->GetOutlineProperty()->SetOpacity(0.5);	
}

// EED 2022-08-04	throw
//void CutModelData::setTransform(vtkImageData* img)throw( CutModelException)
void CutModelData::setTransform(vtkImageData* img)
{	
	try {
		currentmatrix 			= vtkTransform::New();
		modeltransform 			= vtkTransform::New();
		inversModel				= vtkTransform::New();    
		vtkMatrix4x4* matrix 	= vtkMatrix4x4::New();  
		matrix->Identity();
		double *ori 			= img->GetOrigin();
		int *ext 				= img->GetExtent();
		double *spc 			= img->GetSpacing();
		matrix->SetElement(0,0,(ext[1]-ext[0])/4*spc[0]);
		matrix->SetElement(1,1,(ext[3]-ext[2])/4*spc[1]);	
		matrix->SetElement(2,2,(ext[5]-ext[4])/4*spc[2]);
		double orgx = (ext[1]+ext[0])/2.*spc[0];
	 	double orgy = (ext[3]+ext[2])/2.*spc[1];
		double orgz = (ext[5]+ext[4])/2.*spc[2];
		matrix->SetElement(0,3,orgx);
		matrix->SetElement(1,3,orgy);
		matrix->SetElement(2,3,orgz);
		currentmatrix->SetMatrix(matrix);
		_boxWidgetVolume->SetTransform(currentmatrix);			
	} catch (...) {
	  throw ;
	}
}

void CutModelData::createActor(){	

	_Mapper	= vtkPolyDataMapper::New();	
	_Actor	= vtkActor::New();
	_Actor->SetMapper(_Mapper);	
	_Actor->GetProperty()->SetColor(1, 0, 0);
	_Actor->GetProperty()->SetOpacity(0.5);		
}

// EED 2022-08-04	throw
//void CutModelData::udapteActorDirection()throw( CutModelException)
void CutModelData::udapteActorDirection()
{
	try {
	checkInvariant();	
	_boxWidgetVolume->GetTransform(currentmatrix);		
	_Actor->SetUserMatrix(currentmatrix->GetMatrix());  //SetUserTransform(currentmatrix );
	} catch (...) {
	  throw ;
	}
}

void CutModelData::createShapes()
{
	_cubefigure = new CutModelFigureCube();
	_cylinderfigure = new CutModelFigureCylinder();
	_spherefigure = new CutModelFigureSphere();
}

// EED 2022-08-04	throw
//void CutModelData::changeOpacity(int opacity)throw( CutModelException)
void CutModelData::changeOpacity(int opacity)
{
	try {
		checkInvariant();	
		_Actor->GetProperty()->SetOpacity((double)opacity/100.0);
	} catch (...) {
	  throw ;
	}
}

// EED 2022-08-04	throw
//void CutModelData::ShowViewBox(bool check)throw( CutModelException)
void CutModelData::ShowViewBox(bool check)
{
	try {
		checkInvariant();
		if(check){
			_boxWidgetVolume->On();
		}else{
			_boxWidgetVolume->Off();
		}    
	} catch (...) {
	  throw ;
	}
}

// EED 2022-08-04	throw
//void CutModelData::ChangeShape(int selection)throw( CutModelException)
void CutModelData::ChangeShape(int selection)
{
	try {
				checkInvariant();
				if(selection == 0)
				{
			//EED 2017-01-01 Migration VTK7
			#if VTK_MAJOR_VERSION <= 5
					_Mapper->SetInput(_spherefigure->getPolyData());
			#else
					_Mapper->SetInputData(_spherefigure->getPolyData());
			#endif
				}else if(selection == 1){

			#if VTK_MAJOR_VERSION <= 5
					_Mapper->SetInput(_cylinderfigure->getPolyData());
			#else
					_Mapper->SetInputData(_cylinderfigure->getPolyData());
			#endif

				}else if(selection == 2){
			//EED 2017-01-01 Migration VTK7
			#if VTK_MAJOR_VERSION <= 5
					_Mapper->SetInput(_cubefigure->getPolyData());
			#else
					_Mapper->SetInputData(_cubefigure->getPolyData());
			#endif
				}else{
					throw CutModelException("Shape type not found");
				}
				_currentshape=selection;
	} catch (...) {
	  throw ;
	}

}

// EED 2022-08-04	throw
//void CutModelData::checkInvariant()throw( CutModelException)
void CutModelData::checkInvariant()
{
	try {
		if(_boxWidgetVolume==NULL){
			throw CutModelException("Box Widget not created");
		}
		if(_Mapper==NULL){
			throw CutModelException("Mapper not created");
		}
		if(_Actor==NULL){
			throw CutModelException("Actor not created");
		}
		if(_cubefigure==NULL){
			throw CutModelException("Cube not created");
		}
		if(_cylinderfigure==NULL){
			throw CutModelException("Cylinder not created");
		}
		if(_spherefigure==NULL){
			throw CutModelException("Sphere not created");
		}
	} catch (...) {
	  throw ;
	}
}

// EED 2022-08-04	throw
//vtkActor* CutModelData::getActor()throw( CutModelException)
vtkActor* CutModelData::getActor()
{
	try {
		checkInvariant();
		return _Actor;
	} catch (...) {
	  throw ;
	}
}

// EED 2022-08-04	throw
//void CutModelData::changeColor(double r,double g,double b)throw( CutModelException)
void CutModelData::changeColor(double r,double g,double b)
{
	try {
		checkInvariant();		
		_Actor->GetProperty()->SetColor( r,g,b );	
	} catch (...) {
	  throw ;
	}
}

CutModelFigure* CutModelData::getCurentCuttingModel()
{
	checkInvariant();	
	if(_currentshape == 0){
		return _spherefigure;
	}else if(_currentshape == 1){
		return _cylinderfigure;
	}else if(_currentshape == 2){
		return _cubefigure;
	}else{
		throw CutModelException("Shape type not found");
	}	
}

// EED 2022-08-04	throw
//void CutModelData::ExecuteCut( double* range, bool isinside, vtkImageData* copyimage)throw( CutModelException)
void CutModelData::ExecuteCut( double* range, bool isinside, vtkImageData* copyimage)
{
	try {
		wxBusyCursor wait;
		CutModelFigure* actualCuttingModel = getCurentCuttingModel();
		//_boxWidgetVolume->GetTransform(currentmatrix);	
		actualCuttingModel->SetVtkTransform(getModelTransform(copyimage));
		//actualCuttingModel->SetVtkTransform(currentmatrix);
		actualCuttingModel->SetInversVtkTransform(getModelTransformInvers());	
		bool			inside;
		//bool			volInt, volExt; // unused
		int			xx,yy,zz;
		unsigned short* 	pOrg;	
		int			ext[6];
		//double			spc[3]; // unused
		//long int		contAfter = 0; // unused
		//long int		contBefor = 0; // unused
		double minvalue = range[0]; 
		double value    = range[1];
		double maxvalue = range[2]; 
		copyimage->GetExtent(ext);    	  
		for (xx=ext[0];xx<=ext[1]; xx++)
		{
		            for (yy=ext[2];yy<=ext[3]; yy++)
			{
		                    for (zz=ext[4];zz<=ext[5];zz++)
				{
					inside=actualCuttingModel->IfPointInside(xx,yy,zz);
					if (  ((inside==true)&&(isinside==true)) || ((!inside==true)&&(!isinside)) )
					{
						pOrg=(unsigned short*)copyimage->GetScalarPointer (xx,yy,zz); 

						//std::cout<<"xx,yy,zz "<<xx<<","<<yy<<","<<zz<<" "<<*pOrg<<std::endl;
						if ((unsigned short)minvalue <=(*pOrg)&&(*pOrg)<=(unsigned short)maxvalue)
						{	
							
							*pOrg=(unsigned short)value;
							
						} 					
					} // if inside
				} // for zz
			} // for yy
		} // for xx
	} catch (...) {
	  throw ;
	}
}

// EED 2022-08-04	throw
//void CutModelData::ExecuteUnCut(bool isinside, vtkImageData* image, vtkImageData* copyimage)throw( CutModelException)
void CutModelData::ExecuteUnCut(bool isinside, vtkImageData* image, vtkImageData* copyimage)
{
	try {
		wxBusyCursor wait;
		CutModelFigure* actualCuttingModel = getCurentCuttingModel();
		actualCuttingModel->SetVtkTransform(getModelTransform(copyimage));
		actualCuttingModel->SetInversVtkTransform(getModelTransformInvers());	
		bool			inside;
		// bool			volInt, volExt; // unused // JPRx
		int			xx,yy,zz;
		unsigned short* pOrg;	
		int			ext[6];
		//double		spc[3]; // unused // JPRx
		//long int		contAfter = 0; // unused // JPRx
		//long int		contBefor = 0;	 // unused // JPRx
		double value;
		copyimage->GetExtent(ext);    	  
		for (xx=ext[0];xx<=ext[1]; xx++)
		{
		            for (yy=ext[2];yy<=ext[3]; yy++)
			{
		                    for (zz=ext[4];zz<=ext[5];zz++)
				{
					inside=actualCuttingModel->IfPointInside(xx,yy,zz);
					if (  ((inside==true)&&(isinside==true)) || ((!inside==true)&&(!isinside)) )
					{
						value = *((unsigned short*)image->GetScalarPointer (xx,yy,zz));
						pOrg=(unsigned short*)copyimage->GetScalarPointer (xx,yy,zz); 
						//std::cout<<"CutModelData::ExecuteUnCut() "<<value<<" "<<*pOrg<<std::endl;
						*pOrg=(unsigned short)value;
					} // if inside
				} // for zz
			} // for yy
		} // for xx
	} catch (...) {
	  throw ;
	}

}

vtkTransform* CutModelData::getModelTransform(vtkImageData* copyimage)
{
	double *spc = copyimage->GetSpacing();
	modeltransform->Identity();
    vtkMatrix4x4* matrix = currentmatrix->GetMatrix();
	vtkMatrix4x4* matrixmodel = modeltransform->GetMatrix();
	matrixmodel->SetElement(0,0,matrix->GetElement(0,0)/spc[0]);
	matrixmodel->SetElement(1,0,matrix->GetElement(1,0)/spc[0]);
	matrixmodel->SetElement(2,0,matrix->GetElement(2,0)/spc[0]);
	matrixmodel->SetElement(0,1,matrix->GetElement(0,1)/spc[1]);
	matrixmodel->SetElement(1,1,matrix->GetElement(1,1)/spc[1]);
	matrixmodel->SetElement(2,1,matrix->GetElement(2,1)/spc[1]);
	matrixmodel->SetElement(0,2,matrix->GetElement(0,2)/spc[2]);
	matrixmodel->SetElement(1,2,matrix->GetElement(1,2)/spc[2]);
	matrixmodel->SetElement(2,2,matrix->GetElement(2,2)/spc[2]);		
	matrixmodel->SetElement(0,3,matrix->GetElement(0,3)/spc[0]);
	matrixmodel->SetElement(1,3,matrix->GetElement(1,3)/spc[1]);
	matrixmodel->SetElement(2,3,matrix->GetElement(2,3)/spc[2]);	
	/*
	double* pos = currentmatrix->GetPosition();
	modeltransform->Translate(pos[0]/spc[0],pos[1]/spc[1],pos[2]/spc[2]);	
	double* scal = currentmatrix->GetScale();	
	modeltransform->Scale(scal[0]/spc[0],scal[1]/spc[1],scal[2]/spc[2]);
	//double* orient = currentmatrix->GetOrientation();
	//modeltransform->RotateZ(orient[2]);	
	//modeltransform->RotateY(orient[1]);	
	//modeltransform->RotateX(orient[0]);	
	double* orient = currentmatrix->GetOrientationWXYZ();	
	modeltransform->RotateWXYZ(orient[0],orient[1],orient[2],orient[3]);	
	*/
	modeltransform->Update();
	return modeltransform;
}

// EED 2022-08-04	throw
//void CutModelData::setTransform(vtkTransform* transform, vtkImageData* img)throw( CutModelException)
void CutModelData::setTransform(vtkTransform* transform, vtkImageData* img)
{	
	try {
		//	double *spc = img->GetSpacing();  // spc unused // JPR
			currentmatrix->Identity();	
			/*double* orient = transform->GetOrientationWXYZ();
			currentmatrix->RotateWXYZ(orient[0],orient[1],orient[2],orient[3]);	
			double* pos = transform->GetPosition();
			currentmatrix->Translate(pos[0]/spc[0],pos[1]/spc[1],pos[2]/spc[2]);
			double* scal = transform->GetScale();	
			currentmatrix->Scale(scal[0]/spc[0],scal[1]/spc[1],scal[2]/spc[2]);
			currentmatrix->Update();	*/
			_boxWidgetVolume->SetTransform(transform);
			/*vtkMatrix4x4* matrix = transform->GetMatrix();
			vtkMatrix4x4* matrixcurrent = currentmatrix->GetMatrix();
			matrixcurrent->SetElement(0,0,matrix->GetElement(0,0)/spc[0]);
			matrixcurrent->SetElement(1,0,matrix->GetElement(1,0)/spc[0]);
			matrixcurrent->SetElement(2,0,matrix->GetElement(2,0)/spc[0]);
			matrixcurrent->SetElement(0,1,matrix->GetElement(0,1)/spc[1]);
			matrixcurrent->SetElement(1,1,matrix->GetElement(1,1)/spc[1]);
			matrixcurrent->SetElement(2,1,matrix->GetElement(2,1)/spc[1]);
			matrixcurrent->SetElement(0,2,matrix->GetElement(0,2)/spc[2]);
			matrixcurrent->SetElement(1,2,matrix->GetElement(1,2)/spc[2]);
			matrixcurrent->SetElement(2,2,matrix->GetElement(2,2)/spc[2]);		
			matrixcurrent->SetElement(0,3,matrix->GetElement(0,3)/spc[0]);
			matrixcurrent->SetElement(1,3,matrix->GetElement(1,3)/spc[1]);
			matrixcurrent->SetElement(2,3,matrix->GetElement(2,3)/spc[2]);	*/
			udapteActorDirection();
			getModelTransform(img);
	} catch (...) {
	  throw ;
	}
}

vtkTransform* CutModelData::getModelTransformInvers()
{
		inversModel->Identity ();
		inversModel->Concatenate ( modeltransform );
		inversModel->Inverse();	
		return inversModel;
}

// EED 2022-08-04	throw
//vtkPolyData* CutModelData::getPolyData()throw( CutModelException)
vtkPolyData* CutModelData::getPolyData()
{
	try {
		return _Mapper->GetInput();
	} catch (...) {
	  throw ;
	}
}

