#*************************************************************************
#*                                  *                                    *
#* NAME     : ibrowser.tcl          * PROJECT  : MARACAS                 *
#* AUTHOR   : Leonardo Flrez/Kyron * TYPE     : TCL/TK widget def.      *
#* VERSION  : v2.0                  * CREATION : 06/05/2001              *
#* LANGUAGE : TCL                   * REVISION : 16/07/2001              *
#*                                  *                                    *
#*************************************************************************
#*                                                                       *
#*  Description : This file defines a new TCL/TK widget that allows the  *
#*                user to browse a serie of image thumbnails. It has a   *
#*                dinamical scroll bar, so, don't worry about use one.   *
#*                                                                       *
#*  The basic use of this widget is:                                     *
#*                                                                       *
#*      1. Create a new widget: "ibrowser <name> <options>"              *
#*         Options given are a list of '-<option> <value>' pairs.        *
#*         Supported options are:                                        *
#*         +-----------------+--------------------------------------+    *
#*         | OPTION          | DESCRIPTION                          |    *
#*         +-----------------+--------------------------------------+    *
#*         | -background     | Background color                     |    *
#*         | -borderwidth    | Border width                         |    *
#*         | -closeenough    | Proximity value                      |    *
#*         +-----------------+--------------------------------------+    *
#*         | -cache          | Indicates if ibrowser must save real |    *
#*         |                 | images.                              |    *
#*         | -cursor         | Cursor value                         |    *
#*         | -fontcolor      | Font color                           |    *
#*         +-----------------+--------------------------------------+    *
#*         | -gap            | Distance between thumbnails          |    *
#*         | -height         | Height of widget                     |    *
#*         | -multisel       | Allows multiselection                |    *
#*         +-----------------+--------------------------------------+    *
#*         | -primarycolor   | Primary selection color              |    *
#*         | -relief         | Type of border                       |    *
#*         | -secondarycolor | Secondary selection color            |    *
#*         +-----------------+--------------------------------------+    *
#*         | -takefocus      | Indicates if widget have focus       |    *
#*         | -thumbheight    | Height of thumbnails                 |    *
#*         | -thumbwidth     | Width of thumbnails                  |    *
#*         +-----------------+--------------------------------------+    *
#*         | -width          | Width of widget                      |    *
#*         +-----------------+--------------------------------------+    *
#*                                                                       *
#*      2. Pack this new widget in your hierarchy:                       *
#*         (pack/place/grid) <name> <pack options>                       *
#*                                                                       *
#*      3. Interact with the new widget by using their sub-commands      *
#*         interface. Sub-commands defined are:                          *
#*         add, cget, configure, curselection, delete, find, lastimage,  *
#*         select, size.                                                 *
#*                                                                       *
#*      4. Optional: use the <<AfterSelectImage>> event definition to    *
#*         grab mouse interaction. Mouse events supported are:           *
#*          Button1       = Basic single selection                       *
#*          Button2       = Multiple selection                           *
#*          Button3       = Multiple selection                           *
#*          Shift+Button1 = Multiple selection                           *
#*                                                                       *
#*************************************************************************
#*                                                                       *
#*  USED MODULES :                                                       *
#*                     TK >= v8.0                                        *
#*                                                                       *
#*************************************************************************
#*                                                                       *
#* REVISIONS :                                                           *
#* (NOTE: Please, don't let this file became a mess. ;-) )               *
#*                                                                       *
#* +------------+----------------+-------------------------------------+ *
#* | DATE       | AUTHOR         | DESCRIPTION                         | *
#* +------------+----------------+-------------------------------------+ *
#* | 06/05/2001 | Kyron          | Initial implementation.             | *
#* +------------+----------------+-------------------------------------+ *
#* | 16/07/2001 | Kyron          | Documentation & conflicts revision. | *
#* +------------+----------------+-------------------------------------+ *
#*                                                                       *
#*************************************************************************

package require Tk 8.0

#* NAMESPACE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser (namespace)                                                *
#*                                                                       *
#* DESCRIPTION : Global namespace that contains all ibrowser widgets in  *
#*               current interpreter (actual TCL work instance).         *
#*                                                                       *
#* SYNTAX : -NONE-                                                       *
#*                                                                       *
#* RETURN :                                                              *
#*        EXPORTS : proc ibrowser { name options }                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*            Namespace components :                                     *
#*              widgetOptions  : list. List of supported options.        *
#*              widgetCommands : list. List of supported sub-commands.   *
#*                                                                       *
#******************************************************* END DESCRIPTION *
package provide ibrowser 2.0
namespace eval ::ibrowser {

    # public interface
    namespace export ibrowser
    
    # variables
    variable widgetOptions
    variable widgetCommands

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::ibrowser (procedure)                                      *
#*                                                                       *
#* DESCRIPTION : Creator of new widgets. Call it in an TK hierarchy      *
#*               creation process.                                       *
#*                                                                       *
#* SYNTAX : ibrowser <name> -<option1> <value1> ... -<optionn> <valuen>  *
#*                                                                       *
#* RETURN : New widget name, if success.                                 *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      name : string. Name for the new widget. To use it in a TK widget *
#*                     hierarchy, this name should be ".<f1>.<f2>...<n>" *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::ibrowser { name args } {

    # Namespace variables used in this procedure
    upvar ::ibrowser::widgetOptions widgetOptions

    # If global namespace doesn't exists yet then initialize it
    if { ![ info exists widgetOptions ] } { ::ibrowser::initIbrowser }

    # Given name exists?. If so, raise an error and finish
    if { [ winfo exists $name ] } {
        error "Widget \"$name\" already exists."
    }

    # Create the new command and return success
    set name [ eval ::ibrowser::buildIbrowser $name $args ]
    return $name

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::initIbrowser (procedure)                                  *
#*                                                                       *
#* DESCRIPTION : Initializes the class manager, i.e., creates the global *
#*               namespace.                                              *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::initIbrowser                                     *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      -NONE-                                                           *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::initIbrowser { } {

    # Namespace variables used in this procedure
    upvar ::ibrowser::widgetOptions  widgetOptions
    upvar ::ibrowser::widgetCommands widgetCommands

    # All posible options for the widget
    array set widgetOptions [ list                     \
        -background     { background     Background  } \
        -borderwidth    { borderWidth    BorderWidth } \
        -closeenough    { closeEnough    CloseEnough } \
        -cache          { cache          Thumbnails  } \
        -cursor         { cursor         Cursor      } \
        -fontcolor      { fontColor      Thumbnails  } \
        -gap            { gap            Thumbnails  } \
        -height         { height         Height      } \
        -multisel       { multiSel       Thumbnails  } \
        -primarycolor   { primaryColor   Thumbnails  } \
        -relief         { relief         Relief      } \
        -secondarycolor { secondaryColor Thumbnails  } \
        -takefocus      { takeFocus      TakeFocus   } \
        -thumbheight    { thumbHeight    Thumbnails  } \
        -thumbwidth     { thumbWidth     Thumbnails  } \
        -width          { width          Width       } \
    ]

    # All posible commands for the widget
    set widgetCommands [ list                    \
        add     cget    configure   curselection \
        delete  find    lastimage   select       \
        size                                     \
    ]

    # New event definition
    event add <<AfterSelectImage>> \
        <ButtonPress-1>            \
        <ButtonPress-2>            \
        <ButtonPress-3>            \
        <Shift-ButtonPress-1>

    # Default initialization... only if Tk exists
    if { [ lsearch -exact [ package names ] "Tk" ] != -1 } {

        option add *Ibrowser.background      #c0c0c0 widgetDefault
        option add *Ibrowser.borderWidth     0       widgetDefault
        option add *Ibrowser.closeEnough     1.0     widgetDefault
        option add *Ibrowser.cache           0       widgetDefault
        option add *Ibrowser.cursor          {}      widgetDefault
        option add *Ibrowser.fontColor       #ffff00 widgetDefault
        option add *Ibrowser.gap             5       widgetDefault
        option add *Ibrowser.height          100     widgetDefault
        option add *Ibrowser.multiSel        0       widgetDefault
        option add *Ibrowser.primaryColor    #ff0000 widgetDefault
        option add *Ibrowser.relief          flat    widgetDefault
        option add *Ibrowser.secondaryColor  #00ff00 widgetDefault
        option add *Ibrowser.takeFocus       0       widgetDefault
        option add *Ibrowser.thumbHeight     50      widgetDefault
        option add *Ibrowser.thumbWidth      50      widgetDefault
        option add *Ibrowser.width           100     widgetDefault

    }

    # set global bindings
    ::ibrowser::setClassIbrowserBindings

}                                                   

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::setClassIbrowserBindings (procedure)                      *
#*                                                                       *
#* DESCRIPTION : Default namespace bindings.                             *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::setClassIbrowserBindings                         *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      -NONE-                                                           *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::setClassIbrowserBindings { } {

    bind Ibrowser <Destroy> [ list ::ibrowser::ibrowserDestroyHandler %W ]

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::buildIbrowser (procedure)                                 *
#*                                                                       *
#* DESCRIPTION : This does all of the work necessary to create a basic   *
#*               ibrowser widget. Creates a new command (widget) with    *
#*               the given name. Also creates a new namespace as a child *
#*               namespace of ::ibrowser.                                *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : set wname [ ::ibrowser::buildIbrowser $name $options ]       *
#*                                                                       *
#* RETURN : New widget hierarchy name                                    *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. New widget name.                                  *
#*      args : list. Option/value pairs list.                            *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::buildIbrowser { w args } {

    # Namespace variables used in this procedure
    upvar ::ibrowser::widgetOptions widgetOptions

    # Child namespace. There's one for each defined widget.
    namespace eval ::ibrowser::$w {

        variable this
        variable options
        variable widgets
        variable currsel {}
        variable lastsel {}
        variable images  {}
        variable nMaxX   0

    }

    # import variables, for programming facilities
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::options options

    # Main frame that contains an ibrowser
    set widgets(this) [     \
        frame $w            \
            -class Ibrowser \
            -takefocus 0    \
            -relief flat    \
            -borderwidth 0  \
    ]

    # Canvas that contains all graphical data
    set widgets(canvas) [ canvas $w.canvas -takefocus 1 ]

    # Dinamical vertical scroll
    set widgets(scroll) ""

    # Set default values
    foreach name [ array names widgetOptions ] {

        set optName  [ lindex $widgetOptions($name) 0 ]
        set optClass [ lindex $widgetOptions($name) 1 ]
        set value [ option get $w $optName $optClass ]
        set options($name) $value

    }

    # set user values...
    if { [ llength $args ] > 0 } { array set options $args }

    # move the name to ibrowser class' namespace...
    set widgets(frame) ::ibrowser::${w}::$w
    rename ::$w $widgets(frame)

    # set canvas options...
    $widgets(canvas) configure -background  $options(-background)
    $widgets(canvas) configure -borderwidth $options(-borderwidth)
    $widgets(canvas) configure -closeenough $options(-closeenough)
    $widgets(canvas) configure -cursor      $options(-cursor)
    $widgets(canvas) configure -height      $options(-height)
    $widgets(canvas) configure -relief      $options(-relief)
    $widgets(canvas) configure -takefocus   $options(-takefocus)
    $widgets(canvas) configure -width       $options(-width)

    # pack the canvas...
    pack $widgets(canvas) -fill both -expand 1

    # local event binding stuff...
    bind $widgets(canvas) <ButtonPress-1>       "::ibrowser::sFImage $widgets(this) %x %y"
    bind $widgets(canvas) <Shift-ButtonPress-1> "::ibrowser::sLImage $widgets(this) %x %y"
    bind $widgets(canvas) <ButtonPress-2>       "::ibrowser::sLImage $widgets(this) %x %y"
    bind $widgets(canvas) <ButtonPress-3>       "::ibrowser::sLImage $widgets(this) %x %y"
    bind $widgets(canvas) <Configure>           "::ibrowser::resize  $widgets(this) %w %h"

    # >>>>>>>>>>>>>>>>>>> HERE, AT LAST, THE NEW COMMAND IS CREATED <<<<<<<<<<<<<<<<<< #
    proc ::$w { command args } "eval ::ibrowser::ibrowserWidgetProc $w \$command \$args"
    # >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< #

    # Last configuration stuff
    if { [ catch "::ibrowser::configureIbrowser $widgets(this) [ array get options ]" error ] } {

        catch { destroy $w }
        error $error

    }

    # have fun ;-)
    return ""

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::configureIbrowser (procedure)                             *
#*                                                                       *
#* DESCRIPTION : This does the configuration process, i.e., change of    *
#*               any option.                                             *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : set ret [ ::ibrowser::configureIbrowser $widget $options ]   *
#*                                                                       *
#* RETURN : All options, if args is empty. If length args == 1 then      *
#*          returns current value. Empty string otherwise.               *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Option/value pairs list.                            *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::configureIbrowser { w args } {

    # For namespace access
    upvar ::ibrowser::widgetOptions widgetOptions
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    # Sends all information to the user...
    if { [ llength $args ] == 0 } {

        set results {}
        foreach opt [ lsort [ array names widgetOptions ] ] {

            if { [ llength $widgetOptions($opt) ] == 1 } {

                set alias $widgetOptions($opt)
                set optName $widgetOptions($alias)
                lappend results [ list $opt $optName ]

            } else {

                set optName     [ lindex $widgetOptions($opt) 0 ]
                set optClass    [ lindex $widgetOptions($opt) 1 ]
                set default     [ option get $w $optName $optClass ]
                lappend results [ list $opt $optName $optClass $default $options($opt) ]

            }

        }
        return $results

    }

    # or single information...
    if { [ llength $args ] == 1 } {

        set opt      [ ::ibrowser::canonizeIbrowser $w option [ lindex $args 0 ] ]
        set optName  [ lindex $widgetOptions($opt) 0 ]
        set optClass [ lindex $widgetOptions($opt) 1 ]
        set default  [ option get $w $optName $optClass ]
        set results  [ list $opt $optName $optClass $default $options($opt) ]
        return $results

    }

    # check if the list is given in pairs
    if { [ expr { [ llength $args ] % 2 } ] == 1 } {
        error "some values for \"$args\" are missing"
    }

    # check if all given options exists...
    foreach { name value } $args {

        set name [ ::ibrowser::canonizeIbrowser $w option $name ]
        set opts($name) $value

    }

    # and set values...
    foreach option [ array names opts ] {

        set newValue $opts($option)
        switch -- $option {

            -cache {

                if { $newValue == 0 || $newValue == 1 } {
                    set options(-cache) $newValue
                } else { error "\"-cache\" option must be: 0/1" }

            }
            -multisel {

                if { $newValue == 0 || $newValue == 1 } {
                    set options(-multisel) $newValue
                } else { error "\"-multisel\" option must be: 0/1" }

            }
            -gap {

                set options(-gap) $newValue
                set nMaxX [ expr int( floor ( $options(-width) / ( $options(-gap) + \
                                                                   $options(-thumbwidth) \
                                                                 ) \
                                            ) \
                                    ) \
                ]
                if { $nMaxX == 0 } {

                    set options(-width) [ expr $options(-thumbwidth) + $options(-gap) ]
                    set nMaxX [ expr int( floor ( $options(-width) / ( $options(-gap) + \
                                                                       $options(-thumbwidth) \
                                                                     ) \
                                                ) \
                                        ) \
                    ]
                    $widgets(canvas) configure -width $options(-width)

                }

            }
            -thumbwidth {

                set options(-thumbwidth) $newValue
                set nMaxX [ expr int( floor ( $options(-width) / ( $options(-gap) + \
                                                                   $options(-thumbwidth) \
                                                                 ) \
                                            ) \
                                    ) \
                ]
                if { $nMaxX == 0 } {

                    set options(-width) [ expr $options(-thumbwidth) + $options(-gap) ]
                    set nMaxX [ expr int( floor ( $options(-width) / ( $options(-gap) + \
                                                                       $options(-thumbwidth) \
                                                                     ) \
                                                ) \
                                        ) \
                    ]
                    $widgets(canvas) configure -width $options(-width)

                }

            }
            -fontcolor      { set options(-fontcolor)      $newValue }
            -primarycolor   { set options(-primarycolor)   $newValue }
            -secondarycolor { set options(-secondarycolor) $newValue }
            -thumbheight    { set options(-thumbheight)    $newValue }
            default         { eval "$widgets(canvas) configure $option $newValue" }

        }

    }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::canonizeIbrowser (procedure)                              *
#*                                                                       *
#* DESCRIPTION : Takes a option or command and canonizes it. Returns     *
#*               either the canonical form of an option or command, or   *
#*               raises an error if the option or command is unknown or  *
#*               ambiguous.                                              *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : set c [ ::ibrowser::canonizeIbrowser $w option $args ]       *
#*                                                                       *
#* RETURN : Option or command canonical form                             *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*      object : string. option/command id.                              *
#*      opt    : string. Option/command value.                           *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::canonizeIbrowser { w object opt } {

    # Namespace variables used in this procedure
    upvar ::ibrowser::widgetOptions  widgetOptions
    upvar ::ibrowser::widgetCommands widgetCommands

    switch $object {
    
        command {

            if { [ lsearch -exact $widgetCommands $opt ] >= 0 } { return $opt }
            set list $widgetCommands
            foreach element $list { set tmp($element) "" }
            set matches [ array names tmp ${opt}* ]

        }
        option {

            if { [ info exists widgetOptions($opt) ] && \
                 [ llength $widgetOptions($opt) ] == 2 \
            } { return $opt }
            set list [ array names widgetOptions ]
            set matches [ array names widgetOptions ${opt}* ]

        }

    }
    if { [ llength $matches ] == 0 } {
        error "unknown $object \"$opt\"; must be one of $list"
    } elseif { [ llength $matches ] == 1 } {

        set opt [ lindex $matches 0 ]

        switch $object {

            option {

                set opt [ lindex $matches 0 ]
                if { [ llength $widgetOptions($opt) ] == 1 } { set opt $widgetOptions($opt) }

            }

        }
        return $opt

    } else { error "ambiguous $object \"$opt\"; must be one of $list" }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::ibrowserDestroyHandler (procedure)                        *
#*                                                                       *
#* DESCRIPTION : Handles the destroy event.                              *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::ibrowserDestroyHandler $w                        *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::ibrowserDestroyHandler { w } {

    if { [ string compare [ winfo class $w ] "Ibrowser" ] == 0 } {
 
        # Namespace variables used in this procedure
        upvar ::ibrowser::${w}::options options
        upvar ::ibrowser::${w}::widgets widgets
        upvar ::ibrowser::${w}::currsel currsel
        upvar ::ibrowser::${w}::lastsel lastsel
        upvar ::ibrowser::${w}::images  images
        upvar ::ibrowser::${w}::nMaxX   nMaxX

        # Deletes all images... if any
        foreach img $images {
        
            if { [ lindex $img 1 ] != "" } { image delete [ lindex $img 1 ] }

        }
        namespace delete ::ibrowser::$w
        rename $w {}

    }
    return ""

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::ibrowserWidgetProc (procedure)                            *
#*                                                                       *
#* DESCRIPTION : Main procedure. This executes all sub-commands for the  *
#*               actual widget.                                          *
#*                                                                       *
#* SYNTAX : ::ibrowser::ibrowserWidgetProc $widget $command $args        *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* RETURN : Depends on each sub-command                                  *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w       : string. Widget name.                                   *
#*      command : string. Sub-command name.                              *
#*      args    : list. Arguments for sub-command.                       *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::ibrowserWidgetProc { w command args } {

    # guess... ;-)
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    # given command exists?
    set command [ ::ibrowser::canonizeIbrowser $w command $command ]

    set result {}

    # execute subcommands
    switch $command {

        add          { set result [ eval ::ibrowser::addImageIbrowser     {$w} $args ] }
        cget         { set result [ eval ::ibrowser::cgetIbrowser         {$w} $args ] }
        configure    { set result [ eval ::ibrowser::configureIbrowser    {$w} $args ] }
        curselection { set result [ eval ::ibrowser::curSelectionIbrowser {$w} $args ] }
        delete       { set result [ eval ::ibrowser::deleteImageIbrowser  {$w} $args ] }
        find         { set result [ eval ::ibrowser::findImageIbrowser    {$w} $args ] }
        lastimage    { set result $lastsel }
        select       { set result [ eval ::ibrowser::selectImageIbrowser  {$w} $args ] }
        size         { set result [ llength $images ] }

    }
    return $result;

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::addImageIbrowser (procedure)                              *
#*                                                                       *
#* DESCRIPTION : Executes the "add" sub-command. This can add a tkimage, *
#*               load an image from disk, make references with tags, and *
#*               gives a title to show with the image.                   *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::ibrowser::addImageIbrowser $w $args ]            *
#*          <widget> add                                                 *
#*                       ?-image <tkimage>?                              *
#*                       ?-file <filename> -format <fileformat>?         *
#*                       ?-tags <tags>?                                  *
#*                       ?-title <title>?                                *
#*                                                                       *
#* RETURN : 0/1 Error/Success                                            *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::addImageIbrowser { w args } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    set ret -1
    if { [ llength $args ] % 2 == 0 } {

        # an associative array is easier...
        array set opt $args

        # look at options...
        set inds [ array names opt ]
        foreach ind $inds {

            if { $ind != "-image"  && \
                 $ind != "-file"   && \
                 $ind != "-format" && \
                 $ind != "-tags"   && \
                 $ind != "-title"     \
            } { error "unknown add option \"$ind\"" }

        }

        # set up title...
        if { [ lsearch -exact $inds "-title" ] != -1 } {
        set txt $opt(-title)
        } else { set txt "NO_TITLE" }

        # extract real image...
        set img_erase 0
        if { [ lsearch -exact $inds "-image" ] != -1 } {
        set img $opt(-image)
        } elseif { [ lsearch -exact $inds "-file" ] != -1 && \
                   [ lsearch -exact $inds "-format" ] != -1  \
        } {

            set img [ image create photo -file $opt(-file) -format $opt(-format) ]
            set img_erase 1

        } else { error "no image to add" }

        # get tags...
        if { [ lsearch -exact $inds "-tags" ] != -1 } {
        set tags $opt(-tags)
        } else { error "\"-tags\" options not found" }

        # subsample real image and generate thumbnail...
        set dimX [ image width $img ]
        set dimY [ image height $img ]
        set rx [ expr $dimX / $options(-thumbwidth) ]
        set ry [ expr $dimY / $options(-thumbheight) ]
        set thumb [ image create photo     \
            -width $options(-thumbwidth)   \
            -height $options(-thumbheight) \
        ]
        if { $rx == 0 || $ry == 0 } {
        $thumb copy $img
        } else { $thumb copy $img -subsample $rx $ry }

        # add nImg to canvas...
        set pos [ llength $images ]
        set px  [ expr ( ( $pos % $nMaxX ) * ( $options(-gap) + $options(-thumbwidth) ) ) + \
                         $options(-gap)
        ]
        set py  [ expr ( ( $pos / $nMaxX ) * ( $options(-gap) + $options(-thumbheight) ) ) + \
                         $options(-gap)
        ]

        set id [ $widgets(canvas) create image $px $py \
            -image $thumb \
            -anchor nw \
            -tags "image"
        ]
        $widgets(canvas) create text $px $py \
            -text $txt \
            -fill $options(-fontcolor) \
            -anchor nw \
            -tags "text_$id text"

        # add image data...
        if { $options(-cache) == 0 } {

            lappend images [ list $id {} $tags ]
            if { $img_erase == 1 } { image delete $img }

        } else { lappend images [ list $id $img $tags ] }
        repos $w

    } else { error "\"add\" command with incorrect number of arguments \"$args\"" }

    return "1"

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::cgetIbrowser (procedure)                                  *
#*                                                                       *
#* DESCRIPTION : Executes the "cget" sub-command. Returns information    *
#*               about certain widget option.                            *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::ibrowser::cgetIbrowser $w $args ]                *
#*          <widget> cget ?-<option>?                                    *
#*                                                                       *
#* RETURN : Option value.                                                *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::cgetIbrowser { w args } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    set ret {}
    if { [ llength $args ] == 1 } {

        set opt [ ::ibrowser::canonizeIbrowser $w option $args ]
        set ret $options($opt)

    } else { error "\"cget\" command only accepts one argument" }
    return $ret

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::curSelectionIbrowser (procedure)                          *
#*                                                                       *
#* DESCRIPTION : Executes the "curselection" sub-command. Returns a list *
#*               with all selected thumbnails. This list has the format: *
#*                [ <id tkimage <tags>> ].                               *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::ibrowser::curSelectionIbrowser $w $args ]        *
#*          <widget> curselection                                        *
#*                                                                       *
#* RETURN : Selected images.                                             *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::curSelectionIbrowser { w args } {

    # again, guess!
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    # Get range. To manage this range, the widget uses two rectangles to
    # show the selection, the range is defined as the first rentangle
    # (frect) to the last (lrect).
    set f_id [ $widgets(canvas) find withtag "frect" ]
    set l_id [ $widgets(canvas) find withtag "lrect" ]
    if { $f_id != "" } {
        set f_id [ lindex [ $widgets(canvas) gettags $f_id ] 1 ]
    } else { set f_id [ llength $images ] }
    if { $l_id != "" } {
        set l_id [ lindex [ $widgets(canvas) gettags $l_id ] 1 ]
    } else { set l_id -1 }
    set f $f_id
    set l $l_id
    set i 0
    foreach img $images {

        if { [ lindex $img 0 ] == $f } { set f $i }
        if { [ lindex $img 0 ] == $l } { set l $i }
        incr i

    }

    # f > l? avoid it!
    set f [ expr ( $f == [ llength $images ] )? $l: $f ]
    set l [ expr ( $l == -1 )? $f: $l ]
    set min [ expr ( $f < $l )? $f: $l ]
    set max [ expr ( $f >= $l )? $f: $l ]

    # set return variable...
    set ret [ list ]
    for { set i $min } { $i <= $max && $i >= 0 } { incr i } {

        lappend ret [ lindex $images $i ]
        
    }

    # end
    return $ret

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::deleteImageIbrowser (procedure)                           *
#*                                                                       *
#* DESCRIPTION : Executes the "delete" sub-command. Starting from a tag, *
#*               delete one thumbnail.                                   *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::ibrowser::deleteImageIbrowser $w $args ]         *
#*          <widget> delete -tags <tags>                                 *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::deleteImageIbrowser { w args } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    if { [ llength $args ] % 2 == 0 } {

        # an associative array is easier...
        array set opt $args

        # look at options...
        set inds [ array names opt ]
        foreach ind $inds {

            if { $ind != "-tags" } { error "unknown delete option \"$ind\"" }

        }

        # select images to delete
        set del [ list ]
        if { [ lsearch -exact $inds "-tags" ] != -1 } {

            set tags $opt(-tags)
            foreach img $images {

                set itags [ lindex $img 2 ]
                set c 0
                foreach t $itags { if { [ lsearch -exact $tags $t ] != -1 } { incr c } }
                if { $c == [ llength $tags ] } { lappend del $img }

            }

        } else { error "\"delete\" subcommand without \"-tags\" option" }

        # real delete process
        $widgets(canvas) delete "frect"
        $widgets(canvas) delete "lrect"
        foreach d $del {

            # delete from canvas...
            set ind [ lindex $d 0 ]
            $widgets(canvas) delete $ind
            $widgets(canvas) delete "text_$ind"

            # from memory...
            if { [ lindex $d 1 ] != "" } { image delete [ lindex $d 1 ] }

            # from list...
            set i [ lsearch -exact $images $d ]
            set images [ lreplace $images $i $i ]

        }

        # Re-arrange widget
        ::ibrowser::repos $w

    } else { error "wrong number of arguments in \"delete\" command" }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::findImageIbrowser (procedure)                             *
#*                                                                       *
#* DESCRIPTION : Executes the "find" sub-command. Starting from a tag,   *
#*               searchs for thumbnails. Returns a list with images.     *
#*                [ <id tkimage <tags>> ].                               *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::ibrowser::findImageIbrowser $w $args ]           *
#*          <widget> find -tags <tags>                                   *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::findImageIbrowser { w args } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    set ret {}
    set tags [ lindex $args 1 ]
    if { [ llength $tags ] > 0 } {

        foreach img $images {

            set itags [ lindex $img 2 ]

            # if counter c, equals the number of given tags, then
            # actual image is of our interest
            set c 0
            foreach t $itags { if { [ lsearch -exact $tags $t ] != -1 } { incr c } }
            if { $c == [ llength $tags ] } { lappend ret $img }

        }

    }
    return $ret

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::selectImageIbrowser (procedure)                           *
#*                                                                       *
#* DESCRIPTION : Executes the "select" sub-command. Starting from a tag, *
#*               selects the first thumbnail. If the -secondary option   *
#*               is specified, then do a secondary selection             *
#*               (right mouse button).                                   *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::ibrowser::selectImageIbrowser $w $args ]         *
#*          <widget> select -tags <tags> ?-secondary?                    *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::selectImageIbrowser { w args } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    # arguments verif...
    set ind -1
    set l [ llength $args ]
    if { $l == 2 || $l == 3 } {

        set tind [ lsearch -exact $args "-tags" ]
        set prim [ expr ( [ lsearch -exact $args "-secondary" ] != -1 )? 0: 1 ]
        if { $tind != -1 } {

            set tags [ lindex $args [ expr $tind + 1 ] ]
            if { $tags != "" } {

                foreach img $images {

                    set itags [ lindex $img 2 ]

                    # if counter c, equals the number of given tags, then
                    # actual image is of our interest
                    set c 0
                    foreach t $itags { if { [ lsearch -exact $tags $t ] != -1 } { incr c } }
                    if { $c == [ llength $tags ] } {

                        set ind [ lindex $img 0 ]
                        break

                    }

                }
                if { $prim == 1 } { sFImage $w 0 0 $ind } else { sLImage $w 0 0 $ind }

            }

        }

    } else { error "Wrong number of arguments for \"select\" subcommand." }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::sFImage (procedure)                                       *
#*                                                                       *
#* DESCRIPTION : Event callback. Mouse left button action.               *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::sFImage %W %x %y $i                              *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w : string. Widget name.                                         *
#*      x : string. x-coordinate.                                        *
#*      y : string. y-coordinate.                                        *
#*      i : string. (optional) Image index to select.                    *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::sFImage { w x y { i -1 } } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    # is manual selection?
    if { $i == -1 } {

        set ima [ $widgets(canvas) gettags current ]
        set i [ $widgets(canvas) find withtag current ]

    } else { set ima [ $widgets(canvas) gettags $i ] }

    $widgets(canvas) delete "frect"

    # ok, select thumb
    if { [ string compare [ lindex $ima 0 ] "image" ] == 0 } {

        set co [ $widgets(canvas) coords $i ]
        set x [ lindex $co 0 ]
        set y [ lindex $co 1 ]
        set px [ expr $x - $options(-gap) / 2 ]
        set py [ expr $y - $options(-gap) / 2 ]
        set fx [ expr $px + $options(-thumbwidth) + $options(-gap) / 2 ]
        set fy [ expr $py + $options(-thumbheight) + $options(-gap) / 2 ]
        $widgets(canvas) create rect  $px $py $fx $fy \
            -outline $options(-primarycolor) \
            -width 2 \
            -tags "frect $i rects"

        set lastsel [ list -1 "" {} ]
        foreach img $images {

            if { [ lindex $img 0 ] == $i } {

                set lastsel $img
                break

            }

        }

        # fire selection event
        event generate $widgets(this) <<AfterSelectImage>>

    }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::sLImage (procedure)                                       *
#*                                                                       *
#* DESCRIPTION : Event callback. Mouse right button action.              *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::sLImage %W %x %y $i                              *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w : string. Widget name.                                         *
#*      x : string. x-coordinate.                                        *
#*      y : string. y-coordinate.                                        *
#*      i : string. (optional) Image index to select.                    *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::sLImage { w x y { i -1 } } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    if { $options(-multisel) == 1 } {

        # is manual selection?
        if { $i == -1 } {

            set ima [ $widgets(canvas) gettags current ]
            set i [ $widgets(canvas) find withtag current ]

        } else { set ima [ $widgets(canvas) gettags $i ] }

        # ok, select thumb
        $widgets(canvas) delete "lrect"

        if { [ string compare [ lindex $ima 0 ] "image" ] == 0 } {

            set co [ $widgets(canvas) coords $i ]
            set x [ lindex $co 0 ]
            set y [ lindex $co 1 ]
            set px [ expr $x - $options(-gap) / 2 ]
            set py [ expr $y - $options(-gap) / 2 ]
            set fx [ expr $px + $options(-thumbwidth) + $options(-gap) / 2 ]
            set fy [ expr $py + $options(-thumbheight) + $options(-gap) / 2 ]
            $widgets(canvas) create rect  $px $py $fx $fy \
                -outline $options(-secondarycolor) \
                -width 2 \
                -tags "lrect $i rects"

            set lastsel [ list -1 "" {} ]
            foreach img $images {

                if { [ lindex $img 0 ] == $i } {

                    set lastsel $img
                    break

                }

            }

            # fire selection event
            event generate $widgets(this) <<AfterSelectImage>>

        }

    }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::repos (procedure)                                         *
#*                                                                       *
#* DESCRIPTION : Event callback. Repositionate images.                   *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::repos %W                                         *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w : string. Widget name.                                         *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::repos { w } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    set itms [ $widgets(canvas) find withtag "image" ]
    set pos 0
    foreach itm $itms {

        set px [ expr ( ( $pos % $nMaxX ) * ( $options(-gap) + $options(-thumbwidth) ) ) + \
                        $options(-gap) \
        ]
        set py [ expr ( ( $pos / $nMaxX ) * ( $options(-gap) + $options(-thumbheight) ) ) + \
                        $options(-gap) \
        ]
        $widgets(canvas) coords $itm $px $py
        incr pos

    }

    set itms [ $widgets(canvas) find withtag "text" ]
    set pos 0
    foreach itm $itms {

        set px [ expr ( ( $pos % $nMaxX ) * ( $options(-gap) + $options(-thumbwidth) ) ) + \
                        $options(-gap) \
        ]
        set py [ expr ( ( $pos / $nMaxX ) * ( $options(-gap) + $options(-thumbheight) ) ) + \
                        $options(-gap) \
        ]
        $widgets(canvas) coords $itm $px $py
        incr pos

    }

    set rects [ $widgets(canvas) find withtag "rects" ]
    foreach r $rects {

        set info [ $widgets(canvas) gettags $r ]
        set co   [ $widgets(canvas) coords [ lindex $info 1 ] ]
        set x [ lindex $co 0 ]
        set y [ lindex $co 1 ]
        set px [ expr $x - $options(-gap) / 2 ]
        set py [ expr $y - $options(-gap) / 2 ]
        set fx [ expr $px + $options(-thumbwidth) + $options(-gap) / 2 ]
        set fy [ expr $py + $options(-thumbheight) + $options(-gap) / 2 ]
        $widgets(canvas) coords $r $px $py $fx $fy

    }
    ::ibrowser::configureSlider $w

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::resize (procedure)                                        *
#*                                                                       *
#* DESCRIPTION : Event callback. Resize widget.                          *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::resize %W %w %h                                  *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*      width  : string. Widget width.                                   *
#*      height : string. Widget height.                                  *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::resize { w width height } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    if { $options(-width) != $width || $options(-height) != $height } {

        set options(-width)  $width
        set options(-height) $height
        set nMaxX [ expr int( floor ( $options(-width) / ( $options(-gap) + \
                                                           $options(-thumbwidth) \
                                                         ) \
                                    ) \
                            ) \
        ]
        if { $nMaxX == 0 } {

            set options(-width) [ expr $options(-thumbwidth) + $options(-gap) ]
            set nMaxX [ expr int( floor ( $options(-width) / ( $options(-gap) + \
                                                               $options(-thumbwidth) \
                                                             ) \
                                        ) \
                                ) \
            ]
            $widgets(canvas) configure -width $options(-width)

        }
        ::ibrowser::repos $w

    }
    

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::ibrowser::configureSlider (procedure)                               *
#*                                                                       *
#* DESCRIPTION : Puts or erases a slider, if necessary.                  *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::ibrowser::configureSlider $widget                          *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::ibrowser::configureSlider { w } {

    # For namespace access
    upvar ::ibrowser::${w}::options options
    upvar ::ibrowser::${w}::widgets widgets
    upvar ::ibrowser::${w}::currsel currsel
    upvar ::ibrowser::${w}::lastsel lastsel
    upvar ::ibrowser::${w}::images  images
    upvar ::ibrowser::${w}::nMaxX   nMaxX

    set s    [ llength $images ]
    set y    [ expr ceil ( $s.0 / $nMaxX.0 ) ]
    set need [ expr $y * ( $options(-thumbheight) + $options(-gap) ) + $options(-gap) ]
    set rh   $options(-height)

    $widgets(canvas) configure -scrollregion "0 0 $options(-width) $need"
    if { $need >= $rh && ! [ winfo exists $w.scroll ] } {

        set widgets(scroll) [ scrollbar $w.scroll -command "$widgets(canvas) yview" ]
        $widgets(canvas) configure -yscrollcommand "$widgets(scroll) set"
        grid $widgets(canvas)  \
            -in $widgets(this) \
            -row 0             \
            -column 0          \
            -rowspan 1         \
            -columnspan 1      \
            -sticky news
        grid $widgets(scroll) -row 0 -column 1 -rowspan 1 -columnspan 1 -sticky news
        grid rowconfig    $widgets(this) 0 -weight 1 -minsize 0
        grid columnconfig $widgets(this) 0 -weight 1 -minsize 0

    } elseif { $need < $rh && [ winfo exists $w.scroll ] } {
    
        $widgets(canvas) configure -yscrollcommand ""
        destroy $w.scroll
        set widgets(scroll) ""

    }

}

# EOF - ibrowser.tcl
