#*************************************************************************
#*                                  *                                    *
#* NAME     : imagepp.tcl           * PROJECT  : MARACAS                 *
#* AUTHOR   : Leonardo Flrez/Kyron * TYPE     : TCL/TK widget def.      *
#* VERSION  : v1.0                  * CREATION : 06/05/2001              *
#* LANGUAGE : TCL                   * REVISION : 16/07/2001              *
#*                                  *                                    *
#*************************************************************************
#*                                                                       *
#*  Description : This file defines a new TCL/TK widget that allows the  *
#*                user to work with an image, it's a viewer. It has a    *
#*                dinamical scroll bar, so, don't worry about use one.   *
#*                                                                       *
#*  The basic use of this widget is:                                     *
#*                                                                       *
#*      1. Create a new widget: "imagepp <name> <options>"               *
#*         Options given are a list of '-<option> <value>' pairs.        *
#*         Supported options are:                                        *
#*         +-----------------+--------------------------------------+    *
#*         | OPTION          | DESCRIPTION                          |    *
#*         +-----------------+--------------------------------------+    *
#*         |  -background    | Background color.                    |    *
#*         |  -borderwidth   | Border width.                        |    *
#*         |  -height        | Height.                              |    *
#*         +-----------------+--------------------------------------+    *
#*         |  -initialroi    | Initial Region Of Interest.          |    *
#*         |  -relief        | Relief.                              |    *
#*         |  -takefocus     | Take focus?                          |    *
#*         +-----------------+--------------------------------------+    *
#*         |  -width         | Width                                |    *
#*         +-----------------+--------------------------------------+    *
#*                                                                       *
#*      2. Pack this new widget in your hierarchy:                       *
#*         (pack/place/grid) <name> <pack options>                       *
#*                                                                       *
#*      3. Interact with the new widget by using their sub-commands      *
#*         interface. Sub-commands defined are:                          *
#*         add , addtext, cget, clean, configure, delete,                *
#*         find, getprofildata, resetroi, roi, show,                     *
#*         setlinear, setarea, setnone.                                  *
#*                                                                       *
#*      4. Optional: use the <<AfterProfil>> event definition to         *
#*         grab mouse interaction. Mouse events supported are:           *
#*         Button1, Button3                                              *
#*                                                                       *
#*************************************************************************
#*                                                                       *
#*  USED MODULES :                                                       *
#*                     TK >= v8.0                                        *
#*                                                                       *
#*************************************************************************
#*                                                                       *
#* REVISIONS :                                                           *
#* (NOTE: Please, don't let this file became a mess. ;-) )               *
#*                                                                       *
#* +------------+----------------+-------------------------------------+ *
#* | DATE       | AUTHOR         | DESCRIPTION                         | *
#* +------------+----------------+-------------------------------------+ *
#* | 06/05/2001 | Kyron          | Initial implementation.             | *
#* +------------+----------------+-------------------------------------+ *
#* | 16/07/2001 | Kyron          | Documentation & conflicts revision. | *
#* +------------+----------------+-------------------------------------+ *
#*                                                                       *
#*************************************************************************

package require Tk 8.0

#* NAMESPACE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp (namespace)                                                 *
#*                                                                       *
#* DESCRIPTION : Global namespace that contains all imagepp widgets in   *
#*               current interpreter (actual TCL work instance).         *
#*                                                                       *
#* SYNTAX : -NONE-                                                       *
#*                                                                       *
#* RETURN :                                                              *
#*        EXPORTS : proc imagepp { name options }                        *
#*                                                                       *
#* PARAMETERS :                                                          *
#*            Namespace components :                                     *
#*              widgetOptions  : list. List of supported options.        *
#*              widgetCommands : list. List of supported sub-commands.   *
#*                                                                       *
#******************************************************* END DESCRIPTION *
package provide imagepp 1.0
namespace eval ::imagepp {

    # public interface
    namespace export imagepp
    
    # variables
    variable widgetOptions
    variable widgetCommands

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::imagepp (procedure)                                        *
#*                                                                       *
#* DESCRIPTION : Creator of new widgets. Call it in an TK hierarchy      *
#*               creation process.                                       *
#*                                                                       *
#* SYNTAX : imagepp <name> -<option1> <value1> ... -<optionn> <valuen>   *
#*                                                                       *
#* RETURN : New widget name, if success.                                 *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      name : string. Name for the new widget. To use it in a TK widget *
#*                     hierarchy, this name should be ".<f1>.<f2>...<n>" *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::imagepp { name args } {

    # Namespace variables used in this procedure
    upvar ::imagepp::widgetOptions widgetOptions
    
    # If global namespace doesn't exists yet then initialize it
    if { ![ info exists widgetOptions ] } initImagePP

    # Given name exists?. If so, raise an error and finish
    if { [ winfo exists $name ] } {
        error "Widget \"$name\" already exists."
    }

    # Create the new command and return success
    set name [ eval ::imagepp::buildImagePP $name $args ]
    return $name

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::initImagePP (procedure)                                    *
#*                                                                       *
#* DESCRIPTION : Initializes the class manager, i.e., creates the global *
#*               namespace.                                              *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::initImagePP                                       *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      -NONE-                                                           *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::initImagePP { } {

    # Namespace variables used in this procedure
    upvar ::imagepp::widgetOptions  widgetOptions
    upvar ::imagepp::widgetCommands widgetCommands

    # All posible options for the widget
    array set widgetOptions [ list               \
        -background  { background Background }   \
        -borderwidth { borderWidth BorderWidth } \
        -height      { height Height }           \
        -initialroi  { initialROI InitialROI }   \
        -relief      { relief Relief }           \
        -takefocus   { takeFocus TakeFocus }     \
        -width       { width Width }             \
    ]

    # All posible commands for the widget
    set widgetCommands [ list                     \
        add         addtext cget    clean         \
        configure   delete  find    getprofildata \
        resetroi    roi     show    setlinear     \
        setarea     setnone                       \
    ]

    event add <<AfterProfil>> \
        <ButtonRelease-1> \
        <ButtonRelease-3>

    # Default initialization... only if Tk exists
    if { [ lsearch -exact [ package names ] "Tk" ] != -1 } {

        option add *ImagePP.background  #c0c0c0       widgetDefault
        option add *ImagePP.borderWidth 0             widgetDefault
        option add *ImagePP.height      100           widgetDefault
        option add *ImagePP.initialROI  "-1 -1 -1 -1" widgetDefault
        option add *ImagePP.relief      flat          widgetDefault
        option add *ImagePP.takeFocus   0             widgetDefault
        option add *ImagePP.width       100           widgetDefault

    }

    # set global bindings
    ::imagepp::setClassImagePPBindings

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::setClassImagePPBindings (procedure)                        *
#*                                                                       *
#* DESCRIPTION : Default namespace bindings.                             *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::setClassImagePPBindings                           *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      -NONE-                                                           *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::setClassImagePPBindings { } {

    bind ImagePP <Destroy> [ list ::imagepp::imagePPDestroyHandler %W ]

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::buildImagePP (procedure)                                   *
#*                                                                       *
#* DESCRIPTION : This does all of the work necessary to create a basic   *
#*               imagepp widget. Creates a new command (widget) with     *
#*               the given name. Also creates a new namespace as a child *
#*               namespace of ::imagepp.                                 *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : set wname [ ::imagepp::buildImagePP $name $options ]         *
#*                                                                       *
#* RETURN : New widget hierarchy name                                    *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. New widget name.                                  *
#*      args : list. Option/value pairs list.                            *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::buildImagePP { w args } {

    variable widgetOptions
    
    # New namespace...
    namespace eval ::imagepp::$w {

        variable this
        variable options
        variable widgets
        variable localIds {}
        variable hNeed    0
        variable vNeed    0
        variable actROI   {}
        variable lastX    -1
        variable lastY    -1
        variable s_width  0
        variable s_height 0

    }

    # import variables, for programming facilities
    upvar ::imagepp::${w}::widgets widgets
    upvar ::imagepp::${w}::options options

    # definition of TK widgets...
    set widgets(this)   [ frame  $w -class ImagePP \
                                    -takefocus 0   \
                                    -relief flat   \
                                    -borderwidth 0 \
    ]
    set widgets(canvas) [ canvas $w.canvas -takefocus 1 ]
    set widgets(hs) ""
    set widgets(vs) ""
    set widgets(profildata) ""
    set widgets(movingroi) 0
    
    # set all the default values...
    foreach name [ array names widgetOptions ] {

        set optName  [ lindex $widgetOptions($name) 0 ]
        set optClass [ lindex $widgetOptions($name) 1 ]
        set value [ option get $w $optName $optClass ]
        set options($name) $value

    }

    # set user values...
    if { [ llength $args ] > 0 } { array set options $args }

    # move the name to imagepp class' namespace...
    set widgets(frame) ::imagepp::${w}::$w
    rename ::$w $widgets(frame)

    # set canvas options...
    $widgets(canvas) configure -background  $options(-background)
    $widgets(canvas) configure -borderwidth $options(-borderwidth)
    $widgets(canvas) configure -height      $options(-height)
    $widgets(canvas) configure -relief      $options(-relief)
    $widgets(canvas) configure -takefocus   $options(-takefocus)
    $widgets(canvas) configure -width       $options(-width)

    # pack the canvas...
    pack $widgets(canvas) -fill both -expand 1

    # local event stuff...
    bind $widgets(canvas) <Configure> "::imagepp::resize  $widgets(this) %w %h"

    # >>>>>>>>>>>>>>>>>>> HERE, AT LAST, THE NEW COMMAND IS DEFINED <<<<<<<<<<<<<<<<<< #
    proc ::$w { command args } "eval ::imagepp::imagePPWidgetProc $w \$command \$args"
    # >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< #

    # Last configuration stuff
    if { [ catch "::imagepp::configureImagePP $widgets(this) [ array get options ]" \
           error \
    ] } {

        catch { destroy $w }
        error $error

    }

    # have fun ;-)
    return ""

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::configureImagePP (procedure)                               *
#*                                                                       *
#* DESCRIPTION : This does the configuration process, i.e., change of    *
#*               any option.                                             *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::configureImagePP $widget $options ]     *
#*                                                                       *
#* RETURN : All options, if args is empty. If length args == 1 then      *
#*          returns current value. Empty string otherwise.               *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Option/value pairs list.                            *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::configureImagePP { w args } {

    variable widgetOptions

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    # Sends all information to the user...
    if { [ llength $args ] == 0 } {

        set results {}
        foreach opt [ lsort [ array names widgetOptions ] ] {

            if { [ llength $widgetOptions($opt) ] == 1 } {

                set alias $widgetOptions($opt)
                set optName $widgetOptions($alias)
                lappend results [ list $opt $optName ]

            } else {

                set optName     [ lindex $widgetOptions($opt) 0 ]
                set optClass    [ lindex $widgetOptions($opt) 1 ]
                set default     [ option get $w $optName $optClass ]
                lappend results [ list $opt $optName $optClass $default $options($opt) ]

            }

        }
        return $results

    }

    # or single information...
    if { [ llength $args ] == 1 } {

        set opt      [ ::imagepp::canonizeImagePP $w option [ lindex $args 0 ] ]
        set optName  [ lindex $widgetOptions($opt) 0 ]
        set optClass [ lindex $widgetOptions($opt) 1 ]
        set default  [ option get $w $optName $optClass ]
        set results  [ list $opt $optName $optClass $default $options($opt) ]
        return $results

    }
    
    if { [ expr { [ llength $args ] % 2 } ] == 1 } {
    error "some values for \"$args\" are missing"
    }

    # check if all given options exists...
    foreach { name value } $args {

        set name [ ::imagepp::canonizeImagePP $w option $name ]
        set opts($name) $value

    }

    # and set values...
    foreach option [ array names opts ] {

        set newValue $opts($option)
        switch -- $option {

            -initialroi {
            
                if { [ llength $newValue ] == 4 } {
                set options(-initialroi) $newValue
                } else { error "wrong ROI value" }
            
            }
            default  { eval "$widgets(canvas) configure $option $newValue" }

        }
        
    }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::canonizeImagePP (procedure)                                *
#*                                                                       *
#* DESCRIPTION : Takes a option or command and canonizes it. Returns     *
#*               either the canonical form of an option or command, or   *
#*               raises an error if the option or command is unknown or  *
#*               ambiguous.                                              *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : set c [ ::imagepp::canonizeImagePP $w option $args ]         *
#*                                                                       *
#* RETURN : Option or command canonical form                             *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*      object : string. option/command id.                              *
#*      opt    : string. Option/command value.                           *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::canonizeImagePP { w object opt } {

    variable widgetOptions
    variable widgetCommands

    switch $object {
    
        command {

            if { [ lsearch -exact $widgetCommands $opt ] >= 0 } { return $opt }
            set list $widgetCommands
            foreach element $list { set tmp($element) "" }
            set matches [ array names tmp ${opt}* ]

        }
        option {

            if { [ info exists widgetOptions($opt) ] && \
                 [ llength $widgetOptions($opt) ] == 2 \
            } { return $opt }
            set list [ array names widgetOptions ]
            set matches [ array names widgetOptions ${opt}* ]

        }
    
    }
    if { [ llength $matches ] == 0 } {
    error "unknown $object \"$opt\"; must be one of $list"
    } elseif { [ llength $matches ] == 1 } {

        set opt [ lindex $matches 0 ]

        switch $object {

            option {

                set opt [ lindex $matches 0 ]
                if { [ llength $widgetOptions($opt) ] == 1 } { set opt $widgetOptions($opt) }

            }

        }
        return $opt

    } else { error "ambiguous $object \"$opt\"; must be one of $list" }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::imagePPDestroyHandler (procedure)                          *
#*                                                                       *
#* DESCRIPTION : Handles the destroy event.                              *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::imagePPDestroyHandler $w                          *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::imagePPDestroyHandler { w } {

    if { [ string compare [ winfo class $w ] "ImagePP" ] == 0 } {
 
        # For namespace access
        upvar ::imagepp::${w}::options  options
        upvar ::imagepp::${w}::widgets  widgets
        upvar ::imagepp::${w}::localIds localIds
        upvar ::imagepp::${w}::hNeed    hNeed
        upvar ::imagepp::${w}::vNeed    vNeed
        upvar ::imagepp::${w}::actROI   actROI
        upvar ::imagepp::${w}::lastX    lastX
        upvar ::imagepp::${w}::lastY    lastY


        namespace delete ::imagepp::$w
        rename $w {}
 
    }
    return ""

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::imagePPWidgetProc (procedure)                              *
#*                                                                       *
#* DESCRIPTION : Main procedure. This executes all sub-commands for the  *
#*               actual widget.                                          *
#*                                                                       *
#* SYNTAX : ::imagepp::imagePPWidgetProc $widget $command $args          *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* RETURN : Depends on each sub-command                                  *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w       : string. Widget name.                                   *
#*      command : string. Sub-command name.                              *
#*      args    : list. Arguments for sub-command.                       *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::imagePPWidgetProc { w command args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    # given command exists?
    set command [ ::imagepp::canonizeImagePP $w command $command ]

    set result {}

    # execute subcommands
    switch $command {

        add           { set result [ eval ::imagepp::addImagePP           {$w} $args ] }
        addtext       { set result [ eval ::imagepp::addTextImagePP       {$w} $args ] }
        cget          { set result [ eval ::imagepp::cgetImagePP          {$w} $args ] }
        clean         { set result [ eval ::imagepp::cleanImagePP         {$w} $args ] }
        configure     { set result [ eval ::imagepp::configureImagePP     {$w} $args ] }
        delete        { set result [ eval ::imagepp::deleteImagePP        {$w} $args ] }
        find          { set result [ eval ::imagepp::findImagePP          {$w} $args ] }
        getprofildata { set result [ eval ::imagepp::getprofildataImagePP {$w} $args ] }
        resetroi      { set result [ eval ::imagepp::resetROIImagePP      {$w} $args ] }
        roi           { set result [ eval ::imagepp::ROIImagePP           {$w} $args ] }
        setroi        { set result [ eval ::imagepp::setROIImagePP        {$w} $args ] }
        show          { set result [ eval ::imagepp::showImagePP          {$w} $args ] }
        setlinear     { set result [ eval ::imagepp::setlinearImagePP     {$w} $args ] }
        setarea       { set result [ eval ::imagepp::setareaImagePP       {$w} $args ] }
        setnone       { set result [ eval ::imagepp::setnoneImagePP       {$w} $args ] }

    }
    return $result
 
}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::addImagePP (procedure)                                     *
#*                                                                       *
#* DESCRIPTION : Executes the "add" sub-command. This can add a tkimage, *
#*               load an image from disk, make references with id's.     *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::addImagePP $w $args ]                   *
#*          <widget> add                                                 *
#*                       ?-image <tkimage>?                              *
#*                       ?-file <filename> -format <fileformat>?         *
#*                       ?-id <id>?                                      *
#*                                                                       *
#* RETURN : Empty string on error.                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::addImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY
    upvar ::imagepp::${w}::s_width  s_width
    upvar ::imagepp::${w}::s_height s_height

    # arguments parsing...
    if { [ llength $args ] == 4 } {

        array set opc $args
        set l_opc [ array names opc ]
        if { [ lsearch -exact $l_opc "-image" ] != -1 && \
             [ lsearch -exact $l_opc "-id" ] != -1       \
        } {

            set new_img $opc(-image)
            set new_id $opc(-id)

        } else { error "error in \"add\" command arguments" }

    } elseif { [ llength $args ] == 6 } {

        array set opc $args
        set l_opc [ array names opc ]
        if { [ lsearch -exact $l_opc "-file" ] != -1 &&   \
             [ lsearch -exact $l_opc "-format" ] != -1 && \
             [ lsearch -exact $l_opc "-id" ] != -1        \
        } {

            set new_img [ image create photo -file $opc(-file) -format $opc(-format) ]
            set new_id $opc(-id)

        } else { error "error in \"add\" command arguments" }

    } else { error "error in \"add\" command arguments" }

    # given id already exists?
    array set ids $localIds
    set l_localIds [ array names ids ]
    if { [ lsearch -exact $l_localIds $new_id ] == -1 } {

        # ok, do it
        set s_width  [ image width  $new_img ]
        set s_height [ image height $new_img ]
        set canvas_id [ \
            $widgets(canvas) create image 0 0 \
            -image $new_img \
            -anchor nw \
            -tags "id_$new_id"
        ]
        lappend localIds $new_id
        lappend localIds $canvas_id
        set hNeed [ expr ( $s_width  > $hNeed )? $s_width : $hNeed ]
        set vNeed [ expr ( $s_height > $vNeed )? $s_height: $vNeed ]
        $widgets(canvas) configure -scrollregion "0 0 $hNeed $vNeed"
        foreach e $actROI { $widgets(canvas) raise $e }

    } else {
    
        set item [ $widgets(canvas) find withtag "id_$new_id" ]
        $widgets(canvas) itemconfigure $item -image $new_img
        $widgets(canvas) raise $item
        foreach e $actROI { $widgets(canvas) raise $e }

    }
    return ""

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::addTextImagePP (procedure)                                 *
#*                                                                       *
#* DESCRIPTION : Executes the "addtext" sub-command.                     *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::addTextImagePP $w $args ]               *
#*          <widget> addtext <text> <x> <y>                              *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::addTextImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    catch { $widgets(canvas) delete "textErase" }
    $widgets(canvas) create text     \
        [ lindex $args 1 ]           \
        [ lindex $args 2 ]           \
        -text [ lindex $args 0 ]     \
        -tags "textErase"            \
        -fill yellow                 \
        -justify left                \
        -anchor nw                   \
        -font { Helvetica -12 bold }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::cgetImagePP (procedure)                                    *
#*                                                                       *
#* DESCRIPTION : Executes the "cget" sub-command. Returns information    *
#*               about certain widget option.                            *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::cgetImagePP $w $args ]                  *
#*          <widget> cget ?-<option>?                                    *
#*                                                                       *
#* RETURN : Option value.                                                *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::cgetImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY
    
    set ret {}
    if { [ llength $args ] == 1 } {

        set opt [ ::imagepp::canonizeImagePP $w option $args ]
        set ret $options($opt)

    } else { error "\"cget\" command only accepts one argument" }
    return $ret

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::cleanImagePP (procedure)                                   *
#*                                                                       *
#* DESCRIPTION : Executes the "clean" sub-command.                       *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::cleanImagePP $w $args ]                 *
#*          <widget> clean                                               *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::cleanImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    catch { $widgets(canvas) delete $widgets(actual_line) }
    catch { $widgets(canvas) delete "textErase" }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::deleteImagePP (procedure)                                  *
#*                                                                       *
#* DESCRIPTION : Executes the "delete" sub-command.                      *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::deleteImagePP $w $args ]                *
#*          <widget> delete -id <id/all>                                 *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::deleteImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    set ret -1

    # arguments parsing...
    if { [ llength $args ] == 2 } {

        if { [ string compare [ lindex $args 0 ] "-id" ] == 0 } {

            if { [ string compare [ lindex $args 1 ] "all" ] == 0 } {

                set all_items [ $widgets(canvas) find withtag all ]
                foreach item $all_items { $widgets(canvas) delete $item }

            } else {
            }

        } else { error "wrong argument for \"delete\" command" }

    } else { error "wrong number of arguments for \"delete\" command" }
    return ""

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::findImagePP (procedure)                                    *
#*                                                                       *
#* DESCRIPTION : Executes the "find" sub-command.                        *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::findImagePP $w $args ]                  *
#*          <widget> delete -id <id>                                     *
#*                                                                       *
#* RETURN : Index.                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::findImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    set ret -1

    # arguments parsing...
    if { [ llength $args ] == 2 } {

        if { [ string compare [ lindex $args 0 ] "-id" ] == 0 } {

            array set ids $localIds
            set l_localIds [ array names ids ]
            set ret [ lsearch -exact $l_localIds [ lindex $args 1 ] ]

        } else { error "wrong argument for \"find\" command" }

    } else { error "wrong number of arguments for \"find\" command" }
    return $ret

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::getprofildataImagePP (procedure)                           *
#*                                                                       *
#* DESCRIPTION : Executes the "find" sub-command.                        *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::getprofildataImagePP $w $args ]         *
#*          <widget> getprofildata -id <id>                              *
#*                                                                       *
#* RETURN : Limits for a profil measure.                                 *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::getprofildataImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    return $widgets(profildata)

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::resetROIImagePP (procedure)                                *
#*                                                                       *
#* DESCRIPTION : Executes the "resetroi" sub-command.                    *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::resetROIImagePP $w $args ]              *
#*          <widget> resetroi                                            *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::resetROIImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    # erase
    foreach e $actROI { $widgets(canvas) delete $e }

    # coordinates
    set ux [ lindex $options(-initialroi) 0 ]
    set uy [ lindex $options(-initialroi) 1 ]
    set bx [ lindex $options(-initialroi) 2 ]
    set by [ lindex $options(-initialroi) 3 ]

    # four rectangles...
    set actROI [ list ]

    # 1) ux, uy
    lappend actROI [ $widgets(canvas) create rectangle \
        [ expr $ux - 6 ] [ expr $uy - 6 ]      \
        [ expr $ux + 0 ] [ expr $uy + 0 ]      \
        -fill #ffff00                          \
        -tags "rect_fill"                      \
    ]

    # 2) bx, uy
    lappend actROI [ $widgets(canvas) create rectangle \
        [ expr $bx - 0 ] [ expr $uy - 6 ]      \
        [ expr $bx + 6 ] [ expr $uy + 0 ]      \
        -fill #ffff00                          \
        -tags "rect_fill"                      \
    ]

    # 3) bx, by
    lappend actROI [ $widgets(canvas) create rectangle \
        [ expr $bx - 0 ] [ expr $by - 0 ]      \
        [ expr $bx + 6 ] [ expr $by + 6 ]      \
        -fill #ffff00                          \
        -tags "rect_fill"                      \
    ]

    # 4) ux, by
    lappend actROI [ $widgets(canvas) create rectangle \
        [ expr $ux - 6 ] [ expr $by - 0 ]      \
        [ expr $ux + 0 ] [ expr $by + 6 ]      \
        -fill #ffff00                          \
        -tags "rect_fill"                      \
    ]

    # rectangle for area
    lappend actROI [ $widgets(canvas) create rectangle \
        [ expr $ux ] [ expr $uy ]      \
        [ expr $bx ] [ expr $by ]      \
        -outline #ffff00               \
        -width 2                       \
        -tags "rect_area"              \
    ]

    # item binds
    $widgets(canvas) bind [ lindex $actROI 0 ] <Enter>     "$widgets(canvas) itemconfigure current -fill #ff0000"
    $widgets(canvas) bind [ lindex $actROI 0 ] <Leave>     "$widgets(canvas) itemconfigure current -fill #ffff00"
    $widgets(canvas) bind [ lindex $actROI 0 ] <Button-1>  "::imagepp::startMotion $w %x %y"
    $widgets(canvas) bind [ lindex $actROI 0 ] <B1-Motion> "::imagepp::moveCorner $w %x %y 0"

    $widgets(canvas) bind [ lindex $actROI 1 ] <Enter>     "$widgets(canvas) itemconfigure current -fill #ff0000"
    $widgets(canvas) bind [ lindex $actROI 1 ] <Leave>     "$widgets(canvas) itemconfigure current -fill #ffff00"
    $widgets(canvas) bind [ lindex $actROI 1 ] <Button-1>  "::imagepp::startMotion $w %x %y"
    $widgets(canvas) bind [ lindex $actROI 1 ] <B1-Motion> "::imagepp::moveCorner $w %x %y 1"

    $widgets(canvas) bind [ lindex $actROI 2 ] <Enter>     "$widgets(canvas) itemconfigure current -fill #ff0000"
    $widgets(canvas) bind [ lindex $actROI 2 ] <Leave>     "$widgets(canvas) itemconfigure current -fill #ffff00"
    $widgets(canvas) bind [ lindex $actROI 2 ] <Button-1>  "::imagepp::startMotion $w %x %y"
    $widgets(canvas) bind [ lindex $actROI 2 ] <B1-Motion> "::imagepp::moveCorner $w %x %y 2"

    $widgets(canvas) bind [ lindex $actROI 3 ] <Enter>     "$widgets(canvas) itemconfigure current -fill #ff0000"
    $widgets(canvas) bind [ lindex $actROI 3 ] <Leave>     "$widgets(canvas) itemconfigure current -fill #ffff00"
    $widgets(canvas) bind [ lindex $actROI 3 ] <Button-1>  "::imagepp::startMotion $w %x %y"
    $widgets(canvas) bind [ lindex $actROI 3 ] <B1-Motion> "::imagepp::moveCorner $w %x %y 3"

    $widgets(canvas) bind [ lindex $actROI 4 ] <Enter>     "$widgets(canvas) itemconfigure current -outline #ff0000"
    $widgets(canvas) bind [ lindex $actROI 4 ] <Leave>     "$widgets(canvas) itemconfigure current -outline #ffff00"
    $widgets(canvas) bind [ lindex $actROI 4 ] <Button-1>  "::imagepp::startMotion $w %x %y"
    $widgets(canvas) bind [ lindex $actROI 4 ] <B1-Motion> "::imagepp::moveRect $w %x %y"

    foreach e $actROI { $widgets(canvas) raise $e }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::ROIImagePP (procedure)                                     *
#*                                                                       *
#* DESCRIPTION : Executes the "roi" sub-command.                         *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::ROIImagePP $w $args ]                   *
#*          <widget> roi                                                 *
#*                                                                       *
#* RETURN : Actual Region Of Interest.                                   *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::ROIImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY
    upvar ::imagepp::${w}::s_width  s_width
    upvar ::imagepp::${w}::s_height s_height

    set ux $s_width; set uy $s_height
    set bx -1; set by -1
	set c  [ $widgets(canvas) coords [ lindex $actROI 4 ] ]
    set ux [ expr ( [ lindex $c 0 ] < $ux )? [ lindex $c 0 ]: $ux ]
    set uy [ expr ( [ lindex $c 1 ] < $uy )? [ lindex $c 1 ]: $uy ]
    set bx [ expr ( [ lindex $c 2 ] > $bx )? [ lindex $c 2 ]: $bx ]
    set by [ expr ( [ lindex $c 3 ] > $by )? [ lindex $c 3 ]: $by ]

    set ux [ expr ( $ux < 0 )? 0: $ux ]
    set uy [ expr ( $uy < 0 )? 0: $uy ]
    set bx [ expr ( $bx > $s_width )? $s_width - 1: $bx ]
    set by [ expr ( $by > $s_height )? $s_height - 1: $by ]

    set ret [ list $ux $uy $bx $by ]
    return $ret

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::showImagePP (procedure)                                    *
#*                                                                       *
#* DESCRIPTION : Executes the "show" sub-command.                        *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::showImagePP $w $args ]                  *
#*          <widget> roi                                                 *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::showImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    # arguments parsing...
    if { [ llength $args ] == 2 } {

        if { [ string compare [ lindex $args 0 ] "-id" ] == 0 } {

            array set ids $localIds
            set l_localIds [ array names ids ]
            set local_id [ lindex $args 1 ]
            if { [ lsearch -exact $l_localIds $local_id ] != -1 } {

                $widgets(canvas) raise $ids($local_id)
                foreach e $actROI { $widgets(canvas) raise $e }

            }

        } else { error "wrong argument for \"show\" command" }

    } else { error "wrong number of arguments for \"show\" command" }
    return ""

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::setlinearImagePP (procedure)                               *
#*                                                                       *
#* DESCRIPTION : Executes the "setlinear" sub-command.                   *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::setlinearImagePP $w $args ]             *
#*          <widget> setlinear                                           *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::setlinearImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    catch { $widgets(canvas) delete $widgets(actual_line) }
    catch { $widgets(canvas) delete "textErase" }
    bind $widgets(canvas) <ButtonPress-3>   "::imagepp::startMotion $widgets(this) %x %y 1"
    bind $widgets(canvas) <B3-Motion>       "::imagepp::goMotion $widgets(this) %x %y 1"
    bind $widgets(canvas) <ButtonRelease-3> "::imagepp::finishMotion $widgets(this) %x %y 1"

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::setareaImagePP (procedure)                                 *
#*                                                                       *
#* DESCRIPTION : Executes the "setarea" sub-command.                     *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::setareaImagePP $w $args ]               *
#*          <widget> setarea                                             *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::setareaImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    catch { $widgets(canvas) delete $widgets(actual_line) }
    catch { $widgets(canvas) delete "textErase" }
    bind $widgets(canvas) <ButtonPress-3>   "::imagepp::startMotion $widgets(this) %x %y 2"
    bind $widgets(canvas) <B3-Motion>       "::imagepp::goMotion $widgets(this) %x %y 2"
    bind $widgets(canvas) <ButtonRelease-3> "::imagepp::finishMotion $widgets(this) %x %y 2"

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::setnoneImagePP (procedure)                                 *
#*                                                                       *
#* DESCRIPTION : Executes the "setnone" sub-command.                     *
#*               This is a dummy proc, don't call it in your code.       *
#*               Use your widget definition and the sub-command.         *
#*                                                                       *
#* SYNTAX : set ret [ ::imagepp::setnoneImagePP $w $args ]               *
#*          <widget> setnone                                             *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w    : string. Widget name.                                      *
#*      args : list. Arguments for sub-command.                          *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::setnoneImagePP { w args } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    catch { $widgets(canvas) delete $widgets(actual_line) }
    catch { $widgets(canvas) delete "textErase" }
    bind $widgets(canvas) <ButtonPress-1>   "::imagepp::startMotion $widgets(this) %x %y 3"
    bind $widgets(canvas) <B1-Motion>       "::imagepp::goMotion $widgets(this) %x %y 3"
    bind $widgets(canvas) <ButtonRelease-1> "::imagepp::finishMotion $widgets(this) %x %y 3"

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::resize (procedure)                                         *
#*                                                                       *
#* DESCRIPTION : Event callback. Resize widget.                          *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::resize %W %w %h                                   *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*      width  : string. Widget width.                                   *
#*      height : string. Widget height.                                  *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::resize { w width height } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    if { $options(-width) != $width || $options(-height) != $height } {

        set options(-width)  $width
        set options(-height) $height
        $widgets(canvas) configure -width $options(-width)
        ::imagepp::configureSlider $w

    }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::configureSlider (procedure)                                *
#*                                                                       *
#* DESCRIPTION : Puts or erases a slider, if necessary.                  *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::configureSlider $widget                           *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w      : string. Widget name.                                    *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::configureSlider { w } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    set rh $options(-height)
    set rw $options(-width)

    if { $vNeed >= $rh && ! [ winfo exists $w.vs ] } {

        set widgets(vs) [ scrollbar $w.vs -command "$widgets(canvas) yview" ]
        $widgets(canvas) configure -yscrollcommand "$widgets(vs) set"
        grid $widgets(canvas) \
            -in $widgets(this) \
            -row 0 \
            -column 0 \
            -rowspan 1 \
            -columnspan 1 \
            -sticky news
        grid $widgets(vs) -row 0 -column 1 -rowspan 1 -columnspan 1 -sticky news
        grid rowconfig    $widgets(this) 0 -weight 1 -minsize 0
        grid columnconfig $widgets(this) 0 -weight 1 -minsize 0
    
    } elseif { $vNeed < $rh && [ winfo exists $w.vs ] } {
    
        $widgets(canvas) configure -yscrollcommand ""
        destroy $w.vs
        set widgets(vs) ""
    
    }

    if { $hNeed >= $rw && ! [ winfo exists $w.hs ] } {

        set widgets(hs) [ scrollbar $w.hs -orient horizontal -command "$widgets(canvas) xview" ]
        $widgets(canvas) configure -xscrollcommand "$widgets(hs) set"
        grid $widgets(canvas) \
            -in $widgets(this) \
            -row 0 \
            -column 0 \
            -rowspan 1 \
            -columnspan 1 \
            -sticky news
        grid $widgets(hs) -row 1 -column 0 -rowspan 1 -columnspan 1 -sticky news
        grid rowconfig    $widgets(this) 0 -weight 1 -minsize 0
        grid columnconfig $widgets(this) 0 -weight 1 -minsize 0
    
    } elseif { $hNeed < $rw && [ winfo exists $w.hs ] } {
    
        $widgets(canvas) configure -xscrollcommand ""
        destroy $w.hs
        set widgets(hs) ""
    
    }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::startMotion (procedure)                                    *
#*                                                                       *
#* DESCRIPTION : Event callback. Start mouse motion.                     *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::startMotion %W %x %y $typ                         *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w   : string. Widget name.                                       *
#*      x   : string. x-coordinate.                                      *
#*      y   : string. y-coordinate.                                      *
#*      typ : string. (optional) Motion type.                            *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::startMotion { w x y { typ 0 } } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    set lastX $x
    set lastY $y

    catch { $widgets(canvas) delete $widgets(actual_line) }
    catch { $widgets(canvas) delete "textErase" }
    if { $widgets(movingroi) == 0 } {

        if { $typ == 1 } {

            set widgets(actual_line) [ \
                $widgets(canvas) create line $lastX $lastY $lastX $lastY \
                    -arrow both   \
                    -fill #0000ff \
                    -width 1 \
            ]

        } elseif { $typ == 2 } {

            set widgets(actual_line) [ \
                $widgets(canvas) create rectangle \
                    $lastX $lastY $lastX $lastY   \
                    -outline #0000ff              \
                    -width 1                      \
            ]
    
        }

    }

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::goMotion (procedure)                                       *
#*                                                                       *
#* DESCRIPTION : Event callback. Do mouse motion.                        *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::goMotion %W %x %y $typ                            *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w   : string. Widget name.                                       *
#*      x   : string. x-coordinate.                                      *
#*      y   : string. y-coordinate.                                      *
#*      typ : string. (optional) Motion type.                            *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::goMotion { w x y typ } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    if { $widgets(movingroi) == 0 } {

        if { $typ == 1 || $typ == 2 } {

            $widgets(canvas) coords \
                $widgets(actual_line) \
                $lastX $lastY $x $y

        } elseif { $typ == 3 } {

            set widgets(profildata) [ list 3 $lastX $lastY $x $y ]
            event generate $widgets(this) <<AfterProfil>>

        }

    }
}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::finishMotion (procedure)                                   *
#*                                                                       *
#* DESCRIPTION : Event callback. Finish mouse motion.                    *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::finishMotion %W %x %y $typ                        *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w   : string. Widget name.                                       *
#*      x   : string. x-coordinate.                                      *
#*      y   : string. y-coordinate.                                      *
#*      typ : string. (optional) Motion type.                            *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::finishMotion { w x y typ } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    if { $widgets(movingroi) == 0 } {

        if { $typ == 3 } { incr typ }
        set widgets(profildata) [ list $typ $lastX $lastY $x $y ]

        event generate $widgets(this) <<AfterProfil>>

    }
    set widgets(movingroi) 0

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::moveRect (procedure)                                       *
#*                                                                       *
#* DESCRIPTION : Event callback. Move a rectangle.                       *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::moveRect %W %x %y                                 *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w   : string. Widget name.                                       *
#*      x   : string. x-coordinate.                                      *
#*      y   : string. y-coordinate.                                      *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::moveRect { w x y } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    set widgets(movingroi) 1
    $widgets(canvas) move [ lindex $actROI 4 ] \
        [ expr $x - $lastX ] \
        [ expr $y - $lastY ]
    $widgets(canvas) move [ lindex $actROI 0 ] \
        [ expr $x - $lastX ] \
        [ expr $y - $lastY ]
    $widgets(canvas) move [ lindex $actROI 1 ] \
        [ expr $x - $lastX ] \
        [ expr $y - $lastY ]
    $widgets(canvas) move [ lindex $actROI 2 ] \
        [ expr $x - $lastX ] \
        [ expr $y - $lastY ]
    $widgets(canvas) move [ lindex $actROI 3 ] \
        [ expr $x - $lastX ] \
        [ expr $y - $lastY ]

    set lastX $x
    set lastY $y

}

#* PROCEDURE DESCRIPTION *************************************************
#*                                                                       *
#* ::imagepp::moveCorner (procedure)                                     *
#*                                                                       *
#* DESCRIPTION : Event callback. Move a rectangle corner.                *
#*               This is a dummy proc, don't call it in your code.       *
#*                                                                       *
#* SYNTAX : ::imagepp::moveCorner %W %x %y $i                            *
#*                                                                       *
#* RETURN : -NONE-                                                       *
#*                                                                       *
#* PARAMETERS :                                                          *
#*      w   : string. Widget name.                                       *
#*      x   : string. x-coordinate.                                      *
#*      y   : string. y-coordinate.                                      *
#*      i   : string. Corner index.                                      *
#*                                                                       *
#******************************************************* END DESCRIPTION *
proc ::imagepp::moveCorner { w x y i } {

    # For namespace access
    upvar ::imagepp::${w}::options  options
    upvar ::imagepp::${w}::widgets  widgets
    upvar ::imagepp::${w}::localIds localIds
    upvar ::imagepp::${w}::hNeed    hNeed
    upvar ::imagepp::${w}::vNeed    vNeed
    upvar ::imagepp::${w}::actROI   actROI
    upvar ::imagepp::${w}::lastX    lastX
    upvar ::imagepp::${w}::lastY    lastY

    set widgets(movingroi) 1

    $widgets(canvas) move [ lindex $actROI $i ] \
        [ expr $x - $lastX ] \
        [ expr $y - $lastY ]

    if { $i == 0 } {

        $widgets(canvas) move [ lindex $actROI 1 ] \
            0 [ expr $y - $lastY ]

        $widgets(canvas) move [ lindex $actROI 3 ] \
            [ expr $x - $lastX ] 0

    } elseif { $i == 1 } {

        $widgets(canvas) move [ lindex $actROI 0 ] \
            0 [ expr $y - $lastY ]

        $widgets(canvas) move [ lindex $actROI 2 ] \
            [ expr $x - $lastX ] 0

    } elseif { $i == 2 } {

        $widgets(canvas) move [ lindex $actROI 3 ] \
            0 [ expr $y - $lastY ]

        $widgets(canvas) move [ lindex $actROI 1 ] \
            [ expr $x - $lastX ] 0

    } elseif { $i == 3 } {

        $widgets(canvas) move [ lindex $actROI 2 ] \
            0 [ expr $y - $lastY ]

        $widgets(canvas) move [ lindex $actROI 0 ] \
            [ expr $x - $lastX ] 0

    }

    # set area
    set c0 [ $widgets(canvas) coords [ lindex $actROI 0 ] ]
    set ux [ lindex $c0 0 ]; set uy [ lindex $c0 1 ]
    set bx [ lindex $c0 2 ]; set by [ lindex $c0 3 ]

    set c0 [ $widgets(canvas) coords [ lindex $actROI 1 ] ]
    set ux [ expr ( [ lindex $c0 0 ] < $ux )? [ lindex $c0 0 ]: $ux ]
    set uy [ expr ( [ lindex $c0 1 ] < $uy )? [ lindex $c0 1 ]: $uy ]
    set bx [ expr ( [ lindex $c0 2 ] > $bx )? [ lindex $c0 2 ]: $bx ]
    set by [ expr ( [ lindex $c0 3 ] > $by )? [ lindex $c0 3 ]: $by ]

    set c0 [ $widgets(canvas) coords [ lindex $actROI 2 ] ]
    set ux [ expr ( [ lindex $c0 0 ] < $ux )? [ lindex $c0 0 ]: $ux ]
    set uy [ expr ( [ lindex $c0 1 ] < $uy )? [ lindex $c0 1 ]: $uy ]
    set bx [ expr ( [ lindex $c0 2 ] > $bx )? [ lindex $c0 2 ]: $bx ]
    set by [ expr ( [ lindex $c0 3 ] > $by )? [ lindex $c0 3 ]: $by ]

    set c0 [ $widgets(canvas) coords [ lindex $actROI 3 ] ]
    set ux [ expr ( [ lindex $c0 0 ] < $ux )? [ lindex $c0 0 ]: $ux ]
    set uy [ expr ( [ lindex $c0 1 ] < $uy )? [ lindex $c0 1 ]: $uy ]
    set bx [ expr ( [ lindex $c0 2 ] > $bx )? [ lindex $c0 2 ]: $bx ]
    set by [ expr ( [ lindex $c0 3 ] > $by )? [ lindex $c0 3 ]: $by ]

    $widgets(canvas) coords [ lindex $actROI 4 ] \
        [ expr $ux + 6 ] \
        [ expr $uy + 6 ] \
        [ expr $bx - 6 ] \
        [ expr $by - 6 ]

    set lastX $x
    set lastY $y

}

# EOF - imagepp.tcl
