# ev_userzoom.tcl
# ====================================================================================================
# User interface events for a vtkTkRenderWidget widget. Based on TkInteractor.tcl
# ====================================================================================================
# Author  : Leonardo Florez-Valencia (lflorez@creatis.insa-lyon.fr)
# Created : 02/03/2001
# ====================================================================================================
# Log :
#	02/03/2001 ==> Initial implementation.
# ====================================================================================================

set gv_rendererFound 0;
set gv_deltaStep 0.001;
set gv_alfa 0.1;
set gv_actualPoint -1;

# -- PROCEDURE "bindBasicEvents { widget }" -----------------------------------------------------------
# -- Binds basic behavior the graphical VTK-widget.
# --
# -- IN  : widget = vtkTkRenderWidget
# -- OUT :
proc bindBasicEvents { widget } {

	bind $widget <w>      { evb_wireframe %W }
	bind $widget <s>      { evb_surface %W }
	bind $widget <Enter>  { evb_enter %W %x %y }
	bind $widget <Expose> { evb_expose %W }

}; # proc bindBasicEvents { widget } {
# ----------------------------------------------------------------------------------------------------

# -- PROCEDURE "bindUserEvents { widget }" -----------------------------------------------------------
# -- Binds mouse and some keyboard interaction to the graphical VTK-widget.
# --
# -- Mouse motion :
# --		    Left button + motion	 = Rotates
# --		    Middle button + motion or 
# --		    Shift + Left button + motion = Side moves camera
# --		    Right button + motion	 = Moves
# --
# -- Keyboard :
# --		UpArrow 	    = Forward
# --		DownArrow	    = Backward
# --		LeftArrow	    = Rotate left
# --		RightArrow	    = Rotate right
# --		Shift + UpArrow     = Rotate up
# --		Shift + DownArrow   = Rotate down
# --		Shift + LeftArrow   = Side move left
# --		Shift + RightArrow  = Side move right
# --		Control + UpArrow   = Side move up
# --		Control + DownArrow = Side move down
# --
# -- IN  : widget = vtkTkRenderWidget
# -- OUT :
proc bindUserEvents { widget { rot 1 } } {

	bind $widget <Any-ButtonPress>	 { ev_startMotion %W %x %y}
	bind $widget <Any-ButtonRelease> { ev_endMotion %W %x %y}
	if { $rot == 1 } { bind $widget <B1-Motion>	 { evu_rotate %W %x %y } }
	bind $widget <Shift-B1-Motion>	 { evu_sideMove %W %x %y }
	bind $widget <B2-Motion>	 { evu_sideMove %W %x %y }
	bind $widget <B3-Motion>	 { evu_move %W %x %y }
	bind $widget <Up>		 { evu_forward %W }
	bind $widget <Down>		 { evu_backward %W }
	bind $widget <Left>		 { evu_rotateLeft %W }
	bind $widget <Right>		 { evu_rotateRight %W }
	bind $widget <Shift-Up> 	 { evu_rotateUp %W }
	bind $widget <Shift-Down>	 { evu_rotateDown %W }
	bind $widget <Shift-Left>	 { evu_sideMoveLeft %W }
	bind $widget <Shift-Right>	 { evu_sideMoveRight %W }
	bind $widget <Control-Up>	 { evu_sideMoveUp %W }
	bind $widget <Control-Down>	 { evu_sideMoveDown %W }

}; # proc bindUserEvents { widget } {
# ----------------------------------------------------------------------------------------------------

# -- PROCEDURE "bindZoomEvents { widget }" -----------------------------------------------------------
# -- Binds mouse and some keyboard interaction to the camera zoom on a VTK-widget.
# --
# -- Mouse motion :
# --		    Left button + motion	 = Rotates
# --		    Middle button + motion or 
# --		    Shift + Left button + motion = Pan
# --		    Right button + motion	 = Zoom
# --
# -- IN  : widget = vtkTkRenderWidget
# -- OUT :
proc bindZoomEvents { widget { rot 1 } } {

	bind $widget <Any-ButtonPress>	 { ev_startMotion %W %x %y}
	bind $widget <Any-ButtonRelease> { ev_endMotion %W %x %y}
    bind $widget <B1-Motion>
	if { $rot == 1 } { bind $widget <B1-Motion>	 { evz_rotate %W %x %y } }
	bind $widget <B2-Motion>	 { evz_pan %W %x %y }
	bind $widget <B3-Motion>	 { evz_zoom %W %x %y }

}; # proc bindZoomEvents { widget } {
# ----------------------------------------------------------------------------------------------------

proc updateRenderer { widget x y } {

	global gv_currentCamera gv_currentLight;
	global gv_currentRenderWindow gv_currentRenderer;
	global gv_rendererFound gv_initX gv_initY gv_moving;
	global gv_windowCenterX gv_windowCenterY;

	# Get the renderer window dimensions
	set WindowX [ lindex [ $widget configure -width ] 4 ];
	set WindowY [ lindex [ $widget configure -height ] 4 ];

	# Find which renderer event has occurred in
	set gv_currentRenderWindow [ $widget GetRenderWindow ];
	set renderers [ $gv_currentRenderWindow GetRenderers ];
	set numRenderers [ $renderers GetNumberOfItems ];

	$renderers InitTraversal;
	set gv_rendererFound 0;

	for { set i 0 } { $i < $numRenderers } { incr i } {

		set gv_currentRenderer [ $renderers GetNextItem ];
		set vx [ expr double( $x ) / $WindowX ];
		set vy [expr ( $WindowY - double( $y ) ) / $WindowY ];
		set viewport [ $gv_currentRenderer GetViewport ];
		set vpxmin [ lindex $viewport 0 ];
		set vpymin [ lindex $viewport 1 ];
		set vpxmax [ lindex $viewport 2 ];
		set vpymax [ lindex $viewport 3 ];

		if { $vx >= $vpxmin && $vx <= $vpxmax && \
		     $vy >= $vpymin && $vy <= $vpymax} {
			set gv_rendererFound 1;
			set gv_windowCenterX [ expr double( $WindowX ) * ( ( $vpxmax - $vpxmin ) / 2.0 \
					       + $vpxmin ) ];
			set gv_windowCenterY [ expr double( $WindowY ) * ( ( $vpymax - $vpymin ) / 2.0 \
					       + $vpymin ) ];
			break;

		}; # fi

	}; # rof

	set gv_currentCamera [ $gv_currentRenderer GetActiveCamera ];
	set lights [ $gv_currentRenderer GetLights ];
	$lights InitTraversal;
	set gv_currentLight [ $lights GetNextItem ];
   
	set gv_initX $x;
	set gv_initY $y;
	set gv_moving 0;

}; # proc updateRenderer { widget x y } {
# ----------------------------------------------------------------------------------------------------

proc render { } {

	global gv_currentCamera gv_currentLight gv_currentRenderWindow;

	eval $gv_currentLight SetPosition [ $gv_currentCamera GetPosition ];
	eval $gv_currentLight SetFocalPoint [ $gv_currentCamera GetFocalPoint ];
	$gv_currentCamera SetClippingRange 0.1 1000;
	$gv_currentRenderWindow Render;

}; # proc render { } {
# ----------------------------------------------------------------------------------------------------

# -- PROCEDURE "setFollowPath { path } ---------------------------------------------------------------
# -- Sets a point set to interpolate a path for the camera.
#
# -- IN : A set of points that defines the path
# -- OUT : The user has been enabled to walk over the path, the normal view goes over
# --	   the pendant direction.
proc setFollowPath { path } {

	global gv_pointSet;
	global gv_actualPoint;
	
	set gv_pointSet $path;
	set gv_actualPoint 0;

}; # proc setFollowPath { path } {
# ----------------------------------------------------------------------------------------------------

proc unsetFollowPath { } {

	set gv_actualPoint -1;

}; # proc unsetFollowPath { } {
# ----------------------------------------------------------------------------------------------------

proc ev_startMotion { widget x y } {

	global gv_currentRenderWindow;
	global gv_rendererFound;

	updateRenderer $widget $x $y;
	if { ! $gv_rendererFound } { return };
	$gv_currentRenderWindow SetDesiredUpdateRate 1.0;

}; # proc evz_startMotion { widget x y } {
# ----------------------------------------------------------------------------------------------------

proc ev_endMotion { widget x y } {

	global gv_currentRenderWindow;
	global gv_rendererFound;

	if { ! $gv_rendererFound } { return };
	$gv_currentRenderWindow SetDesiredUpdateRate 0.01;
	render;

}; # proc evz_endMotion { widget x y } {
# ----------------------------------------------------------------------------------------------------

proc evb_wireframe { widget } {

	global gv_currentRenderer;

	set actors [ $gv_currentRenderer GetActors ];
	$actors InitTraversal;
	set actor [ $actors GetNextItem ];

	while { $actor != "" } {

		[ $actor GetProperty ] SetRepresentationToWireframe;
		set actor [ $actors GetNextItem ];

	}; # fwhile

	render;

}; # proc evb_wireframe { widget } {
# ----------------------------------------------------------------------------------------------------

proc evb_surface { widget } {

	global gv_currentRenderer;

	set actors [ $gv_currentRenderer GetActors ];
	$actors InitTraversal;
	set actor [ $actors GetNextItem ];

	while { $actor != "" } {

		[ $actor GetProperty ] SetRepresentationToSurface;
		set actor [ $actors GetNextItem ];

	}; #fwhile

	render;

}; # proc evb_surface { widget } {
# ----------------------------------------------------------------------------------------------------

proc evb_enter { widget x y } {

	#focus $widget;
	#updateRenderer $widget $x $y;

}; # proc evb_enter { widget x y } {
# ----------------------------------------------------------------------------------------------------

proc evb_expose { widget } {

   update;
   [ $widget GetRenderWindow ] Render;

}; # proc evb_expose { widget } {
# ----------------------------------------------------------------------------------------------------

proc evu_rotate { widget x y } {

	global gv_currentCamera;
	global gv_initX gv_initY gv_deltaStep;
	global gv_rendererFound;

	if { ! $gv_rendererFound } { return };
	
	set dx [ expr $gv_initX - $x ];
	set dy [ expr $gv_initY - $y ];
	set ori [ $gv_currentCamera GetOrientation ];
	set ori [ list 0 0 0 ];
	set yaw [ lindex $ori 0 ];
	set pitch [ lindex $ori 1 ];

	# Yaw rotation
	set yaw [ expr $yaw + fmod( $dx * $gv_deltaStep, 6.283185 ) ];
	$gv_currentCamera Yaw $yaw;
	
	# Pitch rotation	
	#set pitch [ expr $pitch + fmod( $dy * $gv_deltaStep, 6.283185 ) ];
	#$gv_currentCamera Pitch $pitch;
	#puts "$pitch";

	render;

}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_sideMove { widget x y } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_sideMove { widget x y } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_move { widget x y } {

	global gv_currentCamera;
	global gv_initX gv_initY;
	global gv_deltaStep;
	global gv_rendererFound;

	if { ! $gv_rendererFound } { return };
	
	set dx [ expr $gv_initY - $y ];
	set pos [ $gv_currentCamera GetPosition ];
	set foc [ $gv_currentCamera GetFocalPoint ];
	set nor [ $gv_currentCamera GetViewPlaneNormal ];

	# Calculates the new position and focal point
	set px [ expr [ lindex $pos 0 ] + ( [ lindex $nor 0 ] * $dx * 0.1 ) ];
	set py [ expr [ lindex $pos 1 ] + ( [ lindex $nor 1 ] * $dx * 0.1 ) ];
	set pz [ expr [ lindex $pos 2 ] + ( [ lindex $nor 2 ] * $dx * 0.1 ) ];
	set fx [ expr [ lindex $foc 0 ] + ( [ lindex $nor 0 ] * $dx * 0.1 ) ];
	set fy [ expr [ lindex $foc 1 ] + ( [ lindex $nor 1 ] * $dx * 0.1 ) ];
	set fz [ expr [ lindex $foc 2 ] + ( [ lindex $nor 2 ] * $dx * 0.1 ) ];
#	set px [ expr [ lindex $pos 0 ] + ( [ lindex $nor 0 ] * $dx * $gv_deltaStep ) ];
#	set py [ expr [ lindex $pos 1 ] + ( [ lindex $nor 1 ] * $dx * $gv_deltaStep ) ];
#	set pz [ expr [ lindex $pos 2 ] + ( [ lindex $nor 2 ] * $dx * $gv_deltaStep ) ];
#	set fx [ expr [ lindex $foc 0 ] + ( [ lindex $nor 0 ] * $dx * $gv_deltaStep ) ];
#	set fy [ expr [ lindex $foc 1 ] + ( [ lindex $nor 1 ] * $dx * $gv_deltaStep ) ];
#	set fz [ expr [ lindex $foc 2 ] + ( [ lindex $nor 2 ] * $dx * $gv_deltaStep ) ];
	
	$gv_currentCamera SetPosition $px $py $pz;
	$gv_currentCamera SetFocalPoint $fx $fy $fz;
	
	render;

}; # proc evu_move { widget x y } {
# ----------------------------------------------------------------------------------------------------

proc evu_forward { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_backward { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_rotateLeft { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_rotateRight { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_rotateUp { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_rotateDown { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_sideMoveLeft { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_sideMoveRight { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_sideMoveUp { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evu_sideMoveDown { widget } {
}; # 
# ----------------------------------------------------------------------------------------------------

proc evz_rotate { widget x y } {

	global gv_currentCamera;
	global gv_initX gv_initY;
	global gv_rendererFound;

	if { ! $gv_rendererFound } { return };

	$gv_currentCamera Azimuth [ expr ( $gv_initX - $x ) ];
	$gv_currentCamera Elevation [ expr ( $y - $gv_initY ) ];
	$gv_currentCamera OrthogonalizeViewUp;

	set gv_initX $x;
	set gv_initY $y;

	render;

}; # proc evz_rotate { widget x y } {
# ----------------------------------------------------------------------------------------------------

proc evz_pan { widget x y } {

	global gv_currentRenderer gv_currentCamera;
	global gv_windowCenterX gv_windowCenterY gv_initX gv_initY;
	global gv_rendererFound;

	if { ! $gv_rendererFound } { return };

	set FPoint [ $gv_currentCamera GetFocalPoint ];
	set FPoint0 [ lindex $FPoint 0 ];
	set FPoint1 [ lindex $FPoint 1 ];
	set FPoint2 [ lindex $FPoint 2 ];

	set PPoint [ $gv_currentCamera GetPosition ];
	set PPoint0 [ lindex $PPoint 0 ];
	set PPoint1 [ lindex $PPoint 1 ];
	set PPoint2 [ lindex $PPoint 2 ];

	$gv_currentRenderer SetWorldPoint $FPoint0 $FPoint1 $FPoint2 1.0;
	$gv_currentRenderer WorldToDisplay;
	set DPoint [ $gv_currentRenderer GetDisplayPoint ];
	set focalDepth [ lindex $DPoint 2 ];

	set APoint0 [ expr $gv_windowCenterX + ( $x - $gv_initX ) ];
	set APoint1 [ expr $gv_windowCenterY - ( $y - $gv_initY ) ];

	$gv_currentRenderer SetDisplayPoint $APoint0 $APoint1 $focalDepth;
	$gv_currentRenderer DisplayToWorld;
	set RPoint [ $gv_currentRenderer GetWorldPoint ];
	set RPoint0 [ lindex $RPoint 0 ];
	set RPoint1 [ lindex $RPoint 1 ];
	set RPoint2 [ lindex $RPoint 2 ];
	set RPoint3 [ lindex $RPoint 3 ];

	if { $RPoint3 != 0.0 } {

		set RPoint0 [ expr $RPoint0 / $RPoint3 ];
		set RPoint1 [ expr $RPoint1 / $RPoint3 ];
		set RPoint2 [ expr $RPoint2 / $RPoint3 ];

	}; # fi

	$gv_currentCamera SetFocalPoint \
					[ expr ( $FPoint0 - $RPoint0 ) / 2.0 + $FPoint0 ] \
					[ expr ( $FPoint1 - $RPoint1 ) / 2.0 + $FPoint1 ] \
					[ expr ( $FPoint2 - $RPoint2 ) / 2.0 + $FPoint2 ];

	$gv_currentCamera SetPosition \
				      [ expr ( $FPoint0 - $RPoint0 ) / 2.0 + $PPoint0 ] \
				      [ expr ( $FPoint1 - $RPoint1 ) / 2.0 + $PPoint1 ] \
				      [ expr ( $FPoint2 - $RPoint2 ) / 2.0 + $PPoint2 ];

	set gv_initX $x;
	set gv_initY $y;

	render;

}; # proc evz_pan { widget x y } {
# ----------------------------------------------------------------------------------------------------

proc evz_zoom { widget x y } {

	global gv_currentCamera;
	global gv_initX gv_initY;
	global gv_rendererFound;

	if { ! $gv_rendererFound } { return };

	set zoomFactor [ expr pow( 1.02, ( 0.5 * ( $gv_initY - $y ) ) ) ];

	if { [ $gv_currentCamera GetParallelProjection ] } {

		set parallelScale [ expr [ $gv_currentCamera GetParallelScale ] * $zoomFactor ];
		$gv_currentCamera SetParallelScale $parallelScale;

	} else {

		$gv_currentCamera SetClippingRange \
						   [ expr 0.1 / $zoomFactor ] \
						   [ expr 1000 / $zoomFactor ];
		$gv_currentCamera Dolly $zoomFactor;

	}; # fi

	set gv_initX $x;
	set gv_initY $y;

	render;

}; # proc evz_zoom { widget x y } {
# ----------------------------------------------------------------------------------------------------

# EOF - ev_userzoom.tcl
