package require BLT
package require combobox
catch { namespace import blt::* }
catch { namespace import combobox::* }

namespace eval u_surf {

    # public interface
    namespace export        \
        create              \
        positionate         \
        forget
    
    # variables
    variable widgets

    variable reference_actor
    variable actual_actor
    variable surf_data
    variable bounds
    variable intRange
    variable axis_index -1
    variable indexes

}

proc u_surf::reset { parent } {

    # For programming facilities
    upvar u_surf::widgets widgets

    set widgets(base)            "$parent"
    set widgets(work3D)          "$parent\.surf"
    set widgets(controls)        "$parent\.surfControls"
    set widgets(frButtons)       "$parent\.surfControls.frButtons"
    set widgets(btnExtract)      "$parent\.surfControls.frButtons.btnExtract"
    set widgets(btnErase)        "$parent\.surfControls.frButtons.btnErase"
    set widgets(btnAdd)          "$parent\.surfControls.frButtons.btnAdd"
    set widgets(btnContinue)     "$parent\.surfControls.frButtons.btnContinue"
    set widgets(frOther)         "$parent\.surfControls.frOther"
    set widgets(frSurface)       "$parent\.surfControls.frOther.frSurface"
    set widgets(btnOpaque)       "$parent\.surfControls.frOther.frSurface.btnOpaque"
    set widgets(btnTransparent)  "$parent\.surfControls.frOther.frSurface.btnTransparent"
    set widgets(btnEdgePoints)   "$parent\.surfControls.frOther.frSurface.btnEdgePoints"
    set widgets(frSurfaceValues) "$parent\.surfControls.frOther.frSurfaceValues"
    set widgets(sclIso)          "$parent\.surfControls.frOther.frSurfaceValues.sclIso"
    set widgets(sclOpacity)      "$parent\.surfControls.frOther.frSurfaceValues.sclOpacity"
    set widgets(btnColor)        "$parent\.surfControls.frOther.btnColor"
    set widgets(lbl001)          "$parent\.surfControls.frOther.lbl001"
    set widgets(lbl002)          "$parent\.surfControls.frOther.lbl002"
    set widgets(lbl003)          "$parent\.surfControls.frOther.lbl003"
    set widgets(frChooseAxis)    "$parent\.surfControls.frOther.frChooseAxis"
    set widgets(lblChooseAxis)   "$parent\.surfControls.frOther.frChooseAxis.01"
    set widgets(edtChooseAxis)   "$parent\.surfControls.frOther.frChooseAxis.02"
    set widgets(btnSaveAsVtk)   "$parent\.surfControls.frOther.btnSaveAsVtk"

}

proc u_surf::create { parent } {

    # For programming facilities
    upvar u_surf::widgets widgets

    u_surf::reset $parent

    # frames
    frame $widgets(controls)        -borderwidth 2 -height 75 -relief groove -width 125 
    frame $widgets(frButtons)       -borderwidth 2 -height 75 -relief groove -width 125 
    frame $widgets(frOther)         -borderwidth 2 -height 75 -relief groove -width 125 
    frame $widgets(frSurface)       -borderwidth 2 -height 75 -relief groove -width 125 
    frame $widgets(frSurfaceValues) -borderwidth 2 -height 75 -relief groove -width 125 
    frame $widgets(frChooseAxis)    -height 30 -width 30 

    # buttons
    button $widgets(btnExtract)  -relief flat -image $global_window::images(extract)  -command "u_surf::extract_axis"
    button $widgets(btnErase)    -relief flat -image $global_window::images(del_axis) -command "u_surf::remove_axis"
    button $widgets(btnAdd)      -relief flat -image $global_window::images(add)      -command "u_surf::add_axis"
    button $widgets(btnContinue) -relief flat -image $global_window::images(continue) -command "u_surf::continue_axis"
    button $widgets(btnColor)    -command "u_surf::change_color"
    button $widgets(btnSaveAsVtk) -text "Save as VTK..." -command "u_surf::saveasvtk"

    bind $widgets(btnExtract)  <Enter> { if { [ string compare [ %W cget -state ] "normal" ] == 0 } { %W configure -relief raised } }
    bind $widgets(btnErase)    <Enter> { if { [ string compare [ %W cget -state ] "normal" ] == 0 } { %W configure -relief raised } }
    bind $widgets(btnAdd)      <Enter> { if { [ string compare [ %W cget -state ] "normal" ] == 0 } { %W configure -relief raised } }
    bind $widgets(btnContinue) <Enter> { if { [ string compare [ %W cget -state ] "normal" ] == 0 } { %W configure -relief raised } }

    bind $widgets(btnExtract)  <Leave> { %W configure -relief flat }
    bind $widgets(btnErase)    <Leave> { %W configure -relief flat }
    bind $widgets(btnAdd)      <Leave> { %W configure -relief flat }
    bind $widgets(btnContinue) <Leave> { %W configure -relief flat }

    # radiobuttons
    radiobutton $widgets(btnOpaque)      -text $string_table::str_opaque      -variable surf_type  -value 1 -command { u_surf::set_IsoActor $surf_type }
    radiobutton $widgets(btnTransparent) -text $string_table::str_transparent -variable surf_type  -value 2 -command { u_surf::set_IsoActor $surf_type }
    radiobutton $widgets(btnEdgePoints)  -text $string_table::str_edge_points -variable surf_type  -value 3 -command { u_surf::set_IsoActor $surf_type }

    # scales
    scale $widgets(sclIso)     -label $string_table::str_isovalue -orient horizontal 
    scale $widgets(sclOpacity) -label $string_table::str_opacity -orient horizontal -from 0.0 -to 1.0 -resolution 0.01

    # labels
    label $widgets(lbl001)        -borderwidth 0 -text $string_table::str_surface 
    label $widgets(lbl002)        -borderwidth 0 -text $string_table::str_surface_values
    label $widgets(lbl003)        -borderwidth 0 -text $string_table::str_surface_color 
    label $widgets(lblChooseAxis) -anchor w -borderwidth 0 -text $string_table::str_axis 

    # comboboxes
    combobox $widgets(edtChooseAxis) \
        -editable true      \
        -command  "u_surf::select_axis"

}

proc u_surf::positionate { } {

    # For programming facilities
    upvar u_surf::widgets widgets

    set global_window::show_params 0

    pack  $widgets(work3D)          -anchor nw -expand 1 -fill both -side left
    pack  $widgets(controls)        -anchor nw -expand 0 -fill both -side left 
    pack  $widgets(frButtons)       -anchor center -expand 0 -fill none -side top 
    pack  $widgets(btnExtract)      -anchor nw -expand 0 -fill none -side left 
    pack  $widgets(btnErase)        -anchor nw -expand 0 -fill none -side left 
    pack  $widgets(btnAdd)          -anchor nw -expand 0 -fill none -side left 
    pack  $widgets(btnContinue)     -anchor nw -expand 0 -fill none -side left 
    pack  $widgets(frOther)         -anchor center -expand 1 -fill both -side top 
    place $widgets(frSurface)       -x 10 -y 55 -width 165 -height 80 -anchor nw -bordermode ignore 
    place $widgets(btnOpaque)       -x 15 -y 10 -anchor nw -bordermode ignore 
    place $widgets(btnTransparent)  -x 15 -y 30 -anchor nw -bordermode ignore 
    place $widgets(btnEdgePoints)   -x 15 -y 50 -anchor nw -bordermode ignore 
    place $widgets(frSurfaceValues) -x 10 -y 155 -width 165 -height 140 -anchor nw -bordermode ignore 
    place $widgets(sclIso)          -x 5 -y 10 -width 153 -height 59 -anchor nw -bordermode ignore 
    place $widgets(sclOpacity)      -x 5 -y 70 -width 153 -height 59 -anchor nw -bordermode ignore 
    place $widgets(btnColor)        -x 10 -y 320 -width 165 -height 48 -anchor nw -bordermode ignore 
    place $widgets(lbl001)          -x 15 -y 45 -anchor nw -bordermode ignore 
    place $widgets(lbl002)          -x 15 -y 145 -anchor nw -bordermode ignore 
    place $widgets(lbl003)          -x 10 -y 300 -anchor nw -bordermode ignore 
    place $widgets(frChooseAxis)    -x 5 -y 10 -width 170 -height 25 -anchor nw -bordermode ignore 
    #place $widgets(btnSaveAsVtk)    -x 15 -y 400 -width 150 -height 25 -anchor nw -bordermode ignore 
    pack  $widgets(lblChooseAxis)   -anchor center -expand 0 -fill none -padx 2 -pady 2 -side left 
    pack  $widgets(edtChooseAxis)   -anchor center -expand 1 -fill x -padx 2 -pady 2 -side right 

    if { $u_mpr::stPoint != "" } {

        u_surf::select_initial_2 [ lindex $u_mpr::stPoint 0 ] [ lindex $u_mpr::stPoint 1 ] [ lindex $u_mpr::stPoint 2 ]

    }

}

proc u_surf::forget { } {

    # For programming facilities
    upvar u_surf::widgets widgets

    pack  forget $widgets(edtChooseAxis)
    pack  forget $widgets(lblChooseAxis)
    place forget $widgets(btnSaveAsVtk)
    place forget $widgets(frChooseAxis)
    place forget $widgets(lbl003)
    place forget $widgets(lbl002)
    place forget $widgets(lbl001)
    place forget $widgets(btnColor)
    place forget $widgets(sclOpacity)
    place forget $widgets(sclIso)
    place forget $widgets(frSurfaceValues)
    place forget $widgets(btnEdgePoints)
    place forget $widgets(btnTransparent)
    place forget $widgets(btnOpaque)
    place forget $widgets(frSurface)
    pack  forget $widgets(frOther)
    pack  forget $widgets(btnContinue)
    pack  forget $widgets(btnAdd)
    pack  forget $widgets(btnErase)
    pack  forget $widgets(btnExtract)
    pack  forget $widgets(frButtons)
    pack  forget $widgets(controls)
    pack  forget $widgets(work3D)

}

proc u_surf::set_data { } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::axis_index      axis_index

    set axis_index 0

    $widgets(edtChooseAxis) list delete 0 end
    $widgets(edtChooseAxis) delete 0 end

    # render widget
    destroy $widgets(work3D)
    vtkTkRenderWidget $widgets(work3D) -width 30 -height 30

    # TODO: change
    bindBasicEvents $widgets(work3D)

    bind $widgets(work3D) <Double-Button-1> "u_surf::select_initial %x %y"
    bind $widgets(work3D) <Double-Button-3> "u_surf::select_axis_point %x %y"

    catch { renderer_$widgets(work3D) Delete }
    vtkRenderer renderer_$widgets(work3D)
    set render [ $widgets(work3D) GetRenderWindow ]
    renderer_$widgets(work3D) SetBackground 1 1 1
    #renderer_$widgets(work3D) SetBackground 0.023 0.239 0.909
    #renderer_$widgets(work3D) SetBackground 0 0 0.627
    $render AddRenderer renderer_$widgets(work3D)
    $render SetSize 1 1

    set range    [ GetImageRange_dll ]
    set resample [ GetVTKVolume_dll ]
    set bounds   [ $resample GetBounds ]
    $widgets(btnColor) configure -background #faebd6
    set r 0.9803
    set g 0.9215
    set b 0.8392

    catch { renderer_$widgets(work3D) RemoveActor outlineActor_$widgets(work3D) }
    catch { renderer_$widgets(work3D) RemoveActor isoActor_$widgets(work3D) }
    catch { renderer_$widgets(work3D) RemoveActor edgeActor_$widgets(work3D) }
    catch { renderer_$widgets(work3D) RemoveVolume opaqueVol_$widgets(work3D) }
    catch { outlineActor_$widgets(work3D)  Delete }
    catch { outlineMapper_$widgets(work3D) Delete }
    catch { outlineSource_$widgets(work3D) Delete }
    catch { opaqueVol_$widgets(work3D)     Delete }
    catch { isoCast_$widgets(work3D)       Delete }
    catch { isoFunc_$widgets(work3D)       Delete }
    catch { opaqueProp_$widgets(work3D)    Delete }
    catch { colorFunc_$widgets(work3D)     Delete }
    catch { constFunc_$widgets(work3D)     Delete }
    catch { edgeActor_$widgets(work3D)     Delete }
    catch { edgeMapper_$widgets(work3D)    Delete }
    catch { edgePoints_$widgets(work3D)    Delete }
    catch { isoActor_$widgets(work3D)      Delete }
    catch { isoMapper_$widgets(work3D)     Delete }
    catch { cubes_$widgets(work3D)         Delete }
    catch { struct_$widgets(work3D)        Delete }

    # 3D outline
    vtkCubeSource outlineSource_$widgets(work3D)
        outlineSource_$widgets(work3D) SetXLength [ expr [ lindex $bounds 1 ] - [ lindex $bounds 0 ] ]
        outlineSource_$widgets(work3D) SetYLength [ expr [ lindex $bounds 3 ] - [ lindex $bounds 2 ] ]
        outlineSource_$widgets(work3D) SetZLength [ expr [ lindex $bounds 5 ] - [ lindex $bounds 4 ] ]
        outlineSource_$widgets(work3D) SetCenter  \
            [ expr ( [ lindex $bounds 0 ] + [ lindex $bounds 1 ] ) / 2 ] \
            [ expr ( [ lindex $bounds 2 ] + [ lindex $bounds 3 ] ) / 2 ] \
            [ expr ( [ lindex $bounds 4 ] + [ lindex $bounds 5 ] ) / 2 ]

    vtkPolyDataMapper outlineMapper_$widgets(work3D)
        outlineMapper_$widgets(work3D) SetInput [ outlineSource_$widgets(work3D) GetOutput ]
        outlineMapper_$widgets(work3D) ImmediateModeRenderingOn

    vtkActor outlineActor_$widgets(work3D)
        outlineActor_$widgets(work3D) SetMapper outlineMapper_$widgets(work3D)
        [ outlineActor_$widgets(work3D) GetProperty ] SetRepresentationToWireframe
        [ outlineActor_$widgets(work3D) GetProperty ] SetColor    0.7 0.0 0.9
        [ outlineActor_$widgets(work3D) GetProperty ] SetAmbient  1
        [ outlineActor_$widgets(work3D) GetProperty ] SetDiffuse  0
        [ outlineActor_$widgets(work3D) GetProperty ] SetSpecular 0

    renderer_$widgets(work3D) AddActor outlineActor_$widgets(work3D)

    # Surface
    vtkImageToStructuredPoints struct_$widgets(work3D)
        struct_$widgets(work3D) SetInput $resample
        struct_$widgets(work3D) Update

    vtkMarchingCubes cubes_$widgets(work3D)
        cubes_$widgets(work3D) SetInput [ struct_$widgets(work3D) GetOutput ]
        cubes_$widgets(work3D) SetValue 0 [ expr [ lindex $range 1 ] / 4 ]

    vtkPolyDataMapper isoMapper_$widgets(work3D)
        isoMapper_$widgets(work3D) SetInput [ cubes_$widgets(work3D) GetOutput ]
        isoMapper_$widgets(work3D) ScalarVisibilityOff
        isoMapper_$widgets(work3D) ImmediateModeRenderingOn

    vtkActor isoActor_$widgets(work3D)
        isoActor_$widgets(work3D) SetMapper isoMapper_$widgets(work3D)
        eval [ isoActor_$widgets(work3D) GetProperty ] SetColor $r $g $b
        eval [ isoActor_$widgets(work3D) GetProperty ] SetOpacity 0.5

    # Edge points
    vtkEdgePoints edgePoints_$widgets(work3D)
        edgePoints_$widgets(work3D) SetInput $resample
        edgePoints_$widgets(work3D) SetValue [ expr [ lindex $range 1 ] / 4 ]

    vtkDataSetMapper edgeMapper_$widgets(work3D)
        edgeMapper_$widgets(work3D) SetInput [ edgePoints_$widgets(work3D) GetOutput ]
        edgeMapper_$widgets(work3D) ScalarVisibilityOff
        edgeMapper_$widgets(work3D) ImmediateModeRenderingOn
    
    vtkActor edgeActor_$widgets(work3D)
        edgeActor_$widgets(work3D) SetMapper edgeMapper_$widgets(work3D)
        eval [ edgeActor_$widgets(work3D) GetProperty ] SetColor $r $g $b

    # Opaque
    vtkPiecewiseFunction constFunc_$widgets(work3D)
        constFunc_$widgets(work3D) AddPoint  0                  1.0
        constFunc_$widgets(work3D) AddPoint  [ lindex $range 1 ]  1.0

    vtkColorTransferFunction colorFunc_$widgets(work3D)
        colorFunc_$widgets(work3D) AddRGBPoint 0.0                 1.0 1.0 1.0
        colorFunc_$widgets(work3D) AddRGBPoint [ lindex $range 1 ] 1.0 1.0 1.0

    vtkVolumeProperty opaqueProp_$widgets(work3D)
        opaqueProp_$widgets(work3D) SetColor colorFunc_$widgets(work3D)
        opaqueProp_$widgets(work3D) SetScalarOpacity constFunc_$widgets(work3D)
        opaqueProp_$widgets(work3D) ShadeOn
        opaqueProp_$widgets(work3D) SetInterpolationTypeToLinear

    vtkVolumeRayCastIsosurfaceFunction isoFunc_$widgets(work3D)
        isoFunc_$widgets(work3D) SetIsoValue [ expr [ lindex $range 1 ] / 4 ]

    vtkVolumeRayCastMapper isoCast_$widgets(work3D)
        isoCast_$widgets(work3D) SetInput $resample
        isoCast_$widgets(work3D) SetVolumeRayCastFunction isoFunc_$widgets(work3D)

    vtkVolume opaqueVol_$widgets(work3D)
        opaqueVol_$widgets(work3D) SetMapper isoCast_$widgets(work3D)
        opaqueVol_$widgets(work3D) SetProperty opaqueProp_$widgets(work3D)

    $widgets(sclIso) configure \
        -from [ lindex $range 0 ] \
        -to   [ lindex $range 1 ] \
        -resolution 1.0

    $widgets(sclOpacity) configure \
        -from 0.0 \
        -to   1.0 \
        -resolution 0.01

    bind $widgets(sclIso) <Any-ButtonRelease>
    bind $widgets(sclIso) <Any-ButtonRelease> "u_surf::change_intensity 0"
    
    bind $widgets(sclOpacity) <Any-ButtonRelease>
    bind $widgets(sclOpacity) <Any-ButtonRelease> "u_surf::change_opacity 0"

    u_surf::change_intensity 1
    u_surf::change_opacity 1

    $widgets(btnOpaque)      deselect
    $widgets(btnTransparent) deselect
    $widgets(btnEdgePoints)  deselect
    $widgets(btnTransparent) invoke

    [ $widgets(work3D) GetRenderWindow ] Render
    u_3D::controls 0

}

proc u_surf::set_IsoActor { typ } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::actual_actor   actual_actor

    catch { renderer_$widgets(work3D) RemoveActor isoActor_$widgets(work3D) }
    catch { renderer_$widgets(work3D) RemoveActor edgeActor_$widgets(work3D) }
    catch { renderer_$widgets(work3D) RemoveVolume opaqueVol_$widgets(work3D) }

    set actual_actor $typ

    if { $actual_actor == 1 } {

        renderer_$widgets(work3D) AddVolume opaqueVol_$widgets(work3D)
    
    } elseif { $actual_actor == 2 } {

        renderer_$widgets(work3D) AddActor isoActor_$widgets(work3D)

    } elseif { $actual_actor == 3 } {

        renderer_$widgets(work3D) AddActor edgeActor_$widgets(work3D)

    }
    [ $widgets(work3D) GetRenderWindow ] Render

}

proc u_surf::change_color { } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::reference_actor reference_actor
    upvar u_surf::actual_actor    actual_actor
    upvar u_surf::surf_data       surf_data
    upvar u_surf::bounds          bounds
    upvar u_surf::intRange        intRange

    set col [ $widgets(btnColor) cget -background ]
    set col [ \
     tk_chooseColor \
         -initialcolor $col \
         -parent $widgets(btnColor) \
         -title "Choose a color..."
    ]
    if { $col != "" } { 
        
        $widgets(btnColor) configure -background $col            
        set r [ expr "0x[ string index $col 1 ][ string index $col 2 ]" / 255.0 ]
        set g [ expr "0x[ string index $col 3 ][ string index $col 4 ]" / 255.0 ]
        set b [ expr "0x[ string index $col 5 ][ string index $col 6 ]" / 255.0 ]
        [ isoActor_$widgets(work3D) GetProperty ] SetColor $r $g $b
        [ edgeActor_$widgets(work3D) GetProperty ] SetColor $r $g $b
        
        [ $widgets(work3D) GetRenderWindow ] Render

    }

}

proc u_surf::intersectionSurface { data p n o dir extent distance } {

    # For programming facilities
    upvar u_surf::widgets widgets

    vtkCellLocator locator_tmp_$widgets(work3D)
        locator_tmp_$widgets(work3D) SetDataSet $data
        locator_tmp_$widgets(work3D) Initialize

    set p1x [ expr ( $dir == 0 )? [ lindex $p 0 ] - [ lindex $n 0 ]: [ lindex $p 0 ] + [ lindex $n 0 ] ]
    set p1y [ expr ( $dir == 0 )? [ lindex $p 1 ] - [ lindex $n 1 ]: [ lindex $p 1 ] + [ lindex $n 1 ] ]
    set p1z [ expr ( $dir == 0 )? [ lindex $p 2 ] - [ lindex $n 2 ]: [ lindex $p 2 ] + [ lindex $n 2 ] ]

    set iter 1
    set i 0
    while { $iter == 1 } {
    
        set p2x [ expr ( $dir == 0 )? [ expr $p1x - [ lindex $n 0 ] * $i ]: [ expr $p1x + [ lindex $n 0 ] * $i ] ]
        set p2y [ expr ( $dir == 0 )? [ expr $p1y - [ lindex $n 1 ] * $i ]: [ expr $p1y + [ lindex $n 1 ] * $i ] ]
        set p2z [ expr ( $dir == 0 )? [ expr $p1z - [ lindex $n 2 ] * $i ]: [ expr $p1z + [ lindex $n 2 ] * $i ] ]
        incr i

        set index [ $data FindPoint $p2x $p2y $p2z ]
        set voxx [ lindex [ split [ expr $p2x - [ lindex $o 0 ] ] . ] 0 ]
        set voxy [ lindex [ split [ expr $p2y - [ lindex $o 1 ] ] . ] 0 ]
        set voxz [ lindex [ split [ expr $p2z - [ lindex $o 2 ] ] . ] 0 ]

        if {
            $voxx >= 0 && $voxx <= [ lindex $extent 1 ] - [ lindex $extent 0 ] &&
            $voxy >= 0 && $voxy <= [ lindex $extent 3 ] - [ lindex $extent 2 ] &&
            $voxz >= 0 && $voxz <= [ lindex $extent 5 ] - [ lindex $extent 4 ]
        } {

            locator_tmp_$widgets(work3D) Update
            set res [ IntersectWithLine_dll locator_tmp_$widgets(work3D)  $p1x $p1y $p1z $p2x $p2y $p2z 0.001 ]
            if { [ lindex $res 0 ] == 1 } {

                set ret [ list        \
                    1                 \
                    [ lindex $res 1 ] \
                    [ lindex $res 2 ] \
                    [ lindex $res 3 ] \
                    $i                \
                ]
                set iter 0

            } elseif { $i >= $distance } { set iter 0; set ret [ list 0 ] }

 
        } else { set iter 0; set ret [ list 0 ] }

    }
    catch { locator_tmp_$widgets(work3D) Delete }
    return $ret

}

proc u_surf::select_initial { x y } {

    # For programming facilities
    upvar u_surf::widgets widgets

    busy hold .
    update

    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_1_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_2_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_3_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_4_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_5_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_ROI_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor line_actor_1_$widgets(work3D) }
    catch { renderer_$widgets(work3D) RemoveActor line_actor_2_$widgets(work3D) }
    DeleteSphere sphere_1_$widgets(work3D)
    DeleteSphere sphere_2_$widgets(work3D)
    DeleteSphere sphere_3_$widgets(work3D)
    DeleteSphere sphere_4_$widgets(work3D)
    DeleteSphere sphere_5_$widgets(work3D)
    DeleteSphere sphere_ROI_$widgets(work3D)
    catch { line_actor_1_$widgets(work3D)  Delete }
    catch { line_mapper_1_$widgets(work3D) Delete }
    catch { tube_normal_1_$widgets(work3D) Delete }
    catch { poly_normal_1_$widgets(work3D) Delete }
    catch { line_1_$widgets(work3D)        Delete }
    catch { points_1_$widgets(work3D)      Delete }
    catch { line_actor_2_$widgets(work3D)  Delete }
    catch { line_mapper_2_$widgets(work3D) Delete }
    catch { tube_normal_2_$widgets(work3D) Delete }
    catch { poly_normal_2_$widgets(work3D) Delete }
    catch { line_2_$widgets(work3D)        Delete }
    catch { points_2_$widgets(work3D)      Delete }

    set resample [ GetVTKVolume_dll ]
    set bounds   [ $resample GetBounds ]

    renderer_$widgets(work3D) IsInViewport 0 0
    set y [ expr [ lindex [ $widgets(work3D) configure -height ] 4 ] - $y - 1 ]

    set surfPD [ cubes_$widgets(work3D) GetOutput ]
    $surfPD ComputeBounds

    set pointData    [ $surfPD GetPointData ]
    set normalsSurf  [ $pointData GetNormals ]

    [ $widgets(work3D) GetRenderWindow ] Render
    
    set pickWPPos [ pick_point_local_actor $x $y renderer_$widgets(work3D) isoActor_$widgets(work3D) 0.001 ]

    set xc [ lindex $pickWPPos 0 ]; set yc [ lindex $pickWPPos 1 ]; set zc [ lindex $pickWPPos 2 ]
    
    set indP    [ $surfPD FindPoint $xc $yc $zc ]
    set coordsP [ $surfPD GetPoint $indP ]

    set xc [ lindex $coordsP 0 ]; set yc [ lindex $coordsP 1 ]; set zc [ lindex $coordsP 2 ]
    set x1 $xc; set y1 $yc; set z1 $zc

    renderer_$widgets(work3D) AddActor [ DrawSphere sphere_1_$widgets(work3D) 0.5 $xc $yc $zc 1.0 0.0 0.0 ]

    set normalP [ $normalsSurf GetTuple3 $indP ]
    set xN [ lindex $normalP 0 ]; set yN [ lindex $normalP 1 ]; set zN [ lindex $normalP 2 ]
    set xO [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 0 ] . ] 0 ]
    set yO [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 1 ] . ] 0 ]
    set zO [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 2 ] . ] 0 ]

    set largmax 30

    set resInt [ u_surf::intersectionSurface \
        $surfPD                              \
        $coordsP                             \
        $normalP                             \
        [ list $xO $yO $zO ]                 \
        0                                    \
        $bounds                              \
        $largmax                             \
    ]

    set trouve [ lindex $resInt 0 ]
    set xI     [ lindex $resInt 1 ]
    set yI     [ lindex $resInt 2 ]
    set zI     [ lindex $resInt 3 ]
    set i      [ lindex $resInt 4 ]

    if { $trouve == 0 } {

        tk_messageBox \
            -icon error \
            -message "$string_table::str_please_select_a_different_point" \
            -title "ERROR"

    } else {

        renderer_$widgets(work3D) AddActor [ DrawSphere sphere_2_$widgets(work3D) 0.5 $xI $yI $zI 1.0 0.0 0.0 ]

        vtkPoints points_1_$widgets(work3D)
            points_1_$widgets(work3D) InsertPoint 0 $xI $yI $zI
            points_1_$widgets(work3D) InsertPoint 1 $xc $yc $zc
 
        vtkCellArray line_1_$widgets(work3D)
            line_1_$widgets(work3D) InsertNextCell 2
        for { set j 0 } { $j < 2 } { incr j } { line_1_$widgets(work3D) InsertCellPoint $j }

        vtkPolyData poly_normal_1_$widgets(work3D)
            poly_normal_1_$widgets(work3D) SetPoints points_1_$widgets(work3D)
            poly_normal_1_$widgets(work3D) SetLines line_1_$widgets(work3D)

        vtkTubeFilter tube_normal_1_$widgets(work3D)
            tube_normal_1_$widgets(work3D) SetInput poly_normal_1_$widgets(work3D)
            tube_normal_1_$widgets(work3D) SetRadius 0.25

        vtkPolyDataMapper line_mapper_1_$widgets(work3D)
            line_mapper_1_$widgets(work3D) SetInput [ tube_normal_1_$widgets(work3D) GetOutput ]
            line_mapper_1_$widgets(work3D) ImmediateModeRenderingOn

        vtkActor line_actor_1_$widgets(work3D)
            line_actor_1_$widgets(work3D) SetMapper line_mapper_1_$widgets(work3D)
            [ line_actor_1_$widgets(work3D) GetProperty ] SetColor 1 0 0
            [ line_actor_1_$widgets(work3D) GetProperty ] BackfaceCullingOff

        renderer_$widgets(work3D) AddActor line_actor_1_$widgets(work3D)

        set distPoints $i
        set largVaisseau1 [ DistPoints $xc $yc $zc $xI $yI $zI ]
        set vox_size [ GetActualVoxelSize_dll ]
        set coordVoxelPc  [ VoxelCoord $xc $yc $zc $xO $yO $zO $vox_size $vox_size $vox_size ]
        set xcvol         [ lindex $coordVoxelPc 0 ]
        set ycvol         [ lindex $coordVoxelPc 1 ]
        set zcvol         [ lindex $coordVoxelPc 2 ]
        set coordVoxelPI [ VoxelCoord $xI $yI $zI $xO $yO $zO $vox_size $vox_size $vox_size ]
        set xIvol        [ lindex $coordVoxelPI 0 ]
        set yIvol        [ lindex $coordVoxelPI 1 ]
        set zIvol        [ lindex $coordVoxelPI 2 ]
        set distPointsVol1 [ DistPoints $xcvol $ycvol $zcvol $xIvol $yIvol $zIvol ]

        set xc [ expr ( $xc - ( $xN * ( $largVaisseau1 / 2 ) ) ) ]
        set yc [ expr ( $yc - ( $yN * ( $largVaisseau1 / 2 ) ) ) ]
        set zc [ expr ( $zc - ( $zN * ( $largVaisseau1 / 2 ) ) ) ]

        renderer_$widgets(work3D) AddActor [ DrawSphere sphere_3_$widgets(work3D) 0.5 $xc $yc $zc 1.0 0.0 0.0 ]

        set vectsPerp [ TclPerpendiculars_dll $xN $yN $zN 0 ]
        set xP1 [ lindex $vectsPerp 0 ]; set yP1 [ lindex $vectsPerp 1 ]; set zP1 [ lindex $vectsPerp 2 ];
        set xP2 [ lindex $vectsPerp 3 ]; set yP2 [ lindex $vectsPerp 4 ]; set zP2 [ lindex $vectsPerp 5 ];

        set resIntP11 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP1 $yP1 $zP1 ]              \
            [ list $xO $yO $zO ]                 \
            0                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP11 [ lindex $resIntP11 0 ]
        set xP11      [ lindex $resIntP11 1 ]
        set yP11      [ lindex $resIntP11 2 ]
        set zP11      [ lindex $resIntP11 3 ]
        set i         [ lindex $resIntP11 4 ]

        set resIntP12 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP1 $yP1 $zP1 ]              \
            [ list $xO $yO $zO ]                 \
            1                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP12 [ lindex $resIntP12 0 ]
        set xP12      [ lindex $resIntP12 1 ]
        set yP12      [ lindex $resIntP12 2 ]
        set zP12      [ lindex $resIntP12 3 ]
        set i         [ lindex $resIntP12 4 ]

        if { ( $trouveP11 == 1 ) && ( $trouveP12 == 1 ) } {
            set largP1 [ DistPoints $xP11 $yP11 $zP11 $xP12 $yP12 $zP12 ]
        } else { set largP1 10000 }

        set resIntP21 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP2 $yP2 $zP2 ]              \
            [ list $xO $yO $zO ]                 \
            0                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP21 [ lindex $resIntP21 0 ]
        set xP21      [ lindex $resIntP21 1 ]
        set yP21      [ lindex $resIntP21 2 ]
        set zP21      [ lindex $resIntP21 3 ]
        set i         [ lindex $resIntP21 4 ]

        set resIntP22 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP2 $yP2 $zP2 ]              \
            [ list $xO $yO $zO ]                 \
            1                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP22 [ lindex $resIntP22 0 ]
        set xP22      [ lindex $resIntP22 1 ]
        set yP22      [ lindex $resIntP22 2 ]
        set zP22      [ lindex $resIntP22 3 ]
        set i         [ lindex $resIntP22 4 ]

        if { ( $trouveP21 == 1 ) && ( $trouveP22 == 1 ) } {
            set largP2 [ DistPoints $xP21 $yP21 $zP21 $xP22 $yP22 $zP22 ]
        } else { set largP2 10000 }

        if { ( $largP1 == 10000 ) && ( $largP2 == 10000 ) } {
        
            set xP1 0; set yP1 0; set zP1 0
            set xP2 0; set yP2 0; set zP2 0
            set largdef 0

        } elseif { ( $largP1 < $largP2 ) } {
        
            set xP1 $xP11; set yP1 $yP11; set zP1 $zP11
            set xP2 $xP12; set yP2 $yP12; set zP2 $zP12
            set largdef $largP1

        } elseif { ( $largP2 < $largP1 ) } {

            set xP1 $xP21; set yP1 $yP21; set zP1 $zP21
            set xP2 $xP22; set yP2 $yP22; set zP2 $zP22
            set largdef $largP2
        } 
        set largVaisseau2 $largdef

        if { $largVaisseau2 > 0 } {
        
            set largVaisseau [ expr ( $largVaisseau1 + $largVaisseau2 ) / 2 ]
            set vox_size [ GetActualVoxelSize_dll ]
            set coordVoxelP1 [ VoxelCoord $xP1 $yP1 $zP1 $xO $yO $zO $vox_size $vox_size $vox_size ]
            set xP1vol [ lindex $coordVoxelP1 0 ]; set yP1vol [ lindex $coordVoxelP1 1 ]; set zP1vol [ lindex $coordVoxelP1 2 ]

            set coordVoxelP2 [VoxelCoord $xP2 $yP2 $zP2 $xO $yO $zO $vox_size $vox_size $vox_size ]
            set xP2vol [ lindex $coordVoxelP2 0 ]; set yP2vol [ lindex $coordVoxelP2 1 ]; set zP2vol [ lindex $coordVoxelP2 2 ]
            set distPointsVol2 [ DistPoints $xP1vol $yP1vol $zP1vol $xP2vol $yP2vol $zP2vol ]

        } else {
        
            set largVaisseau 0
            set distPointsVol2 0
        }

        if { $largVaisseau2 > 0 } {

            renderer_$widgets(work3D) AddActor [ DrawSphere sphere_4_$widgets(work3D) 0.5 $xP1 $yP1 $zP1 0.0 1.0 0.0 ]
            renderer_$widgets(work3D) AddActor [ DrawSphere sphere_5_$widgets(work3D) 0.5 $xP2 $yP2 $zP2 0.0 1.0 0.0 ]

            vtkPoints points_2_$widgets(work3D)
                points_2_$widgets(work3D) InsertPoint 0 $xP1 $yP1 $zP1
                points_2_$widgets(work3D) InsertPoint 1 $xP2 $yP2 $zP2
 
            vtkCellArray line_2_$widgets(work3D)
                line_2_$widgets(work3D) InsertNextCell 2
            for { set j 0 } { $j < 2 } { incr j } { line_2_$widgets(work3D) InsertCellPoint $j }

            vtkPolyData poly_normal_2_$widgets(work3D)
                poly_normal_2_$widgets(work3D) SetPoints points_2_$widgets(work3D)
                poly_normal_2_$widgets(work3D) SetLines line_2_$widgets(work3D)

            vtkTubeFilter tube_normal_2_$widgets(work3D)
                tube_normal_2_$widgets(work3D) SetInput poly_normal_2_$widgets(work3D)
                tube_normal_2_$widgets(work3D) SetRadius 0.25

            vtkPolyDataMapper line_mapper_2_$widgets(work3D)
                line_mapper_2_$widgets(work3D) SetInput [ tube_normal_2_$widgets(work3D) GetOutput ]
                line_mapper_2_$widgets(work3D) ImmediateModeRenderingOn

            vtkActor line_actor_2_$widgets(work3D)
                line_actor_2_$widgets(work3D) SetMapper line_mapper_2_$widgets(work3D)
                [ line_actor_2_$widgets(work3D) GetProperty ] SetColor 0.0 1.0 0.0
                [ line_actor_2_$widgets(work3D) GetProperty ] BackfaceCullingOff

            renderer_$widgets(work3D) AddActor line_actor_2_$widgets(work3D)

        }

        if { $distPointsVol2 > 0 } {
            set distPointsVol [ expr ( $distPointsVol1 + $distPointsVol2 ) / 2 ]
        } else { set distPointsVol $distPointsVol1 }
        SetROIStep_dll $distPointsVol
        array set arr [ params_dll ]
        set act [ \
            DrawSphere \
                sphere_ROI_$widgets(work3D) \
                [ expr $arr(e_roi_dimension) / 2.0 ] \
                $xc $yc $zc \
                1.0 0.0 0.0 \
        ]
        [ $act GetProperty ] SetOpacity 0.2
        renderer_$widgets(work3D) AddActor $act

    }

    busy release .
    update
    [ $widgets(work3D) GetRenderWindow ] Render

}

proc u_surf::select_initial_2 { xc yc zc } {

    # For programming facilities
    upvar u_surf::widgets widgets

    busy hold .
    update

    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_1_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_2_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_3_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_4_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_5_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_ROI_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor line_actor_1_$widgets(work3D) }
    catch { renderer_$widgets(work3D) RemoveActor line_actor_2_$widgets(work3D) }
    DeleteSphere sphere_1_$widgets(work3D)
    DeleteSphere sphere_2_$widgets(work3D)
    DeleteSphere sphere_3_$widgets(work3D)
    DeleteSphere sphere_4_$widgets(work3D)
    DeleteSphere sphere_5_$widgets(work3D)
    DeleteSphere sphere_ROI_$widgets(work3D)
    catch { line_actor_1_$widgets(work3D)  Delete }
    catch { line_mapper_1_$widgets(work3D) Delete }
    catch { tube_normal_1_$widgets(work3D) Delete }
    catch { poly_normal_1_$widgets(work3D) Delete }
    catch { line_1_$widgets(work3D)        Delete }
    catch { points_1_$widgets(work3D)      Delete }
    catch { line_actor_2_$widgets(work3D)  Delete }
    catch { line_mapper_2_$widgets(work3D) Delete }
    catch { tube_normal_2_$widgets(work3D) Delete }
    catch { poly_normal_2_$widgets(work3D) Delete }
    catch { line_2_$widgets(work3D)        Delete }
    catch { points_2_$widgets(work3D)      Delete }

    set resample [ GetVTKVolume_dll ]
    set bounds   [ $resample GetBounds ]

#    renderer_$widgets(work3D) IsInViewport 0 0
#    set y [ expr [ lindex [ $widgets(work3D) configure -height ] 4 ] - $y - 1 ]
#
    set surfPD [ cubes_$widgets(work3D) GetOutput ]
    $surfPD ComputeBounds
#
    set pointData    [ $surfPD GetPointData ]
    set normalsSurf  [ $pointData GetNormals ]
#
#    [ $widgets(work3D) GetRenderWindow ] Render
#    
#    set pickWPPos [ pick_point_local_actor $x $y renderer_$widgets(work3D) isoActor_$widgets(work3D) 0.001 ]
#
#    set xc [ lindex $pickWPPos 0 ]; set yc [ lindex $pickWPPos 1 ]; set zc [ lindex $pickWPPos 2 ]
#    
    set indP    [ $surfPD FindPoint $xc $yc $zc ]
    set coordsP [ $surfPD GetPoint $indP ]
#
#    set xc [ lindex $coordsP 0 ]; set yc [ lindex $coordsP 1 ]; set zc [ lindex $coordsP 2 ]
    set x1 $xc; set y1 $yc; set z1 $zc

    renderer_$widgets(work3D) AddActor [ DrawSphere sphere_1_$widgets(work3D) 0.5 $xc $yc $zc 1.0 0.0 0.0 ]

    set normalP [ $normalsSurf GetTuple3 $indP ]
    set xN [ lindex $normalP 0 ]; set yN [ lindex $normalP 1 ]; set zN [ lindex $normalP 2 ]
    set xO [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 0 ] . ] 0 ]
    set yO [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 1 ] . ] 0 ]
    set zO [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 2 ] . ] 0 ]

    set largmax 30

    set resInt [ u_surf::intersectionSurface \
        $surfPD                              \
        $coordsP                             \
        $normalP                             \
        [ list $xO $yO $zO ]                 \
        0                                    \
        $bounds                              \
        $largmax                             \
    ]

    set trouve [ lindex $resInt 0 ]
    set xI     [ lindex $resInt 1 ]
    set yI     [ lindex $resInt 2 ]
    set zI     [ lindex $resInt 3 ]
    set i      [ lindex $resInt 4 ]

    if { $trouve == 0 } {

        tk_messageBox \
            -icon error \
            -message "$string_table::str_please_select_a_different_point" \
            -title "ERROR"

    } else {

        renderer_$widgets(work3D) AddActor [ DrawSphere sphere_2_$widgets(work3D) 0.5 $xI $yI $zI 1.0 0.0 0.0 ]

        vtkPoints points_1_$widgets(work3D)
            points_1_$widgets(work3D) InsertPoint 0 $xI $yI $zI
            points_1_$widgets(work3D) InsertPoint 1 $xc $yc $zc
 
        vtkCellArray line_1_$widgets(work3D)
            line_1_$widgets(work3D) InsertNextCell 2
        for { set j 0 } { $j < 2 } { incr j } { line_1_$widgets(work3D) InsertCellPoint $j }

        vtkPolyData poly_normal_1_$widgets(work3D)
            poly_normal_1_$widgets(work3D) SetPoints points_1_$widgets(work3D)
            poly_normal_1_$widgets(work3D) SetLines line_1_$widgets(work3D)

        vtkTubeFilter tube_normal_1_$widgets(work3D)
            tube_normal_1_$widgets(work3D) SetInput poly_normal_1_$widgets(work3D)
            tube_normal_1_$widgets(work3D) SetRadius 0.25

        vtkPolyDataMapper line_mapper_1_$widgets(work3D)
            line_mapper_1_$widgets(work3D) SetInput [ tube_normal_1_$widgets(work3D) GetOutput ]
            line_mapper_1_$widgets(work3D) ImmediateModeRenderingOn

        vtkActor line_actor_1_$widgets(work3D)
            line_actor_1_$widgets(work3D) SetMapper line_mapper_1_$widgets(work3D)
            [ line_actor_1_$widgets(work3D) GetProperty ] SetColor 1 0 0
            [ line_actor_1_$widgets(work3D) GetProperty ] BackfaceCullingOff

        renderer_$widgets(work3D) AddActor line_actor_1_$widgets(work3D)

        set distPoints $i
        set largVaisseau1 [ DistPoints $xc $yc $zc $xI $yI $zI ]
        set vox_size [ GetActualVoxelSize_dll ]
        set coordVoxelPc  [ VoxelCoord $xc $yc $zc $xO $yO $zO $vox_size $vox_size $vox_size ]
        set xcvol         [ lindex $coordVoxelPc 0 ]
        set ycvol         [ lindex $coordVoxelPc 1 ]
        set zcvol         [ lindex $coordVoxelPc 2 ]
        set coordVoxelPI [ VoxelCoord $xI $yI $zI $xO $yO $zO $vox_size $vox_size $vox_size ]
        set xIvol        [ lindex $coordVoxelPI 0 ]
        set yIvol        [ lindex $coordVoxelPI 1 ]
        set zIvol        [ lindex $coordVoxelPI 2 ]
        set distPointsVol1 [ DistPoints $xcvol $ycvol $zcvol $xIvol $yIvol $zIvol ]

        set xc [ expr ( $xc - ( $xN * ( $largVaisseau1 / 2 ) ) ) ]
        set yc [ expr ( $yc - ( $yN * ( $largVaisseau1 / 2 ) ) ) ]
        set zc [ expr ( $zc - ( $zN * ( $largVaisseau1 / 2 ) ) ) ]

        renderer_$widgets(work3D) AddActor [ DrawSphere sphere_3_$widgets(work3D) 0.5 $xc $yc $zc 1.0 0.0 0.0 ]

        set vectsPerp [ TclPerpendiculars_dll $xN $yN $zN 0 ]
        set xP1 [ lindex $vectsPerp 0 ]; set yP1 [ lindex $vectsPerp 1 ]; set zP1 [ lindex $vectsPerp 2 ];
        set xP2 [ lindex $vectsPerp 3 ]; set yP2 [ lindex $vectsPerp 4 ]; set zP2 [ lindex $vectsPerp 5 ];

        set resIntP11 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP1 $yP1 $zP1 ]              \
            [ list $xO $yO $zO ]                 \
            0                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP11 [ lindex $resIntP11 0 ]
        set xP11      [ lindex $resIntP11 1 ]
        set yP11      [ lindex $resIntP11 2 ]
        set zP11      [ lindex $resIntP11 3 ]
        set i         [ lindex $resIntP11 4 ]

        set resIntP12 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP1 $yP1 $zP1 ]              \
            [ list $xO $yO $zO ]                 \
            1                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP12 [ lindex $resIntP12 0 ]
        set xP12      [ lindex $resIntP12 1 ]
        set yP12      [ lindex $resIntP12 2 ]
        set zP12      [ lindex $resIntP12 3 ]
        set i         [ lindex $resIntP12 4 ]

        if { ( $trouveP11 == 1 ) && ( $trouveP12 == 1 ) } {
            set largP1 [ DistPoints $xP11 $yP11 $zP11 $xP12 $yP12 $zP12 ]
        } else { set largP1 10000 }

        set resIntP21 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP2 $yP2 $zP2 ]              \
            [ list $xO $yO $zO ]                 \
            0                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP21 [ lindex $resIntP21 0 ]
        set xP21      [ lindex $resIntP21 1 ]
        set yP21      [ lindex $resIntP21 2 ]
        set zP21      [ lindex $resIntP21 3 ]
        set i         [ lindex $resIntP21 4 ]

        set resIntP22 [ u_surf::intersectionSurface \
            [ cubes_$widgets(work3D) GetOutput ]          \
            [ list $xc $yc $zc ]                 \
            [ list $xP2 $yP2 $zP2 ]              \
            [ list $xO $yO $zO ]                 \
            1                                    \
            $bounds                              \
            [ expr $largmax / 2 ]                \
        ]
        set trouveP22 [ lindex $resIntP22 0 ]
        set xP22      [ lindex $resIntP22 1 ]
        set yP22      [ lindex $resIntP22 2 ]
        set zP22      [ lindex $resIntP22 3 ]
        set i         [ lindex $resIntP22 4 ]

        if { ( $trouveP21 == 1 ) && ( $trouveP22 == 1 ) } {
            set largP2 [ DistPoints $xP21 $yP21 $zP21 $xP22 $yP22 $zP22 ]
        } else { set largP2 10000 }

        if { ( $largP1 == 10000 ) && ( $largP2 == 10000 ) } {
        
            set xP1 0; set yP1 0; set zP1 0
            set xP2 0; set yP2 0; set zP2 0
            set largdef 0

        } elseif { ( $largP1 < $largP2 ) } {
        
            set xP1 $xP11; set yP1 $yP11; set zP1 $zP11
            set xP2 $xP12; set yP2 $yP12; set zP2 $zP12
            set largdef $largP1

        } elseif { ( $largP2 < $largP1 ) } {

            set xP1 $xP21; set yP1 $yP21; set zP1 $zP21
            set xP2 $xP22; set yP2 $yP22; set zP2 $zP22
            set largdef $largP2
        } 
        set largVaisseau2 $largdef

        if { $largVaisseau2 > 0 } {
        
            set largVaisseau [ expr ( $largVaisseau1 + $largVaisseau2 ) / 2 ]
            set vox_size [ GetActualVoxelSize_dll ]
            set coordVoxelP1 [ VoxelCoord $xP1 $yP1 $zP1 $xO $yO $zO $vox_size $vox_size $vox_size ]
            set xP1vol [ lindex $coordVoxelP1 0 ]; set yP1vol [ lindex $coordVoxelP1 1 ]; set zP1vol [ lindex $coordVoxelP1 2 ]

            set coordVoxelP2 [VoxelCoord $xP2 $yP2 $zP2 $xO $yO $zO $vox_size $vox_size $vox_size ]
            set xP2vol [ lindex $coordVoxelP2 0 ]; set yP2vol [ lindex $coordVoxelP2 1 ]; set zP2vol [ lindex $coordVoxelP2 2 ]
            set distPointsVol2 [ DistPoints $xP1vol $yP1vol $zP1vol $xP2vol $yP2vol $zP2vol ]

        } else {
        
            set largVaisseau 0
            set distPointsVol2 0
        }

        if { $largVaisseau2 > 0 } {

            renderer_$widgets(work3D) AddActor [ DrawSphere sphere_4_$widgets(work3D) 0.5 $xP1 $yP1 $zP1 0.0 1.0 0.0 ]
            renderer_$widgets(work3D) AddActor [ DrawSphere sphere_5_$widgets(work3D) 0.5 $xP2 $yP2 $zP2 0.0 1.0 0.0 ]

            vtkPoints points_2_$widgets(work3D)
                points_2_$widgets(work3D) InsertPoint 0 $xP1 $yP1 $zP1
                points_2_$widgets(work3D) InsertPoint 1 $xP2 $yP2 $zP2
 
            vtkCellArray line_2_$widgets(work3D)
                line_2_$widgets(work3D) InsertNextCell 2
            for { set j 0 } { $j < 2 } { incr j } { line_2_$widgets(work3D) InsertCellPoint $j }

            vtkPolyData poly_normal_2_$widgets(work3D)
                poly_normal_2_$widgets(work3D) SetPoints points_2_$widgets(work3D)
                poly_normal_2_$widgets(work3D) SetLines line_2_$widgets(work3D)

            vtkTubeFilter tube_normal_2_$widgets(work3D)
                tube_normal_2_$widgets(work3D) SetInput poly_normal_2_$widgets(work3D)
                tube_normal_2_$widgets(work3D) SetRadius 0.25

            vtkPolyDataMapper line_mapper_2_$widgets(work3D)
                line_mapper_2_$widgets(work3D) SetInput [ tube_normal_2_$widgets(work3D) GetOutput ]
                line_mapper_2_$widgets(work3D) ImmediateModeRenderingOn

            vtkActor line_actor_2_$widgets(work3D)
                line_actor_2_$widgets(work3D) SetMapper line_mapper_2_$widgets(work3D)
                [ line_actor_2_$widgets(work3D) GetProperty ] SetColor 0.0 1.0 0.0
                [ line_actor_2_$widgets(work3D) GetProperty ] BackfaceCullingOff

            renderer_$widgets(work3D) AddActor line_actor_2_$widgets(work3D)

        }

        if { $distPointsVol2 > 0 } {
            set distPointsVol [ expr ( $distPointsVol1 + $distPointsVol2 ) / 2 ]
        } else { set distPointsVol $distPointsVol1 }
        SetROIStep_dll $distPointsVol
        array set arr [ params_dll ]
        set act [ \
            DrawSphere \
                sphere_ROI_$widgets(work3D) \
                [ expr $arr(e_roi_dimension) / 2.0 ] \
                $xc $yc $zc \
                1.0 0.0 0.0 \
        ]
        [ $act GetProperty ] SetOpacity 0.2
        renderer_$widgets(work3D) AddActor $act

    }

    busy release .
    update
    [ $widgets(work3D) GetRenderWindow ] Render

}

proc u_surf::change_intensity { typ } {

    # For programming facilities
    upvar u_surf::widgets widgets

    if { $typ == 1 } {

        set slc [ expr ( [ $widgets(sclIso) cget -to ] + ( [ $widgets(sclIso) cget -from ] * 3 ) ) / 4 ]
        $widgets(sclIso) set $slc

    } else { set slc [ $widgets(sclIso) get ] }

    cubes_$widgets(work3D) SetValue 0 $slc
    edgePoints_$widgets(work3D) SetValue $slc
    [ $widgets(work3D) GetRenderWindow ] Render
    

}

proc u_surf::change_opacity { typ } {

    # For programming facilities
    upvar u_surf::widgets widgets

    if { $typ == 1 } {

        set slc [ expr ( [ $widgets(sclOpacity) cget -to ] + [ $widgets(sclOpacity) cget -from ] ) / 2 ]
        $widgets(sclOpacity) set $slc
    
    } else { set slc [ $widgets(sclOpacity) get ] }

    [ isoActor_$widgets(work3D) GetProperty ] SetOpacity $slc
    [ edgeActor_$widgets(work3D) GetProperty ] SetOpacity $slc
    [ $widgets(work3D) GetRenderWindow ] Render

}

proc u_surf::show_axis { points } {

    # For programming facilities
    upvar u_surf::widgets widgets

    set resample [ GetVTKVolume_dll ]
    set bounds   [ $resample GetBounds ]
    set nP [ expr [ llength $points ] / 3 ]

    if { $nP > 0 } {

        # Poly-line actor construction...
        catch { renderer_$widgets(work3D) RemoveActor axisActor_$widgets(work3D) }
        catch { axisActor_$widgets(work3D)    Delete }
        catch { axisMapper_$widgets(work3D)   Delete }
        catch { axisGrid_$widgets(work3D)     Delete }
        catch { axisPolyLine_$widgets(work3D) Delete }
        catch { axisPoints_$widgets(work3D)   Delete }
        catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_6_$widgets(work3D) ] }
        catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_7_$widgets(work3D) ] }
        DeleteSphere sphere_6_$widgets(work3D)
        DeleteSphere sphere_7_$widgets(work3D)

        vtkPoints axisPoints_$widgets(work3D)
        vtkPolyLine axisPolyLine_$widgets(work3D)
        [ axisPolyLine_$widgets(work3D) GetPointIds ] SetNumberOfIds $nP
        set vox_size [ GetActualVoxelSize_dll ]
        for { set i 0 } { $i < $nP } { incr i } {

            [ axisPolyLine_$widgets(work3D) GetPointIds ] SetId $i $i
            set p [ \
                PointCoord \
                [ lindex $points [ expr $i * 3 + 0 ] ] \
                [ lindex $points [ expr $i * 3 + 1 ] ] \
                [ lindex $points [ expr $i * 3 + 2 ] ] \
                [ lindex $bounds 0 ] \
                [ lindex $bounds 2 ] \
                [ lindex $bounds 4 ] \
                $vox_size $vox_size $vox_size \
            ]
            axisPoints_$widgets(work3D) InsertNextPoint [ lindex $p 0 ] [ lindex $p 1 ] [ lindex $p 2 ]

            #catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_cell_int_$widgets(work3D) ] }
            #DeleteSphere sphere_cell_int_$widgets(work3D)
            #set act [ \
            #    DrawSphere \
            #        sphere_cell_int_$widgets(work3D) \
            #        [ expr [ getCellIntensitySize_dll ] / 2.0 ] \
            #        [ lindex $p 0 ] [ lindex $p 1 ] [ lindex $p 2 ] \
            #        1.0 0.0 0.0 \
            #]
            #[ $act GetProperty ] SetOpacity 0.2
            #renderer_$widgets(work3D) AddActor $act
            #[ $widgets(work3D) GetRenderWindow ] Render
            #update

        }

        vtkUnstructuredGrid axisGrid_$widgets(work3D)
            axisGrid_$widgets(work3D) Allocate 1 1
            axisGrid_$widgets(work3D) InsertNextCell \
                [ axisPolyLine_$widgets(work3D) GetCellType ] \
                [ axisPolyLine_$widgets(work3D) GetPointIds ]
            axisGrid_$widgets(work3D) SetPoints axisPoints_$widgets(work3D)

        vtkDataSetMapper axisMapper_$widgets(work3D)
            axisMapper_$widgets(work3D) SetInput axisGrid_$widgets(work3D)
         axisMapper_$widgets(work3D) ImmediateModeRenderingOn

        vtkActor axisActor_$widgets(work3D)
            axisActor_$widgets(work3D) SetMapper axisMapper_$widgets(work3D)
            [ axisActor_$widgets(work3D) GetProperty] BackfaceCullingOn
            [ axisActor_$widgets(work3D) GetProperty] SetDiffuseColor 1 0 0
#            [ axisActor_$widgets(work3D) GetProperty]  SetLineWidth 3
			
        renderer_$widgets(work3D) AddActor axisActor_$widgets(work3D)

        set p [ \
            PointCoord \
            [ lindex $points 0 ] \
            [ lindex $points 1 ] \
            [ lindex $points 2 ] \
            [ lindex $bounds 0 ] \
            [ lindex $bounds 2 ] \
            [ lindex $bounds 4 ] \
            $vox_size $vox_size $vox_size \
        ]
        renderer_$widgets(work3D) AddActor [ \
            DrawSphere \
                sphere_6_$widgets(work3D) \
                1.0 \
                [ lindex $p 0 ] [ lindex $p 1 ] [ lindex $p 2 ] \
                0.0 0.0 1.0 \
        ]

        set i [ expr $nP - 1 ]
        set p [ \
            PointCoord \
            [ lindex $points [ expr $i * 3 + 0 ] ] \
            [ lindex $points [ expr $i * 3 + 1 ] ] \
            [ lindex $points [ expr $i * 3 + 2 ] ] \
            [ lindex $bounds 0 ] \
            [ lindex $bounds 2 ] \
            [ lindex $bounds 4 ] \
            $vox_size $vox_size $vox_size \
        ]
        renderer_$widgets(work3D) AddActor [ \
            DrawSphere \
                sphere_7_$widgets(work3D) \
                1.0 \
                [ lindex $p 0 ] [ lindex $p 1 ] [ lindex $p 2 ] \
                0.0 1.0 0.0 \
        ]
        [ $widgets(work3D) GetRenderWindow ] Render
        u_3D::controls 2

    }

}

proc u_surf::extract_axis { } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::axis_index      axis_index

    busy hold .
    u_3D::controls
    update

    catch { set center [ sphere_3_$widgets(work3D) GetCenter ] } err
    set err [ expr ( [ string compare [ lindex $err 0 ] invalid ] != 0 )? 0: 1 ]

    catch { renderer_$widgets(work3D) RemoveActor axesActor_$widgets(work3D) }
    catch { axesActor_$widgets(work3D) Delete }
    catch { axesMapper_$widgets(work3D) Delete }

    if { $err == 0 } {

        set vox_size [ GetActualVoxelSize_dll ]
        set coord [ \
            VoxelCoord \
                [ lindex $center 0 ] \
                [ lindex $center 1 ] \
                [ lindex $center 2 ] \
                [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 0 ] . ] 0 ] \
                [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 1 ] . ] 0 ] \
                [ lindex [ split [ lindex [ [ struct_$widgets(work3D) GetOutput ] GetOrigin ] 2 ] . ] 0 ] \
                $vox_size $vox_size $vox_size \
        ]

#        set resample [ GetVTKVolume_dll ]
#        set bounds   [ $resample GetBounds ]

        $widgets(edtChooseAxis) delete 0  end
        $widgets(edtChooseAxis) insert end ""

        SetStartPoint_dll [ lindex $coord 0 ] [ lindex $coord 1 ] [ lindex $coord 2 ]
        ExtractAxes_dll

        set axes [ GetAllAxes_dll ]

        vtkPolyDataMapper axesMapper_$widgets(work3D)
        axesMapper_$widgets(work3D) SetInput $axes

        vtkActor axesActor_$widgets(work3D)
        axesActor_$widgets(work3D) SetMapper axesMapper_$widgets(work3D)
        [ axesActor_$widgets(work3D) GetProperty ] SetColor 1 0 0

        renderer_$widgets(work3D) AddActor axesActor_$widgets(work3D)
        [ $widgets(work3D) GetRenderWindow ] Render

#        set points [ getAxis_dll ]
#        u_surf::show_axis $points

        busy release .
        update
        u_3D::controls 2

    } else {

        busy release .
        update

        tk_messageBox \
            -icon error \
            -message "$string_table::str_please_select_a_point" \
            -title "ERROR"

    }

}

proc u_surf::add_axis { } {

    # For programming facilities
    upvar u_surf::widgets    widgets
    upvar u_surf::axis_index axis_index
    upvar u_surf::indexes    indexes

    
    if { [ string compare [ $widgets(edtChooseAxis) get ] "" ] != 0 } {
    
        set name [ $widgets(edtChooseAxis) get ]
        $widgets(edtChooseAxis) list insert end $name
        set indexes($name) [ commitAxis_dll $name ]
        u_3D::controls 2

    } else {

        set name [ dlg_name_axis::show $widgets(base) ]
        if { [ string compare $name "" ] != 0 } {
        
            $widgets(edtChooseAxis) delete 0  end
            $widgets(edtChooseAxis) insert end $name
            u_surf::add_axis
        
        }

    }

}

proc u_surf::select_axis_point { x y } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::axis_index axis_index

    catch { tmp_world_picker_$widgets(work3D) Delete }
    renderer_$widgets(work3D) IsInViewport 0 0

    set y [ expr [ lindex [ $widgets(work3D) configure -height ] 4 ] - $y - 1 ]

    set pickWPPos [ pick_point_local_actor $x $y renderer_$widgets(work3D) isoActor_$widgets(work3D) 0.001 ]
    set xc [ lindex $pickWPPos 0 ]
    set yc [ lindex $pickWPPos 1 ]
    set zc [ lindex $pickWPPos 2 ]

    #Chercher le point de l'axe le plus proche au point choisi sur la surface
    set indP [ axisGrid_$widgets(work3D) FindPoint $xc $yc $zc ]
    set coordsP [ axisGrid_$widgets(work3D) GetPoint $indP ]
    set xc [ lindex $coordsP 0 ]
    set yc [ lindex $coordsP 1 ]
    set zc [ lindex $coordsP 2 ]

    renderer_$widgets(work3D) AddActor [ DrawSphere sphere_tmp_$widgets(work3D) 1.0 $xc $yc $zc 1.0 0.0 0.0 ]

    [ $widgets(work3D) GetRenderWindow ] Render

    set dir [ choose_direction::show $widgets(work3D) ]
    if { $dir == -1 || $dir == 1 } {

        cutAxis_dll $indP $dir
        set points [ getAxis_dll ]
        u_surf::show_axis $points
        set ans [ \
            tk_messageBox \
                -icon question \
                -message "$string_table::str_accept_changes" \
                -type yesno \
                -title "Question" \
        ]
        if { [ string compare $ans no ] == 0 } {
        
            undoAxis_dll
            u_surf::show_axis [ getAxis_dll ]

        } else {

            $widgets(edtChooseAxis) delete 0  end
            $widgets(edtChooseAxis) insert end ""

        }

    }
    [ $widgets(work3D) GetRenderWindow ] Render
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_tmp_$widgets(work3D) ] }
    DeleteSphere sphere_tmp_$widgets(work3D)

}

proc u_surf::remove_axis { } {

    # For programming facilities
    upvar u_surf::widgets    widgets
    upvar u_surf::axis_index axis_index
    upvar u_surf::indexes    indexes

    #   Delete from data...
    set name [ $widgets(edtChooseAxis) get ]
    cleanAxis_dll
    if { $name != "" } {

        deleteAxis_dll $indexes($name)

        #   Reorder combobox
        $widgets(edtChooseAxis) delete 0  end
        $widgets(edtChooseAxis) insert end ""
        $widgets(edtChooseAxis) list delete $indexes($name) $indexes($name)

        #   Reorder array
        set l [ getAxisDescriptions_dll ]
        array set indexes [ list [ lindex $l 0 ] 0 ]
        for { set i 1 } { $i < [ llength $l ] } { incr i } {

            set indexes([ lindex $l $i ]) $i

        }

    }

    #   3D data reset
    catch { renderer_$widgets(work3D) RemoveActor axisActor_$widgets(work3D) }
    catch { axisActor_$widgets(work3D)    Delete }
    catch { axisMapper_$widgets(work3D)   Delete }
    catch { axisGrid_$widgets(work3D)     Delete }
    catch { axisPolyLine_$widgets(work3D) Delete }
    catch { axisPoints_$widgets(work3D)   Delete }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_6_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_7_$widgets(work3D) ] }
    catch { renderer_$widgets(work3D) RemoveActor [ GetSphereActorName sphere_cell_int_$widgets(work3D) ] }
    DeleteSphere sphere_6_$widgets(work3D)
    DeleteSphere sphere_7_$widgets(work3D)
    DeleteSphere sphere_cell_int_$widgets(work3D)

    [ $widgets(work3D) GetRenderWindow ] Render
    u_3D::controls 0

}

proc u_surf::continue_axis { } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::axis_index axis_index

    busy hold .
    u_3D::controls
    update

    catch { set center [ sphere_3_$widgets(work3D) GetCenter ] } err
    set err [ expr ( [ string compare [ lindex $err 0 ] invalid ] != 0 )? 0: 1 ]

    if { $err == 0 } {

        set resample [ GetVTKVolume_dll ]
        set bounds   [ $resample GetBounds ]
        set vox_size [ GetActualVoxelSize_dll ]
        set coord [ \
            VoxelCoord \
                [ lindex $center 0 ] \
                [ lindex $center 1 ] \
                [ lindex $center 2 ] \
                [ lindex $bounds 0 ] \
                [ lindex $bounds 2 ] \
                [ lindex $bounds 4 ] \
                $vox_size $vox_size $vox_size \
        ]

        continueAxis_dll $coord
        set points [ getAxis_dll ]
        u_surf::show_axis $points
        set ans [ \
            tk_messageBox \
                -icon question \
                -message "$string_table::str_accept_changes" \
                -type yesno \
                -title "Question" \
        ]
        if { [ string compare $ans no ] == 0 } {
        
            undoAxis_dll
            u_surf::show_axis [ getAxis_dll ]

        } else {

            $widgets(edtChooseAxis) delete 0  end
            $widgets(edtChooseAxis) insert end ""

        }


        busy release .
        update
        u_3D::controls 0

    } else {

        busy release .
        update
        u_3D::controls 0

        tk_messageBox \
            -icon error \
            -message "$string_table::str_please_select_a_point" \
            -title "ERROR"

    }
}

proc u_surf::set_mouse_left_events { mask } {

    # For programming facilities
    upvar u_surf::widgets widgets

    # TODO : change
    catch { bind $widgets(work3D) <Any-ButtonPress>   { ev_startMotion %W %x %y } }
    catch { bind $widgets(work3D) <Any-ButtonRelease> { ev_endMotion %W %x %y } }

    if { [ expr $mask & 0x4 ] == 0x4 } { catch { bind $widgets(work3D) <B1-Motion> { evz_rotate %W %x %y } } }
    if { [ expr $mask & 0x2 ] == 0x2 } { catch { bind $widgets(work3D) <B1-Motion> { evz_pan %W %x %y } } }

}

proc u_surf::set_mouse_right_events { mask } {

    # For programming facilities
    upvar u_surf::widgets widgets

    catch { bind $widgets(work3D) <Any-ButtonPress>   { ev_startMotion %W %x %y } }
    catch { bind $widgets(work3D) <Any-ButtonRelease> { ev_endMotion %W %x %y } }

    if { [ expr $mask & 0x4 ] == 0x4 } { catch { bind $widgets(work3D) <B3-Motion> { evz_zoom %W %x %y } } }

}

proc u_surf::select_axis { args } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::axis_index axis_index
    upvar u_surf::indexes         indexes

    if { [ array names indexes [ $widgets(edtChooseAxis) get ] ] != "" } {
    
        setAxis_dll $indexes([ $widgets(edtChooseAxis) get ])
        u_surf::show_axis [ getAxis_dll ]

    }

}

proc u_surf::get_isoval { } {

    # For programming facilities
    upvar u_surf::widgets widgets

    return [ $widgets(sclIso) get ]

}

proc u_surf::reload_axis { mask step } {

    # For programming facilities
    upvar u_surf::widgets    widgets
    upvar u_surf::axis_index axis_index

    if { $axis_index != -1 } {

        if { $step == 1 } {

            redoAxisSpline_dll

        } elseif { $mask == 1 } { recalculeAxisSignal_dll }
        set points [ getAxis_dll ]
        u_surf::show_axis $points

    }

}

proc u_surf::load_axes { } {

    # For programming facilities
    upvar u_surf::widgets widgets
    upvar u_surf::indexes indexes

    u_surf::show_axis [ getAxis_dll ]

    $widgets(edtChooseAxis) list delete 0 end
    set names [ getAxisDescriptions_dll ]
    set i 0
    array set indexes [ list [ lindex $names 0 ] 0 ]
    foreach name $names {

        $widgets(edtChooseAxis) list insert end $name
        set indexes($name) $i
        incr i

    }
    
    $widgets(edtChooseAxis) delete 0 end
    $widgets(edtChooseAxis) insert end  [ getAxisDescription_dll ]
    [ $widgets(work3D) GetRenderWindow ] Render
    u_3D::controls 2

}

proc u_surf::saveasvtk { } {

    # For programming facilities
    upvar u_surf::widgets widgets

#    catch { objexp Delete }
#
#    vtkOBJExporter objexp
#    objexp SetInput [ $widgets(work3D) GetRenderWindow ]
#    objexp SetFilePrefix "./maracas"
#    objexp Write

    set a [cubes_$widgets(work3D) GetOutput]
    vtkPolyDataWriter b

    set f [ tk_getSaveFile -defaultextension "vtk" \
        -initialdir "." \
			   -title "Save surface..." \
			   -filetypes { { {vtk Files} {.vtk} } }
    ]
    if { [ string compare $f "" ] != 0 } {
        b SetInput $a
        b SetFileName $f
        b SetFileTypeToASCII
        b Write
    }
    b Delete
}


# EOF - u_surf.tcl
