/*# ---------------------------------------------------------------------
#
# Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
#                        pour la Sant�)
# Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
# Previous Authors : Laurent Guigues, Jean-Pierre Roux
# CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
#
#  This software is governed by the CeCILL-B license under French law and
#  abiding by the rules of distribution of free software. You can  use,
#  modify and/ or redistribute the software under the terms of the CeCILL-B
#  license as circulated by CEA, CNRS and INRIA at the following URL
#  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
#  or in the file LICENSE.txt.
#
#  As a counterpart to the access to the source code and  rights to copy,
#  modify and redistribute granted by the license, users are provided only
#  with a limited warranty  and the software's author,  the holder of the
#  economic rights,  and the successive licensors  have only  limited
#  liability.
#
#  The fact that you are presently reading this means that you have had
#  knowledge of the CeCILL-B license and that you accept its terms.
# ------------------------------------------------------------------------ */

/**
 *  \file
 *  \brief Class bbtk::ThresholdImageView .
 */

#include "LayerImageBase.h"

#include "wxMPRBaseData.h"
#include "wxVtk2DBaseView.h"
#include "creawxVTKRenderWindowInteractor.h"
#include "vtkTextProperty.h"


//---------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------

// Callback for the interaction
class ColorLayerObserver : public vtkCommand{
  public:
	virtual char const *GetClassName() const { return "ColorLayerObserver";}

	static ColorLayerObserver *New()	{      return new ColorLayerObserver; 	}

	ColorLayerObserver() 
	{
		_layerimagebase = NULL;
  	}

	virtual void Execute(vtkObject *wdg, unsigned long event, void* calldata) 	
	{ 
		if (_layerimagebase!= NULL)
		{

			wxVtkBaseView		*wxvtkbaseview = _layerimagebase->GetwxVtkBaseView();
		  	if (wxvtkbaseview!=NULL)
			{

				vtkMPRBaseData 	*vtkmprbasedata =	(vtkMPRBaseData*)wxvtkbaseview->GetVtkBaseData();
			  	if (vtkmprbasedata!=NULL)
				{
					_layerimagebase->SetX( vtkmprbasedata->GetX() );
					_layerimagebase->SetY( vtkmprbasedata->GetY() );
					_layerimagebase->SetZ( vtkmprbasedata->GetZ() );
	 	    	   	_layerimagebase->onThresholdChange();
				} // if basedata
				
			} // if wxvtkbaseview
		} //if layerimagebase
	}

	void SetLayerImageBase(LayerImageBase* layerimagebase)
	{
   	_layerimagebase	=	layerimagebase;
	}

private:
	LayerImageBase* _layerimagebase;
};



//---------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------
//---------------------------------------------------------------------------------------------



LayerImageBase::LayerImageBase()
{
	_actorPresent			=	false;
	_X						=	-1;
	_Y						=	-1;
	_Z						=	-1;
	_opacity				=	1;
	_thresholdTable			=	NULL;
	_thresholdMapper		=	NULL;
	_scalarbarActor			=	NULL;
	_thresholdActor			=	NULL;
	_image					=	NULL;
	_baseView				=	NULL;
	_imageReslicer			=	vtkImageReslice::New();
	_imageChangeInformation	= 	vtkImageChangeInformation::New();
}

//----------------------------------------------------------------------------
LayerImageBase::~LayerImageBase()
{
}

//----------------------------------------------------------------------------
void LayerImageBase::SetX(int x)
{
	_X = x;
}

//----------------------------------------------------------------------------
void LayerImageBase::SetY(int y)
{
	_Y = y;
}

//----------------------------------------------------------------------------
void LayerImageBase::SetZ(int z)
{
	_Z = z;
}

//----------------------------------------------------------------------------
int LayerImageBase::GetX()  // virtual
{
	return _X;
}

//----------------------------------------------------------------------------
int LayerImageBase::GetY()  // virtual
{
	return _Y;
}
//----------------------------------------------------------------------------
int LayerImageBase::GetZ()  // virtual
{
	return _Z;
}


//----------------------------------------------------------------------------
vtkImageData* LayerImageBase::GetImage()
{
//	return _image;
	vtkImageData *result;
	if (_image==NULL)
	{
		result = NULL;
	} else {
		result = _imageChangeInformation->GetOutput();
	}
	return result;
}

//----------------------------------------------------------------------------
bool LayerImageBase::GetActorPresent()
{
	return _actorPresent;
}


//----------------------------------------------------------------------------
void LayerImageBase::SetImage(vtkImageData* image)
{
	int 	ext[6];
	double 	spcBase[3];
	int 	dimensionOriginalLayer[3];
	int 	dimBase[6];

	ext[0]=0;
	ext[1]=0;
	ext[2]=0;
	ext[3]=0;
	ext[4]=0;
	ext[5]=0;
	spcBase[0]=0;
	spcBase[1]=0;
	spcBase[2]=0;
	spcBase[3]=0;
	spcBase[4]=0;
	spcBase[5]=0;

	_image = image;

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	_image->Update();
	_image->GetWholeExtent(ext);
#else
	_image->GetExtent(ext);
#endif

	dimensionOriginalLayer[0] = ext[1]-ext[0]+1;
	dimensionOriginalLayer[1] = ext[3]-ext[2]+1;
	dimensionOriginalLayer[2] = ext[5]-ext[4]+1;
	SetDimensionOriginalLayer(dimensionOriginalLayer);
	double 	spcOriginalLayer[3];
	_image->GetSpacing(spcOriginalLayer);
	SetSpcOriginalLayer(spcOriginalLayer);
	if (GetwxVtkBaseView()!=NULL)
	{
		vtkImageData 	*imagebase		=	GetwxVtkBaseView()->GetVtkBaseData()->GetImageData();
		if (imagebase!=NULL) 
		{	

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
			imagebase->GetWholeExtent(ext);
#else
			imagebase->GetExtent(ext);
#endif

			dimBase[0]=ext[1]-ext[0]+1;				
			dimBase[1]=ext[3]-ext[2]+1;				
			dimBase[2]=ext[5]-ext[4]+1;				
			SetDimensionBase(dimBase);
			imagebase->GetSpacing(spcBase);
			SetSpcBase(spcBase);

			if ((_X<0) || (_X>dimBase[0])) { SetX( dimBase[0]/2 ); } 
			if ((_Y<0) || (_Y>dimBase[1])) { SetY( dimBase[1]/2 ); } 
			if ((_Z<0) || (_Z>dimBase[2])) { SetZ( dimBase[2]/2 ); } 


		} // if imagebase
	} // if wxvtkbaseview
	_image->GetScalarRange( _range );
//	_thresholdTable = vtkLookupTable::New();


//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
	_imageChangeInformation->SetInput(_image);
#else
	_imageChangeInformation->SetInputData(_image);
	_imageChangeInformation->Update();
#endif

}

//------------------------------------------------------------------------------
void LayerImageBase::SetSpcOriginalLayer(double spc[3])
{
	_spcOriginalLayer[0] = spc[0];
	_spcOriginalLayer[1] = spc[1];
	_spcOriginalLayer[2] = spc[2];
}

//------------------------------------------------------------------------------
void LayerImageBase::SetNewSpacingLayer(double spc[3])
{
	_newSpcLayer[0] = spc[0];
	_newSpcLayer[1] = spc[1];
	_newSpcLayer[2] = spc[2];
	_imageChangeInformation->SetOutputSpacing( _newSpcLayer );  
_imageChangeInformation->Update();
}


//------------------------------------------------------------------------------
void LayerImageBase::SetSpcBase(double spc[3])
{
	_spcBase[0] = spc[0];
	_spcBase[1] = spc[1];
	_spcBase[2] = spc[2];
}

//------------------------------------------------------------------------------
void LayerImageBase::SetDimensionOriginalLayer(int dim[3])
{
	_dimOriginalLayer[0] = dim[0];
	_dimOriginalLayer[1] = dim[1];
	_dimOriginalLayer[2] = dim[2];
}

//------------------------------------------------------------------------------
void LayerImageBase::SetDimensionBase(int dim[3])
{
	_dimBase[0] = dim[0];
	_dimBase[1] = dim[1];
	_dimBase[2] = dim[2];
}



//------------------------------------------------------------------------------
void LayerImageBase::GetSpcOriginalLayer(double *spc)
{
	spc[0] = _spcOriginalLayer[0];
	spc[1] = _spcOriginalLayer[1];
	spc[2] = _spcOriginalLayer[2];
}

//------------------------------------------------------------------------------
void LayerImageBase::GetSpcBase(double *spc)
{
	spc[0] = _spcBase[0];
	spc[1] = _spcBase[1];
	spc[2] = _spcBase[2];
}

//------------------------------------------------------------------------------
void LayerImageBase::GetDimensionOriginalLayer(int *dim)
{
	dim[0] = _dimOriginalLayer[0];
	dim[1] = _dimOriginalLayer[1];
	dim[2] = _dimOriginalLayer[2];
}

//------------------------------------------------------------------------------
void LayerImageBase::GetDimensionBase(int *dim)
{
	dim[0] = _dimBase[0];
	dim[1] = _dimBase[1];
	dim[2] = _dimBase[2];
}

//------------------------------------------------------------------------------
void LayerImageBase::SetColorBarPosition(std::vector<int> &colorbarposition)
{
   _colorBarPosition=colorbarposition;
   UpdateColorBarPosition();
}

//------------------------------------------------------------------------------
void LayerImageBase::UpdateColorBarPosition()
{
	if ( _scalarbarActor!=NULL)
	{
		if (_colorBarPosition.size()==2)
		{ 
			_scalarbarActor->SetDisplayPosition( _colorBarPosition[0] , _colorBarPosition[1] );
		} else {
			_scalarbarActor->SetDisplayPosition(0,0);
		}
	} // if   
}//

//------------------------------------------------------------------------------
void LayerImageBase::SetRangeForColorBar(std::vector<double> &range)
{
   _rangeForColorBar=range;
}


//------------------------------------------------------------------------------
void LayerImageBase::GetRangeForColorBar(double &minRange, double &maxRange)
{
	if (_rangeForColorBar.size()==2)
	{
		minRange = _rangeForColorBar[0];	
		maxRange = _rangeForColorBar[1];
	} else {
		minRange = _range[0];	
		maxRange = _range[1];
	}
}



//----------------------------------------------------------------------------
void LayerImageBase::SetwxVtkBaseView(wxVtkBaseView *baseview)
{
	if (_baseView==NULL)
   {
		_baseView = baseview;
		ColorLayerObserver *colorlayerobserver = ColorLayerObserver::New();
		_baseView->GetVtkBaseData()->AddObserver( 1, colorlayerobserver );
		colorlayerobserver->SetLayerImageBase(this);
	} // if _baseView
}

//----------------------------------------------------------------------------
wxVtkBaseView *LayerImageBase::GetwxVtkBaseView()
{
	return _baseView;
}

//----------------------------------------------------------------------------
void LayerImageBase::Refresh()
{
    if (_baseView!=NULL)
    {
        _baseView->Refresh();
    }
}


//----------------------------------------------------------------------------
vtkLookupTable* LayerImageBase::GetThresholdTable()
{
	return _thresholdTable;
}

//----------------------------------------------------------------------------
void LayerImageBase::CleanXYZ(double &x, double &y, double &z)
{
	if (x<0)
	{
		x=0;
	}
	if (y<0)
	{
		y=0;
	}
	if (z<0)
	{
		z=0;
	}
	double maxSize;
	maxSize=_dimOriginalLayer[0]*_newSpcLayer[0];
	if ( x > maxSize  )
	{
		x=maxSize;
	}
	maxSize=_dimOriginalLayer[1]*_newSpcLayer[1];
	if ( y > maxSize  )
	{
		y = maxSize;
	}
	maxSize=_dimOriginalLayer[2]*_newSpcLayer[2];
	if ( z > maxSize  )
	{
		z = maxSize;
	}
}

//----------------------------------------------------------------------------
void LayerImageBase::onThreshold()
{
	if ((_image!=NULL) && (_baseView!=NULL))
	{
		double x=GetX();
		double y=GetY();
		double z=GetZ();

		x = x*_spcBase[0];
		y = y*_spcBase[1];
		z = z*_spcBase[2];

		CleanXYZ(x,y,z);

		vtkCamera *camera = _baseView->GetRenderer()->GetActiveCamera();
		int directionViewer=0;
		if (camera->GetParallelProjection()==1)
		{
			wxVtk2DBaseView *wxvtk2dbasevie = (wxVtk2DBaseView *)_baseView;
			directionViewer =  wxvtk2dbasevie->GetDirection();
		} // ParallelProjection


		if (!GetActorPresent())
		{
			if (_thresholdTable==NULL)
			{
				//Lookup Table
				_thresholdTable = vtkLookupTable::New();
			} // _thresholdTable

			if (_thresholdMapper==NULL)
			{
				_thresholdMapper = vtkImageMapToColors::New( );
			}

			if (_thresholdActor==NULL)
			{
				_thresholdActor = vtkImageActor::New( );
				_scalarbarActor = vtkScalarBarActor::New();
				UpdateColorBarPosition();
				_scalarbarActor->SetHeight(0.4);  	//default  0.8
				_scalarbarActor->SetWidth(0.10);    //default  0.17
				_thresholdActor->SetOpacity( 1 );
				_thresholdActor->InterpolateOn(  );
				if (directionViewer==0)
				{
					_thresholdActor->SetPosition( 900-1,0,0 );
					_thresholdActor->RotateY( -90 );
				}
				if (directionViewer==1)
				{
					_thresholdActor->SetPosition( 0,-900-1, 0 );
					_thresholdActor->RotateX( 90 );
				}
				if (directionViewer==2)
				{
					_thresholdActor->SetPosition( 0,0, -900-1 );
				}
			} // _thresholdActor
			_baseView->GetRenderer()->AddActor( _thresholdActor );
			_baseView->GetRenderer()->AddActor2D( _scalarbarActor );
			_actorPresent = true;
		}  // !GetActorPresent()

		ConfigLookupTable();  // virtual method

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		_imageReslicer->SetInput( GetImage() );
#else
		_imageReslicer->SetInputData( GetImage() );
#endif

		_imageReslicer->SetInformationInput( GetImage() );

		if (directionViewer==0)
		{
			_imageReslicer->SetResliceAxesDirectionCosines( 0,0,1, 0,1,0,  -1,0,0 );
			_imageReslicer->SetResliceAxesOrigin(x,0,0);
		}

		if (directionViewer==1)
		{
			_imageReslicer->SetResliceAxesDirectionCosines( 1,0,0 ,0,0,1,  0,-1,0 );
			_imageReslicer->SetResliceAxesOrigin(0,y,0);
		}

		if (directionViewer==2)
		{
			_imageReslicer->SetResliceAxesDirectionCosines( 1,0,0, 0,1,0 ,0,0,1 );
			_imageReslicer->SetResliceAxesOrigin(0,0,z);
		}

		_imageReslicer->SetOutputDimensionality(2);
//		_imageReslicer->SetInterpolationModeToLinear();
		_imageReslicer->SetInterpolationModeToNearestNeighbor();
		_imageReslicer->Modified();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		// ..
#else
		_imageReslicer->Update();
#endif
		vtkImageData *img = _imageReslicer->GetOutput();
//		img->Update();
//		img->UpdateInformation();
//		_thresholdTable->Update();

//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		_thresholdMapper->SetInput( img );
#else
		_thresholdMapper->SetInputData( img );
#endif
		_thresholdMapper->SetLookupTable( _thresholdTable );


//EED 2017-01-01 Migration VTK7
#if VTK_MAJOR_VERSION <= 5
		_thresholdActor->SetInput( _thresholdMapper->GetOutput() );
#else
		_thresholdMapper->Update();
		_thresholdActor->SetInputData( _thresholdMapper->GetOutput() );
#endif
		
		_thresholdActor->SetOpacity(_opacity);
		_scalarbarActor->SetLookupTable( _thresholdTable );
//		_scalarbarActor->SetLookupTable( _thresholdMapper->GetLookupTable() );
		_scalarbarActor->SetTitle("Value");
		_scalarbarActor->SetNumberOfLabels(4);
		int fontsize = _scalarbarActor->GetLabelTextProperty()->GetFontSize();
		_scalarbarActor->GetLabelTextProperty()->SetFontSize(fontsize/2);
//		_scalarbarActor->SetTextPad(4);  ??
		} // _image
}



//----------------------------------------------------------------------------
void LayerImageBase::onThresholdChange()
{
	if (_actorPresent)
	{
		onThreshold();
	}
}

//----------------------------------------------------------------------------
void LayerImageBase::onThresholdInterpolation(bool interpolate)
{
	if (_thresholdActor!=NULL)
	{
		if (interpolate)
		{
			_thresholdActor->InterpolateOn( );
		}
		else
		{
			_thresholdActor->InterpolateOff( );
		}
	}
}

//----------------------------------------------------------------------------
void LayerImageBase::onThresholdChangeOpacity (int opacity)
{
	if (_actorPresent)
	{
		_thresholdActor->SetOpacity(opacity/100.0);
	}
// EED 2017-12-17
	_opacity = (double)opacity/100.0;
}

//----------------------------------------------------------------------------
void LayerImageBase::onThresholdRemove()
{
	if (_actorPresent)
	{
		wxVtkBaseView * baseView = _baseView;
		baseView->GetRenderer()->RemoveActor( _thresholdActor );
		baseView->GetRenderer()->RemoveActor( _scalarbarActor );
		_actorPresent = false;
	}  // if _actorPresent
}

//----------------------------------------------------------------------------
vtkImageReslice *LayerImageBase::GetvtkImageReslice()
{
    return _imageReslicer;
}

//----------------------------------------------------------------------------
void LayerImageBase::GetImageScalarRange()
{
	double min,max;
	GetRangeForColorBar(min,max);
//	_image->GetScalarRange(_range);
	_range[0]=min;
	_range[1]=max;
}

vtkScalarsToColors* LayerImageBase::GetLookupTable()
{
	return _thresholdTable;
}

// EOF

