/*# ---------------------------------------------------------------------
 #
 # Copyright (c) CREATIS (Centre de Recherche en Acquisition et Traitement de l'Image
 #                        pour la Sant�)
 # Authors : Eduardo Davila, Frederic Cervenansky, Claire Mouton
 # Previous Authors : Laurent Guigues, Jean-Pierre Roux
 # CreaTools website : www.creatis.insa-lyon.fr/site/fr/creatools_accueil
 #
 #  This software is governed by the CeCILL-B license under French law and
 #  abiding by the rules of distribution of free software. You can  use,
 #  modify and/ or redistribute the software under the terms of the CeCILL-B
 #  license as circulated by CEA, CNRS and INRIA at the following URL
 #  http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.html
 #  or in the file LICENSE.txt.
 #
 #  As a counterpart to the access to the source code and  rights to copy,
 #  modify and redistribute granted by the license, users are provided only
 #  with a limited warranty  and the software's author,  the holder of the
 #  economic rights,  and the successive licensors  have only  limited
 #  liability.
 #
 #  The fact that you are presently reading this means that you have had
 #  knowledge of the CeCILL-B license and that you accept its terms.
 # ------------------------------------------------------------------------ */

/*!
 * @file buttonContainerSettings.h
 * @brief This contains the ButtonContainerSettings class.
 * @author Diego CACERES (diego.caceres[AT]creatis.insa-lyon.fr)
 * @date  2011-06-02
 */

#ifndef BUTTONCONTAINERSETTINGS_H_
#define BUTTONCONTAINERSETTINGS_H_

#include <wx/panel.h>
#include <wx/bitmap.h>
//#include <wx/listctrl.h>

#include <list>
#include <map>
#include <vector>
#include <iostream>

#include "structBCPSettings.h"
#include "containerSettings.h"
#include "functor.h"

/*!	@namespace <creaPanelButtonContainer>
 *	@brief Contains the creaPanelButtonContainer library included in creaMaracasVisu.
 */
namespace creaPanelButtonContainer
{
	/*! @class ButtonContainerSettings buttonContainerSettings.h "buttonContainerSettings.h"
	 *	@brief This class contains the settings of the buttons.
	 */
	class ButtonContainerSettings
	{
		public:
			//typedef definition.
			// ----------------------------------------------------------------------------------
			/*!	@typedef std::string* StringType;
			 *	@brief Defines the StringType type.
			 */
			typedef std::string StringType;
			// ----------------------------------------------------------------------------------
			/*!	typedef creaButtonContainer::model::TFunctor TFunctor;
			 *	@brief Defines the TFunctor type.
			 */
			typedef creaButtonContainer::model::TFunctor TFunctor;
			// ----------------------------------------------------------------------------------
			/*!	@typedef creaButtonContainer::model::ContainerSettings ButtonGroupSettings;
			 *	@brief Defines the ButtonGroupSettings type.
			 */
			typedef creaButtonContainer::model::ContainerSettings ButtonGroupSettings;
			// ----------------------------------------------------------------------------------
			/*!	@typedef wxPanel* PanelButton;
			 *	@brief Defines the PanelButton type.
			 */
			typedef wxPanel* PanelButton;
			// ----------------------------------------------------------------------------------
			/*!	@typedef std::pair< std::string, PanelButton > ActionButton;
			 *	@brief Defines the ActionButton type.
			 *	First is the button name, Second FunctionEventType
			 */
			typedef std::pair<std::string, PanelButton> ActionButton;

			// ----------------------------------------------------------------------------------
			/*!	@typedef std::pair< std::string, wxBitmap > ButtonInfo;
			 *	@brief Defines the ButtonInfo type.
			 *	First is the ButtonName, Second is the ImageIcon
			 */
			typedef std::pair<std::string, wxBitmap> ButtonInfo;

			// ----------------------------------------------------------------------------------
			/*!	@typedef std::pair< ButtonInfo*, ActionButton* > ButtonPair;
			 *	@brief Defines the ButtonPair type.
			 */
			typedef std::pair<ButtonInfo*, ActionButton*> ButtonPair;
			// ----------------------------------------------------------------------------------
			/*!	@typedef std::list< std::string > KeyMapList;
			 *	@brief Defines the KeyMapList type.
			 */
			typedef std::list<std::string> KeyMapList;

			// ----------------------------------------------------------------------------------
			/*!	@typedef std::list< ButtonPair* > ButtonList;
			 *	@brief Defines the ButtonList type.
			 */
			typedef std::list<ButtonPair*> ButtonList;

			// ----------------------------------------------------------------------------------
			/*!	@typedef std::map< std::string, ButtonList > ButtonGroupMap;
			 *	@brief Defines the ButtonGroupMap type.
			 */
			typedef std::map<std::string, ButtonList> ButtonGroupMap;
			// ----------------------------------------------------------------------------------
			/*!	@typedef std::vector< BCPSettingsStruct* > BCStructVectorType;
			 *	@brief Defines the BCStructVectorType type.
			 */
			typedef std::vector<BCPSettingsStruct*> BCStructVectorType;
			//MLER
			// ----------------------------------------------------------------------------------
			/*!	@typedef std::pair<std::string, wxPanel*> ListAction;
			 *	@brief Defines the ListAction type.
			 *	First is the button name(item list), Second FunctionEventType
			 */
			typedef std::pair<std::string, wxPanel*> ListAction;
			// ----------------------------------------------------------------------------------
			/*!	@typedef std::vector<ListAction> ItemsVector;
			 *	@brief Defines the Items Vector type.
			 */
			typedef std::vector<ListAction> ItemsVector;
			//MLER
			//end typedef definition
			// ----------------------------------------------------------------------------------

		public:
			// ----------------------------------------------------------------------------------
			/*! @fn ButtonContainerSettings( );
			 *	@brief This is the default constructor.
			 */
			ButtonContainerSettings();
			// ----------------------------------------------------------------------------------
			/*! @fn virtual ~ButtonContainerSettings( );
			 *	@brief This is the destructor.
			 */
			virtual
			~ButtonContainerSettings();

			// ----------------------------------------------------------------------------------
			/*! @fn ButtonGroupMap GetButtonGroupContainer( );
			 *	@brief This method returns ButtonGroupMap with all buttons.
			 * @return
			 */
			ButtonGroupMap
			GetButtonGroupContainer();
			// ----------------------------------------------------------------------------------
			/*! @fn KeyMapList GetGroupNameList( );
			 *	@brief This method returns a KeyMapsList with the name of the groups.
			 * @return
			 */
			KeyMapList
			GetGroupNameList();
			// ----------------------------------------------------------------------------------
			/*! @fn PanelButton GetPanelButton( const std::string &buttonName );
			 * 	@brief This method returns the panel of a button.
			 * 	@param buttonName
			 * 	@exception std::bad_alloc
			 * 	@return
			 */
			PanelButton
			GetPanelButton(const std::string &buttonName);
			// ----------------------------------------------------------------------------------
			/*! @fn ButtonGroupSettings* GetButtonGroupSettings( TFunctor* functor );
			 *	@brief this method return the settings the generic settings for creaButtonContainer.
			 * @param functor The function to be called in the action button.
			 * @exception std::bad_alloc
			 * @return
			 */
			ButtonGroupSettings*
			GetButtonGroupSettings(TFunctor* functor);
			// ----------------------------------------------------------------------------------
			/*! @fn void SetButtonGroupContainer( ButtonGroupMap bGroupContainer );
			 *	@brief This method sets the button group container.
			 * @param bGroupContainer
			 */
			// ----------------------------------------------------------------------------------
			void
			SetButtonGroupContainer(ButtonGroupMap bGroupContainer);
			// ----------------------------------------------------------------------------------
			/*! @fn void SetGroupNameList( KeyMapList gNameList );
			 *	@brief this method sets the groupName list
			 * @param gNameList
			 */
			void
			SetGroupNameList(KeyMapList gNameList);
			// ----------------------------------------------------------------------------------
			/*! @fn void AddButton( const std::string &groupName, const std::string &buttonName,
			 const std::string &iconpath, const std::string &buttonDescription,
			 PanelButton panel );
			 * @brief This method adds a new button into a group of buttons.
			 * @param groupName The button groupName
			 * @param buttonName The buttonName
			 * @param iconpath The path of the icon
			 * @param buttonDescription The button tooltip
			 * @param panel The panel to be called when the button is clicked
			 * @exception std::bad_alloc
			 */
			void
			AddButton(const std::string &groupName, const std::string &buttonName,
					const wxBitmap &icon, const std::string &buttonDescription,
					PanelButton panel);
			// ----------------------------------------------------------------------------------
			/*! @fn void AddButton( BCPSettingsStruct* info );
			 * @brief This method adds a new button into a group of buttons.
			 * @param info The pointer to BCPSettingsStruct.
			 * @exception std::bad_alloc
			 */
			void
			AddButton(BCPSettingsStruct* info);
			// ----------------------------------------------------------------------------------
			/*! @fn void InitModel( BCStructVectorType infoList, int type );
			 * @brief This method adds new buttons or items into the container (Depends type).
			 * @param infoList The BCStructVectorType
			 * @param type If 1:Buttons 2:List 3:Configurable List
			 * @exception std::bad_alloc
			 */
			void
			InitModel(BCStructVectorType infoList, int type);
			// ----------------------------------------------------------------------------------
			//MLER
			/*! @fn void AddItems( BCPSettingsStruct* info );
			 * @brief This method adds a new item into a list.
			 * @param info The pointer to BCPSettingsStruct.
			 * @exception std::bad_alloc
			 */
			void
			AddItems(BCPSettingsStruct* info);
			// ----------------------------------------------------------------------------------
			/*! @fn PanelButton GetPanelList( const std::string &buttonName );
			 * 	@brief This method returns the panel of a list.
			 * 	@param buttonName (item element)
			 * 	@exception std::bad_alloc
			 * 	@return
			 */
			ItemsVector
			GetItemsVector();
			// ----------------------------------------------------------------------------------
			/*! @fn PanelButton GetPanelList( const std::string &buttonName );
			 * 	@brief This method returns the panel of a list.
			 * 	@param buttonName (item element)
			 * 	@exception std::bad_alloc
			 * 	@return
			 */
			wxPanel*
			GetPanelList(const StringType &buttonName);
			//MLER
			// ----------------------------------------------------------------------------------
		private:
			ButtonGroupMap m_ButtonGroupContainer; //!<This is the map with the groups of buttons.
			KeyMapList m_GroupNameList; //!<This is a container with the name of the groups.

			//MLER
			ItemsVector m_ItemsVector; //!<This is the vector containing a pair of each item name and its wxPanel.

	};
} //ecapseman
#endif /* BUTTONCONTAINERSETTINGS_H_ */
